/*
 * linux/arch/arm/drivers/char/gc_keyb.c
 *
 * Keyboard driver for GraphicsClient ARM Linux.
 *
 * Changelog:
 */

#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/kbd_ll.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/kbd_kern.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/keyboard.h>

#include "gc_kbmap.h"

static void send_SSP_msg(unchar *pBuf, int num)
{
	ushort	tmp;
	int		i;

	for (i=0;i<num;i++) {
		while ((Ser4SSSR & SSSR_TNF) == 0);
		tmp = pBuf[i];
		Ser4SSDR = (tmp << 8);
	}
	
	// Throw away Echo
	for (i=0;i<num;i++) {
		while ((Ser4SSSR & SSSR_RNE) == 0);
		tmp = Ser4SSDR;
	}
}

static unchar ReadSSPByte(void)
{
	if (Ser4SSSR & SSSR_ROR) {
		printk("%s() : Overrun\n", __FUNCTION__);
		return 0;
	}

	Ser4SSDR = 0x00;

	while ((Ser4SSSR & SSSR_RNE) == 0);

	return ((unchar) Ser4SSDR);
}

static ulong read_SSP_response(int num)
{
	int		i;
	ulong	ret;
	
	while (ReadSSPByte() == 0);
	ReadSSPByte();

	ret = 0;
	for (i=0;i<num;i++) {
		ret <<= 8;
		ret |= ReadSSPByte();
	}

	return ret;
}

typedef	struct	t_AVR_CMD {
	unchar	Group, Code; 
}	AVR_CMD;

static void gc_kbd_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	AVR_CMD	RD_INT_CMD = { 0x83, 0x01 };
	AVR_CMD	RD_KBD_CMD = { 0x83, 0x02 };
	unchar	code;

#ifdef CONFIG_VT
	kbd_pt_regs = regs;
#endif
	
	/* Read AVR Interrupt Status to check which Interrupt is occurred 
	 * and Clear AVR Interrupt */
	send_SSP_msg((unchar *) &RD_INT_CMD, 2);
	code = (unchar) (read_SSP_response(1) & 0xFF);

	if (code  & (1 << 2)) {
		/* Read Scan code */
		send_SSP_msg((unchar *) &RD_KBD_CMD, 2);
		code = (unchar) (read_SSP_response(1) & 0xFF);
		handle_scancode( code, (code & 0x80) ? 0 : 1 );
	}
}

void gc_kbd_disable_irq( void )
{
}

void gc_kbd_enable_irq( void )
{
}


int gc_kbd_translate(unsigned char scancode, unsigned char *keycode_p, char raw_mode)
{
	*keycode_p = scancode & 0x7f;
	return 1;
}

void gc_kbd_leds(unsigned char leds)
{
}

static inline void gc_kbd_init(void)
{
	GPDR |= (GPIO_GPIO10 | GPIO_GPIO12 | GPIO_GPIO13); 	// Output
	GPDR &= ~GPIO_GPIO11;

	// Alternative Function
	GAFR |= (GPIO_GPIO10 | GPIO_GPIO11 | GPIO_GPIO12 | GPIO_GPIO13);

	Ser4SSCR0 = 0xA707;
	Ser4SSSR = SSSR_ROR;
	Ser4SSCR1 = 0x0010;
	Ser4SSCR0 = 0xA787;

	// Reset AVR
	ADS_AVR_REG &= 0xFE;
	mdelay(10);			// 10 mSec
	ADS_AVR_REG |= 0x01;
	mdelay(10);			// 10 mSec
}

#define	ADS_AVR_IRQ			63

void __init gc_kbd_init_hw(void)
{
	printk (KERN_INFO "Graphics Client keyboard driver v1.0\n");

	gc_kbd_init();

	if (request_irq (ADS_AVR_IRQ, gc_kbd_interrupt, 0, "keyboard", NULL) != 0)
		printk("Could not allocate keyboard IRQ!\n");
}
