/*
 * Common Flash Interface support:
 *   Intel Extended Vendor Command Set (ID 0x0001)
 *
 * (C) 2000 Red Hat. GPL'd
 *
 * $Id: cfi_cmdset_0001.c,v 1.42 2000/10/30 20:36:11 dwmw2 Exp $
 *
 * 
 * 10/10/2000	Nicolas Pitre <nico@cam.org>
 * 	- completely revamped method functions so they are aware and
 * 	  independent of the flash geometry (buswidth, interleave, etc.)
 * 	- scalability vs code size is completely set at compile-time
 * 	  (see include/linux/mtd/cfi.h for selection)
 *	- optimized write buffer method
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <asm/io.h>
#include <asm/byteorder.h>

#include <linux/errno.h>
#include <linux/malloc.h>
#include <linux/delay.h>
#include <linux/mtd/map.h>
#include <linux/mtd/cfi.h>
#include <linux/mtd/compatmac.h>

#ifndef min
#define min(x,y) ( (x)<(y)?(x):(y) )
#endif

static int cfi_intelext_read (struct mtd_info *, loff_t, size_t, size_t *, u_char *);
static int cfi_intelext_write_words(struct mtd_info *, loff_t, size_t, size_t *, const u_char *);
static int cfi_intelext_write_buffers(struct mtd_info *, loff_t, size_t, size_t *, const u_char *);
static int cfi_intelext_erase (struct mtd_info *, struct erase_info *);
static void cfi_intelext_sync (struct mtd_info *);
static int cfi_intelext_suspend (struct mtd_info *);
static void cfi_intelext_resume (struct mtd_info *);

static void cfi_intelext_destroy(struct mtd_info *);

void cfi_cmdset_0001(struct map_info *, int, unsigned long);
EXPORT_SYMBOL_NOVERS(cfi_cmdset_0001);

struct mtd_info *cfi_intelext_setup (struct map_info *);


void cfi_cmdset_0001(struct map_info *map, int primary, unsigned long base)
{
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct cfi_pri_intelext *extp;

	__u16 adr = primary?cfi->cfiq.P_ADR:cfi->cfiq.A_ADR;

	//printk(" Intel/Sharp Extended Query Table at 0x%4.4X\n", adr);

	if (!adr)
		return;

	/* Switch it into Query Mode */
	switch(CFIDEV_BUSWIDTH) {
	case 1:
		map->write8(map, 0x98, 0x55);
		break;
	case 2:
		map->write16(map, 0x9898, 0xaa);
		break;
	case 4:
		map->write32(map, 0x98989898, 0x154);
		break;
	}

	extp = kmalloc(sizeof(*extp), GFP_KERNEL);
	if (!extp) {
		printk("Failed to allocate memory\n");
		return;
	}

	/* Read in the Extended Query Table */
	for (i=0; i<sizeof(*extp); i++) {
		((unsigned char *)extp)[i] = 
		    map->read8(map, (base+((adr+i)*cfi->interleave*cfi->device_type)));
	}

	if (extp->MajorVersion != '1' || 
	    (extp->MinorVersion < '0' || extp->MinorVersion > '2')) {
		printk("  Unknown IntelExt Extended Query version %c.%c.\n",
		       extp->MajorVersion, extp->MinorVersion);
		kfree(extp);
		return;
	}

	/* Do some byteswapping if necessary */
	extp->FeatureSupport = le32_to_cpu(extp->FeatureSupport);
	extp->BlkStatusRegMask = le32_to_cpu(extp->BlkStatusRegMask);

	
	/* Tell the user about it in lots of lovely detail */
#if 0  
	printk("  Feature/Command Support: %4.4X\n", extp->FeatureSupport);
	printk("     - Chip Erase:         %s\n", extp->FeatureSupport&1?"supported":"unsupported");
	printk("     - Suspend Erase:      %s\n", extp->FeatureSupport&2?"supported":"unsupported");
	printk("     - Suspend Program:    %s\n", extp->FeatureSupport&4?"supported":"unsupported");
	printk("     - Legacy Lock/Unlock: %s\n", extp->FeatureSupport&8?"supported":"unsupported");
	printk("     - Queued Erase:       %s\n", extp->FeatureSupport&16?"supported":"unsupported");
	printk("     - Instant block lock: %s\n", extp->FeatureSupport&32?"supported":"unsupported");
	printk("     - Protection Bits:    %s\n", extp->FeatureSupport&64?"supported":"unsupported");
	printk("     - Page-mode read:     %s\n", extp->FeatureSupport&128?"supported":"unsupported");
	printk("     - Synchronous read:   %s\n", extp->FeatureSupport&256?"supported":"unsupported");
	for (i=9; i<32; i++) {
		if (extp->FeatureSupport & (1<<i)) 
			printk("     - Unknown Bit %X:      supported\n", i);
	}
	
	printk("  Supported functions after Suspend: %2.2X\n", extp->SuspendCmdSupport);
	printk("     - Program after Erase Suspend: %s\n", extp->SuspendCmdSupport&1?"supported":"unsupported");
	for (i=1; i<8; i++) {
		if (extp->SuspendCmdSupport & (1<<i))
			printk("     - Unknown Bit %X:               supported\n", i);
	}
	
	printk("  Block Status Register Mask: %4.4X\n", extp->BlkStatusRegMask);
	printk("     - Lock Bit Active:      %s\n", extp->BlkStatusRegMask&1?"yes":"no");
	printk("     - Valid Bit Active:     %s\n", extp->BlkStatusRegMask&2?"yes":"no");
	for (i=2; i<16; i++) {
		if (extp->BlkStatusRegMask & (1<<i))
			printk("     - Unknown Bit %X Active: yes\n",i);
	}
	
	printk("  Vcc Logic Supply Optimum Program/Erase Voltage: %d.%d V\n", 
	       extp->VccOptimal >> 8, extp->VccOptimal & 0xf);
	if (extp->VppOptimal)
		printk("  Vpp Programming Supply Optimum Program/Erase Voltage: %d.%d V\n", 
		       extp->VppOptimal >> 8, extp->VppOptimal & 0xf);
#endif	
	/* OK. We like it. Take over the control of it. */

	/* Switch it into Read Mode */
	switch(CFIDEV_BUSWIDTH) {
	case 1:
		map->write8(map, 0xff, 0x55);
		break;
	case 2:
		map->write16(map, 0xffff, 0xaa);
		break;
	case 4:
		map->write32(map, 0xffffffff, 0x154);
		break;
	}


	/* If there was an old setup function, decrease its use count */
	if (cfi->cmdset_setup)
		put_module_symbol((unsigned long)cfi->cmdset_setup);
	if (cfi->cmdset_priv)
		kfree(cfi->cmdset_priv);

	for (i=0; i< cfi->numchips; i++) {
		cfi->chips[i].word_write_time = 128;
		cfi->chips[i].buffer_write_time = 128;
		cfi->chips[i].erase_time = 1024;
	}		
		

	cfi->cmdset_setup = cfi_intelext_setup;
	cfi->cmdset_priv = extp;
	MOD_INC_USE_COUNT; /* So the setup function is still there 
			    * by the time it's called */
	
	return;
}

struct mtd_info *cfi_intelext_setup(struct map_info *map)
{
	struct cfi_private *cfi = map->fldrv_priv;
	struct mtd_info *mtd;

	mtd = kmalloc(sizeof(*mtd), GFP_KERNEL);
	//printk("number of CFI chips: %d\n", cfi->numchips);

	if (!mtd) {
	  printk("Failed to allocate memory for MTD device\n");
	  kfree(cfi->cmdset_priv);
	  return NULL;
	}

	memset(mtd, 0, sizeof(*mtd));
	mtd->priv = map;
	mtd->type = MTD_NORFLASH;

	/* We have to assume that all the erase blocks are the same size. */
	mtd->erasesize = le16_to_cpu( (u16)(cfi->cfiq.EraseRegionInfo[0] 
					    >> 16) )*256 * cfi->interleave;
	/* Also select the correct geometry setup too */ 
	mtd->size = (1 << cfi->cfiq.DevSize) * cfi->numchips * cfi->interleave;
	mtd->erase = cfi_intelext_erase;
	mtd->read = cfi_intelext_read;
	if ( cfi->cfiq.BufWriteTimeoutTyp ) {
		//printk( KERN_INFO"Using buffer write method\n" );
		mtd->write = cfi_intelext_write_buffers;
	} else {
		//printk( KERN_INFO"Using word write method\n" );
		mtd->write = cfi_intelext_write_words;
	}
	mtd->sync = cfi_intelext_sync;
	mtd->suspend = cfi_intelext_suspend;
	mtd->resume = cfi_intelext_resume;

	mtd->flags = MTD_CAP_NORFLASH;
	map->fldrv_destroy = cfi_intelext_destroy;
	mtd->name = map->name;
	return mtd;
}


#define CMD(x)  cfi_build_cmd((x), map, cfi->interleave)

static inline int do_read_onechip(struct map_info *map, struct flchip *chip, loff_t adr, size_t len, u_char *buf)
{
	__u32 status, status_OK;
	unsigned long timeo = jiffies + HZ;
	DECLARE_WAITQUEUE(wait, current);
	int suspendedstate = 0;
	loff_t cmd_addr;
	struct cfi_private *cfi = map->fldrv_priv;

	adr += chip->start;

	/* Ensure cmd read/writes are aligned. */ 
	cmd_addr = adr & ~(CFIDEV_BUSWIDTH-1); 

	/* Let's determine this according to the interleave only once */
	status_OK = CMD(0x80);

 retry:
	spin_lock_bh(chip->mutex);

	/* Check that the chip's ready to talk to us.
	 * If it's in FL_ERASING state, suspend it and make it talk now.
	 */
	switch (chip->state) {
	case FL_ERASING:
		cfi_write (map, CMD(0xb0), cmd_addr);
		chip->oldstate = FL_ERASING;
		chip->state = FL_ERASE_SUSPENDING;
		timeo = jiffies + HZ;

		status = cfi_read(map, cmd_addr);
		while (( (status = cfi_read(map, cmd_addr)) & status_OK ) != status_OK) {
			if (time_after(jiffies, timeo)) {
				/* Urgh */
				cfi_write(map, CMD(0xd0), cmd_addr);
				chip->state = FL_ERASING;
				spin_unlock_bh(chip->mutex);
				printk("Chip not ready after erase suspended\n");
				return -EIO;
			}
			spin_unlock_bh(chip->mutex);
			udelay(1);
			spin_lock_bh(chip->mutex);
		}
		chip->state = FL_ERASE_SUSPENDED;
		/* Remember the status so we know whether to restart
		   the erase later */
		suspendedstate = status;
		cfi_write(map, CMD(0xff), cmd_addr);
		chip->state = FL_READY;
		break;
#if 0
	case FL_WRITING:
		/* Not quite yet */
#endif

	case FL_READY:
		break;
	case FL_CFI_QUERY:
	case FL_JEDEC_QUERY:
		cfi_write(map, CMD(0x70), cmd_addr);
		chip->state = FL_STATUS;

	case FL_STATUS:
		status = cfi_read(map, cmd_addr);

		if ((status & status_OK) != status_OK) {
			static int z=0;
			/* Urgh. Chip not yet ready to talk to us. */
			if (time_after(jiffies, timeo)) {
				spin_unlock_bh(chip->mutex);
				printk("waiting for chip to be ready timed out in read. WSM status = %x", status);
				return -EIO;
			}

			/* Latency issues. Drop the lock, wait a while and retry */
			spin_unlock_bh(chip->mutex);

			z++;
			if ( 0 && !(z % 100 )) 
				printk("chip not ready yet before read. looping\n");

			udelay(1);

			goto retry;
		}
		cfi_write(map, CMD(0xff), cmd_addr);
		chip->state = FL_READY;
		break;

	default:
	  //		printk("Waiting for chip, status = %d\n", chip->state);

		/* Stick ourselves on a wait queue to be woken when
		   someone changes the status */

		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
		
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);
#if 0
		if(signal_pending(current))
			return -EINTR;
#endif
		timeo = jiffies + HZ;

		goto retry;
	}

	map->copy_from(map, buf, adr, len);

	if (suspendedstate) {
		chip->state = chip->oldstate;
		/* Resume if it hadn't already completed. */
		if (suspendedstate & CMD(0x40)) 
			cfi_write(map, CMD(0xd0), cmd_addr);
		else
			cfi_write(map, CMD(0x70), cmd_addr);
	}

	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

	return 0;
}

static int cfi_intelext_read (struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long ofs;
	int chipnum;
	int ret = 0;

	/* ofs: offset within the first chip that the first read should start */
	chipnum = (from >> cfi->chipshift);
	ofs = from - (chipnum <<  cfi->chipshift);

	*retlen = 0;

	while (len) {
		unsigned long thislen;

		if (chipnum >= cfi->numchips)
			break;

		if ((len + ofs -1) >> cfi->chipshift)
			thislen = (1<<cfi->chipshift) - ofs;
		else
			thislen = len;

		ret = do_read_onechip(map, &cfi->chips[chipnum], ofs, thislen, buf);
		if (ret)
			break;

		*retlen += thislen;
		len -= thislen;
		buf += thislen;
		
		ofs = 0;
		chipnum++;
	}
	return ret;
}

static int do_write_oneword(struct map_info *map, struct flchip *chip, unsigned long adr, __u32 datum)
{
	struct cfi_private *cfi = map->fldrv_priv;
	__u32 status, status_OK;
	unsigned long timeo = jiffies + HZ;
	DECLARE_WAITQUEUE(wait, current);
	int z = 0;

	adr += chip->start;

	/* Let's determine this according to the interleave only once */
	status_OK = CMD(0x80);

 retry:
	spin_lock_bh(chip->mutex);

	/* Check that the chip's ready to talk to us.
	 * Later, we can actually think about interrupting it
	 * if it's in FL_ERASING state.
	 * Not just yet, though.
	 */
	switch (chip->state) {
	case FL_CFI_QUERY:
	case FL_JEDEC_QUERY:
	case FL_READY:
		cfi_write(map, CMD(0x70), adr);
		chip->state = FL_STATUS;
		timeo = jiffies + HZ;

	case FL_STATUS:
		status = cfi_read(map, adr);

		if ((status & status_OK) != status_OK) {

			/* Urgh. Chip not yet ready to talk to us. */
			if (time_after(jiffies, timeo)) {
				spin_unlock_bh(chip->mutex);
				printk("waiting for chip to be ready timed out in read");
				return -EIO;
			}

			/* Latency issues. Drop the lock, wait a while and retry */
			spin_unlock_bh(chip->mutex);

			z++;
			if ( 0 && !(z % 100 ))
				printk("chip not ready yet before write. looping\n");
			
			udelay(1);

			goto retry;
		}
		break;

	default:
	  //		printk("Waiting for chip, status = %d\n", chip->state);

		/* Stick ourselves on a wait queue to be woken when
		   someone changes the status */

		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
		
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);
#if 0
		if(signal_pending(current))
			return -EINTR;
#endif
		timeo = jiffies + HZ;

		goto retry;
	}
	ENABLE_VPP(map);

	cfi_write(map, CMD(0x40), adr);
	cfi_write(map, datum, adr);
	chip->state = FL_WRITING;

	spin_unlock_bh(chip->mutex);

	timeo = jiffies + (HZ/2);
	udelay(chip->word_write_time);

	spin_lock_bh(chip->mutex);

	z = 0;
	while ( chip->state != FL_WRITING ||
		( (status = cfi_read(map, adr)) & status_OK ) != status_OK ) {

		if (chip->state != FL_WRITING) {
			/* Someone's suspended the write. Sleep */
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			
			schedule();
			remove_wait_queue(&chip->wq, &wait);
#if 0			
			if (signal_pending(current)) {
				DISABLE_VPP(map);
				return -EINTR;
#endif			
			timeo = jiffies + (HZ / 2); /* FIXME */

			spin_lock_bh(chip->mutex);
			continue;
		}

		/* OK Still waiting */
		if (time_after(jiffies, timeo)) {
			chip->state = FL_STATUS;
			spin_unlock_bh(chip->mutex);
			printk("waiting for chip to be ready timed out in word write");
			DISABLE_VPP(map);
			return -EIO;
		}
		
		/* Latency issues. Drop the lock, wait a while and retry */
		spin_unlock_bh(chip->mutex);
		
		z++;
		if ( 0 && !(z % 100 )) 
		  printk("chip not ready yet after word write. looping\n");
		
		udelay(1);
		
		spin_lock_bh(chip->mutex);
		continue;
	}
	if (!z) {
		chip->word_write_time--;
		if (!chip->word_write_time)
			chip->word_write_time++;
	}
	if (z > 1) 
		chip->word_write_time++;

	DISABLE_VPP(map);
	/* Done and happy. */
	chip->state = FL_STATUS;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);
	//	printk("write ret OK at %lx\n", adr);
	return 0;
}


static int cfi_intelext_write_words (struct mtd_info *mtd, loff_t to , size_t len, size_t *retlen, const u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int ret = 0;
	int chipnum;
	unsigned long ofs;

	*retlen = 0;
	if (!len)
		return 0;

	chipnum = to >> cfi->chipshift;
	ofs = to  - (chipnum << cfi->chipshift);

	/* If it's not bus-aligned, do the first byte write */
	if (ofs & (CFIDEV_BUSWIDTH-1)) {
		unsigned long bus_ofs = ofs & ~(CFIDEV_BUSWIDTH-1);
		int gap = ofs - bus_ofs;
		int i = 0, n = 0;
		u_char tmp_buf[4];
		__u32 datum;

		while (gap--)
			tmp_buf[i++] = 0xff;
		while (len && i < CFIDEV_BUSWIDTH)
			tmp_buf[i++] = buf[n++], len--;
		while (i < CFIDEV_BUSWIDTH)
			tmp_buf[i++] = 0xff;

		if (cfi_buswidth_is_2()) {
			datum = *(__u16*)tmp_buf;
		} else if (cfi_buswidth_is_4()) {
			datum = *(__u32*)tmp_buf;
		} else {
			return -EINVAL;  /* should never happen, but be safe */
		}

		ret = do_write_oneword(map, &cfi->chips[chipnum],
					       bus_ofs, datum);
		if (ret) 
			return ret;
		
		ofs += n;
		buf += n;
		(*retlen) += n;

		if (ofs >> cfi->chipshift) {
			chipnum ++; 
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
		}
	}
	
	while(len >= CFIDEV_BUSWIDTH) {
		__u32 datum;

		if (cfi_buswidth_is_1()) {
			datum = *(__u8*)buf;
		} else if (cfi_buswidth_is_2()) {
			datum = *(__u16*)buf;
		} else if (cfi_buswidth_is_4()) {
			datum = *(__u32*)buf;
		} else {
			return -EINVAL;
		}

		ret = do_write_oneword(map, &cfi->chips[chipnum],
				ofs, datum);
		if (ret)
			return ret;

		ofs += CFIDEV_BUSWIDTH;
		buf += CFIDEV_BUSWIDTH;
		(*retlen) += CFIDEV_BUSWIDTH;
		len -= CFIDEV_BUSWIDTH;

		if (ofs >> cfi->chipshift) {
			chipnum ++; 
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
		}
	}

	if (len & (CFIDEV_BUSWIDTH-1)) {
		int i = 0, n = 0;
		u_char tmp_buf[4];
		__u32 datum;

		while (len--)
			tmp_buf[i++] = buf[n++];
		while (i < CFIDEV_BUSWIDTH)
			tmp_buf[i++] = 0xff;

		if (cfi_buswidth_is_2()) {
			datum = *(__u16*)tmp_buf;
		} else if (cfi_buswidth_is_4()) {
			datum = *(__u32*)tmp_buf;
		} else {
			return -EINVAL;  /* should never happen, but be safe */
		}

		ret = do_write_oneword(map, &cfi->chips[chipnum],
					       ofs, datum);
		if (ret) 
			return ret;
		
		(*retlen) += n;
	}

	return 0;
}


static inline int do_write_buffer(struct map_info *map, struct flchip *chip, 
				  unsigned long adr, const u_char *buf, int len)
{
	struct cfi_private *cfi = map->fldrv_priv;
	int interleave = cfi->interleave;
	__u32 status, status_OK;
	unsigned long cmd_adr, timeo = jiffies + HZ;
	DECLARE_WAITQUEUE(wait, current);
	int wbufsize, z = 0;

	/* FIXME: assume write buffer of 16 */
	wbufsize = 16 * cfi->device_type * CFIDEV_INTERLEAVE;

	adr += chip->start;
	cmd_adr = adr & ~(wbufsize-1);
	
	/* Let's determine this according to the interleave only once */
	status_OK = CMD(0x80);

 retry:
	spin_lock_bh(chip->mutex);

	/* Check that the chip's ready to talk to us.
	 * Later, we can actually think about interrupting it
	 * if it's in FL_ERASING state.
	 * Not just yet, though.
	 */
	switch (chip->state) {
	case FL_CFI_QUERY:
	case FL_JEDEC_QUERY:
	case FL_READY:
		cfi_write(map, CMD(0x70), cmd_adr);
		chip->state = FL_STATUS;
		timeo = jiffies + HZ;

	case FL_STATUS:
		status = cfi_read(map, cmd_adr);

		if ((status & status_OK) != status_OK) {

			/* Urgh. Chip not yet ready to talk to us. */
			if (time_after(jiffies, timeo)) {
				spin_unlock_bh(chip->mutex);
				printk("waiting for chip to be ready timed out in buffer write");
				return -EIO;
			}

			/* Latency issues. Drop the lock, wait a while and retry */
			spin_unlock_bh(chip->mutex);

			z++;
			if ( 0 && !(z % 100 ))
				printk("chip not ready yet before buffer write. looping\n");
			
			udelay(1);

			goto retry;
		}
		break;

	default:
	  //		printk("Waiting for chip, status = %d\n", chip->state);

		/* Stick ourselves on a wait queue to be woken when
		   someone changes the status */

		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
		
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);
#if 0
		if(signal_pending(current))
			return -EINTR;
#endif
		timeo = jiffies + HZ;

		goto retry;
	}
	ENABLE_VPP(map);
	cfi_write(map, CMD(0xe8), cmd_adr);
	chip->state = FL_WRITING_TO_BUFFER;

	z=0;
	while ( ( (status = cfi_read(map, cmd_adr)) & status_OK ) != status_OK ) {
		spin_unlock_bh(chip->mutex);
		udelay(1);
		spin_lock_bh(chip->mutex);
		z++;
		if (z > 20) {
			/* Argh. Not ready for write to buffer */
			cfi_write(map, CMD(0x70), cmd_adr);
			chip->state = FL_STATUS;
			printk("Chip not ready for buffer write. Xstatus = %x, status = %x\n", status, cfi_read(map, cmd_adr));
			spin_unlock_bh(chip->mutex);
			DISABLE_VPP(map);
			return -EIO;
		}
	}

	/* Write length of data to come */
	cfi_write(map, CMD((len/cfi->device_type/CFIDEV_INTERLEAVE)-1), cmd_adr);
	/* Write data */
	for (z = 0; z < len; z += CFIDEV_BUSWIDTH) {
		if (cfi_buswidth_is_1()) {
			map->write8 (map, *((__u8*)buf)++, adr+z);
		} else if (cfi_buswidth_is_2()) {
			map->write16 (map, *((__u16*)buf)++, adr+z);
		} else if (cfi_buswidth_is_4()) {
			map->write32 (map, *((__u32*)buf)++, adr+z);
		} else {
			DISABLE_VPP(map);
			return -EINVAL;
		}
	}
	/* GO GO GO */
	cfi_write(map, CMD(0xd0), cmd_adr);
	chip->state = FL_WRITING;

	spin_unlock_bh(chip->mutex);
	timeo = jiffies + (HZ/2);
	z = 0;
       	udelay(chip->buffer_write_time);

	spin_lock_bh(chip->mutex);

	while ( chip->state != FL_WRITING ||
		( (status = cfi_read(map, cmd_adr)) & status_OK ) != status_OK ) {

		if (chip->state != FL_WRITING) {
			/* Someone's suspended the write. Sleep */
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			
			schedule();
			remove_wait_queue(&chip->wq, &wait);
#if 0			
			if (signal_pending(current))
				return -EINTR;
#endif			
			timeo = jiffies + (HZ / 2); /* FIXME */

			spin_lock_bh(chip->mutex);
			continue;
		}

		/* OK Still waiting */
		if (time_after(jiffies, timeo)) {
			chip->state = FL_STATUS;
			spin_unlock_bh(chip->mutex);
			printk("waiting for chip to be ready timed out in bufwrite");
			DISABLE_VPP(map);
			return -EIO;
		}
		
		/* Latency issues. Drop the lock, wait a while and retry */
		spin_unlock_bh(chip->mutex);
		
		z++;
		if ( !(z % 100 )) 
		  printk("chip not ready yet after bufwrite. looping\n");
		
		udelay(1);
		
		spin_lock_bh(chip->mutex);
		continue;
	}
	if (!z) {
		chip->buffer_write_time--;
		if (!chip->buffer_write_time)
			chip->buffer_write_time++;
	}
	if (z > 1) 
		chip->buffer_write_time++;

	/* Done and happy. */
	chip->state = FL_STATUS;
	DISABLE_VPP(map);
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);
	//	printk("write ret OK at %lx\n", adr);
	return 0;
}

static int cfi_intelext_write_buffers (struct mtd_info *mtd, loff_t to, 
				       size_t len, size_t *retlen, const u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int interleave = cfi->interleave;
	/* FIXME: assume write buffer size of 16 */
	int wbufsize = 16 * cfi->device_type * CFIDEV_INTERLEAVE;
	int ret = 0;
	int chipnum;
	unsigned long ofs;

	*retlen = 0;
	if (!len)
		return 0;

	chipnum = to >> cfi->chipshift;
	ofs = to  - (chipnum << cfi->chipshift);

	/* If it's not bus-aligned, do the first word write */
	if (ofs & (CFIDEV_BUSWIDTH-1)) {
		size_t local_len = (-ofs)&(CFIDEV_BUSWIDTH-1);
		if (local_len > len)
			local_len = len;
		ret = cfi_intelext_write_words(mtd, to, local_len,
					       retlen, buf);
		if (ret)
			return ret;
		ofs += local_len;
		buf += local_len;
		len -= local_len;

		if (ofs >> cfi->chipshift) {
			chipnum ++;
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
		}
	}

	/* Write buffer is worth it only if more than one word to write... */
	while(len > CFIDEV_BUSWIDTH) {
		/* We must not cross write block boundaries */
		int size = wbufsize - (ofs & (wbufsize-1));

		if (size > len)
			size = len & ~(CFIDEV_BUSWIDTH-1);
		ret = do_write_buffer(map, &cfi->chips[chipnum], 
				      ofs, buf, size);
		if (ret)
			return ret;

		ofs += size;
		buf += size;
		(*retlen) += size;
		len -= size;

		if (ofs >> cfi->chipshift) {
			chipnum ++; 
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
		}
	}

	/* ... and write the remaining bytes */
	if (len > 0) {
		size_t local_retlen;
		ret = cfi_intelext_write_words(mtd, ofs + (chipnum << cfi->chipshift),
					       len, &local_retlen, buf);
		if (ret)
			return ret;
		(*retlen) += local_retlen;
	}

	return 0;
}


static inline int do_erase_oneblock(struct map_info *map, struct flchip *chip, unsigned long adr)
{
	struct cfi_private *cfi = map->fldrv_priv;
	__u32 status, status_OK;
	unsigned long timeo = jiffies + HZ;
	DECLARE_WAITQUEUE(wait, current);

	adr += chip->start;

	/* Let's determine this according to the interleave only once */
	status_OK = CMD(0x80);

retry:
	spin_lock_bh(chip->mutex);

	/* Check that the chip's ready to talk to us. */
	switch (chip->state) {
	case FL_CFI_QUERY:
	case FL_JEDEC_QUERY:
	case FL_READY:
		cfi_write(map, CMD(0x70), adr);
		chip->state = FL_STATUS;
		timeo = jiffies + HZ;

	case FL_STATUS:
		status = cfi_read(map, adr);

		if ((status & status_OK) != status_OK) {
			static int z=0;
			/* Urgh. Chip not yet ready to talk to us. */
			if (time_after(jiffies, timeo)) {
				spin_unlock_bh(chip->mutex);
				printk("waiting for chip to be ready timed out in erase");
				return -EIO;
			}

			/* Latency issues. Drop the lock, wait a while and retry */
			spin_unlock_bh(chip->mutex);

			z++;
			if ( 0 && !(z % 100 )) 
				printk("chip not ready yet before erase. looping\n");

			udelay(1);

			goto retry;
		}
		break;

	default:
	  //		printk("Waiting for chip, status = %d\n", chip->state);

		/* Stick ourselves on a wait queue to be woken when
		   someone changes the status */

		set_current_state(TASK_UNINTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
		
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);
#if 0
		if(signal_pending(current))
			return -EINTR;
#endif
		timeo = jiffies + HZ;

		goto retry;
	}
	ENABLE_VPP(map);
	cfi_write(map, CMD(0x20), adr);
	cfi_write(map, CMD(0xD0), adr);

	chip->state = FL_ERASING;
	
	timeo = jiffies + (HZ*2);
	spin_unlock_bh(chip->mutex);
	schedule_timeout(HZ);
	spin_lock_bh(chip->mutex);

	/* FIXME. Use a timer to check this, and return immediately. */
	/* Once the state machine's known to be working I'll do that */

	while ( chip->state != FL_ERASING ||
		( (status = cfi_read(map, adr)) & status_OK ) != status_OK ) {
		static int z=0;

		if (chip->state != FL_ERASING) {
			/* Someone's suspended the erase. Sleep */
			set_current_state(TASK_UNINTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			
			schedule();
			remove_wait_queue(&chip->wq, &wait);
#if 0			
			if (signal_pending(current))
				return -EINTR;
#endif			
			timeo = jiffies + (HZ*2); /* FIXME */
			spin_lock_bh(chip->mutex);
			continue;
		}

		/* OK Still waiting */
		if (time_after(jiffies, timeo)) {
			chip->state = FL_STATUS;
			spin_unlock_bh(chip->mutex);
			printk("waiting for erase to complete timed out.");
			DISABLE_VPP(map);
			return -EIO;
		}
		
		/* Latency issues. Drop the lock, wait a while and retry */
		spin_unlock_bh(chip->mutex);

		z++;
		if ( 0 && !(z % 100 )) 
			printk("chip not ready yet after erase. looping\n");

		udelay(1);
		
		spin_lock_bh(chip->mutex);
		continue;
	}
	
	/* Done and happy. */
	chip->state = FL_STATUS;
	DISABLE_VPP(map);
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);
	//printk("erase ret OK\n");
	return 0;
}

static int cfi_intelext_erase (struct mtd_info *mtd, struct erase_info *instr)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long adr, len;
	int chipnum, ret = 0;

	if (instr->addr & (mtd->erasesize - 1))
		return -EINVAL;

	if (instr->len & (mtd->erasesize -1))
		return -EINVAL;

	if ((instr->len + instr->addr) > mtd->size)
		return -EINVAL;

	chipnum = instr->addr >> cfi->chipshift;
	adr = instr->addr - (chipnum << cfi->chipshift);
	len = instr->len;

	while(len) {
		ret = do_erase_oneblock(map, &cfi->chips[chipnum], adr);
		
		if (ret)
			return ret;

		adr += mtd->erasesize;
		len -= mtd->erasesize;

		if (adr >> cfi->chipshift) {
			adr = 0;
			chipnum++;
			
			if (chipnum >= cfi->numchips)
			break;
		}
	}
		
	if (instr->callback)
		instr->callback(instr);
	
	return 0;
}



static void cfi_intelext_sync (struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;
	DECLARE_WAITQUEUE(wait, current);

	for (i=0; !ret && i<cfi->numchips; i++) {
		chip = &cfi->chips[i];

	retry:
		spin_lock_bh(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_SYNCING;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_SYNCING:
			spin_unlock_bh(chip->mutex);
			break;

		default:
			/* Not an idle state */
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			schedule();
		        remove_wait_queue(&chip->wq, &wait);
			
			goto retry;
		}
	}

	/* Unlock the chips again */

	for (i--; i >=0; i--) {
		chip = &cfi->chips[i];

		spin_lock_bh(chip->mutex);
		
		if (chip->state == FL_SYNCING) {
			chip->state = chip->oldstate;
			wake_up(&chip->wq);
		}
		spin_unlock_bh(chip->mutex);
	}
}


static int cfi_intelext_suspend(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;

	for (i=0; !ret && i<cfi->numchips; i++) {
		chip = &cfi->chips[i];

		spin_lock_bh(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_PM_SUSPENDED;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_PM_SUSPENDED:
			spin_unlock_bh(chip->mutex);
			break;

		default:
			ret = -EAGAIN;
			break;
		}
	}

	/* Unlock the chips again */

	if (ret) {
		for (i--; i >=0; i--) {
			chip = &cfi->chips[i];
			
			spin_lock_bh(chip->mutex);
			
			if (chip->state == FL_PM_SUSPENDED) {
				chip->state = chip->oldstate;
				wake_up(&chip->wq);
			}
			spin_unlock_bh(chip->mutex);
		}
	} 
	
	return ret;
}

static void cfi_intelext_resume(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;

	for (i=0; i<cfi->numchips; i++) {
	
		chip = &cfi->chips[i];

		spin_lock_bh(chip->mutex);
		
		if (chip->state == FL_PM_SUSPENDED) {
			chip->state = chip->oldstate;
			wake_up(&chip->wq);
		}

		spin_unlock_bh(chip->mutex);
	}
}

static void cfi_intelext_destroy(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	kfree(cfi->cmdset_priv);
	kfree(cfi);
}

