/*
 * Adapted from nora.c by Jamey Hicks 9/14/2000
 * $Id: mtdbitsy.c,v 1.5 2000/10/15 21:58:23 dwmw2 Exp $
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/arch/bitsy.h>

#define WINDOW_ADDR 0xe8000000
#define WINDOW_SIZE 0x02000000

static struct mtd_info *mymtd;

__u8 bitsy_read8(struct map_info *map, unsigned long ofs)
{
  return *(__u8 *)(WINDOW_ADDR + ofs);
}

__u16 bitsy_read16(struct map_info *map, unsigned long ofs)
{
  return *(__u16 *)(WINDOW_ADDR + ofs);
}

__u32 bitsy_read32(struct map_info *map, unsigned long ofs)
{
  return *(__u32 *)(WINDOW_ADDR + ofs);
}

void bitsy_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
  memcpy(to, (void *)(WINDOW_ADDR + from), len);
}

void bitsy_write8(struct map_info *map, __u8 d, unsigned long adr)
{
  *(__u8 *)(WINDOW_ADDR + adr) = d;
}

void bitsy_write16(struct map_info *map, __u16 d, unsigned long adr)
{
  *(__u16 *)(WINDOW_ADDR + adr) = d;
}

void bitsy_write32(struct map_info *map, __u32 d, unsigned long adr)
{
  *(__u32 *)(WINDOW_ADDR + adr) = d;
}

void bitsy_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
  memcpy((void *)(WINDOW_ADDR + to), from, len);
}

void bitsy_set_vpp(int vpp)
{
  if (vpp)
    set_bitsy_egpio(EGPIO_BITSY_VPP_ON);
  else
    clr_bitsy_egpio(EGPIO_BITSY_VPP_ON);
}

struct map_info bitsy_map = {
  name: "BITSY",
  size: WINDOW_SIZE,
  buswidth: 4,
  read8: bitsy_read8,
  read16: bitsy_read16,
  read32: bitsy_read32,
  copy_from: bitsy_copy_from,
  write8: bitsy_write8,
  write16: bitsy_write16,
  write32: bitsy_write32,
  copy_to: bitsy_copy_to,
  set_vpp: bitsy_set_vpp
};


static struct mtd_partition bitsy_partitions[7] = {
	{
		name: "BITSY boot firmware",
		size: 0x00040000,
		offset: 0,
		mask_flags: MTD_WRITEABLE  /* force read-only */
	},
	{
		name: "BITSY kernel",
		size: 0x00080000,
		offset: 0x40000
	},
	{
		name: "BITSY params",
		size: 0x00040000,
		offset: 0xC0000
	},
	{
		name: "BITSY initrd",
		size: 0x00100000,
		offset: 0x00100000
	},
	{
		name: "BITSY root cramfs",
		size: 0x00300000,
		offset: 0x00200000
	},
	{
		name: "BITSY usr cramfs",
		size: 0x00800000,
		offset: 0x00500000
	},
	{
		name: "BITSY usr local",
		offset: 0x00d00000,
		size: 0  /* will expand to the end of the flash */
	}
};


int __init init_bitsy(void)
{
       	printk(KERN_NOTICE "bitsy flash device: %x at %x\n", WINDOW_SIZE, WINDOW_ADDR);

	mymtd = do_cfi_probe(&bitsy_map);
	if (mymtd) {
		mymtd->module = THIS_MODULE;
		add_mtd_partitions(mymtd, bitsy_partitions, 7);
		return 0;
	}

	return -ENXIO;
}

static void __exit cleanup_bitsy(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
	}
}

module_init(init_bitsy);
module_exit(cleanup_bitsy);
