/*
 * drivers/pcmcia/sa1100_bitsy.c
 *
 * PCMCIA implementation routines for Bitsy
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/i2c.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/arch/pcmcia.h>
#include <asm/arch/h3600-sleeve.h>

#ifdef CONFIG_MERCURY_BACKPAQ
#include <asm/arch/mercury-backpaq.h>

struct mercury_backpaq_sysctl *mercury_backpaq_sysctl = ((struct mercury_backpaq_sysctl *)SYSCTL_BASE);

static void bit_backpaq_setscl(void *data, int state)
{
        if (state) 
                mercury_backpaq_sysctl->ctl |= MERCURY_BACKPAQ_SMB_CLOCK;
        else
                mercury_backpaq_sysctl->ctl &= ~MERCURY_BACKPAQ_SMB_CLOCK;
}

static void bit_backpaq_setsda(void *data, int state)
{
        mercury_backpaq_sysctl->ctl |= MERCURY_BACKPAQ_SMB_DATA_OE;
        if (state) 
                mercury_backpaq_sysctl->ctl |= MERCURY_BACKPAQ_SMB_DATA;
        else
                mercury_backpaq_sysctl->ctl &= ~MERCURY_BACKPAQ_SMB_DATA;
} 

static int bit_backpaq_getscl(void *data)
{
	return (mercury_backpaq_sysctl->ctl & MERCURY_BACKPAQ_SMB_CLOCK) ? 1 : 0;
}

static int bit_backpaq_getsda(void *data)
{
        mercury_backpaq_sysctl->ctl &= ~MERCURY_BACKPAQ_SMB_DATA_OE;
	return (mercury_backpaq_sysctl->ctl & MERCURY_BACKPAQ_SMB_DATA) ? 1 : 0;
}

/* ---------- */

static int bit_backpaq_reg(struct i2c_client *client)
{
	return 0;
}

static int bit_backpaq_unreg(struct i2c_client *client)
{
	return 0;
}

static void bit_backpaq_inc_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}

static void bit_backpaq_dec_use(struct i2c_adapter *adap)
{
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}

static struct i2c_algo_bit_data bit_backpaq_data = {
        setsda: bit_backpaq_setsda,
	setscl: bit_backpaq_setscl,
	getsda: bit_backpaq_getsda,
	getscl: bit_backpaq_getscl,
	udelay: 2, mdelay: 2, timeout: 100,
};

static struct i2c_adapter bit_backpaq_ops = {
	name: "backPAQ",
	id: I2C_HW_B_IOC,
	algo_data: &bit_backpaq_data,
	inc_use: bit_backpaq_inc_use,
	dec_use: bit_backpaq_dec_use,
	client_register: bit_backpaq_reg,
	client_unregister: bit_backpaq_unreg,
};
#endif /* CONFIG_MERCURY_BACKPAQ */

#ifdef CONFIG_H3600_SLEEVE
static enum sleeve_device sleeve_type = SLEEVE_NO_DEVICE;
#else
/* default to behavior of PCMCIA sleeve (also works for CF) */
static enum sleeve_device sleeve_type = SINGLE_PCMCIA_SLEEVE;
#endif

static struct pcmcia_init sa1100_bitsy_pcmcia_init;

static int bitsy_pcmcia_init(struct pcmcia_init *init)
{
        int irq, res;

        printk(__FUNCTION__ " sleeve_type=%d\n", sleeve_type);

        sa1100_bitsy_pcmcia_init = *init;
        switch (sleeve_type) {
        case SLEEVE_NO_DEVICE: 
                break;

        case MERCURY_BACKPAQ:
#ifdef CONFIG_MERCURY_BACKPAQ
                i2c_bit_add_bus(&bit_backpaq_ops);
                /* fall through */
#endif
        case SINGLE_COMPACTFLASH_SLEEVE:
        case SINGLE_PCMCIA_SLEEVE:

                /* Enable CF bus: */
                set_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
                clr_bitsy_egpio(EGPIO_BITSY_OPT_RESET);

                /* All those are inputs */
                GPDR &= ~(GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1 | GPIO_BITSY_PCMCIA_IRQ0| GPIO_BITSY_PCMCIA_IRQ1);

                /* Set transition detect */
                set_GPIO_IRQ_edge( GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1, GPIO_BOTH_EDGES );
                set_GPIO_IRQ_edge( GPIO_BITSY_PCMCIA_IRQ0| GPIO_BITSY_PCMCIA_IRQ1, GPIO_FALLING_EDGE );

                /* Register interrupts */
                irq = IRQ_GPIO_BITSY_PCMCIA_CD0;
                res = request_irq( irq, init->handler, SA_INTERRUPT, "PCMCIA_CD0", NULL );
                if( res < 0 ) goto irq_err;
                irq = IRQ_GPIO_BITSY_PCMCIA_CD1;
                res = request_irq( irq, init->handler, SA_INTERRUPT, "PCMCIA_CD1", NULL );
                if( res < 0 ) goto irq_err;
        }

        return 2;

 irq_err:
        printk( KERN_ERR __FUNCTION__ ": Request for IRQ %u failed\n", irq );
        return -1;
}

static int bitsy_pcmcia_shutdown(void)
{
        printk(__FUNCTION__ "\n");
        switch (sleeve_type) {
        case MERCURY_BACKPAQ:
                /* fall through */
        case SINGLE_COMPACTFLASH_SLEEVE:
        case SINGLE_PCMCIA_SLEEVE:
                /* disable IRQs */
                free_irq( IRQ_GPIO_BITSY_PCMCIA_CD0, NULL );
                free_irq( IRQ_GPIO_BITSY_PCMCIA_CD1, NULL );
  
                /* Disable CF bus: */
                clr_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
                set_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
                break;
        case SLEEVE_NO_DEVICE:
        }

        return 0;
}

static int bitsy_pcmcia_socket_state(struct pcmcia_state_array *state_array)
{
        unsigned long levels;

        if(state_array->size<2) return -1;

        memset(state_array->state, 0, 
               (state_array->size)*sizeof(struct pcmcia_state));

        if (sleeve_type != SLEEVE_NO_DEVICE) {

                levels=GPLR;

                state_array->state[0].detect=((levels & GPIO_BITSY_PCMCIA_CD0)==0)?1:0;
                state_array->state[0].ready=(levels & GPIO_BITSY_PCMCIA_IRQ0)?1:0;
                if (sleeve_type == MERCURY_BACKPAQ) {
                        
                } else {
                        state_array->state[0].bvd1= 0;
                        state_array->state[0].bvd2= 0;
                        state_array->state[0].wrprot=0; /* Not available on Bitsy. */
                        state_array->state[0].vs_3v=1;
                        state_array->state[0].vs_Xv=0;
                }

                state_array->state[1].detect=((levels & GPIO_BITSY_PCMCIA_CD1)==0)?1:0;
                state_array->state[1].ready=(levels & GPIO_BITSY_PCMCIA_IRQ1)?1:0;
                if (sleeve_type == MERCURY_BACKPAQ) {
                } else {
                        state_array->state[1].bvd1=0;
                        state_array->state[1].bvd2=0;
                        state_array->state[1].wrprot=0; /* Not available on Bitsy. */
                        state_array->state[1].vs_3v=1;
                        state_array->state[1].vs_Xv=1;
                }

        }
        return 1;
}

static int bitsy_pcmcia_get_irq_info(struct pcmcia_irq_info *info)
{

        switch (info->sock) {
        case 0:
                info->irq=IRQ_GPIO_BITSY_PCMCIA_IRQ0;
                break;
        case 1:
                info->irq=IRQ_GPIO_BITSY_PCMCIA_IRQ1;
                break;
        default:
                return -1;
        }
        return 0;
}

static int bitsy_pcmcia_configure_socket(const struct pcmcia_configure *configure)
{

        int sock = configure->sock;
        if(sock>1) return -1;

        if (sleeve_type != SLEEVE_NO_DEVICE)
                printk(__FUNCTION__ ": socket=%d vcc=%d reset=%d\n", sock, configure->vcc, configure->reset);

        switch (sleeve_type) {
        case SINGLE_COMPACTFLASH_SLEEVE:
        case SINGLE_PCMCIA_SLEEVE: {
                unsigned long flags;

                save_flags_cli(flags);

                switch (configure->vcc) {
                case 0:
                        clr_bitsy_egpio(EGPIO_BITSY_OPT_ON);
                        break;

                case 50:
                case 33:
                        set_bitsy_egpio(EGPIO_BITSY_OPT_ON);
                        break;

                default:
                        printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
                               configure->vcc);
                        restore_flags(flags);
                        return -1;
                }

                if (configure->reset)
                        set_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
                else
                        clr_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
                restore_flags(flags);
        } break;

        case MERCURY_BACKPAQ: {
#ifdef CONFIG_MERCURY_BACKPAQ
                struct i2c_adapter *adap = &bit_backpaq_ops;
                struct i2c_msg msg[1];
                char data = MAX1604_PWR_ACTIVE | MAX1604_PWR_MASKFLT;

                msg[1].addr = 0x50 | sock;
                msg[1].flags = 0;
                msg[1].len = 8;
                msg[1].buf = &data;

                switch (configure->vcc) {
                case 0:
                        break;
                case 50:
                        data |= MAX1604_PWR_VCC_ON;
                        break;
                case 33:
                        data |= MAX1604_PWR_VCC_ON | MAX1604_PWR_3V;
                        break;
                default:
                        printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
                               configure->vcc);
                        return -1;
                }
                ret = adap->algo->master_xfer(adap, msg, 1);

                if (configure->reset)
                        set_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
                else
                        clr_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
#endif /* CONFIG_MERCURY_BACKPAQ */
        } break;

        case SLEEVE_NO_DEVICE:
        }

        /* silently ignore vpp and speaker enables. */

        if (sleeve_type != SLEEVE_NO_DEVICE)
          printk(__FUNCTION__ ": finished\n");

        return 0;
}

struct pcmcia_low_level bitsy_pcmcia_ops = { 
        bitsy_pcmcia_init,
        bitsy_pcmcia_shutdown,
        bitsy_pcmcia_socket_state,
        bitsy_pcmcia_get_irq_info,
        bitsy_pcmcia_configure_socket
};

void sa1100_bitsy_change_sleeves(enum sleeve_device new_sleeve_type) 
{
        if (sleeve_type == new_sleeve_type)
                return;
        if (sleeve_type != SLEEVE_NO_DEVICE)
                bitsy_pcmcia_shutdown();
        sleeve_type = new_sleeve_type;
        if (sa1100_bitsy_pcmcia_init.handler != NULL) {
                bitsy_pcmcia_init(&sa1100_bitsy_pcmcia_init);
        }
}
EXPORT_SYMBOL(sa1100_bitsy_change_sleeves);

