/*
 * usb-ohci-sa1111.h
 *
 * definitions and special code for Intel SA-1111 USB OHCI host controller
 *
 * init code for the SA-1111 ohci controller
 * special dma map/unmap code to compensate for SA-1111 h/w bug
 * 
 * 09/26/00 Brad Parker <brad@heeltoe.com>
 */

#include "asm/arch-sa1100/SA-1111.h"
#include "asm/proc/cache.h"

#define SA1111_OHCI_BASE        SA1111_p2v(_SA1111(0x0400))
#define SA1111_OHCI_EXTENT      512
#define SA1111_OHCI_IRQ         NIRQHCIM

#if defined(CONFIG_SA1100_XP860) || defined(CONFIG_ASSABET_NEPONSET)
#define PwrSensePolLow  1
#define PwrCtrlPolLow   1
#else
#define PwrSensePolLow  0
#define PwrCtrlPolLow   0
#endif

#define c_flush(x)                                      \
        do {                                            \
        __asm__ __volatile__(                           \
        "mcr    p15, 0, %0, c7, c6, 1      @ flush D entry"  \
          : : "r" (x));                                 \
        } while (0)

#define dwb(x)                                      \
        do {                                            \
        __asm__ __volatile__(                           \
        "mcr    p15, 0, %0, c7, c10, 4      @ DWB"  \
          : : "r" (x));                                 \
        } while (0)

#undef __arch_getl
#undef __arch_putl

volatile unsigned int __arch_getl(unsigned long a)
{
        c_flush(a);
        return *(volatile unsigned int  *)a;
}

void __arch_putl(unsigned long v,unsigned long a)
{
flush_cache_all();
        *(volatile unsigned int  *)a = v;
        dwb(a);
        c_flush(a);
}

/*
 * The SA-1111 errata says that the DMA hardware needs to be exercised
 * before the clocks are turned on to work properly.  This code does
 * a tiny dma transfer to prime to hardware.
 */
static void
sa1111_hc_dma_setup(void)
{
	static unsigned dmaTxBuff[4];

	/* DMA init & setup */

#define SACR0		(*((volatile Word *) SA1111_p2v (_SACR0)))
#define SACR1		(*((volatile Word *) SA1111_p2v (_SACR1)))
#define SADTCS		(*((volatile Word *) SA1111_p2v (_SADTCS)))
#define SADTSA		(*((volatile Word *) SA1111_p2v (_SADTSA)))
#define SADTCA		(*((volatile Word *) SA1111_p2v (_SADTCA)))

#define SKPCR_UCLKEn	(1<<0)
#define SKPCR_I2SCLKEn	(1<<2)
#define SKPCR_DCLKEn	(1<<7)

	/* prime the dma engine with a tiny dma */
	SKPCR |= SKPCR_I2SCLKEn;
	SKAUD |= 0x00000018;
	
	SACR0 |= 0x00003305;
	SACR1 = 0x00000000; 
	
	SADTSA = (unsigned long)dmaTxBuff;
	SADTCA = sizeof(dmaTxBuff);

	SADTCS |= 0x00000011;
	SKPCR |= SKPCR_DCLKEn;

	/* wait */
	udelay(100);

	SACR0 &= ~(0x00000002);
	SACR0 &= ~(0x00000001);

	/* */
	SACR0 |= 0x00000004;
	SACR0 &= ~(0x00000004);

	SKPCR &= ~SKPCR_I2SCLKEn;
}	

static int
sa1111_hc_start_ohci(void)
{
	struct pci_dev dev;
	memset((char *)&dev, 0, sizeof(dev));
	strcpy(dev.name, "usb-ohci");
	strcpy(dev.slot_name, "builtin");

#if 1
	request_region ( SA1111_OHCI_BASE, SA1111_OHCI_EXTENT, "usb-ohci" );
	return hc_found_ohci (&dev, SA1111_OHCI_IRQ, (void *)SA1111_OHCI_BASE);
#else
	unsigned long mem_base;
	mem_base = (unsigned long)ioremap_nocache(SA1111_OHCI_BASE, 4096);
	return hc_found_ohci (&dev, SA1111_OHCI_IRQ, mem_base );
#endif
}

static void
sa1111_release(void)
{
       release_region ( SA1111_OHCI_BASE, SA1111_OHCI_EXTENT );
}

static void
sa1111_cleanup(void)
{
	/* turn the USB clock off */
	SKPCR &= ~0x1;
}


/*
 * reset the SA-1111 usb controller and turn on it's clocks
 */
int
sa1111_ohci_hcd_init (void) 
{
	volatile unsigned long *Reset = (void *)SA1111_p2v(_SA1111(0x051c));
	volatile unsigned long *Status = (void *)SA1111_p2v(_SA1111(0x0518));

	/* turn on clocks */
	SKPCR = SKPCR_UCLKEn;
	udelay(100);

	/* force a reset */
	*Reset = 0x01;
	*Reset |= 0x02;
	udelay(100);

	*Reset = 0;

	/* take out of reset */
	/* set power sense and control lines (this from the diags code) */
        *Reset = ( PwrSensePolLow << 6 )
               | ( PwrCtrlPolLow << 7 );

	*Status = 0;

	udelay(10);

	/* compensate for dma bug */
	sa1111_hc_dma_setup();

	return sa1111_hc_start_ohci();
}


/* -------------- */

/*
 * simple memory allocator which won't give out memory with A20
 * set.
 *
 * allocated 2MB of kernel pages to accomplish this task.  assumes
 * memory is never freed.
 */

static char *mem;
static int mem_size;
static char *memp;
static int memp_left;

void *sa1111_alloc_consistent(struct pci_dev *hwdev,
			      size_t size,
			      dma_addr_t *handle)
{
	char *p;
	unsigned long m = (unsigned long)memp;
	int offset;

	if (0) printk("sa1111_alloc_consistent(size=%d) "
		      "memp=%p,memp_left=%d\n",
		      size, memp, memp_left);

	if (mem_size == 0) {
		/* what the heck, grab 2mb of pages */
		mem = (char *)__get_free_pages (GFP_KERNEL, 9);
		if (0) printk("allocated pages; mem %p\n", mem);
		mem_size = 512 * PAGE_SIZE;
		memp = mem;
		memp_left = mem_size;
	}

	if (memp_left < size)
		return 0;

	/* check for A20 - if set skip over the memory */
	if (m & 0x100000) {
		if (0) printk("memp %p has a20 set!\n", memp);
		offset = 0x100000 - (m & 0xfffff);
		memp += offset;
		memp_left -= offset;
		if (0) printk("new memp %p, memp_left %d\n", memp, memp_left);
	}

	p = memp;
	memp += size;
	memp_left -= size;
	*handle = virt_to_bus(p);
	return p;
}

void
sa1111_free_consistent(struct pci_dev *hwdev,
		       size_t size, void *vaddr,
		       dma_addr_t dma_handle)
{
}

/* -------------- */

/*
 * simple memory manager; keeps two free lists, one for small buffers
 * and one for big buffers.
 *
 * reserves 8 bytes of memory to remember the size of the chunk and the
 * old buffer pointer.
 */

#define MAX_SAFE_SIZES	2
#define SAFE_COUNT_1	8
#define SAFE_COUNT_2	8
#define SAFE_SIZE_1	(1024)
#define SAFE_SIZE_2	(64*1024)

static long mapped_alloc_size;
static char *safe_buffers[MAX_SAFE_SIZES];
static int safe_size[MAX_SAFE_SIZES] = { SAFE_SIZE_1, SAFE_SIZE_2 };
static int safe_count[MAX_SAFE_SIZES] = { SAFE_COUNT_1, SAFE_COUNT_2 };

/* allocate a 'safe' buffer from the pool */
static char *
alloc_safe_buffer(char *unsafe, int size)
{
	char *safe;
	dma_addr_t dma;
	int i, index;

	index = 0;
	if (size > SAFE_SIZE_1 - 8)
		index = 1;

	/* initial call?  grab some safe buffers */
	if (safe_buffers[index] == 0) {
		for (i = 0; i < safe_count[index]; i++)
		{
			safe = sa1111_alloc_consistent((struct pci_dev *)0,
						       safe_size[index],
						       &dma);

			if (safe == 0)
				break;

			*(char **)safe = safe_buffers[index];
			safe_buffers[index] = safe;
		}

		printk("allocated %d safe buffers of %d bytes @ %p\n",
		       i, safe_size[index], memp);
	}

	if (safe_buffers[index] == 0) {
		return (char *)0;
	}

	safe = safe_buffers[index];
	safe_buffers[index] = *(char **)safe;

	/* place the size index and the old buffer ptr in the first 8 bytes
	 * and return a ptr + 8 to caller
	 */
	((int *)safe)[0] = index;
	((char **)safe)[1] = unsafe;
	safe += sizeof(char*)+sizeof(int);

	return safe;
}

/* determine if a buffer is from our "safe" pool */
static char *
find_safe_buffer(char *buf, char **unsafe)
{
	if (buf >= mem && buf < (mem + mem_size)) {
		*unsafe = *(char **)(buf - sizeof(char *));
		return buf;
	}

	return (char *)0;
}

static void
free_safe_buffer(char *buf)
{
	int index;

	/* retrieve the buffer size index */
	buf -= sizeof(char*)+sizeof(int);
	index = ((int *)buf)[0];

	if (index < 0 || index >= MAX_SAFE_SIZES) {
		printk("usb-ohci: free_safe_buffer() corrupt buffer\n");
		return;
	}

	*(char **)buf = safe_buffers[index];
	safe_buffers[index] = buf;
}

/*
 * see if a buffer address is in an 'unsafe' range.  if it is
 * allocate a 'safe' buffer and copy the unsafe buffer into it.
 * substitute the safe buffer for the unsafe one.
 * (basically move the buffer from an unsafe area to a safe one)
 *
 * we assume calls to map_single are symmetric with calls to unmap_single...
 */
static dma_addr_t
magic_pci_map_single(struct pci_dev *hwdev, void *ptr,
		     size_t size, int direction)
{
	mapped_alloc_size += size;

	if (0) printk("magic_pci_map_single(hwdev=%p,ptr=%p,size=%d,dir=%x) "
		      "alloced=%ld\n",
		      hwdev, ptr, size, direction, mapped_alloc_size);

	/* we assume here that a buffer will never be >=64k */
	if ( (((unsigned long)ptr) & 0x100000) ||
	     ((((unsigned long)ptr)+size) & 0x100000) )
	{
		char *safe;

		safe = alloc_safe_buffer(ptr, size);
		if (safe == 0) {
			printk("unable to map unsafe buffer %p!\n", ptr);
			return 0;
		}

		if (0) printk("unsafe buffer %p mapped to %p (phy=%p)\n",
			      ptr, safe, (void *)virt_to_bus(safe));

		memcpy(safe, ptr, size);
		ptr = safe;
	}

	consistent_sync(ptr, size, direction);

	return virt_to_bus(ptr);
}

/*
 * see if a mapped address was really a "safe" buffer and if so,
 * copy the data from the safe buffer back to the unsafe buffer
 * and free up the safe buffer.
 * (basically return things back to the way they should be)
 */
static void
magic_pci_unmap_single(struct pci_dev *hwdev, dma_addr_t dma_addr,
		       size_t size, int direction)
{
	char *safe, *unsafe;
	void *buf;

	mapped_alloc_size -= size;

	if (0) printk("magic_pci_unmap_single(hwdev=%p,ptr=%p,size=%d,dir=%x) "
		      "alloced=%ld\n",
		      hwdev, (void *)dma_addr, size, direction,
		      mapped_alloc_size);

	buf = bus_to_virt(dma_addr);

	if ((safe = find_safe_buffer(buf, &unsafe))) {
		if (0) printk("copyback unsafe %p, safe %p, size %d\n",
			      unsafe, safe, size);

		consistent_sync(safe, size, PCI_DMA_FROMDEVICE);
		memcpy(unsafe, safe, size);
		free_safe_buffer(buf);
	} else {
		consistent_sync(buf, size, PCI_DMA_FROMDEVICE);
	}
}

