#ifndef _LIST_H_
#define _LIST_H_
/*
**++
**  FACILITY:
**
**      Linked list library
**
**  MODULE NAME:
**
**      list.h
**
**  ABSTRACT:
**
**      Contains function prototype declarations for linked list routines 
**      along with type definitions for generic linked lists
**      In total this module supports Four categories of operation :
**
**      This list has been implemented for use in the kernel and
**      in particular ISR. Thus it does not use dynamic memory 
**      allocation. To use the list you must create a data structure
**      with the first field being a list_node_t, thus the pointer to
**      the data may be cast between data * and list_node_t.
**
**      List Header Operations :
**          o list_header_init     -  Initialise a header structure
**          o list_header_clean    -  clean an empty list.
**          o list_header_getCount -  Return the number of elements in 
**                                    the list.
**
**      List Element Addition Operations 
**          o list_add_first     -  Insert the passed data at the start of the
**                                  list.
**          o list_add_last      -  Insert the passed data at the end of the
**                                  list.
**          o list_add_next      -  Insert the passed data after the item
**                                  indicated.
**          o list_add_previous  -  Insert the passed data before the item
**                                  indicated.
**
**      List Element Read Operations 
**          o list_get_first     -  Return pointer to the data of the first
**                                  element in the list.
**          o list_get_last      -  Return pointer to the data of the last
**                                  element in the list.
**          o list_get_next      -  Return pointer to the data of the next
**                                  element after the item indicated.
**          o list_get_previous  -  Return pointer to the data of the previous
**                                  element to the item indicated.
**
**      List Element Return Operations (i.e. node will be removed from the
**      list.)
**          o list_return_first   -  Return pointer to the data of the first
**                                   element in the list and remove this item
**                                   from the list.
**          o list_return_last    -  Return pointer to the data of the last
**                                   element in the list and remove this item
**                                   from the list.
**          o list_return_current -  Return pointer to the data of the current
**                                   item indicated and remove this item from 
**                                   the list.
** 
**  AUTHORS:
**
**      Peter Dettori : This list impementation started out as John Court's
**      however it proved unsuitable for kernel application and so I have 
**      rewritten/designed it, keeping the same functional interface.
**                      
**
**  CREATION DATE:
**
**      25 February, 1999
**
** Copyright (c) Compaq Computer Corporation, 1999
*/

/*  
 * Typedefs for LIST structures
 */

typedef struct list_node
{
    struct list_node   *next;
    struct list_node   *prev;
    struct list_header *header;
} list_node_t;

typedef struct list_header
{
    list_node_t *head;     /* First node in list */
    int       count;       /* Number of items on this list */
} list_header_t;   

typedef enum 
{
    ELIST_Ok,
    ELIST_NoMem,
    ELIST_InvalidHeaderTypePtr,
    ELIST_InvalidHeaderPtr,
    ELIST_InvalidItemPtr,
    ELIST_InvalidDataPtr,
    ELIST_EmptyList,
    ELIST_NonEmptyList,
    ELIST_NoNextItem,
    ELIST_NoPreviousItem
} list_error_t;


/*
 * Define library routine prototypes.
 */
/*********************************************************************

                LIST HEADER MANIPULATION ROUTINES

*********************************************************************/

/* 
 * Routine to initialise a list header.  The header is 
 * initialised, setting the count of items to zero and links to Null.
 *  
 * Returns :   status value.
 *
 * Errors  :   ELIST_Ok                    -  Header created successfully.
 *             ELIST_InvalidHeaderTypePtr  -  Bad type pointer pointer.
 *
 * SideEffects : If the header is already in use then the data 
 *               will be lost.
 */
extern list_error_t list_header_init(list_header_t *);

/* 
 * Routine to clean a list header.  A pointer to a list_header_t must be 
 * passed in.  The header is checked to ensure it is empty before it can 
 * be freed.
 *  
 * Returns :   status value
 *
 * Errors  :   ELIST_Ok                -  Header deleted successfully.
 *             ELIST_NonEmptyList      -  List not empty so could not 
 *                                        be deleted.
 *             ELIST_InvalidHeaderPtr  -  Bad HeaderPtrPtr.
 * 
 * SideEffects :  
 */
extern list_error_t list_header_clean(list_header_t *);

/* 
 * Routine to return the nuumber of elements on a list.  A pointer to a 
 * list_header_t must be passed in. The HeaderPtr is checked to ensure it is 
 * valid, then the list count is returned.
 *  
 * Returns :   Count of items on list. -1 on failure.
 *
 * Errors  :   ELIST_Ok                -  List count returned successfully.
 *             ELIST_InvalidHeaderPtr  -  Bad HeaderPtr.
 *
 * SideEffects : 
 */
extern int list_header_getCount(list_header_t *);


/*********************************************************************

                LIST ELEMENT INSERTION ROUTINES

*********************************************************************/

/* 
 * Routine to insert data as the first item in a list.
 * The list_node_t and list_header_t are checked to ensure they are valid.
 * The node is then inserted at the beginning of the List.
 *
 * Returns :   Error status
 *
 * Errors  :   ELIST_Ok                -  Data appended to list successfully.
 *             ELIST_InvalidHeaderPtr  -  Bad HeaderPtr.
 *	       ELIST_InvalidDataPtr    -  Invalid data item passed.
 *
 * SideEffects : The List header count of items is incremented by one.
 */
extern list_error_t list_add_first(list_header_t *, list_node_t *);


/* 
 * Routine to insert data as the last item in a list.
 * 
 * Returns :   Error status
 *
 * Errors  :   ELIST_Ok                -  Data appended to list successfully.
 *             ELIST_InvalidHeaderPtr  -  Bad HeaderPtr.
 *             ELIST_InvalidDataPtr    -  Invalid data item passed.
 *
 * SideEffects : The List header count of items is incremented by one.
 */
extern list_error_t list_add_last  (list_header_t *, list_node_t *);

/* 
 * Routine to insert data as the next item to the node indicated.
 * This new node is inserted after the item is then
 * inserted in the list following the item indicated.
 *
 * Returns :   Error status
 *
 * Errors  :   ELIST_Ok              -  Data inserted in list successfully.
 *             ELIST_InvalidDataPtr  -  Invalid data item passed.
 *
 * SideEffects : The List header count of items is incremented by one.
 */
extern list_error_t  list_add_next  (list_node_t *node, list_node_t *new);

/* 
 * Routine to insert data before the indicated node.
 * This new node is inserted after the item is then
 * inserted in the list following the item indicated.
 *
 * Returns :   Error status
 *
 * Errors  :   ELIST_Ok              -  Data inserted in list successfully.
 *             ELIST_InvalidDataPtr  -  Invalid data item passed.
 *
 * SideEffects : The List header count of items is incremented by one.
 */
extern list_error_t list_add_previous(list_node_t *node, list_node_t *new);

/*********************************************************************

                LIST ELEMENT DATA READ ROUTINES

*********************************************************************/

/* 
 * Returns the first node on the list.
 *
 * Returns :   Pointer to first node on success.
 *             NULL on failure.
 *
 * SideEffects : 
 */
extern list_node_t   *list_get_first (list_header_t *);

/* 
 * Returns the last node on the list.
 *
 * Returns :   last node on success.
 *             NULL on failure.
 *
 *
 * SideEffects : 
 */
extern list_node_t *list_get_last  (list_header_t *);

/* 
 * Routine to return a pointer the next node in the last after the node 
 * passed in.
 *
 * Returns :   Pointer to next node on success.
 *             NULL on failure.
 *
 * SideEffects : 
 */
extern list_node_t   *list_get_next     (list_node_t *);


/* 
 * Routine to return a pointer to the previous node in the last as referenced
 * by the node passed in.
 *
 * Returns :   Pointer to previous node on success.
 *             NULL on failure.
 *
 *
 * SideEffects : 
 */
extern list_node_t *list_get_previous (list_node_t *);

/*********************************************************************

                LIST ELEMENT DELETION ROUTINES

*********************************************************************/

/* 
 * Routine to return the first node on the list.
 *
 * 
 * Returns :   Pointer to first node on success.
 *             NULL on failure.
 *
 *
 * SideEffects :  
 */
extern list_node_t *list_return_first (list_header_t *);


/* 
 * Routine to return the last node on the list.
 *
 * Returns :   Pointer to last node on success.
 *             NULL on failure.
 *
 * SideEffects :
 */
extern list_node_t *list_return_last  (list_header_t *);

/* 
 * Routine to return the data of the current node on a list.
 *
 * Returns :   Pointer to node on success.
 *             NULL on failure.
 *
 * SideEffects :  
 */
extern list_node_t  *list_return_current (list_node_t *);


#endif
