/***************************************************************************/

/*
 *	usb_ep1.c  --  
 *
 *	Copyright (C) Compaq Computer Corporation, 1998
 *
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>
#include <asm/atomic.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <asm/system.h>

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/skbuff.h>

#include "usbc_h3600_dma.h"
#include "usbc_h3600_ctl.h"
#include "usbc_h3600_dbg.h"

/* 
 * Implementation :
 * endpoint 1 driver will allocate a page for buffering transfers that 
 * occur before the client driver provides a buffer, after the buffer is
 * full OUT packets to endpoint 1 will be NAK'd.
 *
 * Once the client driver provides a buffer for receive characters it will 
 * have the data copied into it, at a later date it might be made to allow
 * direct dma into the client provided buffer.
 *
 * Presently the client may not queue blocks for receive, they will be 
 * rejected. This may be changed at some stage probably won't be neccessary
 * unless direct dma to the buffer is allowed though.
 *
 * It is neccessary to operate in this manner because we don't know the 
 * logical packet size of the client just the USB packet size. If we are
 * to provide packetisation then the client must ask for specific amounts
 * of data.
 */

#define DMA_OUT_COUNT_MAX 0x100
/* CF put these defines in to get zero warnings */
void _set_dma_mode_sa1100(int channel, int addr);
void * _get_dma_address_sa1100(dmach_t channel);
int enable_dma_sa1100(dmach_t channel,void * ptr, unsigned size);
int disable_dma_sa1100 (dmach_t );
dma_regs_t * get_dma_regbase( dmach_t channel);
int get_dma_residue_sa1100(dmach_t channel);

/*
 * Globals
 */
static int dmachn_rx=-1;
int dmachn_rx_inuse = 0;
struct tq_struct recv_task;

/* 
 * DMA circular buffer 
 * start points to the start of free memory for USB receives
 * end points to the end of the free area.
 * the buffer is full when start may not be advanced without
 * going past the end marker.
 *
 * The interrupt handler is allowed write permission to start and read
 * permission to end.
 * The receive_block is allowed write permission to the end and read 
 * permission on the start. The receive block should take a copy of the 
 * start in case an interrupt occurs and the actual start is advanced.
 */
#define RECEIVE_BUFFER_SIZE (4*PAGE_SIZE)
/* TODO rationalise all these pointers */
char *phys_rxbuf = NULL;	/* global phys buf */
char *phys_start = NULL;
char *receive_page = NULL;
char *receive_page_end = NULL;
char *receive_page_start = NULL;
int receive_page_full = 0;
int reset_copy = 0;

struct sk_buff *recv_skb;
unsigned short pktlength = 0;

/* 
 * add size to the ptr ensuring that the result is aligned
 * to align. Used for advancing the circular receive buffer.
 * CF 
 *		<----align----->
 *		+		+		+		+
	           ^            ^
 *		   |		|
 *		ptr(before)	ptr(after)
 */
#define ADVANCE(ptr, size, align)                              \
        {                                                      \
	        if ((size) % (align))                              \
	        {                                                  \
		        (size) = (size) + (align) - (size % align);    \
	        }                                                  \
	        (ptr) += (size);                                   \
	    }                                                

/*
 * Internal Prototypes
 */
void ep1_copy_task(void *data);
/* TODO find out why we dont use data */
#ifdef TXTEST
void ep1_test_task(void * data);
#endif

/*
 * Macros
 */
#ifndef MIN
#define MIN(a,b)	((a) < (b) ? (a) : (b))
#endif

/* UDC tx/rx data register See 11-76/79 of StrongARM manual */
static volatile unsigned int *UDCCS1_reg, *UDCDR_reg;

/*
 * called by the udc control component after a packet is received, or at
 * bus enumeration. Sets up the DMA channels to use the given packet 
 * for receive.
 *
 * pkt       : the packet to use for receive.
 * returns   : 0 if ok, -1 if a pkt is already queued for recv.
 */
int
ep1_recv(void)
{
    int status = -1;

    CHECK_ADDRESS;
    
    /* check that we are initialised by the local record of the 
     * dma channel in use, and that there isn't a packet in use already.
     */
    if ( (dmachn_rx >= 0) && (!dmachn_rx_inuse) && (!receive_page_full) )
    {
        /* the start page ptr should be ok to start dmaing. */
        dmachn_rx_inuse = 1;
        enable_dma_sa1100(dmachn_rx,(void *)phys_rxbuf,usbd_info.rx_pktsize);
        status = 0;
    }
    else
    {
	printk("WARNING: DMA wasn't started chan=%d %d %d\n",
		dmachn_rx,dmachn_rx_inuse,receive_page_full);
    }
	
    CHECK_ADDRESS;
    return status;
}

/*
 * reset any transfers in progress and clear the buffer
 * to start at the beginning again.
 *
 * returns : 0 if ok, -1 otherwise
 */
int
ep1_reset(void)
{
    int status = 0;
    CHECK_ADDRESS;
    if (dmachn_rx >= 0)
    {
#if 0
        disable_dma_sa1100(dmachn_rx);
#else
        dma_regs_t *pDMA = get_dma_regbase( dmachn_rx );
	pDMA->ClrDCSR = 0x07;
#endif
        /* reset dma */
        _set_dma_mode_sa1100(dmachn_rx, 0);
        _set_dma_mode_sa1100(dmachn_rx, DMA_UDC_RECEIVE);
        dmachn_rx_inuse = 0;
    }
    /* stall and unstall endpoint to reset data toggle */
    do
    {
        *(UDCCS1) |= UDCCS1_FST;
    } while (!(*(UDCCS1) & UDCCS1_FST));
    do
    {
        *(UDCCS1) &= ~UDCCS1_FST;
    } while (*(UDCCS1) & UDCCS1_FST);

    /* now start things up again. */
    if (recv_task.sync == 0)
    {
        if (usbd_info.state ==  USB_STATE_CONFIGURED)
        {
            reset_copy = 1;
            receive_page_end = receive_page_start;
            receive_page_full = 0;
        }
    }
    else
    {
        printk("USB RECV : Copy task running\n");
    }

    ep1_recv();
    printk(">>>> EP1_RESET_EXIT >>>>\n");

    CHECK_ADDRESS;
    return status;
}

/*
 * Initialise the DMA system to use the given chn for receiving
 * packets over endpoint 1.
 *
 * chn     : the dma channel to use, or -1 if the receive engine should
 *           continue using the one it already has.
 * returns : 0 if ok, -1 if channel couldn't be used.
   #define	CF_PG_SIZE	4096  CF PGDIR_SIZE is 2 ** 20 too big ??
   WARNING caller of ep1_init MUST check return CF???
 */
int 
ep1_init(int chn)
{
    int status = 0;

    printk("\nEP1_INIT:");
    if (chn >= 0)
    {
        dmachn_rx = chn;
    }
    dmachn_rx_inuse = 0;

    /* allocate a page for the receive buffer */
    if (!receive_page)
    {
	dma_addr_t dmaphys = 0;

        /* allocate the buffer and make it non cacheable. */
	printk("Allocing %u bytes\n",(unsigned int)RECEIVE_BUFFER_SIZE);
	receive_page = (char *)consistent_alloc(GFP_KERNEL,
					RECEIVE_BUFFER_SIZE,
					&dmaphys);

        phys_start = (char *)dmaphys;
    }

    if (!receive_page)
    {
#if 0
        panic("usb (usb_ep1.c %d) : couldn't allocate a receive buffer\n", 
              __LINE__);
#else
	printk("PANIC CANT ALLOCATE BUFFER >>>>>>>>>>>>>>>>>\n");
	return -1;
#endif
    }
    receive_page_start = receive_page_end = receive_page;
    phys_rxbuf = phys_start;

    if (recv_skb)
    {
        dev_kfree_skb(recv_skb/*, FREE_READ*/);
    }
    recv_skb = NULL;
    pktlength = 0;
    
    /* CS1 page 11-71 DR = 11-76 StrongARM Developer's manual */
    UDCCS1_reg = UDCCS1;
    UDCDR_reg = UDCDR;

#ifdef TXTEST
    recv_task.routine =  ep1_test_task;
#else
    recv_task.routine =  ep1_copy_task;
#endif
    recv_task.data = (void *) NULL;
    recv_task.sync = 0;
    
#if 0
    printk("V=%p P=%p\n", receive_page_start,phys_rxbuf);
#endif
    ep1_recv();		/* ep1_init */

    CHECK_ADDRESS;
    printk(">>>>>>>> EP1_INIT_EXIT >>>>>>>>>\n");
    return status;
}

/*
 * Endpoint 1 interrupts will be directed here.
 *
 * status : the contents of the UDC Status/Interrupt Register.
 */
void inline
ep1_int_hndlr(unsigned int status)
{
    unsigned int ep1_status, pkt_error; 
    unsigned int idx,udc_fifo;

    ep1_status = *(UDCCS1_reg);
    pkt_error = ep1_status & UDCCS1_RPE;

    if (pkt_error) printk("\n----PKT ERROR EP1_STATUS=%08x\n",ep1_status);

    /*
	RPC has 2 meanings:
	1. A good packet has arrived.
	2. An error packet has arrived (in conjunction with error bits)
	TODO TODO
	WARNING if RPC is clear then the error bits mean squat - is this a bug?
    */
    if (ep1_status & UDCCS1_RPC)
    { 
	/*
		The ReceivePacketComplete is set tell DMA
		to stop
		REMEMBER to start it TOOTSWEET or you will 
		loose bytes resulting in a FIFO over-run.
		TODO TODO
		dmachn_rx_inuse MUST return FALSE else PANIC!!!
		Dont rely on disable_dma_sa1100 returning a TRUE
		as it will rely on the DONE bits being set and they
		wont be set if we ask for a packet fragment < USB_PKTSIZE=64
	*/
	disable_dma_sa1100(dmachn_rx);
	dmachn_rx_inuse = 0;

        if (!pkt_error)
        {
	    /*
		Since the DMA isn't using an IRQ we rely on the 
		to tell DMA to empty the FIFO when it reaches past
		the low water mark as indiceted by the RFS bit in
		UDCCS1 pg 11-71 
		TODO TODO REMEMBER we asked DMA to DMA pktsize - 8 
			we may be able to remove this.
		If we were able to disable DMA then all bytes have
		been xfered (residue=0)
		-------------SCRIBBLE-------- delete in 2 days
	    int residue = get_dma_residue_sa1100(dmachn_rx);
	    char * rxbuf= get_dma_address_sa1100(dmachn_rx);
	    char * rxbuf=receive_page_start + (usbd_info.rx_pktsize - 8);
		Here is the prob
		If we xfer packet fragments we cannot rely on the DONEA/B
		but we can rely on DMA being finished
		Why as well does Peter ADVANCE his buffer by pktsize bytes
		does he NOT expect fragments.Example what if the host
		sends 2 bytes!!!!!
		----------------end of SCRIBBLE ------------
		What will happen is that the HOST will send 'N'
		'pktsize' USB packets ( where N can be equal to zero )
		The last USB packet will always be a fragment so DMA
		MUST get disabled irrespective of the DONE bits.
		This means we need to peek the DMA buffer to get
		the place to copy our bytes.
	    */
	    int residue = get_dma_residue_sa1100(dmachn_rx);
	    char * rxbuf=receive_page_start+(usbd_info.rx_pktsize-(residue+8));

            while (ep1_status & UDCCS1_RNE)
            {
                /* have data in the fifo so * pull it out .
		 * this is residual data and rxbuf actually
		 * points to the head of the buffer
		  +---------------------------------------------------+
			^				^ <FIFO_data>
			|<---( this has been DMA'd)---->|
			receive_page_start		rxbuf
                 */
                udc_fifo = *(UDCDR_reg);
                *rxbuf = (char) udc_fifo;
                rxbuf++;
                ep1_status = *(UDCCS1_reg);
            }
	    /* exit here with our buffer filled */
        }
    }
    else
    {
        printk("WARNING:ep1_status & UDCCS1_RPC == 0\n");
    }

    /* Some house keeping stuff */

    /* CF clear the stall bit */
    idx=0;
    /* TODO an index of 1000 may cause a timing glitch */
    while ((ep1_status & UDCCS1_SST) && (idx<1000))
    {
        idx++;
        *(UDCCS1_reg) = UDCCS1_SST;
        ep1_status = *(UDCCS1_reg);
    }
    if( idx >= 1000 ) printk("UNABLE TO CLEAR SST\n");
    
    CHECK_ADDRESS;

    /* clear RPC by writing to it*/
    idx = 0;
    do
    {
        idx++;
        *(UDCCS1_reg) = UDCCS1_RPC;
    } while ( (*(UDCCS1_reg) & UDCCS1_RPC) && (idx < 1000));
    if( idx >= 1000 ) printk("UNABLE TO CLEAR SST\n");

    if (!pkt_error)
    {
        usbd_info.usb_stats.outpkts++;
        
        /* advance the start pointer and go again
         * unless have run out of receive buffer, in which case
         * start NAKing packets....
	 * TODO rationalise Virtual & Phys pointer arithmetic
         */
        ADVANCE(receive_page_start, usbd_info.rx_pktsize,
                usbd_info.rx_pktsize);
        ADVANCE(phys_rxbuf, usbd_info.rx_pktsize,
                usbd_info.rx_pktsize);
	/* Doing pointer arithmetic on our Virtual buffer */
        if (receive_page_start >= (receive_page + RECEIVE_BUFFER_SIZE))
        {
            receive_page_start = receive_page;
	    phys_rxbuf = phys_start;
        }
        
        /* if advancing start makes start == end then the buffer is full */
        if (receive_page_start != receive_page_end)
        {
	    /*
		This starts the next DMA
		TODO
		We shouldn't have it in another function for speed.
	    */
            ep1_recv();		/* ep1_int_hndlr */
        }
        else
        {
            printk("receive_page_full\n");
            receive_page_full = 1;
            /*
		TODO where are IRQs reenabled?
		don't want interrupts for this now
	    */
            disable_irq(IRQ_Ser0UDC);
        }

        /* 
         * queue the copy task to run, if it is not presently * running. 
	 * CF the copy task (ep1_copy_task) allocates a socket buffer
         */
        if (recv_task.sync == 0)
        {
	    /* goto ep1_copy_task  */
            queue_task(&recv_task, &tq_immediate); 
        }
        mark_bh(IMMEDIATE_BH);
    }
    else
    {
	/* pkt_error > 0 */
        usbd_info.usb_stats.outpkt_errs++;
        if (ep1_recv() != 0)	/* ep1_int_handler */
        {
            printk("USB RECV : can't restart, ep1 no good\n");
        }
    }

    CHECK_ADDRESS;
}

#ifdef TXTEST
/*
	Note put all your printks here and not in the ISR
	To use this test
	Fire a BIG packet at the iPAQ
	It should receive all of the 64 bytes packets
	then call this function 
	ASSUMTIONS
	1. 'end' is constant
	2. 'start' never wraps
*/
void ep1_test_task(void * data)
{
    unsigned long flags;
    unsigned char received_char,expected_char;
    static unsigned int idx=2; /* skip the len */
    static unsigned int tst_pktlen=0;
    static unsigned int pktcount=0;
    char * end;
    char * start;

    /*
	Remember we read from the end and write from the start.
    */
    end = receive_page;	/* WARNING in the 'real' world 'end' will move */

    save_flags_cli(flags);
    start = receive_page_start;
    restore_flags(flags);

#if 0
    printk("\n---EP1_TEST_TASK E=%p S=%p\n",end,start);
#endif

    /* This may get called more than once so it must be re-entrant */
    if( !tst_pktlen )
    {
	    /* The pkt length is contained in the data received */
            tst_pktlen = (end[0] & 0xff);
            tst_pktlen += ( (end[1] & 0xff) << 8 );
#if 0
	    printk("\nLen=%02x:%02x:=%d", *end,*(end + 1),tst_pktlen );
#endif
    }

    /* 
	We do not use the ADVANCE macro here since.
	The ADVANCE macro will advance pointers onto pktsize boundaries
	so if you Tx pktsize=65 the start pointer moves 128 bytes.
	This means the test (end+idx) != start is not sufficient
    */
    while( (idx != tst_pktlen) && ( (end + idx) != start ) )
    {
	    expected_char = idx;
	    received_char = *(end + idx);
	    if ( expected_char != received_char )
	    {
		printk("[%d]ERROR:expected=%02x received=%02x\n",
				pktcount,expected_char,received_char);
		/* Bad pkt back to the start */
		tst_pktlen=0;
		idx=2;
		break;
	    }
	    ++idx;
    }

    if( tst_pktlen && (tst_pktlen == idx) )
    {
	/* Packet decoded -this is where we process the packet */
	printk("\n[%d]RXOK: %d bytes\n",pktcount,tst_pktlen);
	tst_pktlen=0;
	idx=2;	/* skip the USB len field */
	save_flags_cli(flags);
#if 1
	/* TODO Dont do this since we also want to test wrap round */
	receive_page_start = receive_page;	/* reset start */
	phys_rxbuf = phys_start;
#endif
	restore_flags(flags);
	/* Start a TXTEST ONLY if first packet  */
	if(!pktcount)
	    ep2_start();
	++pktcount;
    }


#if 0
    printk("processed = %d highest=%02x\n",idx,*(end));
#endif

}

#else 	/* ifdef TXTEST */

/*
 * Checks if any received bytes may be copied out of the internal buffer
 * into the receive buffer. After doing so it trys to start another receive
 * in case that the internal buffer was full.
 *
 * Should be called on the schedule wait queue and may be put there by 
 * the block receive function or the interrupt handler.
 *
 * When the receive block is full the callback will be called.
 *
 * data   : unused, globals variables hold our state.
 */
void 
ep1_copy_task(void *data)
{
    char *tmp_start;
    int copy_size;
    int loops = 0;
    unsigned long flags;
    unsigned char *mac_header;
    int idx;

    CHECK_ADDRESS;

    /* flush the cache of the receive page. */
    /*
    processor.u.armv3v4._flush_cache_area((unsigned int) receive_page, 
					  RECEIVE_BUFFER_SIZE, 0);
    */

    /* printk("ep1_copy_task\n"); */
    /* we have been reset... */
    if (reset_copy)
    {
        /* free the current skb */
        if (recv_skb)
        {
            dev_kfree_skb(recv_skb/*, FREE_READ*/);
        }
        recv_skb = NULL;
        pktlength = 0;
        reset_copy = 0;
	printk("EP!_COPY_TASK_RESET:");
    }
        
    /*
     * this process may be interrupted and another packet received,
     * thus we must copy the start buffer ptr, which is the only one that 
     * interrupt handler may changed. The while loop will handle the cases 
     * where an interrupt occurred.
     *
     */

    /* Interrupts are disabled in this area */
    save_flags_cli(flags);
    do
    {
        loops ++;
        /* the interrupt handler is the only function allowed to change the
         * page start, we take a copy in case a packet arrives and it
         * changes.
         */
        tmp_start = receive_page_start;
    /* Interrupts are disabled in this area */
        restore_flags(flags);
        
        /* this confusing logic should mean that we process a block iff
         * the recv_block is valid, and the receive buffer is not empty.
         */
        if (((tmp_start != receive_page_end) || (receive_page_full)))
        {
            /* if packet length is 0, allocate a new skb */
            if (pktlength == 0)
            {
		/* CF
		   The packek length is embedded in the 1st 2 bytes of the
		   USB layer
		*/
                pktlength = (receive_page_end[0] & 0xff);
                pktlength += (receive_page_end[1] & 0xff) << 8;
#if 0
	        printk("\nRX:pktlen=%d\n",pktlength);
#endif

                if ((pktlength>1504) || (pktlength==0))
                {
                                 printk("BAD packet length %x %x %d\n",
                                  receive_page_end[0],
                                  receive_page_end[1],
                                  pktlength);
                    udc_stallep(1);
                }
                else
                {
                    recv_skb = dev_alloc_skb(1518);
                    /* want the ip packet aligned on a 16 byte boundary so
                     * reserve 12 bytes, 2 empty, and 10 for the address.
                     * 2 bytes are used for packet length on the USB.
                     */
                    skb_reserve(recv_skb, 12);
                }
            }

            if ((pktlength) && (recv_skb))
            {
                /* copy the internal buffer into the receive block*/
		/* CF All this does is find the distance between head & tail */
	    /*

		We always copy ( copy_size bytes ) starting at receive_page_end
		
		This calculates the bytes left to copy.
		bytes_left = pktlength - recv_skb->len

		|<---------------pktlen-------------------------------->|
		|<---------------skb->len------>|<-----bytes_left------>|


			RECEIVE_BUFFER_SIZE
	    	+---------------+-------+-----------------------+
				^	^		  ^ 
				|	|<---bytes_left-->|
				|	|<---copy_size--->|
				START	END

		CASE1: bytes_left do not wrap


			RECEIVE_BUFFER_SIZE
	    	+---------------+-------+-----------------------+
				^	^			^ 
		|---->		|	|<---bytes_left---------|
				|	|<------copy_size------>|
				START	END

		CASE2: bytes_left wrap to start of RECEIVE_BUFFER_SIZE


			RECEIVE_BUFFER_SIZE
		+-------+---------------+---------------+---------------+
			^		^
			|<-copy_size--->|
			|<----------bytes_left---------->
			END		START

		CASE3: START > END with still bytes to read to complete packet

			RECEIVE_BUFFER_SIZE
		+-------+---------------+---------------+---------------+
			^		^		^
			|<--copy_size-->|		|
			|<--bytes_left->|<--next pkt---->
			END				START
		CASE4: START > END reading copy_size will complete packet.



		We always copy from END ( copy_len bytes ) so if
		copy_size(CASE2) > copy_size(CASE1) bang!!
	*/
		/* copy_size is the Minimum od CASE 1 & 2 */
                copy_size = MIN(pktlength-recv_skb->len, 
                                (receive_page+RECEIVE_BUFFER_SIZE) -
                                receive_page_end);
		/* copy_size is the Minimum od CASE 3 & 4 */
                if (tmp_start > receive_page_end)
                {
                    copy_size = MIN(copy_size, 
                                    tmp_start-receive_page_end);
                }
            
                if (copy_size)
                {
                    if (skb_tailroom(recv_skb)<copy_size)
                    {
                        printk("USB RECV : no room skb %p"
                               " length %d copy %d\n",
                               recv_skb, recv_skb->len, copy_size);
                    }

#if 0
		    /* dump some bytes of raw data from receive buffers*/
		    {
		    unsigned char * ptr = receive_page_end;
		    unsigned int len,i;
		    len = (copy_size > 20 ) ? 20 : copy_size;
		    printk("RX:[%d:%d]",copy_size,pktlength);
		    for(i=0; i < len ; i++)
			printk("%02x:",*(ptr+i) );
		    }
		    printk("\n");
		    /* dump the raw data */
#endif

                    memcpy(skb_put(recv_skb, copy_size),
                           receive_page_end, 
                           copy_size);
                    /* XXX mark as read CF discard or recycle this buffer */
                    receive_page_end[0] = 0xde;
                    /* advance the end marker */
                    save_flags_cli(flags);
                    ADVANCE(receive_page_end, copy_size,
                            usbd_info.rx_pktsize);
		    /* advancing the END automatically frees some space */
                    if (receive_page_full)
                    {
                        receive_page_full = 0;
                        enable_irq(IRQ_Ser0UDC);
                    }
                    restore_flags(flags);
                    /* if we have advanced the end marker we cannot 
                     * have a full buffer 
                     */
                    if (receive_page_end >= (receive_page+RECEIVE_BUFFER_SIZE))
                    {
                        receive_page_end = receive_page;
                        /* in this case have gone back to the start of 
                         * the buffer. So check if there is more to copy.
                         */
                        
                        /* must flush the cache, to make sure reading the right
                         * stuff.
                         */
			/*  processor.u.armv3v4._flush_cache_area(
                            (unsigned int) receive_page, 
                            RECEIVE_BUFFER_SIZE, 0);*/
                    
                        if ((recv_skb->len != pktlength) &&
                            (tmp_start != receive_page_end))
                        {
                            /* can copy more data... */
			    /* THIS IS CASE 1 ABOVE ) wrapround */
                            copy_size = MIN(pktlength - recv_skb->len, 
                                            tmp_start-receive_page_end);
/*
 Return the number of bytes of free space at the tail of an sk_buff
*/
                            if (skb_tailroom(recv_skb)<copy_size)
                            {
                                printk("USB RECV : no room skb %p"
                                       " length %d copy %d\n",
                                       recv_skb, recv_skb->len, copy_size);
                            }
                            memcpy(skb_put(recv_skb, copy_size), 
                                   receive_page_end, 
                                   copy_size);
                            /* advance the end marker */
                            save_flags_cli(flags);
			    /* CF using the Vbuf to keep track of buffer*/
                            ADVANCE(receive_page_end, copy_size, 
                                    usbd_info.rx_pktsize);
                            if (receive_page_full)
                            {
                                receive_page_full = 0;
                                enable_irq(IRQ_Ser0UDC);   
                            }
                            restore_flags(flags);
                        }
                    }
                }
                        
                /* if the block is complete, do the callback */
                if (recv_skb->len == pktlength)
                {
#if 0
		    /* dump some bytes of raw data from skb*/
		    {
			unsigned char * ptr = recv_skb->data;
			unsigned int len,i;
			len = (pktlength > 20 ) ? 20 : pktlength;
			printk("SKB:[%d:%d]",pktlength,recv_skb->len);
			for(i=0; i < len ; i++)
			    printk("%02x:",*(ptr+i) );
			printk("\n");
		    }
		/* dump the raw data */
#endif
		  if ( netif_running(usbd_info.dev) )
                    {
                        /* put the mac addresses back in, over the top of the
                         * size that was sent over.
                         */
/*
 	This function extends the used data area of the buffer at the buffer
	start. If this would exceed the total buffer headroom the kernel will
	panic. A pointer to the first byte of the extra data is returned.
*/
                        mac_header = skb_push(recv_skb, (2*ETH_ALEN)-2);
                        for (idx=0;idx<ETH_ALEN;idx++)
                        {
                            mac_header[idx] = usbd_info.dev->dev_addr[idx];
                            mac_header[idx+ETH_ALEN] = usbd_info.host_addr[idx];
                        }
                        recv_skb->dev = usbd_info.dev;
#if 0
			/* dump mac header (plus first 10 bytes from skb */
			{
			unsigned char * ptr = recv_skb->data;
			unsigned int i;
			printk("MAC HDR:");
			for(i=0; i < (2 * ETH_ALEN) + 10 ; i++)
			    printk("%02x:",*(ptr+i) );
			printk("\n");
			}
			/* dump the raw data */
#endif
/*
 *	Determine the packet's protocol ID. The rule here is that we 
 *	assume 802.3 if the type field is short enough to be a length.
 *	This is normal practice and works for any 'now in use' protocol.
 */
                        recv_skb->protocol = eth_type_trans(recv_skb,
                                                            usbd_info.dev);
                        recv_skb->ip_summed = CHECKSUM_UNNECESSARY;

                        usbd_info.eth_stats.rx_packets++;
                        netif_rx(recv_skb);
                        recv_skb = NULL;
                    }
                    else
                    {
                        dev_kfree_skb(recv_skb/*, FREE_READ*/);
                        recv_skb = NULL;
                    }
                    pktlength = 0;
                }
            }
        }
        save_flags_cli(flags);
    } while ( ((receive_page_start != receive_page_end) || (receive_page_full))
              && (loops<1000));
    /* ep1_recv(); TODO TODO causes panic */		/* ep1_copy_task */
#if 1
    if( loops >= 1000 )
	printk("WARNING: loops exceeded\n");
#endif

    restore_flags(flags);
    CHECK_ADDRESS;
}

#endif 	/* ifdef TXTEST */




