/*************************************************************
 *  StrongARM 11xx LCD Frontlight Device
 *
 *  linux/drivers/video/sa1100_frontlight.c 
 *************************************************************
 *
 * Initial Version by: Nicholas Mistry (nmistry@lhsl.com)
 *
 * This driver provides basic functionality for the frontlight
 * provided with the Assabet board.   It will enable and disable
 * the light.  
 * 
 * The module grabs a dynamically assigned Major, so you will have
 * to do a mknod before using the device.  I typically use /dev/sa1100-fl
 * 
 * Because the assabet does not come with its own power supply for the
 * one can be obtained/made.  Below is a list of the parts i used to 
 * enable mine. 
 * 
 * **   The parts/cable described below assume that your assabet 
 * **   is using the Sharp LQ039Q2DS54 Screen. 
 * 
 * Parts List:
 * ------------
 *   1 ERG power 8m122375 DC to AC inverter.
 *     obtain from: Endicott Research Group Inc. (http://www.ergpower.com)
 * 
 *   Assabet Side Connector:
 *     1 Minitek 90312-005 Housing 
 *     3 Minitek 77138-101 Pins
 * 	obtain both from:  FCI Framatome Group (http://www.fciconnect.com)
 * 	* Samples Available *  
 * 
 *   Inverter Side Connector:
 *     1 Standard 0.1" 4 pin Housing
 *     3 Standard 0.1" Pins  (crimp is preferred)
 * 
 *   3 pieces of 26 AWG Stranded Wire each aprox 4"-6" long 
 * 	(3 colors optional: red,black,white)
 * 
 * Cable Diagram:
 * --------------
 * 
 * Inverter:				Assabet:
 *  
 *  _____                                   _______
 *       |      ______     /---------------|  |1  [|  (red wire)
 *  V+  1|---- |]   1 |---/   /------------|_ |2  [|  (black wire)
 *  GND 2|---- |]   2 |------(------\      |_||3  [|
 *  CTL 3|---- |]   3 |-----/        \-----|  |4  [|  (white wire)
 *  NE  4|---- |]___4_|                    |__|5__[|
 *  _____|
 * 
 * 
 * Example Program: 
 * ----------------
 * 
 * // This following example program will blink on and off the  
 * //  frontlight 5 times w/ an one second delay.  	     
 * 
 * #include <stdio.h>
 * #include <stdlib.h>
 * #include <sys/ioctl.h>
 * #include <sys/types.h>
 * #include <fcntl.h>
 * #include <unistd.h>
 * #include <errno.h>
 * 
 * #include <video/sa1100_frontlight.h>
 * 
 * int main(void) {
 * 
 *   int i, fd, retval;
 * 
 *   fd = open ("/dev/sa1100-fl", O_WRONLY);
 * 
 *   if (fd ==  -1) {
 *     perror("/dev/sa1100-fl");
 *     exit(errno);
 *   }
 *   
 *   printf("\n\t\t\tFrontlight Driver Test Example.\n\n");
 *   
 *   for (i=0; i<5; i++) {
 *     printf("Turning OFF Frontlight...\n");
 *     retval = ioctl(fd, _SA1100_FL_OFF, 0);
 *     if (retval == -1) {
 *       perror("ioctl");
 *       exit(errno);
 *     }
 *     
 *     sleep(1);
 *     
 *     printf("Turning ON Frontlight...\n");
 *     retval = ioctl(fd, _SA1100_FL_ON, 0);
 *     if (retval == -1) {
 *       perror("ioctl");
 *       exit(errno);
 *     }
 *     
 *     sleep(1);
 *     
 *   }
 *   
 *   printf("Turning OFF Frontlight...\n");
 *   retval = ioctl(fd, _SA1100_FL_OFF, 0);
 *   if (retval == -1) {
 *     perror("ioctl");
 *     exit(errno);
 *   }
 * 
 *   fprintf(stderr, "Done.\n");
 * 
 * }
 * 
 *************************************************************
 * Revision History:
 * 	07/07/2000 - Nicholas Mistry (nmistry@lhsl.com)
 *		Initial release for assabet, no dimming support, 
 *		can enable and disable the frontlight via ioctl.
 *		Initial	framework for dimming included.
 *
 *************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/init.h>
#include <linux/fb.h>
#include <linux/delay.h>

#include <asm/system.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/proc/pgtable.h>

#include <video/sa1100_frontlight.h>

#ifdef MODULE

MODULE_AUTHOR("Nicholas N. Mistry (nmistry@lhsl.com)");
MODULE_DESCRIPTION("SA1100 Frontlight Driver");

#endif

//------------------------------------------------------------

static int sa1100_fl_ioctl(struct inode * inode, struct file *filp,
			   unsigned int cmd , unsigned long arg);

//------------------------------------------------------------


static struct file_operations sa1100_fl_fops = {
	ioctl:		sa1100_fl_ioctl,
};

//------------------------------------------------------------

static int sa1100_frontlight_major;


/*------------------------------------------------------------
 * Set the output intensity of the frontlight.  
 *   intensity is a value between 0..100  
 *   where: 0 = full off  and    100 = full on 
 */

static int sa1100_fl_setintensity(int intensity)
{

  if(machine_is_assabet()) {
    printk(KERN_WARNING "sa1100_frontlight: intensity(%d) not available"
	   " for assabet!\n", intensity);
    return 0;
  }


				/* if the intensity is set to 0, turn off */
  if(intensity == 0) 
    sa1100_fl_disable();

				/* otherwise set the intensity and turn */
				/* on the frontlight */

				/* TODO:  Add code for dimming here */

  printk(KERN_WARNING "sa1100_frontlight: intensety(%d) not implemented yet\n",
	 intensity);

  sa1100_fl_enable();

  return 0;

}

static int sa1100_fl_enable(void)
{
  BCR_set(BCR_LIGHT_ON);
  return 0;
}

static int sa1100_fl_disable(void)
{
  BCR_clear(BCR_LIGHT_ON);
  return 0;
}


//------------------------------------------------------------

static int sa1100_fl_ioctl(struct inode * inode, struct file *filp,
				   unsigned int cmd , unsigned long arg)
{
	int ret;
	ret = (-EINVAL);
	
	switch(cmd) 
	  {
	  case _SA1100_FL_IOCTL_ON:
	    ret = sa1100_fl_enable();
	    break;
	  case _SA1100_FL_IOCTL_OFF:
	    ret = sa1100_fl_disable();
	    break;
	  case _SA1100_FL_IOCTL_INTENSITY:
	    if ((arg >=0) && (arg <= 100)) 
	      ret = sa1100_fl_setintensity(arg);
	    break;
	  default:
	    break;
	  }
	
	return ret;
}


//------------------------------------------------------------

int __init sa1100frtlight_init(void)
{
  int result;

  if ( !machine_is_assabet() )
    return -ENODEV;

  sa1100_frontlight_major = 0;	/* request a dynamic major */

  result = register_chrdev(sa1100_frontlight_major,"sa1100-fl",&sa1100_fl_fops);
  
  if (result < 0) {
    printk(KERN_WARNING "sa1100_frontlight: cant get major %d\n", 
	   sa1100_frontlight_major);
    return result;
  }
  
  if (sa1100_frontlight_major == 0)  sa1100_frontlight_major = result;

  sa1100_fl_enable();

  printk("Frontlight Driver Initialized...\n");
  
  return 0;

}
module_init(sa1100frtlight_init);

//------------------------------------------------------------

void __exit sa1100frtlight_exit(void)
{

  sa1100_fl_disable();

  unregister_chrdev(sa1100_frontlight_major, "sa1100-fl");

  printk("Frontlight Driver Unloaded\n");
}

module_exit(sa1100frtlight_exit);


