/*
 *
 * Initial Ethernet test driver
 *
 * Copyright (c) Compaq Computer Corporation, 1999
 *
 * Created 
 *        23 July 1999 Peter Dettori
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/malloc.h>

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/skbuff.h>

#include "usb_ctl.h"
#include "usb_dbg.h"

#define ETHERNET_PRODUCT_ID 0x505A

/*
 * Globals
 */
int usbd_dbglvl = DEBUG_DEFAULT;

struct usb_info_t usbd_info;

char usbd_name[16] = "usbf";
struct net_device *usbd_device;

/*
 * Ethernet Driver Entry Points
 */

/*
 * Open and close
 */
int 
usbd_open(struct net_device *dev)
{
   int retval = 0;

   MOD_INC_USE_COUNT;

   netif_start_queue(dev);

   return retval;
}

int 
usbd_release(struct net_device *dev)
{
    netif_stop_queue(dev);

    MOD_DEC_USE_COUNT;

    return 0;
}

/*
 * Configuration changes (passed on by ifconfig)
 */
int 
usbd_config(struct net_device *dev, struct ifmap *map)
{
    DEBUG_KPrint(DEBUG_ETH, ("usbd_config()\n"));
    
    /* can't act on a running interface */
    if (dev->flags & IFF_UP)
    {
        return -EBUSY;
    }
    
    /* ignore other fields */
    return 0;
}

/*
 * Transmit a packet (called by the ip stack above us)
 */
int 
usbd_tx(struct sk_buff *skb, struct net_device *dev)
{
    int retval = 0;

    DEBUG_KPrint(DEBUG_ETH, ("usbd_tx()\n"));

    if (skb == NULL)
    {
        return 0;
    }

    skb->dev = dev;
    if (usbd_info.state == USB_STATE_CONFIGURED)
    {
        dev->trans_start = jiffies; /* save the timestamp */
        if (usb_transmit_skb(skb)) {
	    retval = -EBUSY;
	}
    }
    else
    {
        dev_kfree_skb(skb);
    }
    return retval;
}

/*
 * Ioctl commands.
 */
int usbd_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
 
    DEBUG_KPrint(DEBUG_ETH, ("usbd_ioctl()\n"));
    return 0;
}

/*
 * Return statistics to the caller
 */
struct net_device_stats *
usbd_stats(struct net_device *dev)
{
    struct usb_info_t *priv =  (struct usb_info_t *) dev->priv;
    struct net_device_stats *stats=NULL;
    
    DEBUG_KPrint(DEBUG_ETH, ("usbd_stats()\n"));
    if (priv)
    {
        stats = &priv->eth_stats;
    }
    return stats;
}

/*
 * It is invoked by register_netdev()
 */
int 
usbd_probe(struct net_device *dev)
{
   /* 
    * Assign the hardware address of the board: use 
    * 40 00 00 00 00 XX, where XX is the USB address of the
    * device
    */
   dev->dev_addr[0] = 0x40;
   dev->dev_addr[1] = 0;
   dev->dev_addr[2] = 0;
   dev->dev_addr[3] = 0;
   dev->dev_addr[4] = 0;
   dev->dev_addr[5] = 1; /*usbd_info.address;*/
   
   /* we know the host mac address because it is derived from
    * the USB address.
    */
   usbd_info.host_addr[0] = 0x40;
   usbd_info.host_addr[1] = 0;
   usbd_info.host_addr[2] = 0;
   usbd_info.host_addr[3] = 0;
   usbd_info.host_addr[4] = 0xff;
   usbd_info.host_addr[5] = 1;/*usbd_info.address;*/

   dev->open            = usbd_open;
   dev->stop            = usbd_release;
   dev->set_config      = usbd_config;
   dev->hard_start_xmit = usbd_tx;
   dev->do_ioctl        = usbd_ioctl;
   dev->get_stats       = usbd_stats;

   /*
    * Then, allocate the priv field. This encloses the statistics
    * and a few private fields.
    */
   dev->priv = &usbd_info;
   usbd_info.dev = dev;

   /* clear the statistics */
   memset(&usbd_info.eth_stats, 0, sizeof(struct net_device_stats));

   ether_setup(dev);

   dev->flags &= ~IFF_BROADCAST;

   return 0;
}

static int __init
usbd_init(void)
{
    int retval=0;

    DEBUG_KPrint(DEBUG_ETH, ("Initializing raw USB driver\n"));

    /* create the device structure */
    usbd_device = kmalloc(sizeof(struct net_device), GFP_KERNEL);
    memset(usbd_device, 0, sizeof(struct net_device));
    usbd_device->next = NULL;
    usbd_device->base_addr = (__u32) 0;
    strncpy(usbd_device->name, usbd_name, IFNAMSIZ);
    usbd_device->irq = 0;
    usbd_device->init = usbd_probe;
    if ( (retval = register_netdev(usbd_device)) )
    {
        printk("usbd_eth: error %i registering device \"%s\"\n",
               retval, usbd_device->name);
    }
    /* initialise the controller, and device descriptors */
    usbctl_start();
    return retval;
}

module_init(usbd_init);

static void __exit
usbd_cleanup(void)
{
    /* release the USB */
    usbctl_release();
    
    /* free the device memory */
    if (usbd_device)
    {
        unregister_netdev(usbd_device);
        kfree(usbd_device);
    }
}

module_exit(usbd_cleanup);
