/***************************************************************************/

/*
 *	usb_ep1.c  --  
 *
 *	Copyright (C) Compaq Computer Corporation, 1998
 *
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>
#include <asm/atomic.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/irq.h>
#include <asm/system.h>

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/tcp.h>         /* struct tcphdr */
#include <linux/skbuff.h>

#include "usb_ctl.h"
#include "usb_dbg.h"

/* 
 * Implementation :
 * endpoint 1 driver will allocate a page for buffering transfers that 
 * occur before the client driver provides a buffer, after the buffer is
 * full OUT packets to endpoint 1 will be NAK'd.
 *
 * Once the client driver provides a buffer for receive characters it will 
 * have the data copied into it, at a later date it might be made to allow
 * direct dma into the client provided buffer.
 *
 * Presently the client may not queue blocks for receive, they will be 
 * rejected. This may be changed at some stage probably won't be neccessary
 * unless direct dma to the buffer is allowed though.
 *
 * It is neccessary to operate in this manner because we don't know the 
 * logical packet size of the client just the USB packet size. If we are
 * to provide packetisation then the client must ask for specific amounts
 * of data.
 */

#define DMA_OUT_COUNT_MAX 0x100

/*
 * Globals
 */
static int dmachn_rx=-1;
int dmachn_rx_inuse = 0;
struct tq_struct recv_task;

/* 
 * DMA circular buffer 
 * start points to the start of free memory for USB receives
 * end points to the end of the free area.
 * the buffer is full when start may not be advanced without
 * going past the end marker.
 *
 * The interrupt handler is allowed write permission to start and read
 * permission to end.
 * The receive_block is allowed write permission to the end and read 
 * permission on the start. The receive block should take a copy of the 
 * start in case an interrupt occurs and the actual start is advanced.
 */
#define RECEIVE_BUFFER_SIZE (4*PAGE_SIZE)
char *receive_page = NULL;
volatile char *receive_page_end = NULL;
volatile char *receive_page_start = NULL;

unsigned long receive_page_dma;
#define receive_page_vtop(v) (unsigned)(receive_page_dma + ((v)-receive_page))
#define receive_page_ptov(p) (unsigned)(receive_page + ((p)-receive_page_dma))

int receive_page_full = 0;
int reset_copy = 0;

struct sk_buff *recv_skb;
unsigned short pktlength = 0;

/* 
 * add size to the ptr ensuring that the result is aligned
 * to align. Used for advancing the circular receive buffer.
 */
#define ADVANCE(ptr, size, align)                              \
        {                                                      \
	        if ((size) % (align))                              \
	        {                                                  \
		        (size) = (size) + (align) - (size % align);    \
	        }                                                  \
	        (ptr) += (size);                                   \
	    }                                                

/*
 * Internal Prototypes
 */
void ep1_copy_task(void *data);

/*
 * Macros
 */
#ifndef MIN
#define MIN(a,b)	((a) < (b) ? (a) : (b))
#endif

static volatile unsigned int *UDCCS1_reg, *UDCDR_reg;

/*
 * called by the udc control component after a packet is received, or at
 * bus enumeration. Sets up the DMA channels to use the given packet 
 * for receive.
 *
 * pkt       : the packet to use for receive.
 * returns   : 0 if ok, -1 if a pkt is already queued for recv.
 */
int
ep1_recv(void)
{
    int status = -1;

    CHECK_ADDRESS;
    
    /* check that we are initialised by the local record of the 
     * dma channel in use, and that there isn't a packet in use already.
     */
    if ( (dmachn_rx >= 0) && (!dmachn_rx_inuse) && (!receive_page_full) )
    {
unsigned long flags;
save_flags(flags);
cli();
        /* the start page ptr should be ok to start dmaing. */
        dmachn_rx_inuse = 1;
        set_dma_addr(dmachn_rx, receive_page_vtop(receive_page_start));
        set_dma_count(dmachn_rx, usbd_info.rx_pktsize);
        enable_dma(dmachn_rx); 
restore_flags(flags);
        status = 0;
    }
    CHECK_ADDRESS;
    return status;
}

/*
 * reset any transfers in progress and clear the buffer
 * to start at the beginning again.
 *
 * returns : 0 if ok, -1 otherwise
 */
int
ep1_reset(void)
{
    int status = 0;
    CHECK_ADDRESS;
    if (dmachn_rx >= 0)
    {
        disable_dma(dmachn_rx);
        /* reset dma */
        set_dma_mode_sa1100(dmachn_rx, 0);
        set_dma_mode_sa1100(dmachn_rx, DMA_UDC_RECEIVE);
        dmachn_rx_inuse = 0;
    }
    /* stall and unstall endpoint to reset data toggle */
    do
    {
        *(UDCCS1) |= UDCCS1_FST;
    } while (!(*(UDCCS1) & UDCCS1_FST));
    do
    {
        *(UDCCS1) &= ~UDCCS1_FST;
    } while (*(UDCCS1) & UDCCS1_FST);

    /* now start things up again. */
    if (recv_task.sync == 0)
    {
        if (usbd_info.state ==  USB_STATE_CONFIGURED)
        {
            reset_copy = 1;
            receive_page_end = receive_page_start;
            receive_page_full = 0;
        }
    }
    else
    {
        printk("USB RECV : Copy task running\n");
    }

    ep1_recv();

    CHECK_ADDRESS;
    return status;
}

/*
 * Initialise the DMA system to use the given chn for receiving
 * packets over endpoint 1.
 *
 * chn     : the dma channel to use, or -1 if the receive engine should
 *           continue using the one it already has.
 * returns : 0 if ok, -1 if channel couldn't be used.
 */
int 
ep1_init(int chn)
{
    int status = 0;

    if (dmachn_rx >= 0)
	return status;

    dmachn_rx = chn;
    dmachn_rx_inuse = 0;

    /* allocate a page for the receive buffer */
    if (!receive_page)
    {
        /* allocate the buffer and make it non cacheable. */
#if 0
        receive_page = (char *)__get_dma_pages(GFP_KERNEL, 2);
        allocate_vspace((ulong)virt_to_phys((ulong)receive_page),
                        PGDIR_SIZE, 1, 0);
        DEBUG_KPrint(DEBUG_SEND, ("receive page %p\n", receive_page));
#else
	receive_page = consistent_alloc(GFP_KERNEL, RECEIVE_BUFFER_SIZE,
					(dma_addr_t *)&receive_page_dma);
        DEBUG_KPrint(DEBUG_SEND, ("receive page %p (size %ld); dma %p\n",
				  receive_page, RECEIVE_BUFFER_SIZE,
				  (dma_addr_t *)receive_page_dma));
#endif
    }
    if (!receive_page)
    {
        panic("usb (usb_ep1.c %d) : couldn't allocate a receive buffer\n", 
              __LINE__);
    }
    receive_page_start = receive_page_end = receive_page;

    if (recv_skb)
    {
        dev_kfree_skb(recv_skb);
    }
    recv_skb = NULL;
    pktlength = 0;
    
    UDCCS1_reg = UDCCS1;
    UDCDR_reg = UDCDR;
    
    recv_task.routine =  ep1_copy_task;
    recv_task.data = (void *) NULL;
    recv_task.sync = 0;
    
    ep1_recv();
    CHECK_ADDRESS;

    return status;
}

/*
 * Endpoint 1 interrupts will be directed here.
 *
 * status : the contents of the UDC Status/Interrupt Register.
 */
void inline
ep1_int_hndlr(unsigned int status)
{
    unsigned int ep1_status, pkt_error; 
    unsigned int dma_address;
    unsigned char *dmabuf;
    unsigned int udc_fifo;
    int idx;

    if (UDCCS1_reg == 0 || receive_page_start == 0) {
	return;
    }

    ep1_status = *(UDCCS1_reg);
    pkt_error = ep1_status & UDCCS1_RPE;

    if (ep1_status & UDCCS1_RPC)
    { 
	if (pkt_error)
	  printk("ep1_int_hndlr() UDCCS1_RPC, pkt_error %d, %x\n",
		 pkt_error, ep1_status);

        disable_dma(dmachn_rx);
        dma_address = get_dma_address_sa1100(dmachn_rx);
        dmabuf = (char *) receive_page_ptov(dma_address);

        if (!pkt_error)
        {
            /* alright.. we have a packet empty the FIFO */ 
            while (ep1_status & UDCCS1_RNE)
            {
                /* have data in the fifo so 
                 * pull it out .
                 */
                udc_fifo = *(UDCDR_reg);
                *dmabuf = (char) udc_fifo;
                dmabuf++;
                ep1_status = *(UDCCS1_reg);
            }
        } else {
#if 0
	        disable_dma(dmachn_rx);
		/* reset dma */
		set_dma_mode_sa1100(dmachn_rx, 0);
		set_dma_mode_sa1100(dmachn_rx, DMA_UDC_RECEIVE);
#endif
	}
    }

    /* clear SST */
    idx=0;
    while ((ep1_status & UDCCS1_SST) && (idx<1000))
    {
        idx++;
        *(UDCCS1_reg) = UDCCS1_SST;
        ep1_status = *(UDCCS1_reg);
    }
    
    if (idx>=1000)
    {
        printk("USB RECV : SST loop timed out\n");
    }
    CHECK_ADDRESS;

    /* clear RPC */
    idx = 0;
    do
    {
        idx++;
        *(UDCCS1_reg) = UDCCS1_RPC;
    } while ( (*(UDCCS1_reg) & UDCCS1_RPC) && (idx < 1000));

    if (idx>=1000)
    {
        printk("USB RECV : RPC loop timed out\n");
    }

    dmachn_rx_inuse = 0;
    
    if (!pkt_error)
    {
        usbd_info.usb_stats.outpkts++;
        
        /* advance the start pointer and go again
         * unless have run out of receive buffer, in which case
         * start NAKing packets....
         */
        ADVANCE(receive_page_start, usbd_info.rx_pktsize,
                usbd_info.rx_pktsize);
        if (receive_page_start >= (receive_page + RECEIVE_BUFFER_SIZE))
        {
            receive_page_start = receive_page;
        }
        
        /* if advancing start makes start == end then the buffer is full */
        if (receive_page_start != receive_page_end)
        {
            ep1_recv();
        }
        else
        {
            DEBUG_KPrint(DEBUG_RECV, ("receive_page_full\n"));
            receive_page_full = 1;
            /* don't want interrupts for this now */
            disable_irq(IRQ_Ser0UDC);
        }
        /* 
         * queue the copy task to run, if it is not presently 
         * running. 
         */
        if (recv_task.sync == 0)
        {
            queue_task(&recv_task, &tq_immediate); 
        }
        mark_bh(IMMEDIATE_BH);
    }
    else
    {
        usbd_info.usb_stats.outpkt_errs++;
        if (ep1_recv() != 0)
        {
            printk("USB RECV : can't restart, ep1 no good\n");
        }
    }
    CHECK_ADDRESS;
}

/*
 * Checks if any received bytes may be copied out of the internal buffer
 * into the receive buffer. After doing so it trys to start another receive
 * in case that the internal buffer was full.
 *
 * Should be called on the schedule wait queue and may be put there by 
 * the block receive function or the interrupt handler.
 *
 * When the receive block is full the callback will be called.
 *
 * data   : unused, globals variables hold our state.
 */
void 
ep1_copy_task(void *data)
{
    char *tmp_start;
    int copy_size;
    int loops = 0;
    unsigned long flags;
    unsigned char *mac_header;
    int idx;

    CHECK_ADDRESS;

    /* flush the cache of the receive page. */
    /*
    processor.u.armv3v4._flush_cache_area((unsigned int) receive_page, 
					  RECEIVE_BUFFER_SIZE, 0);
    */
flush_cache_all();

    /* we have been reset... */
    if (reset_copy)
    {
        /* free the current skb */
        if (recv_skb)
        {
            dev_kfree_skb(recv_skb);
        }
        recv_skb = NULL;
        pktlength = 0;
        reset_copy = 0;
    }
        
    /*
     * this process may be interrupted and another packet received,
     * thus we must copy the start buffer ptr, which is the only one that 
     * interrupt handler may changed. The while loop will handle the cases 
     * where an interrupt occurred.
     *
     */
    save_flags(flags);
    cli();
    do
    {
        loops ++;
        /* the interrupt handler is the only function allowed to change the
         * page start, we take a copy in case a packet arrives and it
         * changes.
         */
        tmp_start = receive_page_start;
        restore_flags(flags);

        /* this confusing logic should mean that we process a block iff
         * the recv_block is valid, and the receive buffer is not empty.
         */
        if (tmp_start != receive_page_end || receive_page_full)
        {
            /* if packet length is 0, allocate a new skb */
            if (pktlength == 0)
            {
                pktlength = (receive_page_end[0] & 0xff);
                pktlength += (receive_page_end[1] & 0xff) << 8;

                if (pktlength > 1504 || pktlength == 0)
                {
                    printk("USB RECV : bad packet length %d (%02x %02x)\n",
			   pktlength,
			   receive_page_end[0], receive_page_end[1]);

                    udc_stallep(1);
                }
                else
                {
                    recv_skb = dev_alloc_skb(1518);
                    /* want the ip packet aligned on a 16 byte boundary so
                     * reserve 12 bytes, 2 empty, and 10 for the address.
                     * 2 bytes are used for packet length on the USB.
                     */
                    skb_reserve(recv_skb, 12);
                }
            }

            if (pktlength && recv_skb)
            {
                /* copy the internal buffer into the receive block*/
                copy_size = MIN(pktlength-recv_skb->len, 
                                (receive_page+RECEIVE_BUFFER_SIZE) -
                                receive_page_end);
                if (tmp_start > receive_page_end)
                {
                    copy_size = MIN(copy_size, 
                                    tmp_start-receive_page_end);
                }
            
                if (copy_size)
                {
                    if (skb_tailroom(recv_skb)<copy_size)
                    {
                        printk("USB RECV : no room skb %p"
                               " length %d copy %d\n",
                               recv_skb, recv_skb->len, copy_size);
                    }
                    memcpy(skb_put(recv_skb, copy_size),
                           (void *)receive_page_end, 
                           copy_size);
                    /* XXX mark as read */
                    receive_page_end[0] = 0xde;
                    /* advance the end marker */
                    save_flags(flags);
                    cli();
                    ADVANCE(receive_page_end, copy_size,
                            usbd_info.rx_pktsize);
                    if (receive_page_full)
                    {
                        receive_page_full = 0;
                        enable_irq(IRQ_Ser0UDC);
                    }
                    restore_flags(flags);
                    /* if we have advanced the end marker we cannot 
                     * have a full buffer 
                     */
                    if (receive_page_end >= (receive_page+RECEIVE_BUFFER_SIZE))
                    {
                        receive_page_end = receive_page;
                        /* in this case have gone back to the start of 
                         * the buffer. So check if there is more to copy.
                         */
                        
                        /* must flush the cache, to make sure reading the right
                         * stuff.
                         */
/*                        processor.u.armv3v4._flush_cache_area(
                            (unsigned int) receive_page, 
                            RECEIVE_BUFFER_SIZE, 0);*/

                        if (recv_skb->len != pktlength &&
                            tmp_start != receive_page_end)
                        {
                            /* can copy more data... */
                            copy_size = MIN(pktlength - recv_skb->len, 
                                            tmp_start - receive_page_end);

                            if (skb_tailroom(recv_skb) < copy_size)
                            {
                                printk("USB RECV : no room skb %p"
                                       " length %d copy %d\n",
                                       recv_skb, recv_skb->len, copy_size);
                            }
                            memcpy(skb_put(recv_skb, copy_size), 
                                   (void *)receive_page_end, 
                                   copy_size);
                            /* advance the end marker */
                            save_flags(flags);
                            cli();
                            ADVANCE(receive_page_end, copy_size, 
                                    usbd_info.rx_pktsize);
                            if (receive_page_full)
                            {
                                receive_page_full = 0;
                                enable_irq(IRQ_Ser0UDC);   
                            }
                            restore_flags(flags);
                        }
                    }
                }
                        
                /* if the block is complete, do the callback */
                if (recv_skb->len == pktlength)
                {
                    /* eventually this will pass this up but not now.. */
		  if (netif_running(usbd_info.dev))
                    {
                        /* put the mac addresses back in, over the top of the
                         * size that was sent over.
                         */
                        mac_header = skb_push(recv_skb, (2*ETH_ALEN)-2);
                        for (idx=0;idx<ETH_ALEN;idx++)
                        {
                            mac_header[idx] = usbd_info.dev->dev_addr[idx];
                            mac_header[idx+ETH_ALEN] = usbd_info.host_addr[idx];
                        }
                        recv_skb->dev = usbd_info.dev;
                        recv_skb->protocol = eth_type_trans(recv_skb,
                                                            usbd_info.dev);
                        recv_skb->ip_summed = CHECKSUM_UNNECESSARY;

                        usbd_info.eth_stats.rx_packets++;
                        netif_rx(recv_skb);
                        recv_skb = NULL;
                    }
                    else
                    {
                        dev_kfree_skb(recv_skb);
                        recv_skb = NULL;
                    }
                    pktlength = 0;
                }
            }
        }
        save_flags(flags);
        cli();
    } while ( (receive_page_start != receive_page_end || receive_page_full)
              && loops < 1000 );
    ep1_recv();
    restore_flags(flags);
    CHECK_ADDRESS;
}




