/*
*
* Driver for H3600 Extension Packs
*
* Copyright 2000 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author: Jamey Hicks.
*
*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>

/* SA1100 serial defines */
#include <asm/arch/hardware.h>
#include <asm/arch/serial_reg.h>
#include <asm/arch/irqs.h>
#include <asm/arch/h3600-sleeve.h>

#include <linux/h3600_ts.h>

static void msleep(unsigned int msec)
{
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout( (msec * HZ + 999) / 1000);
}

static int h3600_sleeve_major = 0;

struct {
        char start_of_id; /* 0xaa */
        int data_len;
        char version;
        short vendor_id;
        short device_id;
} sleeve_header;

struct proc_dir_entry *proc_sleeve_dir = NULL;

static int h3600_sleeve_proc_read_eeprom(char *page, char **start, off_t off, int count, int *eof, void *data)
{
        char *p = page;
        int len = 0;
        int i;
        char mark = 0;
        int offset = 1;

        h3600_iic_read(0, &mark, 1);
        h3600_iic_read(1, (char*)&len, 4);
        if (mark != 0xaa || len == 0xFFFFFFFF)
                return 0;
        printk(__FUNCTION__ ": len=%d\n", len);
        for (i = 0; i < len; i+=8) {
                h3600_iic_read(offset, p, 8);
                p += 8;
                offset += 8;
        }
        if (len != (p - page))
                h3600_iic_read(offset, p, len-(p - page));
        return len;
}

static int h3600_sleeve_proc_read_backpaq(char *page, char **start, off_t off, int count, int *eof, void *data)
{
        char *p = page;
        int i;
        p += sprintf(p, "SM2: %08x  %#08lx\n", 0xf1000000, *(long *)0xf1000000);
        for (i = 0; i < 0x14; i += 4)
          p += sprintf(p, "SM4: %#08x  %#08lx\n", (0xf2000000 + i), *(long *)(0xf2000000 + i));
        p += sprintf(p, "PCMCIA SKT1 IO: %#08lx\n", *(long *)0xf6000000);
        p += sprintf(p, "PCMCIA SKT2 IO: %#08lx\n", *(long *)0xf7000000);
        return p-page;
}

int h3600_sleeve_proc_attach_device(struct sleeve_dev *dev)
{
        dev->procent = NULL;
	return 0;
}

int h3600_sleeve_proc_detach_device(struct sleeve_dev *dev)
{
	struct proc_dir_entry *e;

	if ((e = dev->procent)) {
		if (e->count)
			return -EBUSY;
		remove_proc_entry(e->name, proc_sleeve_dir);
		dev->procent = NULL;
	}
	return 0;
}


const struct sleeve_device_id *h3600_sleeve_match_device(const struct sleeve_device_id *ids, struct sleeve_dev *dev)
{
        while (ids->vendor || ids->subvendor) {
                if ((ids->vendor == SLEEVE_ANY_ID || ids->vendor == dev->vendor) &&
		    (ids->device == SLEEVE_ANY_ID || ids->device == dev->device))
			return ids;
                ids++;
        }
        return NULL;
}

static LIST_HEAD(sleeve_drivers);

static struct sleeve_device_id *sleeve_device_id;
static struct sleeve_dev *sleeve_dev;

int h3600_sleeve_register_driver(struct sleeve_driver *drv)
{
 	list_add_tail(&drv->node, &sleeve_drivers);
	if (sleeve_device_id->device != SLEEVE_NO_DEVICE ) {
                const struct sleeve_device_id *did = NULL;
                if ((did = h3600_sleeve_match_device(&drv->id_table[0], sleeve_dev)) != NULL) {
                        if (drv->probe) {
                                printk(__FUNCTION__ ": about to call %p\n", drv->probe);
                                drv->probe(sleeve_dev, did);
                        }
                }
	}
        return 0; 
}
EXPORT_SYMBOL(h3600_sleeve_register_driver);

void
h3600_sleeve_unregister_driver(struct sleeve_driver *drv)
{
        list_del(&drv->node);
        if (sleeve_dev->driver == drv) {
                if (drv->remove)
                        drv->remove(sleeve_dev);
                sleeve_dev->driver = NULL;
        }
}
EXPORT_SYMBOL(h3600_sleeve_unregister_driver);

static int
h3600_sleeve_announce_device(struct sleeve_driver *drv, struct sleeve_dev *dev)
{
	const struct sleeve_device_id *id;

	if (drv->id_table) {
		id = h3600_sleeve_match_device(drv->id_table, dev);
		if (!id)
			return 0;
	} else
		id = NULL;
	if (drv->probe(dev, id) >= 0) {
		dev->driver = drv;
		return 1;
	}
	return 0;
}

void
h3600_sleeve_insert_device(struct sleeve_dev *dev)
{
        struct list_head *ln;
        printk(KERN_CRIT __FUNCTION__ "-1- \n");
#ifdef CONFIG_PROC_FS
        h3600_sleeve_proc_attach_device(dev);
#endif
        for (ln=sleeve_drivers.next; ln != &sleeve_drivers; ln=ln->next) {
                struct sleeve_driver *drv = list_entry(ln, struct sleeve_driver, node);
                if (h3600_sleeve_announce_device(drv, dev))
                        break;
        }
}
EXPORT_SYMBOL(h3600_sleeve_insert_device);

void
h3600_sleeve_remove_device(struct sleeve_dev *dev)
{
	if (dev->driver) {
		if (dev->driver->remove)
			dev->driver->remove(dev);
		dev->driver = NULL;
	}
#if 0
	h3600_sleeve_free_resources(dev);
#endif
#ifdef CONFIG_PROC_FS
	h3600_sleeve_proc_detach_device(dev);
#endif
}
EXPORT_SYMBOL(h3600_sleeve_remove_device);


DECLARE_WAIT_QUEUE_HEAD(h3600_sleeve_wait);
static int h3600_sleeve_active = 1;
static int h3600_sleeve_thread(void *x)
{
        struct task_struct *tsk = current;

        printk(KERN_CRIT __FUNCTION__ "-1- \n");

        tsk->session = 1;
        tsk->pgrp = 1;
        strcpy(tsk->comm, "k_h3600_sleeve");
        sigfillset(&tsk->blocked);
	exit_mm(tsk);
	exit_files(tsk);

        printk(KERN_CRIT __FUNCTION__ "-2- \n");

        while (h3600_sleeve_active) {
                int opt_ndet;

                printk(KERN_CRIT __FUNCTION__ "-3- \n");
                /* debounce */
                msleep(100);
                opt_ndet = (GPLR & GPIO_BITSY_OPT_DET);

                printk(KERN_CRIT __FUNCTION__ "-4- \n");

                if (!opt_ndet) {
                printk(KERN_CRIT __FUNCTION__ "-5- \n");
                        set_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON | EGPIO_BITSY_OPT_ON);
                printk(KERN_CRIT __FUNCTION__ "-6- \n");
			msleep(100);
                printk(KERN_CRIT __FUNCTION__ "-7- \n");
			h3600_iic_read(6, (char*)&sleeve_dev->vendor, 2);
			h3600_iic_read(8, (char*)&sleeve_dev->device, 2);
			printk(KERN_CRIT __FUNCTION__ ": x=%p opt_det=%#x vendorid=%#x devid=%#x\n", x, !opt_ndet, sleeve_dev->vendor, sleeve_dev->device);
			
                printk(KERN_CRIT __FUNCTION__ "-8- \n");
                        h3600_sleeve_insert_device(sleeve_dev);

                } else {
                printk(KERN_CRIT __FUNCTION__ "-9- sleeve_dev->driver=%p\n", sleeve_dev->driver);
                        if (sleeve_dev->driver && sleeve_dev->driver->remove) {
                printk(KERN_CRIT __FUNCTION__ "-10- sleeve_dev->driver->remove=%p\n", sleeve_dev->driver->remove);
                                sleeve_dev->driver->remove(sleeve_dev);
                        }
                        memset(sleeve_dev, 0, sizeof(struct sleeve_dev));
                        clr_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON | EGPIO_BITSY_OPT_ON);
                }

                interruptible_sleep_on(&h3600_sleeve_wait);
        }
        return 0;
}

static void h3600_sleeve_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
        int opt_ndet = (GPLR & GPIO_BITSY_OPT_DET);
        printk(__FUNCTION__ ": OPT_DET=%#08x\n", !opt_ndet);
        /* now clear the interrupt */
        GEDR = GPIO_BITSY_OPT_DET;
        wake_up_interruptible(&h3600_sleeve_wait);
}


int __init h3600_sleeve_init_module(void)
{
	int result;

        if (!machine_is_bitsy())
          return -ENODEV;

        /* register our character device */
#if 0
        printk(__FUNCTION__ ": registering char device\n");
        result = register_chrdev(0, MOD_NAME, &ts_fops);
        if (result < 0) {
                printk(__FUNCTION__ ": can't get major number\n");
                return result;
        }

	h3600_sleeve_major = result;
#endif

        clr_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON | EGPIO_BITSY_OPT_ON);

        sleeve_device_id = (struct sleeve_device_id *)kmalloc(sizeof(struct sleeve_device_id), GFP_KERNEL);
        sleeve_dev = (struct sleeve_dev *)kmalloc(sizeof(struct sleeve_dev), GFP_KERNEL);
        if (sleeve_device_id == NULL || sleeve_dev == NULL) {
          printk(__FUNCTION__ ": kmalloc failed sleeve_device_id=%p sleeve_dev=%p\n", sleeve_device_id, sleeve_dev);
          return -ENOMEM;
        }
        memset(sleeve_device_id, 0, sizeof(struct sleeve_device_id));
        memset(sleeve_dev, 0, sizeof(struct sleeve_dev));

        set_GPIO_IRQ_edge( GPIO_BITSY_OPT_DET, GPIO_BOTH_EDGES );
	result = request_irq(IRQ_GPIO_BITSY_OPT_DET, h3600_sleeve_interrupt, SA_SHIRQ | SA_INTERRUPT | SA_SAMPLE_RANDOM,
                             "h3600_sleeve", h3600_sleeve_interrupt);

#ifdef CONFIG_PROC_FS
        proc_sleeve_dir = proc_mkdir("sleeve", proc_bus);
        if (proc_sleeve_dir) {
                create_proc_read_entry("eeprom", 0, proc_sleeve_dir, h3600_sleeve_proc_read_eeprom, 0);
                create_proc_read_entry("backpaq", 0, proc_sleeve_dir, h3600_sleeve_proc_read_backpaq, 0);
        }
#endif

        printk(KERN_CRIT __FUNCTION__ ": creating kernel thread at %p\n", h3600_sleeve_thread);
        /* this thread calls the handlers, so that they are not called in interrupt context */
        kernel_thread ( h3600_sleeve_thread, NULL, 
                        CLONE_FS | CLONE_FILES | CLONE_SIGHAND );
        wake_up_interruptible( &h3600_sleeve_wait );

	if (!result)
	{	
                printk(KERN_CRIT "init_module successful init major= %d irq=%d\n",
                       h3600_sleeve_major, IRQ_GPIO_BITSY_OPT_DET);

	}
	else
		printk(__FUNCTION__ ": error from irq %d\n", result);
	return 0;
}

void h3600_sleeve_cleanup_module(void)
{
	printk(__FUNCTION__ ":\n");

        free_irq(IRQ_GPIO_BITSY_OPT_DET, h3600_sleeve_interrupt);

#ifdef CONFIG_PROC_FS
        if (proc_sleeve_dir) {
                remove_proc_entry("eeprom", proc_sleeve_dir);
                remove_proc_entry("sleeve", proc_bus);
                remove_proc_entry("backpaq", proc_bus);
        }
#endif

#if 0
	unregister_chrdev(, MOD_NAME);
#endif
}

module_init(h3600_sleeve_init_module);
module_exit(h3600_sleeve_cleanup_module);

