/*
 * Common Flash Interface support:
 *   AMD & Fujitsu Extended Vendor Command Set (ID 0x0002)
 *
 * Copyright (C) 2000 Crossnet Co. <info@crossnet.co.jp>
 *
 * 2_by_8 routines added by Simon Munton
 *
 * This code is GPL
 *
 * $Id: cfi_cmdset_0002.c,v 1.8 2000/10/15 21:53:21 dwmw2 Exp $
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <asm/io.h>
#include <asm/byteorder.h>

#include <linux/errno.h>
#include <linux/malloc.h>
#include <linux/delay.h>
#include <linux/mtd/map.h>
#include <linux/mtd/cfi.h>

#if LINUX_VERSION_CODE < 0x20300
#define set_current_state(x) current->state = (x);
#endif

static int cfi_amdext_read (struct mtd_info *, loff_t, size_t, size_t *, u_char *);
static int cfi_amdext_write(struct mtd_info *, loff_t, size_t, size_t *, const u_char *);
static int cfi_amdext_erase(struct mtd_info *, struct erase_info *);
static void cfi_amdext_sync (struct mtd_info *);
static int cfi_amdext_suspend (struct mtd_info *);
static void cfi_amdext_resume (struct mtd_info *);

static void cfi_amdext_destroy(struct mtd_info *);

void cfi_cmdset_0002(struct map_info *, int, unsigned long);
EXPORT_SYMBOL_NOVERS(cfi_cmdset_0002);

struct mtd_info *cfi_amdext_setup (struct map_info *);

void cfi_cmdset_0002(struct map_info *map, int primary, unsigned long base)
{
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
//	struct cfi_pri_intelext *extp;

	__u16 adr = primary?cfi->cfiq.P_ADR:cfi->cfiq.A_ADR;

	printk(" Amd/Fujitsu Extended Query Table at 0x%4.4X\n", adr);


	/* If there was an old setup function, decrease its use count */
	if (cfi->cmdset_setup)
		put_module_symbol((unsigned long)cfi->cmdset_setup);
	if (cfi->cmdset_priv)
		kfree(cfi->cmdset_priv);

	for (i=0; i< cfi->numchips; i++) {
		cfi->chips[i].word_write_time = cfi->cfiq.WordWriteTimeoutTyp;
		cfi->chips[i].buffer_write_time = cfi->cfiq.BufWriteTimeoutTyp;
		cfi->chips[i].erase_time = cfi->cfiq.BlockEraseTimeoutTyp;
	}		
		

	cfi->cmdset_setup = cfi_amdext_setup;
//	cfi->cmdset_priv = extp;
	MOD_INC_USE_COUNT; /* So the setup function is still there 
			    * by the time it's called */
	
	return;
}

struct mtd_info *cfi_amdext_setup(struct map_info *map)
{
	struct cfi_private *cfi = map->fldrv_priv;
	struct mtd_info *mtd;

	mtd = kmalloc(sizeof(*mtd), GFP_KERNEL);
	printk("number of CFI chips: %d\n", cfi->numchips);

	if (!mtd) {
	  printk("Failed to allocate memory for MTD device\n");
	  kfree(cfi->cmdset_priv);
	  return NULL;
	}

	memset(mtd, 0, sizeof(*mtd));
	mtd->priv = map;
	mtd->type = MTD_NORFLASH;
	/* Also select the correct geometry setup too */ 
	mtd->size = (1 << cfi->cfiq.DevSize) * cfi->numchips * cfi->interleave;
	mtd->erasesize = 0x10000 * cfi->interleave; /* FIXME, assumes sectors of 64KB */

	switch (map->buswidth)
	{
	case 1:
	case 2:
	case 4:
		mtd->erase = cfi_amdext_erase;
		mtd->read = cfi_amdext_read;
		mtd->write = cfi_amdext_write;
		break;

	default:
	        printk("Unsupported buswidth\n");
		kfree(mtd);
		kfree(cfi->cmdset_priv);
		return NULL;
		break;
	}
	mtd->sync = cfi_amdext_sync;
	mtd->suspend = cfi_amdext_suspend;
	mtd->resume = cfi_amdext_resume;
	mtd->flags = MTD_CAP_NORFLASH;
	map->fldrv_destroy = cfi_amdext_destroy;
	mtd->name = map->name;
	return mtd;
}

static inline int do_read_onechip(struct map_info *map, struct flchip *chip, loff_t adr, size_t len, u_char *buf)
{
	DECLARE_WAITQUEUE(wait, current);
	unsigned long timeo = jiffies + HZ;

 retry:
	spin_lock_bh(chip->mutex);

	if (chip->state != FL_READY){
	        printk("Waiting for chip to read, status = %d\n", chip->state);
		set_current_state(TASK_INTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
                
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);

		if(signal_pending(current))
			return -EINTR;

		timeo = jiffies + HZ;

		goto retry;
	}	

	adr += chip->start;

	chip->state = FL_READY;

	map->copy_from(map, buf, adr, len);

	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

	return 0;
}

static int cfi_amdext_read (struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long ofs;
	int chipnum;
	int ret = 0;

	/* ofs: offset within the first chip that the first read should start */

	chipnum = (from >> cfi->chipshift);
	chipnum /= (cfi->interleave);
	ofs = from - (chipnum <<  cfi->chipshift) * (cfi->interleave);

	*retlen = 0;

	while (len) {
		unsigned long thislen;

		if (chipnum >= cfi->numchips)
			break;

		if (((len + ofs -1) >> cfi->chipshift) / (cfi->interleave))
			thislen = (1<<cfi->chipshift) * (cfi->interleave) - ofs;
		else
			thislen = len;

		ret = do_read_onechip(map, &cfi->chips[chipnum], ofs, thislen, buf);
		if (ret)
			break;

		*retlen += thislen;
		len -= thislen;
		buf += thislen;

		ofs = 0;
		chipnum++;
	}
	return ret;
}

static __inline__ void amd_send_cmd(u_char cmd, __u32 ofs, __u32 addr, struct map_info *map, u_char interleave, int type)
{
	cfi_send_gen_cmd(cmd, ofs, addr, map, interleave, type, NULL);
}

static __inline__ void amd_send_cmd_at(u_char cmd, __u32 addr, struct map_info *map, u_char interleave)
{
	__u32 val;

	val = cfi_build_cmd(cmd, map, interleave);
	switch(map->buswidth) {
#ifdef CFIDEV_BUSWIDTH_1
	case CFIDEV_BUSWIDTH_1:
		map->write8(map, val, addr);
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_2
	case CFIDEV_BUSWIDTH_2:
		map->write16(map, cpu_to_le16(val), addr);
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_4
	case CFIDEV_BUSWIDTH_4:
		map->write32(map, cpu_to_le32(val), addr);
		break;
#endif
	}
}

static __inline__ __u32 amd_write_val(__u32 *val, __u32 addr, struct map_info *map)
{
	__u32 retval;

	switch(map->buswidth) {
#ifdef CFIDEV_BUSWIDTH_1
	case CFIDEV_BUSWIDTH_1:
		retval = *((__u8 *) val);
		map->write8(map, retval, addr);
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_2
	case CFIDEV_BUSWIDTH_2:
		retval = *((__u16 *) val);
		map->write16(map, retval, addr);
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_4
	case CFIDEV_BUSWIDTH_4:
		retval = *val;
		map->write32(map, retval, addr);
		break;
	}
#endif
	return retval;
}

static __inline__ __u32 amd_read_val(__u32 addr, struct map_info *map)
{
	__u32 val;

	switch(map->buswidth) {
#ifdef CFIDEV_BUSWIDTH_1
	case CFIDEV_BUSWIDTH_1:
		val = map->read8(map, addr);
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_2
	case CFIDEV_BUSWIDTH_2:
		val = map->read16(map, addr);
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_4
	case CFIDEV_BUSWIDTH_4:
		val = map->read32(map, addr);
		break;
#endif
	}
	return val;
}

static int do_write_oneword(struct map_info *map, struct flchip *chip, unsigned long adr, __u32 *datum, int fast)
{
	unsigned long timeo = jiffies + HZ;
	unsigned int Last[4];
	unsigned long Count = 0;
	struct cfi_private *cfi = map->fldrv_priv;
	DECLARE_WAITQUEUE(wait, current);
	int ret = 0;
	__u32 val;

 retry:
	spin_lock_bh(chip->mutex);

	if (chip->state != FL_READY){
	        printk("Waiting for chip to write, status = %d\n", chip->state);
		set_current_state(TASK_INTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
                
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);
		printk("Wake up to write:\n");
		if(signal_pending(current))
			return -EINTR;

		timeo = jiffies + HZ;

		goto retry;
	}	

	chip->state = FL_WRITING;

	adr += chip->start;
	ENABLE_VPP(map);
	if (fast) { /* Unlock bypass */
		amd_send_cmd(0xA0, 0x0, chip->start, map, cfi->interleave, cfi->device_type);
	}
	else {
		amd_send_cmd(0xAA, 0x555, chip->start, map, cfi->interleave, cfi->device_type);
		amd_send_cmd(0x55, 0x2AA, chip->start, map, cfi->interleave, cfi->device_type);
		amd_send_cmd(0xA0, 0x555, chip->start, map, cfi->interleave, cfi->device_type);
	}
	val = amd_write_val(datum, adr, map);

	spin_unlock_bh(chip->mutex);
	udelay(chip->word_write_time);
	spin_lock_bh(chip->mutex);

	Last[0] = amd_read_val(adr, map);
	Last[1] = amd_read_val(adr, map);
	Last[2] = amd_read_val(adr, map);

	for (Count = 3; Last[(Count - 1) % 4] != Last[(Count - 2) % 4] && Count < 10000; Count++){
		udelay(10);
		
		Last[Count % 4] = amd_read_val(adr, map);
	}
	
	if (Last[(Count - 1) % 4] != val){
		amd_send_cmd(0xF0, adr, chip->start, map, cfi->interleave, cfi->device_type);
		DISABLE_VPP(map);
		ret = -EIO;
	}       
	DISABLE_VPP(map);
	chip->state = FL_READY;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);
	
	return ret;
}

#ifdef CFIDEV_BUSWIDTH_2
static int cfi_amdext_write_byte_16(struct map_info *map, struct flchip *chip, unsigned long ofs, u_char data)
{
	/* destination doesn't start on word boundary, so write the first odd byte */
	__u16 tmp;
	int ret;

	ofs += chip->start;

	if (ofs & 1) {
		tmp = map->read16(map, ofs - 1);
		tmp = (le16_to_cpu(tmp) & 0x00FF) | (data << 8);
		tmp = cpu_to_le16(tmp);
	}
	else {
		tmp = map->read16(map, ofs);
		tmp = (le16_to_cpu(tmp) & 0xFF00) | data;
		tmp = cpu_to_le16(tmp);
	}

	ret = do_write_oneword(map, chip, ofs - chip->start, (__u32 *) &tmp, 0);

	return ret;
}
#endif /* CFIDEV_BUSWIDTH_2 */

#ifdef CFIDEV_BUSWIDTH_4
static int cfi_amdext_write_bytes_32(struct map_info *map, struct flchip *chip, unsigned long ofs, 
				     const u_char *buf, size_t len)
{
	__u32 tmp;
	int size;
	int ret = 0;

	ofs += chip->start;

	while (len > 0 && !ret) {
		/* if write address is aligned */
		if ((ofs & 0x03) == 0) {
			tmp = map->read32(map, ofs);
			tmp = be32_to_cpu(tmp);
			memcpy((u_char *)&tmp, buf, len);
			tmp = cpu_to_be32(tmp);
			size = len;
		}
		else if ((ofs & 0x03) == 3) {
			tmp = map->read32(map, ofs - 3);
			tmp = (be32_to_cpu(tmp) & 0xFFFFFF00) | buf[0];
			tmp = cpu_to_be32(tmp);
			size = 1;
		}
		else if ((ofs & 0x03) == 2) {
			tmp = map->read32(map, ofs - 2);
			tmp = (be32_to_cpu(tmp) & 0xFFFF0000) | buf[1] | (buf[0] << 8);
			tmp = cpu_to_be32(tmp);
			size = 2;
		}
		else if ((ofs & 0x03) == 1) {
			tmp = map->read32(map, ofs - 1);
			tmp = (be32_to_cpu(tmp) & 0xFF000000) | buf[2] | (buf[1] << 8) | buf[0] << 16;
			tmp = cpu_to_be32(tmp);
			size = 3;
		}
		ret = do_write_oneword(map, chip, ofs - chip->start, (__u32 *) &tmp, 0);
		ofs += size;
		len -= size;
	}

	return ret;
}
#endif /* CFIDEV_BUSWIDTH_4 */

static int cfi_amdext_write (struct mtd_info *mtd, loff_t to , size_t len, size_t *retlen, const u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int ret = 0;
	int chipnum;
	unsigned long ofs;
	__u32 tmplen = 0;

	*retlen = 0;

	chipnum = (to >> cfi->chipshift);
	chipnum /= cfi->interleave;
	ofs = (to  - (chipnum << cfi->chipshift) * cfi->interleave);

	/* Handle unaligned writes */
	switch(map->buswidth) {
#ifdef CFIDEV_BUSWIDTH_2
	case CFIDEV_BUSWIDTH_2:
		tmplen = (ofs + cfi->chips[chipnum].start) & 1;
		if (tmplen) {
			ret = cfi_amdext_write_byte_16(map, &cfi->chips[chipnum], ofs, *buf);
		}
		break;
#endif
#ifdef CFIDEV_BUSWIDTH_4
	case CFIDEV_BUSWIDTH_4:
		tmplen = (ofs + cfi->chips[chipnum].start) & 3;
		if (tmplen) {
			ret = cfi_amdext_write_bytes_32(map, &cfi->chips[chipnum], ofs, buf, tmplen);
		}
		break;
#endif
	default:
		ret = -EINVAL;
		break;
	}

	if (ret)
		return ret;

	ofs += tmplen;
	buf += tmplen;
	(*retlen) += tmplen;
	len -= tmplen;

	if ((ofs >> cfi->chipshift) / cfi->interleave) {
		chipnum ++;
		ofs = 0;
		if (chipnum == cfi->numchips)
			return 0;
	}
	/* Go into unlock bypass mode */
	amd_send_cmd(0xAA, 0x555, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0x55, 0x2AA, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0x20, 0x555, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);

	/* We are now aligned, write as much as possible */
	while(len > (map->buswidth-1)) {
		ret = do_write_oneword(map, &cfi->chips[chipnum],
				       ofs, (__u32 *)buf, 1);
		if (ret) {
			/* Get out of unlock bypass mode */
			amd_send_cmd(0x90, 0x0, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
			amd_send_cmd(0x00, 0x0, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
			return ret;
		}

		ofs += map->buswidth;
		buf += map->buswidth;
		(*retlen) += map->buswidth;
		len -= map->buswidth;

		if ((ofs >> cfi->chipshift) / cfi->interleave) {
			/* Get out of unlock bypass mode */
			amd_send_cmd(0x90, 0x0, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
			amd_send_cmd(0x00, 0x0, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);

			chipnum ++; 
			ofs = 0;
			if (chipnum == cfi->numchips)
				return 0;
			/* Go into unlock bypass mode for next set of chips */
			amd_send_cmd(0xAA, 0x555, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
			amd_send_cmd(0x55, 0x2AA, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
			amd_send_cmd(0x20, 0x555, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
		}
	}

	/* Get out of unlock bypass mode */
	amd_send_cmd(0x90, 0x0, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0x00, 0x0, cfi->chips[chipnum].start, map, cfi->interleave, cfi->device_type);

	if (len) {
		/* Final bytes to write */
		switch(map->buswidth) {
#ifdef CFIDEV_BUSWIDTH_2
		case CFIDEV_BUSWIDTH_2:
			ret = cfi_amdext_write_byte_16(map, &cfi->chips[chipnum], ofs, *buf);
			break;
#endif
#ifdef CFIDEV_BUSWIDTH_4
		case CFIDEV_BUSWIDTH_4:
			ret = cfi_amdext_write_bytes_32(map, &cfi->chips[chipnum], ofs, buf, len);
		break;
#endif
		}

		if (ret)
			return ret;
		
		(*retlen) += len;
	}

	return 0;
}


static inline int do_erase_oneblock(struct map_info *map, struct flchip *chip, unsigned long adr)
{
	unsigned int status;
	unsigned long timeo = jiffies + HZ;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned int rdy_mask;
	DECLARE_WAITQUEUE(wait, current);

 retry:
	spin_lock_bh(chip->mutex);

	if (chip->state != FL_READY){
		set_current_state(TASK_INTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);
                
		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);

		if(signal_pending(current))
			return -EINTR;

		timeo = jiffies + HZ;

		goto retry;
	}	

	chip->state = FL_ERASING;

	adr += chip->start;
	ENABLE_VPP(map);
	amd_send_cmd(0xAA, 0x555, chip->start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0x55, 0x2AA, chip->start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0x80, 0x555, chip->start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0xAA, 0x555, chip->start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd(0x55, 0x2AA, chip->start, map, cfi->interleave, cfi->device_type);
	amd_send_cmd_at(0x30, adr, map, cfi->interleave);
	
	timeo = jiffies + (HZ*20);

	spin_unlock_bh(chip->mutex);
	schedule_timeout(HZ);
	spin_lock_bh(chip->mutex);
	
	rdy_mask = cfi_build_cmd(0x80, map, cfi->interleave);

	/* FIXME. Use a timer to check this, and return immediately. */
	/* Once the state machine's known to be working I'll do that */

	while ( ( (status = amd_read_val(adr, map)) & rdy_mask ) != rdy_mask ) {
		static int z=0;

		if (chip->state != FL_ERASING) {
			/* Someone's suspended the erase. Sleep */
			set_current_state(TASK_INTERRUPTIBLE);
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			printk("erase suspended. Sleeping\n");
			
			schedule();
			remove_wait_queue(&chip->wq, &wait);
			
			if (signal_pending(current))
				return -EINTR;
			
			timeo = jiffies + (HZ*2); /* FIXME */
			spin_lock_bh(chip->mutex);
			continue;
		}

		/* OK Still waiting */
		if (time_after(jiffies, timeo)) {
			chip->state = FL_READY;
			spin_unlock_bh(chip->mutex);
			printk("waiting for erase to complete timed out.");
			DISABLE_VPP(map);
			return -EIO;
		}
		
		/* Latency issues. Drop the lock, wait a while and retry */
		spin_unlock_bh(chip->mutex);

		z++;
		if ( 0 && !(z % 100 )) 
			printk("chip not ready yet after erase. looping\n");

		udelay(1);
		
		spin_lock_bh(chip->mutex);
		continue;
	}
	
	/* Done and happy. */
	DISABLE_VPP(map);
	chip->state = FL_READY;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);
	return 0;
}

static int cfi_amdext_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	unsigned long adr, len;
	int chipnum, ret = 0;

//printk("erase : 0x%x 0x%x 0x%x\n", instr->addr, instr->len, mtd->size);

	if (instr->addr & (mtd->erasesize - 1))
		return -EINVAL;

	if (instr->len & (mtd->erasesize -1))
		return -EINVAL;

	if ((instr->len + instr->addr) > mtd->size)
		return -EINVAL;

	chipnum = instr->addr >> cfi->chipshift;
	chipnum /= cfi->interleave;
	adr = instr->addr - (chipnum << cfi->chipshift) * (cfi->interleave);
	len = instr->len;

//	printk("erase : 0x%lx 0x%lx 0x%x %lx\n", adr, len, chipnum, mtd->size);

	while(len) {
//printk("erase : 0x%x 0x%x 0x%x 0x%x\n", chipnum, adr, len, cfi->chipshift);
		ret = do_erase_oneblock(map, &cfi->chips[chipnum], adr);

		if (ret)
			return ret;

		adr += mtd->erasesize;
		len -= mtd->erasesize;

		if ((adr >> cfi->chipshift) / (cfi->interleave)) {
			adr = 0;
			chipnum++;
			
			if (chipnum >= cfi->numchips)
			break;
		}
	}
		
	if (instr->callback)
		instr->callback(instr);
	
	return 0;
}

static void cfi_amdext_sync (struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;
	DECLARE_WAITQUEUE(wait, current);

	for (i=0; !ret && i<cfi->numchips; i++) {
		chip = &cfi->chips[i];

	retry:
		spin_lock_bh(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_SYNCING;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_SYNCING:
			spin_unlock_bh(chip->mutex);
			break;

		default:
			/* Not an idle state */
			add_wait_queue(&chip->wq, &wait);
			
			spin_unlock_bh(chip->mutex);
			schedule();
		        remove_wait_queue(&chip->wq, &wait);
			
			goto retry;
		}
	}

	/* Unlock the chips again */

	for (i--; i >=0; i--) {
		chip = &cfi->chips[i];

		spin_lock_bh(chip->mutex);
		
		if (chip->state == FL_SYNCING) {
			chip->state = chip->oldstate;
			wake_up(&chip->wq);
		}
		spin_unlock_bh(chip->mutex);
	}
}


static int cfi_amdext_suspend(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
	int ret = 0;
//printk("suspend\n");

	for (i=0; !ret && i<cfi->numchips; i++) {
		chip = &cfi->chips[i];

		spin_lock_bh(chip->mutex);

		switch(chip->state) {
		case FL_READY:
		case FL_STATUS:
		case FL_CFI_QUERY:
		case FL_JEDEC_QUERY:
			chip->oldstate = chip->state;
			chip->state = FL_PM_SUSPENDED;
			/* No need to wake_up() on this state change - 
			 * as the whole point is that nobody can do anything
			 * with the chip now anyway.
			 */
		case FL_PM_SUSPENDED:
			spin_unlock_bh(chip->mutex);
			break;

		default:
			ret = -EAGAIN;
			break;
		}
	}

	/* Unlock the chips again */

	if (ret) {
    		for (i--; i >=0; i--) {
			chip = &cfi->chips[i];

			spin_lock_bh(chip->mutex);
		
			if (chip->state == FL_PM_SUSPENDED) {
				chip->state = chip->oldstate;
				wake_up(&chip->wq);
			}
			spin_unlock_bh(chip->mutex);
		}
	}
	
	return ret;
}

static void cfi_amdext_resume(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	int i;
	struct flchip *chip;
//printk("resume\n");

	for (i=0; i<cfi->numchips; i++) {
	
		chip = &cfi->chips[i];

		spin_lock_bh(chip->mutex);
		
		if (chip->state == FL_PM_SUSPENDED) {
			chip->state = chip->oldstate;
			wake_up(&chip->wq);
		}
		else
			printk("Argh. Chip not in PM_SUSPENDED state upon resume()\n");

		spin_unlock_bh(chip->mutex);
	}
}

static void cfi_amdext_destroy(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;
	struct cfi_private *cfi = map->fldrv_priv;
	kfree(cfi->cmdset_priv);
	kfree(cfi);
}

