/* sbc_mediagx.c -- MTD map driver for Arcom Control Systems SBC-MediaGX
 
   Copyright (C) 2000 Arcom Control System Ltd
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

   $Id: sbc_mediagx.c,v 1.8 2000/10/16 20:07:52 nico Exp $

The SBC-MediaGX has up to 16 Mbyte of Intel StrataFlash (28F320/28F640) in x8
mode.  This drivers uses the CFI probe and Intel Extended Command Set drivers.

The flash is accessed as follows:

   16 kbyte memory window at 0xdc000-0xdffff
   Two IO address locations for paging
   0x258
       bit 0-7: address bit 14-21
   0x259
       bit 0-1: address bit 22-23
       bit 7:   0 - reset/powered down
                1 - device enabled

The single flash device is divided into 3 partition which appear as seperate
MTD devices.
*/
#include <linux/module.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <asm/io.h>

#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#define WINDOW_START 0xdc000
/* Number of bits in offset. */
#define WINDOW_SHIFT 14
#define WINDOW_LENGTH (1 << WINDOW_SHIFT)
/* The bits for the offset into the window. */
#define WINDOW_MASK (WINDOW_LENGTH-1)
#define PAGE_IO 0x258
#define PAGE_IO_SIZE 2

/* bit 7 of 0x259 must be 1 to enable device. */
#define DEVICE_ENABLE 0x8000

static volatile int page_in_window = -1; // Current page in window.
static unsigned long iomapadr;
static spinlock_t sbc_mediagx_spin = SPIN_LOCK_UNLOCKED;

/* partition_info gives details on the logical partitions that the split the 
 * single flash device into. If the size if zero we use up to the end of the
 * device. */
const static struct mtd_partition partition_info[]={
    { name: "SBC-MediaGX flash boot partition", 
      offset: 0, 
      size: 640*1024 },
    { name: "SBC-MediaGX flash partition 1", 
      offset: 640*1024, 
      size: 2*1024*1024-640*1024 },
    { name: "SBC-MediaGX flash partition 2", 
      offset: 2*1024*1024 }
};
#define NUM_PARTITIONS 3

static inline void sbc_mediagx_page(struct map_info *map, unsigned long ofs)
{
	unsigned long page = ofs >> WINDOW_SHIFT;

	if( page!=page_in_window ) {
		outw( page | DEVICE_ENABLE, PAGE_IO );
		page_in_window = page;
	}
}


static __u8 sbc_mediagx_read8(struct map_info *map, unsigned long ofs)
{
	__u8 ret;
	spin_lock(&sbc_mediagx_spin);
	sbc_mediagx_page(map, ofs);
	ret = readb(iomapadr + (ofs & WINDOW_MASK));
	spin_unlock(&sbc_mediagx_spin);
	return ret;
}

static __u16 sbc_mediagx_read16(struct map_info *map, unsigned long ofs)
{
	__u16 ret;
	spin_lock(&sbc_mediagx_spin);
	sbc_mediagx_page(map, ofs);
	ret = readw(iomapadr + (ofs & WINDOW_MASK));
	spin_unlock(&sbc_mediagx_spin);
	return ret;
}

static __u32 sbc_mediagx_read32(struct map_info *map, unsigned long ofs)
{
	__u32 ret;
	spin_lock(&sbc_mediagx_spin);
	sbc_mediagx_page(map, ofs);
	ret = readl(iomapadr + (ofs & WINDOW_MASK));
	spin_unlock(&sbc_mediagx_spin);
	return ret;
}

static void sbc_mediagx_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	while(len) {
		unsigned long thislen = len;
		if (len > (WINDOW_LENGTH - (from & WINDOW_MASK)))
			thislen = WINDOW_LENGTH-(from & WINDOW_MASK);
		
		spin_lock(&sbc_mediagx_spin);
		sbc_mediagx_page(map, from);
		memcpy_fromio(to, iomapadr + (from & WINDOW_MASK), thislen);
		spin_unlock(&sbc_mediagx_spin);
		(__u8*)to += thislen;
		from += thislen;
		len -= thislen;
	}
}

static void sbc_mediagx_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	spin_lock(&sbc_mediagx_spin);
	sbc_mediagx_page(map, adr);
	writeb(d, iomapadr + (adr & WINDOW_MASK));
	spin_unlock(&sbc_mediagx_spin);
}

static void sbc_mediagx_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	spin_lock(&sbc_mediagx_spin);
	sbc_mediagx_page(map, adr);
	writew(d, iomapadr + (adr & WINDOW_MASK));
	spin_unlock(&sbc_mediagx_spin);
}

static void sbc_mediagx_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	spin_lock(&sbc_mediagx_spin);
	sbc_mediagx_page(map, adr);
	writel(d, iomapadr + (adr & WINDOW_MASK));
	spin_unlock(&sbc_mediagx_spin);
}

static void sbc_mediagx_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{	
	while(len) {
		unsigned long thislen = len;
		if (len > (WINDOW_LENGTH - (to & WINDOW_MASK)))
			thislen = WINDOW_LENGTH-(to & WINDOW_MASK);
		
		spin_lock(&sbc_mediagx_spin);
		sbc_mediagx_page(map, to);
		memcpy_toio(iomapadr + (to & WINDOW_MASK), from, thislen);
		spin_unlock(&sbc_mediagx_spin);
		to += thislen;
		from += thislen;
		len -= thislen;
	}
}

static struct map_info sbc_mediagx_map = {
	name: "SBC-MediaGX flash",
	size: 16*1024*1024, /* this must be set to a maximum possible amount
			 of flash so the cfi probe routines find all
			 the chips */
	buswidth: 1,
	read8: sbc_mediagx_read8,
	read16: sbc_mediagx_read16,
	read32: sbc_mediagx_read32,
	copy_from: sbc_mediagx_copy_from,
	write8: sbc_mediagx_write8,
	write16: sbc_mediagx_write16,
	write32: sbc_mediagx_write32,
	copy_to: sbc_mediagx_copy_to
};

/* MTD device for all of the flash. */
static struct mtd_info *all_mtd;

#if LINUX_VERSION_CODE < 0x20300
#ifdef MODULE
#define init_sbc_mediagx init_module
#define cleanup_sbc_mediagx cleanup_module
#endif
#define __exit
#endif

void cleanup_sbc_mediagx(void)
{
	if( all_mtd ) {
		del_mtd_partitions( all_mtd );
		map_destroy( all_mtd );
	}

	iounmap((void *)iomapadr);
	release_region(PAGE_IO,PAGE_IO_SIZE);
}

int __init init_sbc_mediagx(void)
{
	int i;

	if (check_region(PAGE_IO,PAGE_IO_SIZE) != 0) {
		printk( KERN_ERR"%s: IO ports 0x%x-0x%x in use\n",
			sbc_mediagx_map.name,
			PAGE_IO, PAGE_IO+PAGE_IO_SIZE-1 );
		return -EAGAIN;
	}
  	iomapadr = (unsigned long)ioremap(WINDOW_START, WINDOW_LENGTH);
	if (!iomapadr) {
		printk( KERN_ERR"%s: failed to ioremap memory region\n",
			sbc_mediagx_map.name );
		return -EIO;
	}
	
	request_region( PAGE_IO, PAGE_IO_SIZE, "SBC-MediaGX flash" );
	
	printk( KERN_INFO"%s: IO:0x%x-0x%x MEM:0x%x-0x%x\n",
		sbc_mediagx_map.name,
		PAGE_IO, PAGE_IO+PAGE_IO_SIZE-1,
		WINDOW_START, WINDOW_START+WINDOW_LENGTH-1 );

	/* Probe for chip. */
	all_mtd = do_cfi_probe( &sbc_mediagx_map );
	if( !all_mtd ) {
		cleanup_sbc_mediagx();
		return -ENXIO;
	}
	
	all_mtd->module=THIS_MODULE;

	/* Create MTD devices for each partition. */
	add_mtd_partitions( all_mtd, partition_info, NUM_PARTITIONS );

	return 0;
}

#if LINUX_VERSION_CODE > 0x20300
module_init(init_sbc_mediagx);
module_exit(cleanup_sbc_mediagx);
#endif
