/*
 * usb-ohci-sa1111.h
 *
 * definitions and special code for Intel SA-1111 USB OHCI host controller
 *
 * 10/24/00 Brad Parker <brad@heeltoe.com>
 * added memory allocation code
 *
 * 09/26/00 Brad Parker <brad@heeltoe.com>
 * init code for the SA-1111 ohci controller
 * special dma map/unmap code to compensate for SA-1111 h/w bug
 * 
 */

#include "asm/arch-sa1100/SA-1111.h"
#include "asm/proc/cache.h"

static int ohci_mem_init (void);

#define SA1111_OHCI_BASE        SA1111_p2v(_SA1111(0x0400))
#define SA1111_OHCI_EXTENT      512
#define SA1111_OHCI_IRQ         NIRQHCIM

#if defined(CONFIG_SA1100_XP860) || defined(CONFIG_ASSABET_NEPONSET)
#define PwrSensePolLow  1
#define PwrCtrlPolLow   1
#else
#define PwrSensePolLow  0
#define PwrCtrlPolLow   0
#endif

#undef __arch_getl
#undef __arch_putl

static volatile inline unsigned int __arch_getl(unsigned long a)
{
	invalidate_dcache_range(a, a);
        return *(volatile unsigned int  *)a;
}

static volatile inline void __arch_putl(unsigned long v,unsigned long a)
{
        *(volatile unsigned int  *)a = v;
	clean_dcache_entry(a);
}

#ifdef OHCI_FLUSH_CACHE

#define RD_CACHE_INV(start, size) \
	invalidate_dcache_range((unsigned long)(start), \
				((unsigned long)(start))+(size))

/* flush_dcache_range really means clean & invalidate */
#define WB_RD_CACHE_RANGE(start, size) \
	flush_dcache_range((unsigned long)(start), \
				((unsigned long)(start))+(size))

#define WRITE_BACK(addr) clean_dcache_entry(addr);

/* flush_cache_all really mean clean & invalidate all */
#define FLUSH_CACHE() flush_cache_all()

#endif	


/*
 * The SA-1111 errata says that the DMA hardware needs to be exercised
 * before the clocks are turned on to work properly.  This code does
 * a tiny dma transfer to prime to hardware.
 */
static void
sa1111_hc_dma_setup(void)
{
	static unsigned dmaTxBuff[4];

	/* DMA init & setup */

#define SACR0		(*((volatile Word *) SA1111_p2v (_SACR0)))
#define SACR1		(*((volatile Word *) SA1111_p2v (_SACR1)))
#define SADTCS		(*((volatile Word *) SA1111_p2v (_SADTCS)))
#define SADTSA		(*((volatile Word *) SA1111_p2v (_SADTSA)))
#define SADTCA		(*((volatile Word *) SA1111_p2v (_SADTCA)))

#define SKPCR_UCLKEn	(1<<0)
#define SKPCR_I2SCLKEn	(1<<2)
#define SKPCR_DCLKEn	(1<<7)

	/* prime the dma engine with a tiny dma */
	SKPCR |= SKPCR_I2SCLKEn;
	SKAUD |= 0x00000018;
	
	SACR0 |= 0x00003305;
	SACR1 = 0x00000000; 
	
	SADTSA = (unsigned long)dmaTxBuff;
	SADTCA = sizeof(dmaTxBuff);

	SADTCS |= 0x00000011;
	SKPCR |= SKPCR_DCLKEn;

	/* wait */
	udelay(100);

	SACR0 &= ~(0x00000002);
	SACR0 &= ~(0x00000001);

	/* */
	SACR0 |= 0x00000004;
	SACR0 &= ~(0x00000004);

	SKPCR &= ~SKPCR_I2SCLKEn;
}	

static int
sa1111_hc_start_ohci(void)
{
	static struct pci_dev dev;
	memset((char *)&dev, 0, sizeof(dev));
	strcpy(dev.name, "usb-ohci");
	strcpy(dev.slot_name, "builtin");

#if 1
	request_region ( SA1111_OHCI_BASE, SA1111_OHCI_EXTENT, "usb-ohci" );
	return hc_found_ohci (&dev, SA1111_OHCI_IRQ, (void *)SA1111_OHCI_BASE);
#else
	unsigned long mem_base;
	mem_base = (unsigned long)ioremap_nocache(SA1111_OHCI_BASE, 4096);
	return hc_found_ohci (&dev, SA1111_OHCI_IRQ, mem_base );
#endif
}

static void
sa1111_release(void)
{
       release_region ( SA1111_OHCI_BASE, SA1111_OHCI_EXTENT );
}

static void
sa1111_cleanup(void)
{
	/* turn the USB clock off */
	SKPCR &= ~0x1;
}


static int _sa1111_ohci_hcd_setup;

/*
 * reset the SA-1111 usb controller and turn on it's clocks
 */
int
sa1111_ohci_hcd_init (void) 
{
	int ret;
	volatile unsigned long *Reset = (void *)SA1111_p2v(_SA1111(0x051c));
	volatile unsigned long *Status = (void *)SA1111_p2v(_SA1111(0x0518));


	if (_sa1111_ohci_hcd_setup)
		return 0;

	/* turn on clocks */
	SKPCR = SKPCR_UCLKEn;
	udelay(100);

	/* force a reset */
	*Reset = 0x01;
	*Reset |= 0x02;
	udelay(100);

	*Reset = 0;

	/* take out of reset */
	/* set power sense and control lines (this from the diags code) */
        *Reset = ( PwrSensePolLow << 6 )
               | ( PwrCtrlPolLow << 7 );

	*Status = 0;

	udelay(10);

	/* compensate for dma bug */
	sa1111_hc_dma_setup();

	/* init memory allocator */
	if ((ret = ohci_mem_init ()) < 0)
		return ret;

	_sa1111_ohci_hcd_setup = 1;

	return sa1111_hc_start_ohci();
}


/* -------------- */

/*

NOTE ON THIS MEMORY ALLOCATION CODE:

This code can be configured to use __get_free_pages() or consistent_alloc().

__get_free_pages() grabs raw pages which are cached and write-buffer
enabled.  This requires the driver to carefully manage the cache/chip
interactions.  If this is done properly most of this code can be
removed in favor of using the slab allocator like the code in
usb-ohci.h.  But, the driver is not quite ready for that.  It's close
but not quite there.

consistent_alloc() grabs pages from GFP_DMA, which is good.  Then it
uses __ioremap.  This creates "alias" mmu entries which have cache &
write-buffer turned off.  It then hands back nice pointers to the
physical memory and the virtual "alias" entries.

why not consistent_alloc() all the time?

Well, bus_to_virt() and virt_to_bus() don't work with the "alias"
entries because they are simple macros.  To fix this I create private
versions of these which "remember" the results from
consistent_alloc().  The trouble is to make these routines fast we
assume a linear mapping and use + & - to convert.  If we call
consistent_alloc() more than once they don't work.  So, we have to
preallocate a large chunk of memory to make it work, which puts us
right back where we started.

[What I really want is a slab allocator which will allocate pages
which are non-cached and non-writeback.]

*/

#ifdef OHCI_MEM_CONSISTENT

#define USE_CONSISTENT_ALLOC

/*
 * simple memory allocator which gives out small portions of memory
 * allocated as pages...
 *
 */

static char *mem;
static int mem_size;
static char *memp;
static int memp_left;
dma_addr_t mem_dma;

#ifdef USE_CONSISTENT_ALLOC
static void *sa1111_alloc_consistent(size_t size, dma_addr_t *handle)
{
	char *p;

	if (0) printk("sa1111_alloc_consistent(size=%d) "
		      "memp=%p,memp_left=%d\n",
		      size, memp, memp_left);

	if (memp_left && memp_left < size) {
		panic("usb-ohci: can't alloc more memory");
		memp_left = 0;
	}

	if (mem_size == 0) {
		mem_size = 16 * PAGE_SIZE;
		mem = consistent_alloc(GFP_KERNEL | GFP_DMA, mem_size,
				       &mem_dma);
		if (1) printk(__FILE__ ": allocated pages; mem %p (%p)\n",
			      mem, (void *)mem_dma);
		memp = mem;
		memp_left = mem_size;
	}

	if (memp_left < size)
		return 0;

	p = memp;
	*handle = mem_dma + (p - mem);

	if (0) printk("sa1111_alloc_consistent() mem %p, dma %p\n",
		      p, (void *)*handle);

	memp += size;
	memp_left -= size;

	return p;
}

void
sa1111_free_consistent(size_t size, void *vaddr, dma_addr_t dma_handle)
{
}

static dma_addr_t ohci_virt_to_bus(struct ohci *ohci, volatile void *obp)
{
	dma_addr_t dma = (long)((char *)mem_dma + ((char *)obp - mem));
	if (0) printk("ohci_virt_to_bus(v=%p) -> %p\n", obp, (void *)dma);
	return dma;
}

static void *ohci_bus_to_virt(struct ohci *ohci, dma_addr_t dma_addr)
{
	void *virt = (void *)(mem + ((char *)dma_addr - (char *)mem_dma));
	if (0) printk("ohci_bus_to_virt(p=%p) -> %p\n", (void *)dma_addr,virt);
	return virt;
}
#else
static void *sa1111_alloc_consistent(size_t size, dma_addr_t *handle)
{
	char *p;

	if (0) printk("sa1111_alloc_consistent(size=%d) "
		      "memp=%p,memp_left=%d\n",
		      size, memp, memp_left);

	if (memp_left && memp_left < size) {
		memp_left = 0;
	}

	if (mem_size == 0) {
		mem_size = 16 * PAGE_SIZE;
		mem = (char *)__get_free_pages (GFP_KERNEL | GFP_DMA, 4);

		if (1) printk(__FILE__ ": allocated pages; mem %p (%p)\n",
			      mem, (void *)mem_dma);
		memp = mem;
		memp_left = mem_size;
	}

	if (memp_left < size)
		return 0;

	p = memp;
	*handle = virt_to_bus(p);

	if (0) printk("sa1111_alloc_consistent() mem %p, dma %p\n", p, (void *)*handle);

	memp += size;
	memp_left -= size;

	return p;
}

void
sa1111_free_consistent(size_t size, void *vaddr, dma_addr_t dma_handle)
{
}

#define ohci_virt_to_bus(o, a) virt_to_bus(a)
#define ohci_bus_to_virt(o, a) bus_to_virt(a)
#endif

/* --------------------------------- */

/*
 * simple memory manager; keeps three free lists, one for very small buffers
 * and two for bigger buffers.
 */

#define MAX_ALLOC_SIZES	3

static long mapped_alloc_size;

static struct {
	int size;
	int count;
	char *buffers;
} alloc[MAX_ALLOC_SIZES] = {
	{ 32, 8 },
	{ 1024, 2 },
	{ 16384, 2 },
};

/* allocate a buffer from the pool */
static char *
__alloc(int index)
{
	char *safe;
	dma_addr_t dma;
	int i;

	/* initial call?  grab some safe buffers */
	if (alloc[index].buffers == 0) {
		for (i = 0; i < alloc[index].count; i++)
		{
			safe = sa1111_alloc_consistent(alloc[index].size,
						       &dma);

			if (safe == 0)
				break;

			if (0) printk("__alloc(%d) gets %p\n", index, safe);

			*(char **)safe = alloc[index].buffers;
			alloc[index].buffers = safe;
		}

		if (0) printk("allocated %d alloc buffers of %d bytes @ %p\n",
			      i, alloc[index].size, memp);
	}

	if (alloc[index].buffers == 0) {
		return (char *)0;
	}

	safe = alloc[index].buffers;
	alloc[index].buffers = *(char **)safe;

	return safe;
}

static void
__free(char *buf, int index)
{
	*(char **)buf = alloc[index].buffers;
	alloc[index].buffers = buf;
}

/*-------------------------------------------------------------------------*/

static int ohci_mem_init (void)
{
	return 0;
}

static void ohci_mem_cleanup (void)
{
}

/* TDs ... */
static inline struct td *
td_alloc (struct ohci *hc)
{
	struct td *td = (struct td *)__alloc(0);
	if (td)
		memset((char *)td, 0, sizeof(*td));
#if 1
	if ((((unsigned long)td) & 0x1f) != 0)
		printk("ALLOCATED UNALIGNED TD %p\n", td);
#endif
	return td;
}

static inline void
td_free (struct ohci *hc, struct td *td)
{
	__free((char *)td, 0);
}


/* DEV + EDs ... only the EDs need to be consistent */
static inline struct ohci_device *
dev_alloc (struct ohci *hc)
{
	struct ohci_device *dev = (struct ohci_device *)__alloc(1);
	if (dev) {
		memset (dev, 0, sizeof (*dev));
#if 1
		if ((((unsigned long)&dev->ed[0]) & 0xf) != 0)
			printk("ALLOCATED UNALIGNED ED %p\n", &dev->ed[0]);
#endif
	}
	return dev;
}

static inline void
dev_free (struct ohci_device *dev)
{
	__free((char *)dev, 1);
}

static inline struct ohci_hcca *
hcca_alloc(void)
{
	struct ohci_hcca *hcca;

	hcca = (struct ohci_hcca *) __alloc(1);
	
	if (hcca != NULL) {
		/* align on 256 byte boundary */
		hcca = (struct ohci_hcca *)((unsigned long)(hcca) & ~0xff);
		memset(hcca, 0, sizeof(struct ohci_hcca));
	}

	return hcca;
}

static inline void
hcca_free(struct ohci_hcca *hcca)
{
	__free((char *)hcca, 1);
}

#endif /* OHCI_MEM_CONSISTENT */

/* -------------- */

/*
 * simple buffer allocator for copying of unsafe to safe buffers
 * uses __alloc/__free for actual buffers
 * keeps track of safe buffers we've allocated so we can recover the
 * unsafe buffers.
 */

#define MAX_SAFE	32

static long mapped_alloc_size;
static char *safe_buffers[MAX_SAFE];

/* allocate a 'safe' buffer and keep track of it */
static char *
alloc_safe_buffer(char *unsafe, int size, dma_addr_t *pbus)
{
	int i, size_index;
	char *safe;
	dma_addr_t busptr;

	size_index = 1;
	if (size > alloc[1].size - 8)
		size_index = 2;

	safe = __alloc(size_index);
	if (safe == 0)
		return 0;

	busptr = ohci_virt_to_bus(NULL, safe) + 8;

	for (i = 0; i < MAX_SAFE; i++)
		if (safe_buffers[i] == 0) {
			safe_buffers[i] = (void *)busptr;
			break;
		}

	if (i == MAX_SAFE) {
		panic("usb-ohci: exceeded MAX_SAFE buffers");
	}

	if (0) printk("alloc_safe_buffer() index %d, ptr %p\n", i, safe);

	/* place the size index and the old buffer ptr in the first 8 bytes
	 * and return a ptr + 8 to caller
	 */
	((int *)safe)[0] = (size_index << 16) | i;
	((char **)safe)[1] = unsafe;
	safe += sizeof(char*)+sizeof(int);

	*pbus = busptr;
	return safe;
}

/* determine if a buffer is from our "safe" pool */
static char *
find_safe_buffer(char *busptr, char **unsafe)
{
	int i;
	char *buf;

	for (i = 0; i < MAX_SAFE; i++) {
		if (safe_buffers[i] == busptr) {
			if (0) printk("find_safe_buffer(%p) found @ %d\n", busptr, i);
			buf = ohci_bus_to_virt(NULL, (dma_addr_t)busptr);
			*unsafe = *(char **)(buf - sizeof(char *));
			return buf;
		}
	}

	return (char *)0;
}

static void
free_safe_buffer(char *buf)
{
	int index, size_index;

	/* retrieve the buffer size index */
	buf -= sizeof(char*)+sizeof(int);
	index = ((int *)buf)[0] & 0xffff;
	size_index = ((int *)buf)[0] >> 16;

	if (0) printk("free_safe_buffer(%p) index %d, size_index %d\n",
	       buf+8, index, size_index);

	if (index < 0 || index >= MAX_SAFE) {
		printk("usb-ohci: free_safe_buffer() corrupt buffer\n");
		return;
	}

	safe_buffers[index] = 0;
	__free(buf, size_index);
}

/*
 * see if a buffer address is in an 'unsafe' range.  if it is
 * allocate a 'safe' buffer and copy the unsafe buffer into it.
 * substitute the safe buffer for the unsafe one.
 * (basically move the buffer from an unsafe area to a safe one)
 *
 * we assume calls to map_single are symmetric with calls to unmap_single...
 */
static dma_addr_t
ohci_pci_map_single(struct pci_dev *hwdev, void *virtptr,
		    size_t size, int direction)
{
	dma_addr_t busptr;

	mapped_alloc_size += size;

	if (0) printk("ohci_pci_map_single(hwdev=%p,ptr=%p,size=%d,dir=%x) "
		      "alloced=%ld\n",
		      hwdev, virtptr, size, direction, mapped_alloc_size);

	busptr = virt_to_bus(virtptr);

	/* we assume here that a buffer will never be >=64k */
	if ( (((unsigned long)busptr) & 0x100000) ||
	     ((((unsigned long)busptr)+size) & 0x100000) )
	{
		char *safe;

		safe = alloc_safe_buffer(virtptr, size, &busptr);
		if (safe == 0) {
			printk("unable to map unsafe buffer %p!\n", virtptr);
			return 0;
		}

		if (0) printk("unsafe buffer %p (phy=%p) mapped to %p (phy=%p)\n",
			      virtptr, (void *)virt_to_bus(virtptr),
			      safe, (void *)busptr);

		memcpy(safe, virtptr, size);
		consistent_sync(safe, size, direction);

		return busptr;
	}

	consistent_sync(virtptr, size, direction);
	return busptr;
}

/*
 * see if a mapped address was really a "safe" buffer and if so,
 * copy the data from the safe buffer back to the unsafe buffer
 * and free up the safe buffer.
 * (basically return things back to the way they should be)
 */
static void
ohci_pci_unmap_single(struct pci_dev *hwdev, dma_addr_t dma_addr,
		      size_t size, int direction)
{
	char *safe, *unsafe;
	void *buf;

	mapped_alloc_size -= size;

	if (0) printk("ohci_pci_unmap_single(hwdev=%p,ptr=%p,size=%d,dir=%x) "
		      "alloced=%ld\n",
		      hwdev, (void *)dma_addr, size, direction,
		      mapped_alloc_size);

	if ((safe = find_safe_buffer((void *)dma_addr, &unsafe))) {
		if (0) printk("copyback unsafe %p, safe %p, size %d\n",
			      unsafe, safe, size);

		consistent_sync(safe, size, PCI_DMA_FROMDEVICE);
		memcpy(unsafe, safe, size);
		free_safe_buffer(safe);
	} else {
		/* assume this is normal memory */
		buf = bus_to_virt(dma_addr);
		consistent_sync(buf, size, PCI_DMA_FROMDEVICE);
	}

	SPECULATIVE_FLUSH_CACHE();
}

