/*
 *  linux/include/asm-arm/arch-integrator/time.h
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/config.h>
#include <asm/system.h>
#include <asm/leds.h>

/*
 * Where is the timer (VA)?
 */
#define TIMER0_VA_BASE (IO_ADDRESS(INTEGRATOR_CT_BASE)+0x00000000)
#define TIMER1_VA_BASE (IO_ADDRESS(INTEGRATOR_CT_BASE)+0x00000100)
#define TIMER2_VA_BASE (IO_ADDRESS(INTEGRATOR_CT_BASE)+0x00000200)

/*
 * How long is the timer interval?
 */
#define TIMER_INTERVAL (TICKS_PER_uSEC * mSEC_10) 

/*
 * What does it look like?
 */
typedef struct TimerStruct {
	unsigned long TimerLoad;
	unsigned long TimerValue;
	unsigned long TimerControl;
	unsigned long TimerClear;
} TimerStruct_t;

/*
 * IRQ handler for the timer
 */
static void integrator_timer_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	volatile TimerStruct_t *timer1 = (volatile TimerStruct_t *)TIMER1_VA_BASE;

	// ...clear the interrupt
	timer1->TimerClear = 1;

	do_leds();
	do_timer(regs);
	do_profile(regs);
}

/*
 * Set up timer interrupt, and return the current time in seconds.
 */
extern __inline__ void setup_timer(void)
{
	volatile TimerStruct_t *timer0 = (volatile TimerStruct_t *)TIMER0_VA_BASE;
	volatile TimerStruct_t *timer1 = (volatile TimerStruct_t *)TIMER1_VA_BASE;
	volatile TimerStruct_t *timer2 = (volatile TimerStruct_t *)TIMER2_VA_BASE;
	unsigned int ticks, control;

	timer_irq.handler = integrator_timer_interrupt;

	/*
	 * Initialise to a known state (all timers off)
	 */
	timer0->TimerControl = 0x0;
	timer1->TimerControl = 0x0;
	timer2->TimerControl = 0x0;

	if (TIMER_INTERVAL >= 0x100000) {
		ticks = TIMER_INTERVAL >> 8;		/* Divide by 256 */
		control = 0x88;				/* Enable, Clock divided by 256 */
	} else if (TIMER_INTERVAL >= 0x10000) {
		ticks = TIMER_INTERVAL >> 4;		/* Divide by 16 */
		control = 0x84;				/* Enable, Clock divided by 16 */
	} else {
		ticks = TIMER_INTERVAL;
		control = 0x80;				/* Enable */
	}

	control |= 0x40;				/* periodic */
	timer1->TimerLoad = ticks;
	timer1->TimerValue = ticks;
	timer1->TimerControl = control;   

	/* 
	 * Make irqs happen for the system timer
	 */
	setup_arm_irq(IRQ_TIMERINT1, &timer_irq);
}
