/*
 * arch/arm/kernel/hw-sa1100.c
 *
 * SA1100-dependent machine specifics
 *
 * Copyright (C) 2000 Nicolas Pitre <nico@cam.org>
 *
 * This will certainly contain more stuff with time... like power management,
 * special hardware autodetection, etc.
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif

#include <asm/delay.h>
#include <asm/hardware.h>
#include <asm/mach-types.h>

/*
 * SA1100 GPIO edge detection for IRQs:
 * IRQs are generated on Falling-Edge, Rising-Edge, or both.
 * This must be called *before* the appropriate IRQ is registered.
 * Use this instead of directly setting GRER/GFER.
 */

int GPIO_IRQ_rising_edge;
int GPIO_IRQ_falling_edge;

void set_GPIO_IRQ_edge( int gpio_mask, int edge )
{
	if( edge & GPIO_FALLING_EDGE )
		GPIO_IRQ_falling_edge |= gpio_mask;
	else
		GPIO_IRQ_falling_edge &= ~gpio_mask;
	if( edge & GPIO_RISING_EDGE )
		GPIO_IRQ_rising_edge |= gpio_mask;
	else
		GPIO_IRQ_rising_edge &= ~gpio_mask;
}

EXPORT_SYMBOL(set_GPIO_IRQ_edge);


#ifdef CONFIG_SA1100_ASSABET

unsigned long BCR_value = BCR_DB1110;
unsigned long SCR_value = SCR_INIT;
EXPORT_SYMBOL(BCR_value);
EXPORT_SYMBOL(SCR_value);

/*
 * Read System Configuration "Register"
 * (taken from "Intel StrongARM SA-1110 Microprocessor Development Board
 * User's Guide", section 4.4.1)
 *
 * This same scan is performed in arch/arm/boot/compressed/head-sa1100.S
 * to set up the serial port for decompression status messages. We 
 * repeat it here because the kernel may not be loaded as a zImage, and
 * also because it's a hassle to communicate the SCR value to the kernel
 * from the decompressor.
 */

void __init get_assabet_scr(void)
{
	unsigned long flags, scr, i;

	local_irq_save(flags);
	GPDR |= 0x3fc;			/* Configure GPIO 9:2 as outputs */
	GPSR = 0x3fc;			/* Write 0xFF to GPIO 9:2 */
	GPDR &= ~(0x3fc);		/* Configure GPIO 9:2 as inputs */
	for(i = 100; i--; scr = GPLR);	/* Read GPIO 9:2 */
	GPDR |= 0x3fc;			/*  restore correct pin direction */
	local_irq_restore(flags);
	scr &= 0x3fc;			/* save as system configuration byte. */

	SCR_value = scr;
}

#endif  /* CONFIG_SA1100_ASSABET */


#ifdef CONFIG_SA1100_HUW_WEBPANEL
unsigned long BCR_value;
EXPORT_SYMBOL(BCR_value);

/*

 */

void __init init_huw_cs3(void)
{
  return;
  // here we can place some initcode
  // BCR_value = 0x1045bf70; //*((volatile unsigned long*)0xf1fffff0);
}

#endif  /* CONFIG_SA1100_HUW_WEBPANEL*/



#if defined(CONFIG_SA1100_FREEBIRD)
unsigned long BCR_value = BCR_DB1110;
EXPORT_SYMBOL(BCR_value);
#endif

#if defined(CONFIG_SA1100_BITSY)
/*
 * Bitsy has extended, write-only memory-mapped GPIO's
 */
static int bitsy_egpio = EGPIO_BITSY_RS232_ON;
void clr_bitsy_egpio(unsigned long x) 
{
  bitsy_egpio &= ~x;
  BITSY_EGPIO = bitsy_egpio;
}
void set_bitsy_egpio(unsigned long x) 
{
  bitsy_egpio |= x;
  BITSY_EGPIO = bitsy_egpio;
}
EXPORT_SYMBOL(clr_bitsy_egpio);
EXPORT_SYMBOL(set_bitsy_egpio);


#ifdef CONFIG_PROC_FS
static int bitsy_proc_read_egpio(char *page, char **start, off_t off, int count, int *eof, void *data)
{
        char *p = page;
        int i;
        p += sprintf(p, "EGPIO: %#08lx\n", bitsy_egpio);
        return p-page;
}
#endif

static void __init hw_bitsy_init(void) 
{
#ifdef CONFIG_PROC_FS
   create_proc_read_entry("egpio", 0, &proc_root, bitsy_proc_read_egpio, 0);
#endif
}

#endif


#ifdef CONFIG_SA1100_YOPY

static unsigned long yopy_egpio =
	GPIO_MASK(GPIO_CF_RESET) |
	GPIO_MASK(GPIO_CLKDIV_CLR1) | GPIO_MASK(GPIO_CLKDIV_CLR2) |
	GPIO_MASK(GPIO_SPEAKER_MUTE) | GPIO_MASK(GPIO_AUDIO_OPAMP_POWER);

static void __init yopy_gpio_init(void)
{
	YOPY_EGPIO = yopy_egpio;

	/* Enable Output */
	PPDR |= PPC_L_BIAS;
	PSDR &= ~PPC_L_BIAS;
	PPSR |= PPC_L_BIAS;

	YOPY_EGPIO = yopy_egpio;
}

int yopy_gpio_test(unsigned int gpio)
{
	return ((yopy_egpio & (1 << gpio)) != 0);
}

void yopy_gpio_set(unsigned int gpio, int level)
{
	unsigned long flags, mask;

	mask = 1 << gpio;

	local_irq_save(flags);

	if (level)
		yopy_egpio |= mask;
	else
		yopy_egpio &= ~mask;
	YOPY_EGPIO = yopy_egpio;

	local_irq_restore(flags);
}

EXPORT_SYMBOL(yopy_gpio_test);
EXPORT_SYMBOL(yopy_gpio_set);

#endif


#ifdef CONFIG_SA1111

static void __init sa1111_init(void){
  unsigned long id=SKID;

  if((id & SKID_ID_MASK) == SKID_SA1111_ID)
    printk(KERN_INFO "SA-1111 Microprocessor Companion Chip: "
	   "silicon revision %lx, metal revision %lx\n",
	   (id & SKID_SIREV_MASK)>>4, (id & SKID_MTREV_MASK));
  else {
    printk(KERN_ERR "Could not detect SA-1111!\n");
    return;
  }

  /* First, set up the 3.6864MHz clock on GPIO 27 for the SA-1111:
   * (SA-1110 Developer's Manual, section 9.1.2.1)
   */
  GAFR |= GPIO_32_768kHz;
  GPDR |= GPIO_32_768kHz;
  TUCR = TUCR_3_6864MHz;

  /* Now, set up the PLL and RCLK in the SA-1111: */
  SKCR = SKCR_PLL_BYPASS | SKCR_RDYEN | SKCR_OE_EN;
  udelay(100);
  SKCR = SKCR_PLL_BYPASS | SKCR_RCLKEN | SKCR_RDYEN | SKCR_OE_EN;

  /* SA-1111 Register Access Bus should now be available. Clocks for
   * any other SA-1111 functional blocks must be enabled separately
   * using the SKPCR.
   */

  /* If the system is going to use the SA-1111 DMA engines, set up
   * the memory bus request/grant pins. Also configure the shared
   * memory controller on the SA-1111 (SA-1111 Developer's Manual,
   * section 3.2.3) and power up the DMA bus clock:
   */
  if(machine_is_assabet()){

    GAFR |= (GPIO_MBGNT | GPIO_MBREQ);
    GPDR |= GPIO_MBGNT;
    GPDR &= ~GPIO_MBREQ;
    TUCR |= TUCR_MR;

    /* Assabet is populated by default with two Samsung KM416S8030T-G8
     * 128Mb SDRAMs, which are organized as 12-bit (row addr) x 9-bit
     * (column addr), according to the data sheet. Apparently, the 
     * bank selects factor into the row address, as Angel sets up the
     * SA-1110 to use 14x9 addresses. The SDRAM datasheet specifies
     * that when running at 100-125MHz, the CAS latency for -8 parts
     * is 3 cycles, which is consistent with Angel.
     */

    SMCR = (SMCR_DTIM | SMCR_MBGE | 
	    FInsrt(FExtr(MDCNFG, MDCNFG_SA1110_DRAC0), SMCR_DRAC) |
	    ((FExtr(MDCNFG, MDCNFG_SA1110_TDL0)==3) ? SMCR_CLAT : 0));

    SKPCR |= SKPCR_DCLKEN;
  }

#ifdef CONFIG_USB_OHCI
  sa1111_ohci_hcd_init();
#endif
}

#else
#define sa1111_init()  printk( "Warning: missing SA1111 support\n" )
#endif


#ifdef CONFIG_SA1100_JORNADA720
long jornada_i;
static void __init hw_jornada720_init(void) 
{
#define JORTUCR_VAL  0x20000400
#define JORSKCR_INIT 0x00002081	/* Turn off VCO to enable PLL, set Ready En
                                 * and enable nOE assertion from DC */
#define JORSKCR_RCLK 0x00002083	/* Add turning on RCLK to above */
#define JORSKCR_VAL  0x0000001B	/* sets the 1101 control register to on */

	GPDR |= GPIO_GPIO20;
    TUCR = JORTUCR_VAL;   /* set the oscillator out to the SA-1101 */

	GPSR = GPIO_GPIO20;	/* Set GPIO 20 of the SA1110 to high */
	for (jornada_i  = 0; jornada_i < 0x800; jornada_i++)
		; /* Wait for 1 us */
	GPCR = GPIO_GPIO20;	/* Set GPIO 20 of the SA1110 to low */
	for (jornada_i  = 0; jornada_i < 0x800; jornada_i++)
		; /* Wait for 1 us */
	GPSR = GPIO_GPIO20;	/* Set GPIO 20 of the SA1110 to high */
	for (jornada_i  = 0; jornada_i < 0x800; jornada_i++)
		; /* Wait for 20 us */

    SKCR = JORSKCR_INIT;   /* Turn on the PLL, enable */
                   /* Ready and enable nOE assertion from DC */
	for (jornada_i  = 0; jornada_i < 0x80000; jornada_i++)
		;

    SKCR = JORSKCR_RCLK;/* turn on the RCLOCK */
    SMCR = 0x35;     /* initialize the SMC (debug SA-1111 reset */
    PCCR = 0;        /* initialize the S2MC (debug SA-1111 reset) */

	/* LDD4 is speaker, LDD3 is microphone */
	PPSR &= ~(PPC_LDD3 | PPC_LDD4);
	PPDR |= PPC_LDD3 | PPC_LDD4;
}
#endif


static int __init hw_sa1100_init(void)
{
	if( machine_is_assabet() ){
#ifdef CONFIG_SA1100_ASSABET
		if(machine_has_neponset()){

		  	/* Angel sets this, but other bootloaders may not.
			 * This must precede any driver calls to BCR_set()
			 * or BCR_clear().
			 */
#ifdef CONFIG_ASSABET
		  	BCR = BCR_value = BCR_DB1111;
#endif

#ifdef CONFIG_ASSABET_NEPONSET
			LEDS = WHOAMI;
			sa1111_init();
#else
			printk( "Warning: Neponset detected but full support "
				"hasn't been configured in the kernel\n" );
#endif
		}
#endif
	} else if (machine_is_bitsy()) {
#ifdef CONFIG_SA1100_BITSY
		hw_bitsy_init();
#endif
	} else if (machine_is_xp860()) {
		sa1111_init();
	} else if (machine_is_yopy()) {
#ifdef CONFIG_SA1100_YOPY
		yopy_gpio_init();
#endif
	} else if (machine_is_jornada720()) {
#ifdef CONFIG_SA1100_JORNADA720
		hw_jornada720_init();
#endif
		sa1111_init();
	}
	return 0;
}

module_init(hw_sa1100_init);
