/*
**++
**  FACILITY:
**
**      Linked list library
**
**  MODULE NAME:
**
**      list.c
**
**  ABSTRACT:
**
**  See list.h for a full description of how to use this component.
**
**  AUTHORS:
**
**      Peter Dettori (rewritten from a version of John Court's
**      lists)
**
**  CREATION DATE:
**
**      25 February, 1999
**
** Copyright (c) Compaq Computer Corporation, 1999
*/

#ifndef __KERNEL__
#include <stdlib.h>
#include <stdio.h>
#endif

#include <linux/malloc.h>
#include "list.h"

#ifndef __KERNEL__
#define printk printf
#endif

/* 
 * Private routines.
 */
static void insert_node_after(list_node_t *, list_node_t *);
static void remove_node(list_node_t *);     


/*********************************************************************

                LIST HEADER MANIPULATION ROUTINES

*********************************************************************/

/* 
 * Algorithm - 
 */
list_error_t 
list_header_init (list_header_t *header)
{
    list_error_t error = ELIST_Ok;
  
    if ( header )
    {
	/* Initialise the header, empty */
	header->head = NULL;
	header->count = 0;
    }
    else
    {
	error = ELIST_InvalidHeaderPtr;
    }
    return error;
}

/* 
 * Algorithm - 
 */
list_error_t  
list_header_clean(list_header_t *header)
{
    list_error_t error = ELIST_Ok;
    
    if (header)
    {
	if (header->count == 0)
	{
	    header->head = NULL;
	}
	else
	{
	    error = ELIST_NonEmptyList;
	}
    }    
    else
    {
	error = ELIST_InvalidHeaderPtr;   
    }
    
    return (error);
}

/* 
 * Algorithm - 
 */
int  
list_header_getCount(list_header_t *header)
{
    int count = -1;

    if (header)
    {
	count = header->count;
    }
    return count;
}

/*********************************************************************

                LIST ELEMENT INSERTION ROUTINES

*********************************************************************/

/* 
 * Algorithm - 
 */
list_error_t  
list_add_last(list_header_t *header, list_node_t *node)
{
    list_error_t error = ELIST_Ok;
    
    if ( header && node )
    {
	node->header = header;

	/* 1. if the list is empty add it at the front */
	if (header->head == NULL)
	{
	    insert_node_after(NULL, node);
	}
	else
	{
	    insert_node_after(header->head->prev, node);
	}
    }
    else
    {
	error = ELIST_InvalidHeaderPtr;
    }
    return error;
}

/* 
 * Algorithm - 
 */
list_error_t  
list_add_first (list_header_t *header, list_node_t *node)
{
    list_error_t error = ELIST_Ok;
    
    if (header && node)
    {
	node->header = header;
	node->next = NULL;
	node->prev = node;
	insert_node_after(NULL, node);
    }
    else
    {
	error = header ? ELIST_InvalidDataPtr : ELIST_InvalidHeaderPtr;
    }
    
    return error;
}

/* 
 * Algorithm - 
 */
list_error_t  
list_add_next(list_node_t *node, list_node_t *new)
{
    list_error_t error = ELIST_Ok;
    
    if (node && new)
    {
	new->header = node->header;
	insert_node_after(node, new);
    }
    else
    {
	error = node ? ELIST_InvalidDataPtr : ELIST_InvalidItemPtr;
    }
    
    return error;
}

/* 
 * Algorithm - 
 */
list_error_t
list_add_previous(list_node_t *node, list_node_t *new)
{
    list_error_t error = ELIST_Ok;
    
    if (node && new)
    {
	new->header  = node->header;
	/* Check if the item we are to be inserted before is also 
	 * the first item in the list !.
	 */
	if (node->prev == node)
	{
	    insert_node_after(NULL, new);
	}
	else
	{
	    insert_node_after(node->prev, new);
	}
    }
    else
    {
	error = node ? ELIST_InvalidDataPtr : ELIST_InvalidItemPtr;
    }
    
    return error;
}

/*********************************************************************

                LIST ELEMENT DATA READ ROUTINES

*********************************************************************/


/* 
 * Algorithm - 
 */
list_node_t *
list_get_first (list_header_t *header)
{
    list_node_t *node = NULL; 

    if (header)
    {
	node = header->head;
    }
    
    return node;

}

/* 
 * Algorithm - 
 */
list_node_t *
list_get_last(list_header_t *header)
{
    list_node_t  *node = NULL; 

    if (header)
    {
	if (header->head)
	{
	    node = header->head->prev;
	}
    }
    
    return node;
}

/* 
 * Algorithm - 
 */
list_node_t *
list_get_next(list_node_t *prev)
{
    list_node_t *node = NULL;

    if (prev)
    {
	node = prev->next;
    }
    
    return node;
}

/* 
 * Algorithm - 
 */
list_node_t *
list_get_previous(list_node_t *next)
{
    list_node_t *node = NULL; 

    if (next)
    {
	node = next->prev;
	if (node == next)
	{
	    node = NULL;
	}
    }
    return node;
}

/*********************************************************************

                LIST ELEMENT DELETION ROUTINES

*********************************************************************/

/* 
 * Algorithm - 
 */
list_node_t *
list_return_first(list_header_t *header)
{
    list_node_t *node = NULL; 
    
    if (header)
    {
	/* Take the data out then remove the linked list item. */
	node = header->head;
	remove_node(node);
    }
    
    return node;
}


/* 
 * Algorithm - 
 */
list_node_t *
list_return_last(list_header_t *header)
{
    list_node_t *node = NULL; 

    if (header)
    {
	/* Take the data out then remove the linked list item. */
	if (header->head)
	{
	    node = header->head->prev;
	    remove_node(node);
	}
    }
    
    return node;
}

/* 
 * Algorithm - 
 */
list_node_t *
list_return_current(list_node_t *node)
{
    remove_node(node);
    return node;
}

/*********************************************************************

                PRIVATE MODULE ROUTINES

*********************************************************************/


/* 
 * Insert an ITEM after the current passed one. 
 * 
 * node    :   the node, which is already on a list, after which 
 *             the new node will be inserted. If node is null,
 *             then new inserted at the head of the list.
 * new     :   the new node, which is currently not on a list.
 *
 * Assumptions : new must have the head field occupied, this is neccessary
 *               to cater for insertion into an empty list where node is 
 *               NULL
 */
void 
insert_node_after(list_node_t *node, list_node_t *new)    
{ 

    if ( (node != NULL) && (new != NULL) )
    {
	/* an already occupied list. */
	new->next = node->next;
	new->prev = node;
	node->next = new;
	
	/* the last node is a special case 
	 * because the head node's prev ptr must 
	 * point to the last node so we can get to 
	 * the end quickly.
	 */
	if (new->next == NULL)
	{
	    /* last node */
	    new->header->head->prev = new;
	}
	else
	{
	    new->next->prev = new;
	} 
	new->header->count++;
    }
    else
    {
	if (new != NULL)
	{
	    if (new->header->head)
	    {
		/* insert at start of list */
		new->next = new->header->head;
		new->prev = new->header->head->prev;
		new->header->head->prev = new;
		new->header->head = new;
	    }
	    else
	    {
		/* empty list */
		new->header->head = new;
		new->prev = new;
		new->next = NULL;
	    }
	    new->header->count++;
	}
    }
}

/* 
 * Remove the node from the list. 
 *
 * node : a node in the list which is to be removed.
 */      
void 
remove_node(list_node_t *node)     
{ 
    list_node_t *new_head = NULL;
    
    if (node)
    {
	if (node->header->head == node)
	{
	    new_head = node->next;
	}

	if (node->next) 
	{  
	    node->next->prev = node->prev; 
	}  
	else  
	{    
	    node->header->head->prev = node->prev; 
	}    

	if ((node->prev) && (node->prev->next != NULL))
	{  
	    node->prev->next = node->next;
	}  

	node->prev = NULL; 
	node->next = NULL; 
	node->header->count--;  

	if (node->header->count == 0)
	{
	    /* just removed the last node, update head. */
	    node->header->head = NULL;
	}
	else
	{
	    /* fix up the head if this was the first node */
	    if (new_head)
	    {
		node->header->head = new_head;
	    }
	}
	node->header = NULL;
    }
}
