/*
 * Ethernet driver for the SA1100 USB client function
 * Copyright (c) 2001 by Nicolas Pitre
 *
 * This code was loosely inspired by the original initial ethernet test driver
 * Copyright (c) Compaq Computer Corporation, 1999
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This is still work in progress...
 * 
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/timer.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>

#include "usb_ctl.h"


#define ETHERNET_PRODUCT_ID 0x505A

struct usb_info_t usbd_info;


static char usb_eth_name[16] = "usbf";
static struct net_device usb_eth_device;
static struct sk_buff *cur_tx_skb, *next_tx_skb;
static struct sk_buff *cur_rx_skb, *next_rx_skb;
static volatile int terminating;


static struct sk_buff * 
usb_new_recv_skb(void)
{
	struct sk_buff *skb = dev_alloc_skb(1518);
	if (skb) {
		/*
		 * In order to align IP addresses to a word, we should 
		 * reserve 2 bytes in addition to the 14-byte ethernet 
		 * header. Since our packets come in with a 2-byte header
		 * instead of the 12-byte MAC addresses, then we need: 
		 * 	reserved = 14 + 2 - (14 - 12 + 2) = 12
		 */
		skb_reserve(skb, 12);
	}
	return skb;
}

static void 
usb_recv_callback(int flag, int size)
{
	struct sk_buff *skb;
	int pktlen, i;
	char *mac_header;
	
	if (terminating) 
		return;
	
	skb = cur_rx_skb;
	pktlen = 0;

	/* flag validation */
	if (flag == 0) {
		skb_put(skb, size);
		if (skb->len > 2) {
			pktlen = le16_to_cpu(*(u16*)skb->data);
			if (pktlen < 2 || pktlen > 1504) {
				pktlen = 0;
				usbd_info.eth_stats.rx_frame_errors++;
			}
		}
	} else if (flag == -EIO) {
		usbd_info.eth_stats.rx_errors++;
	}

	/* validate packet length */
	if (skb->len < pktlen) {
		/* packet not complete yet */
		skb = NULL;
	} else if (pktlen != 0 && skb->len > pktlen) {
		pktlen = 0;
		usbd_info.eth_stats.rx_over_errors++;
	}
	
	if (pktlen == 0) {
		/* 
		 * Error due to bad frame, bad pktlen, etc.
		 * Recycle the current skb and reset USB reception.
		 */
		skb_trim(skb, 0);
		skb = NULL;
		sa1100_usb_recv_reset();
	}
	
	/* 
	 * At this point skb is non null if we have a complete packet.
	 * If so take a fresh skb right away and restart USB receive without
	 * further delays, then process the packet.  Otherwise resume USB
	 * receive on the current skb and exit.
	 */

	if (skb)
		cur_rx_skb = next_rx_skb;
	sa1100_usb_recv(cur_rx_skb->tail, skb_tailroom(cur_rx_skb), 
			usb_recv_callback);
	if (!skb)
		return;

	next_rx_skb = usb_new_recv_skb();
	if (!next_rx_skb) {
		/*
		 * We can't aford loosing buffer space...  
		 * So we drop the current packet and recycle its skb.
		 */
		printk("%s: can't allocate new skb\n", __FUNCTION__);
		usbd_info.eth_stats.rx_dropped++;
		skb_trim(skb, 0);
		next_rx_skb = skb;
		return;
	}
	
	usbd_info.eth_stats.rx_packets++;
	usbd_info.eth_stats.rx_bytes += skb->len;

	/* 
	 * Put the mac addresses back in, over the top of the
	 * size that was sent over.
	 */
	mac_header = skb_push(skb, (2*ETH_ALEN)-2);
	for (i = 0; i < ETH_ALEN; i++) {
		mac_header[i] = usbd_info.dev->dev_addr[i];
		mac_header[i+ETH_ALEN] = usbd_info.host_addr[i];
	}

	skb->dev = usbd_info.dev;
	skb->protocol = eth_type_trans(skb, skb->dev);
	skb->ip_summed = CHECKSUM_UNNECESSARY;
	netif_rx(skb);
}


static void 
usb_send_callback(int flag, int size)
{
	struct net_device *dev = usbd_info.dev;
	struct net_device_stats *stats;
	int ret;

	if (terminating)
		return;

	stats = &usbd_info.eth_stats;
	switch (flag) {
	    case 0:
		stats->tx_packets++;
		stats->tx_bytes += size;
		break;
	    case -EIO:
		stats->tx_errors++;
		break;
	    default:
		stats->tx_dropped++;
		break;
	}

	dev_kfree_skb_irq(cur_tx_skb);
	cur_tx_skb = next_tx_skb;
	next_tx_skb = NULL;
	if (!cur_tx_skb)
		return;
	
	dev->trans_start = jiffies;
	ret = sa1100_usb_send(cur_tx_skb->data, cur_tx_skb->len, usb_send_callback);
	if (ret) {
		/* If the USB core can't accept the packet, we drop it. */
		dev_kfree_skb_irq(cur_tx_skb);
		cur_tx_skb = NULL;
		usbd_info.eth_stats.tx_carrier_errors++;
	}
	netif_wake_queue(dev);
}

static int 
usb_eth_xmit(struct sk_buff *skb, struct net_device *dev)
{
	int ret;
	long flags;
	
	if (next_tx_skb) {
		printk("%s: called with next_tx_skb != NULL\n", __FUNCTION__);
		return 1;
	}

	skb_pull(skb, 10);
	skb->data[0] = skb->len;
	skb->data[1] = skb->len >> 8;

	save_flags_cli(flags);
	if (cur_tx_skb) {
		next_tx_skb = skb;
		netif_stop_queue(dev);
	} else {
		cur_tx_skb = skb;
		dev->trans_start = jiffies;
		ret = sa1100_usb_send(skb->data, skb->len, usb_send_callback);
		if (ret) {
			/* If the USB core can't accept the packet, we drop it. */
			dev_kfree_skb(skb);
			cur_tx_skb = NULL;
			usbd_info.eth_stats.tx_carrier_errors++;
		}
	}
	restore_flags(flags);
	return 0;
}

static void 
usb_xmit_timeout(struct net_device *dev )
{
	sa1100_usb_send_reset();
	dev->trans_start = jiffies;
	netif_wake_queue(dev);
}


static int 
usb_eth_open(struct net_device *dev)
{
	terminating = 0;
	cur_tx_skb = next_tx_skb = NULL;
	cur_rx_skb = usb_new_recv_skb();
	next_rx_skb = usb_new_recv_skb();
	if (!cur_rx_skb || !next_rx_skb) {
		printk("%s: can't allocate new skb\n", __FUNCTION__);
		kfree_skb(cur_rx_skb);
		kfree_skb(next_rx_skb);
		return -ENOMEM;;
	}
	
	MOD_INC_USE_COUNT;
	sa1100_usb_recv(cur_rx_skb->tail, skb_tailroom(cur_rx_skb), 
			usb_recv_callback);
	return 0;
}

static int 
usb_eth_release(struct net_device *dev)
{
	terminating = 1;
	sa1100_usb_send_reset();
	sa1100_usb_recv_reset();
	kfree_skb(cur_tx_skb);
	kfree_skb(next_tx_skb);
	kfree_skb(cur_rx_skb);
	kfree_skb(next_rx_skb);
	MOD_DEC_USE_COUNT;
	return 0;
}

static struct net_device_stats *
usb_eth_stats(struct net_device *dev)
{
	struct usb_info_t *priv =  (struct usb_info_t *) dev->priv;
	struct net_device_stats *stats=NULL;

	if (priv)
		stats = &priv->eth_stats;
	return stats;
}

static int 
usb_eth_probe(struct net_device *dev)
{
	/* 
	 * Assign the hardware address of the board: use 
	 * 40 00 00 00 00 XX, where XX is the USB address of the
	 * device
	 */
	dev->dev_addr[0] = 0x40;
	dev->dev_addr[1] = 0;
	dev->dev_addr[2] = 0;
	dev->dev_addr[3] = 0;
	dev->dev_addr[4] = 0;
	dev->dev_addr[5] = 1; /*usbd_info.address;*/
   
	/* 
	 * we know the host mac address because it is derived from
	 * the USB address.
	 */
	usbd_info.host_addr[0] = 0x40;
	usbd_info.host_addr[1] = 0;
	usbd_info.host_addr[2] = 0;
	usbd_info.host_addr[3] = 0;
	usbd_info.host_addr[4] = 0xff;
	usbd_info.host_addr[5] = 1;/*usbd_info.address;*/

	dev->open = usb_eth_open;
	dev->stop = usb_eth_release;
	dev->hard_start_xmit = usb_eth_xmit;
	dev->get_stats = usb_eth_stats;
	dev->watchdog_timeo = 1*HZ;
	dev->tx_timeout = usb_xmit_timeout;
	dev->priv = &usbd_info;

	usbd_info.dev = dev;

	/* clear the statistics */
	memset(&usbd_info.eth_stats, 0, sizeof(struct net_device_stats));

	ether_setup(dev);
	dev->flags &= ~IFF_MULTICAST;
	dev->flags &= ~IFF_BROADCAST;
	//dev->flags |= IFF_NOARP;

	return 0;
}

static int __init
usb_eth_init(void)
{
	strncpy(usb_eth_device.name, usb_eth_name, IFNAMSIZ);
	usb_eth_device.init = usb_eth_probe;
	if (register_netdev(&usb_eth_device) != 0)
		return -EIO;
	usbctl_start();
	return 0;
}

module_init(usb_eth_init);

static void __exit
usb_eth_cleanup(void)
{
	usbctl_release();
	unregister_netdev(&usb_eth_device);
}

module_exit(usb_eth_cleanup);
