/*****************************************************************************/

/*
 *	usb_ep0.c -- SA1100 USB controller driver. Provides a the USB 
 *                   hardware control for endpoint 0.
 *
 *	Copyright (C)  Compaq Computer Corporation, 1998, 1999
 *
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/tqueue.h>
#include <linux/delay.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <asm/segment.h>

#include <asm/byteorder.h>
#include "usb_ctl.h"
#include "usb_dbg.h"

/* 
 * Prototypes
 */
void inline ep0_set_address(usb_dev_request_t *);
void ep0_set_configuration(usb_dev_request_t *, unsigned int);
void inline ep0_get_desc(usb_dev_request_t *, unsigned int ep0_status);
void inline ep0_in_data(unsigned int);
void inline ep0_end_xfer(unsigned int ep0_status);
void inline ep0_get_status(usb_dev_request_t *, unsigned int ep0_status);
void print_ep0_packet(usb_dev_request_t *);
void print_ep0_packet_char(unsigned char *packet);
int ep0_read_data(usb_dev_request_t *request);

/* 
 * EP0 State Information 
 * This is used to determine how to respond to host requests,
 * e.g. in the case where a data packet is being sent over
 * multiple USB packets.
 */
#define EP0_IDLE 0
#define EP0_IN_DATA_PHASE 1
#define EP0_OUT_DATA_PHASE 2
#define EP0_END_XFER 3
int ep0_state = EP0_IDLE;

static unsigned char *buffer;
static unsigned int buffer_cnt;
static char status_buf[2];

extern char dev_desc[];
extern char config_desc[];
/* unfortunatly there are three scenarios of IN packets,
 * which require different behaviour from the driver.
 * 1. setup requests > 8 bytes, and exact packet size
 *    -> normal case, set DE in END_XFER interrupt
 * 2. setup request <= 8 bytes.
 *    -> this requires setting DE with IPR.
 * 3. setup requests > size of response
 *    -> this requires setting IPR with DE in end_xfer interrupt
 *       so that a 0 length packet is returned instead of a stall.
 */
#define END_XFER_EXACT     0
#define END_XFER_IMMEDIATE 1
#define END_XFER_FORCED    2
static unsigned int end_xfer;

struct tq_struct config_task;

void 
ep0_reset(void)
{
    ep0_state = EP0_IDLE;
}

int get_status=0;
/* 
 * Interrupt Handler for enpoint 0.
 * 
 * This routine is called iff the EIR bit in the UDC 
 * status/interrupt register is et. It implements the 
 * flowchart detailed in the "USB Client Device Validation 
 * for the StrongARM SA-1100 Microprocessor" document.
 *
 */
void 
ep0_int_hndlr(unsigned int status)
{   
    unsigned int ep0_status = *(UDCCS0);
    int idx=0;

    if (ep0_status & UDCCS0_SST)
    {
        DEBUG_KPrint(DEBUG_EP0, ("ep0: sent stall\n"));
    }
    
    /* look for SE */
    if (ep0_status & UDCCS0_SE)
    {
        do
        {
            *(UDCCS0) = UDCCS0_SSE;
            ep0_status |= ep0_status;
            ep0_status |= ep0_status;
            ep0_status |= ep0_status;
            ep0_status = *(UDCCS0);
            idx++;
            if (idx > 1000)
            {
                printk("EP0 : SE loop\n");
            }
        } while (ep0_status & UDCCS0_SE);

        /* spec says to unload any setup packets after 
         * clearing SE.
         */
        if (ep0_status & UDCCS0_OPR)
        {
            *(UDCCS0) = UDCCS0_SO;
        }
        ep0_state = EP0_IDLE;
        end_xfer = END_XFER_EXACT;
    }
    
    if ( (ep0_state != EP0_IDLE) && (ep0_status & UDCCS0_OPR))
    {
        printk("EP0 : OUT packet won't be read %02x...\n", ep0_status);
        ep0_state = EP0_IDLE;
    }
    
    switch (ep0_state)
    {
        case EP0_IDLE :
            ep0_idle(ep0_status);
            break;
        case EP0_IN_DATA_PHASE :
            ep0_in_data(ep0_status);
            break;
        case EP0_END_XFER :
            ep0_end_xfer(ep0_status);
            break;
        default :
            break;
    }
}

void 
ep0_idle(unsigned int ep0_status)
{
    unsigned int setup_cnt;
    usb_dev_request_t request;

    if (ep0_status & UDCCS0_OPR)
    {  
        ep0_read_data(&request);
    
        /* get the length of the requested tranfser */
        setup_cnt = request.wLength;
        if (setup_cnt != 0)
        {
            /* clear OPR */
            *(UDCCS0) = UDCCS0_SO;
        }

        /* Only deal with standard devices 
         * at this point
         */
        if ((request.bmRequestType & 0x60) == 0)
        {
            /* check the request */
            switch (request.bRequest)
            {
                case SET_ADDRESS :    
                    ep0_set_address(&request);
                    break;
                case GET_DESCRIPTOR :
                    ep0_get_desc(&request, ep0_status);
                    break;
                case SET_CONFIGURATION :
                    ep0_set_configuration(&request, ep0_status);
                    break;
                    /* case --> add the others */
                case GET_CONFIGURATION :
                    if (usbd_info.state ==  USB_STATE_CONFIGURED)
                    {
                        status_buf[0] = 1;
                    }
                    else
                    {
                        status_buf[0] = 0;
                    }
                    ep0_state = EP0_IN_DATA_PHASE;
                    buffer = status_buf;
                    buffer_cnt = 1;
                    ep0_in_data(ep0_status);
                    break;
                case GET_STATUS :
                    ep0_get_status(&request, ep0_status);
                    break;
                case CLEAR_FEATURE :
                    /* device is stalled, unstall... */
                    DEBUG_KPrint(DEBUG_EP0,
                                 ("clear feature %d\n", request.wIndex));
                    /* only support endpoint feature */
                    if (request.wValue == 0)
                    {
                        /* ignore the direction */
                        switch (request.wIndex & 0x0f)
                        {
                            case 1 :
                                ep1_reset();
                                break;
                            case 2 :
                                ep2_reset();
                                break;
                            default :
                                break;
                        }
                    }
                    ep0_serviced_opr_data_end();
                    break;
                case SET_FEATURE :
                    /* only support endpoint feature */
                    DEBUG_KPrint(DEBUG_EP0, ("set feature\n"));
                    if (request.wValue == 0)
                    {
                        /* clear the direction bit */
                        udc_stallep(request.wIndex & 0x0f);
                    }
                    ep0_serviced_opr_data_end();
                    break;
                default :
		    printk("ep0: unknown request 0x%x\n", request.bRequest);
                    ep0_serviced_opr_data_end();
                    break;
            }
        }
    }
}

/* 
 *
 */
void 
ep0_set_configuration(usb_dev_request_t *request, unsigned int ep0_status)
{
    ep0_serviced_opr_data_end();

    switch (request->wValue)
    {
        case 1 :
            /* configured ... */
            DEBUG_KPrint(DEBUG_EP0, ("configured\n"));
	    printk("usb_net_func: configured\n");
            usbd_info.state =  USB_STATE_CONFIGURED;
            /* don't want to do this in the interrupt handler so
             * schedule it for later.
             */
            if (config_task.sync == 0)
            {
                config_task.routine =  udc_configure;
                config_task.data = (void *)(unsigned long)request->wValue;
                schedule_task(&config_task);
            }
            break;
        case 0 :
            /* unconfigure ... */
            break;
        default :
            break;
    }
}

void inline 
ep0_get_status(usb_dev_request_t *req, unsigned int ep0_status)
{
    get_status++;
    /* there are three types of status request.
     * 1. device    -
     * 2. interface - 
     * 3. endpoint  - 
     */
    switch (USB_REQUEST_RECIPIENT(req->bmRequestType))
    {
        case USB_RECIPIENT_DEVICE :
            /* device */
            status_buf[0] = 1;   /* self powered */
            status_buf[1] = 0;   /* no remote wakeup */
            break;
        case  USB_RECIPIENT_INTERFACE :
            status_buf[0] = 0;
            status_buf[1] = 0;
            break;
        case USB_RECIPIENT_ENDPOINT :
            /* which endpoint */
            switch (req->wIndex & 0x0f)
            {
                case 1 :
                    status_buf[0] = (*(UDCCS1) & UDCCS1_FST) >> 4;
                    break;
                case 2 :
                    status_buf[0] = (*(UDCCS2) & UDCCS2_FST) >> 5;
                    break;
                default :
                    status_buf[0] = 0;
                    break;
            }
            break;
        default :
            status_buf[0] = 0;
            status_buf[1] = 0;
            break;
    }
    ep0_state = EP0_IN_DATA_PHASE;
    buffer = status_buf;
    buffer_cnt = req->wLength;
    ep0_in_data(ep0_status);
}

/* 
 *
 */
void 
ep0_get_desc(usb_dev_request_t *request, unsigned int ep0_status)
{ 
    
    /* set the default end xfer mechanism */
    end_xfer = END_XFER_EXACT;
    
    /* wValue field specifies the descriptor type in the 
     * high byte and the descriptor index in the low byte.
     * See the USB Spec V1.0 for further details.
     */
    switch (request->wValue >> 8)
    {
        case USB_DESC_DEVICE :
            ep0_state = EP0_IN_DATA_PHASE;
            buffer = dev_desc;
            buffer_cnt = request->wLength;
            if (buffer_cnt > USB_DEVDESC_LENGTH)
            {
                buffer_cnt = USB_DEVDESC_LENGTH;
                end_xfer = END_XFER_FORCED;
            }
            else
            {
                if (buffer_cnt <= 8)
                {
                    end_xfer = END_XFER_IMMEDIATE;
                }
            }
            ep0_in_data(ep0_status);
            break;
        case USB_DESC_CONFIG :
            ep0_state = EP0_IN_DATA_PHASE;
            buffer = config_desc;
            buffer_cnt = request->wLength;
            if (buffer_cnt > (USB_CONFDESC_LENGTH + USB_IFACEDESC_LENGTH +
                  (2 * USB_ENDPDESC_LENGTH)))
            {
                buffer_cnt = (USB_CONFDESC_LENGTH + USB_IFACEDESC_LENGTH +
                              (2 * USB_ENDPDESC_LENGTH));
                end_xfer = END_XFER_FORCED;
            }
            else
            {
                if (buffer_cnt <= 8)
                {
                    end_xfer = END_XFER_IMMEDIATE;
                }
            }
            ep0_in_data(ep0_status);
            break;
        default :
	    printk("ep0: unknown descriptor type %d\n", request->wValue >> 8);
            break;
    }
}

/*
 * Set the address register of the SA1100 after a SET_ADDRESS
 * setup packet has been received.
 */
unsigned int address;
void 
ep0_set_address(usb_dev_request_t *request)
{
/*    unsigned int address;*/

    /* read wValue, the 7 bit address, and clear the 
     * upper bits
     */
    address = request->wValue;
    address = address & 0x7f;
    
    /* set the address */
    DEBUG_KPrint(DEBUG_EP0, ("setting address %d %p\n", address, UDCAR));
    *(UDCAR) = address;
    
    /* clear OPR bit and set DE bit 
     * because there is no data phase
     */ 
    ep0_serviced_opr_data_end();
    
    /* reset the UDCIMP UDCOMP registers */
    udc_write_reg(UDCIMP, usbd_info.tx_pktsize - 1);
    udc_write_reg(UDCOMP, usbd_info.rx_pktsize - 1);

    /* change the USB state */
    usbd_info.state = USB_STATE_ADDRESS;
    usbd_info.address = address;

    /* enable suspend interrupts now, so 
     * we can clear the state.
     */
    udc_enable_interrupts(1);
}

void 
ep0_in_data(unsigned int ep0_status)
{
    int transfer_cnt = 0;
    unsigned int ep0_write_count_s;
    volatile int writes=0;
    int idx;

    /* process iff ep0 is not stalled 
     * no premature end and IPR is clear
     */
    if (!(ep0_status & 0x24))
    {
        /* first check that IPR is not set, if it is then we wait
         * for it to clear.
         */
        if (ep0_status & UDCCS0_IPR)
        {
            DEBUG_KPrint(DEBUG_EP0, ("clearing IPR\n"));
	    idx = 0;
            while ((*(UDCCS0) & UDCCS0_IPR) && idx < 1000)
            {
                /* do nothing */
                writes |= writes;
                writes |= writes;
		idx++;
            }
	    if (idx == 1000)
	      printk("EP0: IPR clear timed out");

            DEBUG_KPrint(DEBUG_EP0, ("cleared IPR\n"));
        }
        
        if (buffer_cnt > 8)
        {
            /* can only send 8 at a time 
             * due to the UDCD0 FIFO size
             */
            transfer_cnt = 8;
            buffer_cnt = buffer_cnt - 8;
        }
        else
        {
            transfer_cnt = buffer_cnt;
            buffer_cnt = 0;
        }
        
        ep0_write_count_s=0;
        writes=0;
        do 
        {
            *(UDCD0) = (unsigned long) buffer[ep0_write_count_s];
            /* for some reason reading the write count register doesn't 
             * always work, presumably it is related to the synch problem 
             * in the UDC. c.f. the technical reference for more 
             * information. As the write count should stabilise
             * after each new data element., three reads seems
             * to do the trick. Based on empirical evidence.
             */
            for (idx=0;idx<3;idx++)
            {
                if ( *(UDCWC) > ep0_write_count_s)
                {
                    ep0_write_count_s = *(UDCWC);
                }
            }
            writes++;
            if (writes>1000)
            {
                DEBUG_KPrint(DEBUG_EP0, ("writes %d\n", writes));
            }
        } while ((ep0_write_count_s < transfer_cnt) && (writes<1000));
        buffer += transfer_cnt;

        if (buffer_cnt == 0)
        {
            /* set data end, and start */
            ep0_state = EP0_END_XFER;
            if (end_xfer != END_XFER_FORCED)
            {
                /* set data end, and start */
                *(UDCCS0) = UDCCS0_DE;
            }
        }

        /* now start the transfer */
        idx = 0;
        do 
        {
            idx++;
            *(UDCCS0) = UDCCS0_IPR;
        } while    (!(*(UDCCS0) & UDCCS0_IPR) && (idx<50));
        if (idx==10)
        {
            printk("EP0 : IPR loop timed out...\n");
        }
    }
    else
    {
        printk("EP0 : in status %02x\n", ep0_status);
    }
}

void 
ep0_end_xfer(unsigned int ep0_status)
{
    ep0_state = EP0_IDLE;
    if (ep0_status & UDCCS0_SST)
    {
        *(UDCCS0) = UDCCS0_SST;
    }
    if (ep0_status & UDCCS0_SE)
    {
        *(UDCCS0) = UDCCS0_SSE;
    }
    if ( !(*(UDCCS0) & UDCCS0_IPR) )
    {
        /* ensure that IPR is clear */
        switch (end_xfer)
        {
            case END_XFER_FORCED :
                *(UDCCS0) = UDCCS0_DE | UDCCS0_IPR;
                break;
            case END_XFER_EXACT :
            case END_XFER_IMMEDIATE :
                /*  *(UDCCS0) = UDCCS0_DE;*/
                break;
            default :
                break;
        }
    }
    end_xfer = END_XFER_EXACT;
}

/* 
 * After an endpoint 0 OUT transaciton completes the data end bit
 * must be set to indicate the end of data, along with the output
 * packet ready bit.
 */
void 
ep0_serviced_opr_data_end(void)
{
    int i=0;
  
    /* workaround for synchronization bug in SA-1100 */
    do
    {
        *UDCCS0 = UDCCS0_SO | UDCCS0_DE;
        i++;
        if (i == 1000)
        {
            printk("EP0 : ep0_serviced_opr_data_end counter turned over\n");
            i = 0;
        }
    } while (*UDCCS0 & UDCCS0_OPR);
}

/*
 * Read an OUT packet from the endpoint 0 fifo's. The maximum
 * length of the packet is 8bytes, the caller must ensure at 
 * least 8 bytes are available in the memory passed here.
 * 
 * returns : the length of the packet.
 */
int
ep0_read_data(usb_dev_request_t *request)
{
    unsigned int ep0_write_count, data;
    unsigned char pkt[8];
    int i, reads, count;

    /* read the write count for ep0 and filter the
     * top bits.
     */
    ep0_write_count = *(UDCWC);
    ep0_write_count = ep0_write_count & 0xff;
    if (ep0_write_count != 8)
    {
        printk("EP0 : have %d bytes; want 8\n", ep0_write_count);
    }

    /* now read the data.
     * For some reason you can't just read the
     * write count once and then read the data 
     * register the appropriate number of times 
     * because you may read the same data twice.
     * So we need to check the write count has
     * decremented before using the data.
     */
    i = 0;
    reads = 0;
    while (ep0_write_count>0)
    {
        /* the register read must be 32 bit 
         * so use this temporary variable to 
         * read the 8bit register.
         */
        do
        {
            data = *(UDCD0);
	    udelay(1);
            reads++;
            count = *(UDCWC); 
        } while (count == ep0_write_count);
        ep0_write_count = count;
        pkt[i] = (char) (data & 0xff);
        i++;
        if (i>=8 && ep0_write_count > 0)
        {
            printk("EP0 : reading too much data\n");
	    break;
        }
    }

    /* now convert the packet into a dev request
     * with the byte ordering adjusted appropriatly.
     */
    memcpy(request, pkt, sizeof(usb_dev_request_t));
    return i;
}

/*
 * Prints the 8 byte control packet passed to it, this is used 
 * for debugging purposes only.
 */
void 
print_ep0_packet_char(unsigned char *packet)
{
    int idx;

    printk("ep0:");
    for (idx = 0; idx < 8; ++idx)
    {
        printk(" %02x", packet[idx]);
    }
    printk("\n");
}

/*
 * Prints the 8 byte control packet passed to it, this is used 
 * for debugging purposes only.
 */
void 
print_ep0_packet(usb_dev_request_t *request)
{
    printk("type %x\nrequest %x\nvalue %x\nindex %x\nlength %x\n", 
       request->bmRequestType,
       request->bRequest,
       request->wValue,
       request->wIndex,
       request->wLength);
}
