/*
 * Generic xmit layer for the SA1100 USB client function
 * Copyright (c) 2001 by Nicolas Pitre
 * 
 * This code was loosely inspired by the original version which was
 * Copyright (c) Compaq Computer Corporation, 1998-1999
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * 
 * This is still work in progress...
 * 
 */

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/errno.h>
#include <asm/hardware.h>
#include <asm/dma.h>
#include <asm/system.h>

#include "usb_ctl.h"


static char *ep2_buf;
static int ep2_len;
static usb_callback_t ep2_callback;
static dma_addr_t ep2_dma;
static dma_addr_t ep2_curdmapos;
static int ep2_curdmalen;
static int ep2_remain;
static int ep2_stall;

static int dmachn_tx;
	

static void 
ep2_start(void)
{
	if (!ep2_len || ep2_stall)
		return;
	
	ep2_curdmalen = usbd_info.tx_pktsize;
	if (ep2_curdmalen > ep2_remain)
		ep2_curdmalen = ep2_remain;
	
	UDC_write(Ser0UDCIMP, ep2_curdmalen-1);
	sa1100_dma_queue_buffer(dmachn_tx, NULL, ep2_curdmapos, ep2_curdmalen);
}

static void
ep2_done(int flag)
{
	int size = ep2_len - ep2_remain;
	if (ep2_len) {
		pci_unmap_single(NULL, ep2_dma, ep2_len, PCI_DMA_TODEVICE);
		ep2_len = 0;
		if (ep2_callback)
			ep2_callback(flag, size);
	}
}

int 
ep2_init(int chn)
{
	dmachn_tx = chn;
	sa1100_dma_flush_all(dmachn_tx);
	ep2_done(-EAGAIN);
	return 0;
}

void
ep2_reset(void)
{
	sa1100_dma_flush_all(dmachn_tx);
	ep2_stall = 1;
	UDC_set(Ser0UDCCS2, UDCCS2_FST);
	ep2_done(-EINTR);
}

void
ep2_int_hndlr(int udcsr)
{
	int status = Ser0UDCCS2;

	if (status & UDCCS2_TPC) {
		sa1100_dma_flush_all(dmachn_tx);
		UDC_flip(Ser0UDCCS2, UDCCS2_TPC|UDCCS2_SST);

		if (status & (UDCCS2_TPE | UDCCS2_TUR)) {
			printk("usb_send: transmit error %x\n", status);
			ep2_done(-EIO);
		} else if (ep2_stall) {
			ep2_stall = 0;
			ep2_start();
		} else {
			ep2_curdmapos += ep2_curdmalen;
			ep2_remain -= ep2_curdmalen;
			if (ep2_remain != 0) {
				ep2_start();
			} else {
				ep2_done(0);
			}
		}
	} else {
		printk("usb_send: UDCCS2 = %x\n", status);
		UDC_flip(Ser0UDCCS2, UDCCS2_SST);
		UDC_flip(Ser0UDCCS2, UDCCS2_TPC);
	}
}

int
sa1100_usb_send(char *buf, int len, usb_callback_t callback)
{
	int flags;
	
	if (usbd_info.state != USB_STATE_CONFIGURED)
		return -ENODEV;

	if (ep2_len)
		return -EBUSY;
	
	local_irq_save(flags);
	ep2_buf = buf;
	ep2_len = len;
	ep2_dma = pci_map_single(NULL, buf, len, PCI_DMA_TODEVICE);
	ep2_callback = callback;
	ep2_remain = len;
	ep2_curdmapos = ep2_dma;
	ep2_start();
	local_irq_restore(flags);
	
	return 0;
}

EXPORT_SYMBOL(sa1100_usb_send);

void
sa1100_usb_send_reset(void)
{
	ep2_reset();
}

EXPORT_SYMBOL(sa1100_usb_send_reset);

