/*
 *  linux/arch/arm/mm/mm-sa1100.c
 *
 *  Copyright (C) 1998-1999 Russell King
 *  Copyright (C) 1999 Hugo Fiennes
 *
 *  Extra MM routines for the SA1100 architecture
 *
 *  1999/12/04 Nicolas Pitre <nico@cam.org>
 *	Converted memory definition for struct meminfo initialisations.
 *	Memory is listed physically now.
 *
 *  2000/04/07 Nicolas Pitre <nico@cam.org>
 *	Reworked for run-time selection of memory definitions
 *
 */
#include <linux/config.h>
#include <linux/mm.h>
#include <linux/init.h>
#include <linux/bootmem.h>

#include <asm/hardware.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <asm/mach-types.h>

#include <asm/mach/map.h>
 
static struct map_desc standard_io_desc[] __initdata = {
 /* virtual     physical    length      domain     r  w  c  b */ \
  { 0xf6000000, 0x20000000, 0x01000000, DOMAIN_IO, 1, 1, 0, 0 }, /* PCMCIA0 IO */
  { 0xf7000000, 0x30000000, 0x01000000, DOMAIN_IO, 1, 1, 0, 0 }, /* PCMCIA1 IO */
  { 0xf8000000, 0x80000000, 0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* PCM */
  { 0xfa000000, 0x90000000, 0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* SCM */
  { 0xfc000000, 0xa0000000, 0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* MER */
  { 0xfe000000, 0xb0000000, 0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* LCD + DMA */
  LAST_DESC
};

/*
 * Typically, static virtual address mappings are as follow:
 * 
 * 0xe8000000-0xefffffff:	flash memory (especially when multiple flash
 * 				banks need to be mapped contigously)
 * 0xf0000000-0xf3ffffff:	miscellaneous stuff (CPLDs, etc.)
 * 0xf4000000-0xf4ffffff:	SA-1111
 * 0xf5000000-0xf5ffffff:	reserved (used by cache flushing area)
 * 0xf6000000-0xffffffff:	reserved (internal SA1100 IO defined above)
 * 
 * Below 0xe8000000 is reserved for vm allocation.
 */

static struct map_desc assabet_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_ASSABET
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x10000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* System Registers */
  { 0xf1000000, 0x12000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* Board Control Register */
  { 0xf2800000, 0x4b800000, 0x00800000, DOMAIN_IO, 1, 1, 0, 0 }, /* MQ200 */
  { 0xf4000000, 0x40000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* SA-1111 */
#endif
  LAST_DESC
};

/**
   memory information (JOR):
   32 MByte - 256KByte bootloader (init at boot time) - 32 kByte save area
   area size = 288 kByte (0x48000 Bytes)
**/
static struct map_desc huw_webpanel_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_HUW_WEBPANEL
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0, neccessary for mtd */
  { 0xf0000000, 0xc1fb8000, 0x00048000, DOMAIN_IO, 1, 1, 0, 0 }, /* Parameter */
  { 0xf1000000, 0x18000000, 0x00100000, DOMAIN_IO, 0, 1, 0, 0 }, /* Paules CS3, write only */
#endif
  LAST_DESC
};

static struct map_desc bitsy_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_BITSY
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x49000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* EGPIO 0 */
  { 0xf1000000, 0x10000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* static memory bank 2 */
  { 0xf2000000, 0x40000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* static memory bank 4 */
#endif
  LAST_DESC
};

static struct map_desc jornada720_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_JORNADA720
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x48000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* Epson registers */
  { 0xf1000000, 0x48200000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* Epson frame buffer */
  { 0xf4000000, 0x40000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* SA-1111 */
#endif
  LAST_DESC
};

/* BRADFIXME The egpio addresses aren't verifiably correct. (i.e. they're most
   likely wrong.
*/
static struct map_desc itsy_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_ITSY
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x49000000, 0x01000000, DOMAIN_IO, 1, 1, 0, 0 }, /* EGPIO 0 */
#endif
  LAST_DESC
};


static struct map_desc freebird_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_FREEBIRD
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x12000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* Board Control Register */
  { 0xf2000000, 0x19000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0},
#endif
   LAST_DESC
};
static struct map_desc cerf_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_CERF
  { 0xe8000000, 0x00000000, 0x01000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x08000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* Crystal Chip */
#endif
  LAST_DESC
};

static struct map_desc empeg_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_EMPEG
  { EMPEG_FLASHBASE, 0x00000000, 0x00200000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash */
#endif
  LAST_DESC
};

static struct map_desc graphicsclient_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_GRAPHICSCLIENT
  { 0xe8000000, 0x08000000, 0x01000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 1 */
  { 0xf0000000, 0x10000000, 0x00400000, DOMAIN_IO, 0, 1, 0, 0 }, /* CPLD */
#endif
  LAST_DESC
};

static struct map_desc lart_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_LART
  { 0xe8000000, 0x00000000, 0x00400000, DOMAIN_IO, 1, 1, 0, 0 }, /* main flash memory */
  { 0xec000000, 0x08000000, 0x00400000, DOMAIN_IO, 1, 1, 0, 0 }, /* main flash, alternative location */
#endif
  LAST_DESC
};

static struct map_desc pleb_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_PLEB
  { 0xe8000000, 0x00000000, 0x00400000, DOMAIN_IO, 1, 1, 0, 0 }, /* main flash memory */
  { 0xe8400000, 0x08000000, 0x00400000, DOMAIN_IO, 1, 1, 0, 0 }, /* main flash, alternative location */
#endif
  LAST_DESC
};

static struct map_desc nanoengine_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_NANOENGINE
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf0000000, 0x10000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* System Registers */
  { 0xf1000000, 0x18A00000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* Internal PCI Config Space */
#endif
  LAST_DESC
};

static struct map_desc sherman_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_SHERMAN
  { 0xe8000000, 0x00000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash*/
#endif
  LAST_DESC
};

static struct map_desc victor_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_VICTOR
  { 0xe8000000, 0x00000000, 0x00200000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash */
#endif
  LAST_DESC
};

static struct map_desc xp860_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_XP860
  { 0xf0000000, 0x10000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* SCSI */
  { 0xf1000000, 0x18000000, 0x00100000, DOMAIN_IO, 1, 1, 0, 0 }, /* LAN */
  { 0xf4000000, 0x40000000, 0x00800000, DOMAIN_IO, 1, 1, 0, 0 }, /* SA-1111 */
#endif
  LAST_DESC
};

static struct map_desc yopy_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_YOPY
  { 0xe8000000, 0x00000000, 0x04000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash 0 */
  { 0xec000000, 0x08000000, 0x04000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash 1 */
  { 0xf0000000, 0x48000000, 0x00300000, DOMAIN_IO, 1, 1, 0, 0 }, /* LCD */
  { 0xf1000000, 0x10000000, 0x00100000, DOMAIN_IO, 0, 1, 0, 0 }, /* EGPIO */
#endif
  LAST_DESC
};

static struct map_desc omnimeter_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_OMNIMETER
  { 0xd2000000, 0x10000000, 0x02000000, DOMAIN_IO, 1, 1, 0, 0 }, /* TS */
#endif
  LAST_DESC
};

static struct map_desc pangolin_io_desc[] __initdata = {
#ifdef CONFIG_SA1100_PANGOLIN
  { 0xe8000000, 0x00000000, 0x04000000, DOMAIN_IO, 1, 1, 0, 0 }, /* Flash bank 0 */
  { 0xf2800000, 0x4b800000, 0x00800000, DOMAIN_IO, 1, 1, 0, 0 }, /* MQ200 */
#endif
  LAST_DESC
};

void __init sa1100_map_io(void)
{
	struct map_desc *desc = NULL;

	iotable_init(standard_io_desc);

	if (machine_is_assabet())
		desc = assabet_io_desc;
	else if (machine_is_bitsy())
		desc = bitsy_io_desc;
	else if (machine_is_freebird())
		desc = freebird_io_desc;
	else if (machine_is_itsy())
		desc = itsy_io_desc;
	else if (machine_is_cerf())
		desc = cerf_io_desc;
	else if (machine_is_empeg())
		desc = empeg_io_desc;
	else if (machine_is_graphicsclient())
		desc = graphicsclient_io_desc;
	else if (machine_is_jornada720())
		desc = jornada720_io_desc;
	else if (machine_is_lart())
	        desc = lart_io_desc;
	else if (machine_is_pleb())
	        desc = pleb_io_desc;
	else if (machine_is_nanoengine())
		desc = nanoengine_io_desc;
	else if (machine_is_omnimeter())
		desc = omnimeter_io_desc;
	else if (machine_is_sherman())
		desc = sherman_io_desc;
	else if (machine_is_victor())
		desc = victor_io_desc;
	else if (machine_is_xp860())
		desc = xp860_io_desc;
	else if (machine_is_yopy())
		desc = yopy_io_desc;
	else if (machine_is_pangolin())
		desc = pangolin_io_desc;
	else if (machine_is_huw_webpanel())
		desc = huw_webpanel_io_desc;

	if (desc)
		iotable_init(desc);
}


#ifdef CONFIG_DISCONTIGMEM

/*
 * Our node_data structure for discontigous memory.
 * There is 4 possible nodes i.e. the 4 SA1100 RAM banks.
 */

static bootmem_data_t node_bootmem_data[4];

pg_data_t sa1100_node_data[4] =
{ { bdata: &node_bootmem_data[0] },
  { bdata: &node_bootmem_data[1] },
  { bdata: &node_bootmem_data[2] },
  { bdata: &node_bootmem_data[3] } };

#endif

  
/*
 * On Assabet, we must probe for the Neponset board *before* paging_init() 
 * has occured to actually determine the amount of RAM available.  To do so, 
 * we map the appropriate IO section in the page table here in order to 
 * access GPIO registers.
 */
void __init map_sa1100_gpio_regs( void )
{
	unsigned long phys = _GPLR & PMD_MASK;
	unsigned long virt = io_p2v(phys);
	int prot = PMD_TYPE_SECT | PMD_SECT_AP_WRITE | PMD_DOMAIN(DOMAIN_IO);
	pmd_t pmd;
	pmd_val(pmd) = phys | prot;
	set_pmd(pmd_offset(pgd_offset_k(virt), virt), pmd);
}

#ifdef CONFIG_SA1100_JORNADA720
/*
 * On Jornada, the parameters are passed far away from the kernel!
 */
void __init map_jornada_params( void )
{
	unsigned long phys = 0xc0000000;
	unsigned long virt = phys;
	int prot = PMD_TYPE_SECT | PMD_SECT_AP_WRITE | PMD_DOMAIN(DOMAIN_KERNEL);
	pmd_t pmd;
	pmd_val(pmd) = phys | prot;
	set_pmd(pmd_offset(pgd_offset_k(virt), virt), pmd);
}
#endif
