/*
 * MTD map driver for pre-CFI Sharp flash chips
 *
 * Copyright 2000 David Schleef <ds@lineo.com>
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/mtd/map.h>
#include <linux/mtd/cfi.h>
#include <linux/delay.h>

struct mtd_info *sharp_probe(struct map_info *);

static int sharp_probe_map(struct map_info *map);

static int sharp_read(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char *buf);
static int sharp_write(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, const u_char *buf);
static int sharp_erase(struct mtd_info *mtd, struct erase_info *instr);
static void sharp_sync(struct mtd_info *mtd);
static int sharp_suspend(struct mtd_info *mtd);
static void sharp_resume(struct mtd_info *mtd);
static void sharp_destroy(struct mtd_info *mtd);

static int sharp_write_oneword(struct map_info *map, struct flchip *chip,
	unsigned long adr, __u32 datum);
static int sharp_erase_oneblock(struct map_info *map, struct flchip *chip,
	unsigned long adr);

static struct map_driver sharp_driver={
	name:		"sharp",
	module:		THIS_MODULE,
	probe:		sharp_probe,
};

struct sharp_info{
	struct flchip *chip;
	int bogus;
	int chipshift;
	int numchips;
	struct flchip chips[1];
};


struct mtd_info *sharp_probe(struct map_info *map)
{
	struct mtd_info *mtd = NULL;
	struct sharp_info *sharp = NULL;
	int width;

	width = sharp_probe_map(map);
	if(!width)
		return NULL;

	mtd = kmalloc(sizeof(*mtd), GFP_KERNEL);
	if(!mtd)
		return NULL;

	sharp = kmalloc(sizeof(*sharp), GFP_KERNEL);
	if(!sharp)
		return NULL;

	memset(mtd, 0, sizeof(*mtd));
	mtd->priv = map;
	mtd->type = MTD_NORFLASH;
	mtd->erasesize = 0x10000 * width;
	mtd->size = 0x200000 * width;
	mtd->erase = sharp_erase;
	mtd->read = sharp_read;
	mtd->write = sharp_write;
	mtd->sync = sharp_sync;
	mtd->suspend = sharp_suspend;
	mtd->resume = sharp_resume;
	mtd->flags = MTD_CAP_NORFLASH;
	mtd->name = map->name;

	sharp->chipshift = 23;
	sharp->numchips = 1;
	sharp->chips[0].start = 0;
	sharp->chips[0].state = FL_READY;
	sharp->chips[0].mutex = &sharp->chips[0]._spinlock;
	init_waitqueue_head(&sharp->chips[0].wq);
	spin_lock_init(&sharp->chips[0]._spinlock);

	map->fldrv_destroy = sharp_destroy;
	map->fldrv_priv = sharp;

	return mtd;
}

static int sharp_probe_map(struct map_info *map)
{
	unsigned long tmp;
	unsigned long base = 0;
	u32 read0, read4;

	tmp = map->read32(map, base+0);

	map->write32(map, 0x90909090, base+0);

	read0=map->read32(map, base+0);
	read4=map->read32(map, base+4);
	if(read0 == 0x89898989){
		printk("Looks like sharp flash\n");
		if((read4 == 0xaaaaaaaa) || (read4 == 0xa0a0a0a0)) {
			/* aa - 16Mb (that's 2MB to you and me) */
			/* a0 - 16Mb -Z4 */
			return 4;
		}else{
			printk("Sort-of looks like sharp flash, 0x%08x 0x%08x\n",
				read0,read4);
		}
	}else if((map->read32(map, base+0) == 0x90909090)){
		/* RAM, probably */
		printk("Looks like RAM\n");
		map->write32(map, tmp, base+0);
	}else{
		printk("Doesn't look like sharp flash, 0x%08x 0x%08x\n",
			read0,read4);
	}

	return 0;
}

/* This function returns with the chip->mutex lock held. */
static int sharp_wait(struct map_info *map, struct flchip *chip)
{
	__u16 status;
	unsigned long timeo = jiffies + HZ;
	DECLARE_WAITQUEUE(wait, current);
	int adr = 0;

retry:
	spin_lock_bh(chip->mutex);

	switch(chip->state){
	case FL_READY:
		map->write32(map,0x70707070,adr);
		chip->state = FL_STATUS;
	case FL_STATUS:
		status = map->read32(map,adr);
//printk("status=%08x\n",status);

		udelay(100);
		if((status & 0x80808080)!=0x80808080){
//printk(".status=%08x\n",status);
			udelay(100);
		}
		break;
	default:
		printk("Waiting for chip\n");

		set_current_state(TASK_INTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);

		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);

		if(signal_pending(current))
			return -EINTR;

		timeo = jiffies + HZ;

		goto retry;
	}

	map->write32(map,0xffffffff, adr);

	chip->state = FL_READY;

	return 0;
}

static void sharp_release(struct flchip *chip)
{
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);
}

/* Needs to be called _with_ lock, and returns with lock
 * unless there is an error */
static int sharp_wait_state(struct map_info *map,struct flchip *chip,
	int state,unsigned long adr)
{
	int timeo;
	DECLARE_WAITQUEUE(wait, current);

	if(chip->state != state){
		set_current_state(TASK_INTERRUPTIBLE);
		add_wait_queue(&chip->wq, &wait);

		spin_unlock_bh(chip->mutex);

		schedule();
		remove_wait_queue(&chip->wq, &wait);

		if (signal_pending(current))
			return -EINTR;
		
		timeo = jiffies + (HZ/2);

		spin_lock_bh(chip->mutex);
	}
	if(chip->state != state){
		return 0;
	}

	return 1;
}

static int sharp_read(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct sharp_info *sharp = map->fldrv_priv;
	int chipnum;
	int ret = 0;
	int ofs = 0;

	chipnum = (from >> sharp->chipshift);
	ofs = from & ((1 << sharp->chipshift)-1);

	*retlen = 0;

	while(len){
		unsigned long thislen;

		if(chipnum>=sharp->numchips)
			break;

		thislen = len;
		if(ofs+thislen >= (1<<sharp->chipshift))
			thislen = (1<<sharp->chipshift) - ofs;

		ret = sharp_wait(map,&sharp->chips[chipnum]);
		if(ret<0)
			break;

		map->copy_from(map,buf,ofs,thislen);

		sharp_release(&sharp->chips[chipnum]);

		*retlen += thislen;
		len -= thislen;
		buf += thislen;

		ofs = 0;
		chipnum++;
	}
	return ret;
}

static int sharp_write(struct mtd_info *mtd, loff_t to, size_t len,
	size_t *retlen, const u_char *buf)
{
	struct map_info *map = mtd->priv;
	struct sharp_info *sharp = map->fldrv_priv;
	int ret = 0;
	int i,j;
	int chipnum;
	unsigned long ofs;
	union { u32 l; unsigned char uc[4]; } tbuf;

	*retlen = 0;

	while(len){
		tbuf.l = 0xffffffff;
		chipnum = to >> sharp->chipshift;
		ofs = to & ((1<<sharp->chipshift)-1);

		j=0;
		for(i=ofs&3;i<4 && len;i++){
			tbuf.uc[i] = *buf;
			buf++;
			to++;
			len--;
			j++;
		}
		sharp_write_oneword(map, &sharp->chips[chipnum], ofs&~3, tbuf.l);
		if(ret<0)
			return ret;
		(*retlen)+=j;
	}

	return 0;
}

static int sharp_write_oneword(struct map_info *map, struct flchip *chip,
	unsigned long adr, __u32 datum)
{
	int ret;
	int timeo;
	int z;

	ret = sharp_wait(map,chip);

	map->write32(map,0x40404040,adr);
	/* cpu_to_le32 -> hack to fix the writel be->le conversion */
	map->write32(map,cpu_to_le32(datum),adr);

	chip->state = FL_WRITING;

	timeo = jiffies + (HZ/2);

	spin_unlock_bh(chip->mutex);
	udelay(chip->word_write_time);
	spin_lock_bh(chip->mutex);

	while(1){
		ret = sharp_wait_state(map,chip,FL_WRITING,adr);
		if(ret < 0)return ret;
		if(ret==1)break;

		if (time_after(jiffies, timeo)) {
			chip->state = FL_STATUS;
			spin_unlock_bh(chip->mutex);
			printk("waiting for chip to be ready timed out in read\n");
			return -EIO;
		}

		spin_unlock_bh(chip->mutex);

		z++;

		udelay(1);

		spin_lock_bh(chip->mutex);
		continue;
	}
	chip->word_write_time += (z)?+1:-1;
	if(chip->word_write_time<1)chip->word_write_time = 1;

	chip->state = FL_STATUS;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

	return 0;
}

static int sharp_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	struct map_info *map = mtd->priv;
	struct sharp_info *sharp = map->fldrv_priv;
	unsigned long adr,len;
	int chipnum, ret=0;

printk("sharp_erase()\n");
	if(instr->addr & (mtd->erasesize - 1))
		return -EINVAL;
	if(instr->len & (mtd->erasesize - 1))
		return -EINVAL;
	if(instr->len + instr->addr > mtd->size)
		return -EINVAL;

	chipnum = instr->addr >> sharp->chipshift;
	adr = instr->addr & ((1<<sharp->chipshift)-1);
	len = instr->len;

	while(len){
		ret = sharp_erase_oneblock(map, &sharp->chips[chipnum], adr);
		if(ret)return ret;

		adr += mtd->erasesize;
		len -= mtd->erasesize;
		if(adr >> sharp->chipshift){
			adr = 0;
			chipnum++;
			if(chipnum>=sharp->numchips)
				break;
		}
	}

printk("callback: %p\n",instr->callback);
	if(instr->callback)
		instr->callback(instr);

	return 0;
}

static int sharp_erase_oneblock(struct map_info *map, struct flchip *chip,
	unsigned long adr)
{
	int ret;
	int timeo;

printk("sharp_erase_oneblock()\n");
	ret = sharp_wait(map,chip);

	map->write32(map,0x20202020,adr);
	map->write32(map,0xd0d0d0d0,adr);

	chip->state = FL_ERASING;

	spin_unlock_bh(chip->mutex);
	schedule_timeout(HZ);
	spin_lock_bh(chip->mutex);

	timeo = jiffies + HZ;

	while(1){
		ret = sharp_wait_state(map,chip,FL_ERASING,adr);
		if(ret < 0)return ret;
		if(ret==1)break;

		if (time_after(jiffies, timeo)) {
			chip->state = FL_STATUS;
			spin_unlock_bh(chip->mutex);
			printk("waiting for chip to be ready timed out in erase\n");
			return -EIO;
		}

		spin_unlock_bh(chip->mutex);

		udelay(1);

		spin_lock_bh(chip->mutex);
		continue;
	}

	chip->state = FL_STATUS;
	wake_up(&chip->wq);
	spin_unlock_bh(chip->mutex);

printk("sharp_erase_oneblock() done\n");

	return 0;
}

static void sharp_sync(struct mtd_info *mtd)
{
	printk("sharp_sync()\n");
	
}

static int sharp_suspend(struct mtd_info *mtd)
{
	printk("sharp_suspend()\n");
	return -EINVAL;
}

static void sharp_resume(struct mtd_info *mtd)
{
	printk("sharp_resume()\n");
	
}

static void sharp_destroy(struct mtd_info *mtd)
{
	printk("sharp_destroy()\n");

}


static int __init sharp_init(void)
{
	printk("MTD Sharp chip driver <ds@lineo.com>\n");
	mtd_register_map_driver(&sharp_driver);
}

static void __exit sharp_cleanup(void)
{
	mtd_unregister_map_driver(&sharp_driver);
}

module_init(sharp_init);
module_cleanup(sharp_cleanup);


