/*
 * drivers/pcmcia/sa1100_backpaq.c
 *
 * PCMCIA implementation routines for Compaq CRL Mercury BackPAQ
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/arch/pcmcia.h>
#include <asm/arch/h3600-sleeve.h>

extern void sa1100_bitsy_change_sleeves(int);
static int __devinit cf_probe_sleeve(struct sleeve_dev *sleeve_dev, const struct sleeve_device_id *ent)
{
        sprintf(sleeve_dev->name, "CompactFlash Sleeve");
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);
        sa1100_bitsy_change_sleeves(SINGLE_COMPACTFLASH_SLEEVE);
        return 0;
}

static void __devexit cf_remove_sleeve(struct sleeve_dev *sleeve_dev)
{
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->driver->name);
        sa1100_bitsy_change_sleeves(SLEEVE_NO_DEVICE);
}

static struct sleeve_device_id cf_tbl[] __devinitdata = {
        { COMPAQ_VENDOR_ID, SINGLE_COMPACTFLASH_SLEEVE, 0, 0, 0 },
        { 0, }
};

static struct sleeve_driver cf_driver = {
        name:   "Compaq Compact Flash Sleeve",
        id_table: cf_tbl,
        probe: cf_probe_sleeve,
        remove: cf_remove_sleeve,
};

static unsigned long pcmcia_sleeve_flash_size = 0x02000000;
static struct mtd_partition pcmcia_sleeve_flash_partitions[] = {
	{
		name: "PCMCIA Sleeve Flash",
		offset: 0x00d00000,
		size: 0  /* will expand to the end of the flash */
	}
};

#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

#define WINDOW_ADDR 0xf1000000 /* static memory bank 1 */

static __u8 sa1100_read8(struct map_info *map, unsigned long ofs)
{
	return *(__u8 *)(WINDOW_ADDR + ofs);
}

static __u16 sa1100_read16(struct map_info *map, unsigned long ofs)
{
	return *(__u16 *)(WINDOW_ADDR + ofs);
}

static __u32 sa1100_read32(struct map_info *map, unsigned long ofs)
{
	return *(__u32 *)(WINDOW_ADDR + ofs);
}

static void sa1100_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
        int i;
        for (i = 0; i < len; i++) {
                *(char *)(to++) = *(char *)(WINDOW_ADDR + from + i);
        }
}

static void sa1100_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	*(__u8 *)(WINDOW_ADDR + adr) = d;
}

static void sa1100_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	*(__u16 *)(WINDOW_ADDR + adr) = d;
}

static void sa1100_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	*(__u32 *)(WINDOW_ADDR + adr) = d;
}

static void sa1100_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
        int i;
        for (i = 0; i < len; i++) {
                *(char *)(WINDOW_ADDR + to + i) = *(char *)(to++);
        }
}

static struct map_info pcmcia_sleeve_flash_map = {
	name:		"PCMCIA Sleeve Flash ",
	read8:		sa1100_read8,
	read16:		sa1100_read16,
	read32:		sa1100_read32,
	copy_from:	sa1100_copy_from,
	write8:		sa1100_write8,
	write16:	sa1100_write16,
	write32:	sa1100_write32,
	copy_to:	sa1100_copy_to,
        size:           0x00200000,
        buswidth:       2
};

static struct mtd_info *pcmcia_sleeve_mtd;

static int pcmcia_sleeve_attach_flash(void)
{
        int i;
        MSC1 = 0xFFF8FFF8;
        for (i = 0; i < 32; i+= 2) {
                printk("h3600_backpaq: probing %p yields %08lx\n", WINDOW_ADDR+i, *(short *)(WINDOW_ADDR + i));
        }
	pcmcia_sleeve_mtd = do_cfi_probe(&pcmcia_sleeve_flash_map);
	if (pcmcia_sleeve_mtd) {
		pcmcia_sleeve_mtd->module = THIS_MODULE;
		add_mtd_partitions(pcmcia_sleeve_mtd, pcmcia_sleeve_flash_partitions, NB_OF(pcmcia_sleeve_flash_partitions));
		printk(KERN_NOTICE "PCMCIA flash access initialized\n");
		return 0;
	}

	return -ENXIO;
}

static void pcmcia_sleeve_detach_flash(void)
{
        if (pcmcia_sleeve_mtd != NULL) {
                del_mtd_partitions(pcmcia_sleeve_mtd);
                map_destroy(pcmcia_sleeve_mtd);
        }
}

static int __devinit pcmcia_probe_sleeve(struct sleeve_dev *sleeve_dev, const struct sleeve_device_id *ent)
{
        sprintf(sleeve_dev->name, "PC Card Sleeve");
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);
        sa1100_bitsy_change_sleeves(SINGLE_PCMCIA_SLEEVE);
        pcmcia_sleeve_attach_flash();
        return 0;
}

static void __devexit pcmcia_remove_sleeve(struct sleeve_dev *sleeve_dev)
{
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);
        pcmcia_sleeve_detach_flash();
        sa1100_bitsy_change_sleeves(SLEEVE_NO_DEVICE);
}

static struct sleeve_device_id pcmcia_tbl[] __devinitdata = {
        { COMPAQ_VENDOR_ID, SINGLE_PCMCIA_SLEEVE, 0, 0, 0 },
        { 0, }
};

static struct sleeve_driver pcmcia_driver = {
        name:   "Compaq PC Card Sleeve",
        id_table: pcmcia_tbl,
        probe: pcmcia_probe_sleeve,
        remove: pcmcia_remove_sleeve,
};

static int __devinit backpaq_probe_sleeve(struct sleeve_dev *sleeve_dev, const struct sleeve_device_id *ent)
{
        sprintf(sleeve_dev->name, "Compaq Mercury Backpaq");
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);
        sa1100_bitsy_change_sleeves(MERCURY_BACKPAQ);
        return 0;
}

static void __devexit backpaq_remove_sleeve(struct sleeve_dev *sleeve_dev)
{
        printk(__FUNCTION__ ": dev->name=%s\n", sleeve_dev->name);
        sa1100_bitsy_change_sleeves(SLEEVE_NO_DEVICE);
}

static struct sleeve_device_id backpaq_tbl[] __devinitdata = {
        { SLEEVE_ANY_ID, MERCURY_BACKPAQ, 0, 0, 0 },
        { 0, }
};

static struct sleeve_driver backpaq_driver = {
        name:   "Compaq Mercury Backpaq",
        id_table: backpaq_tbl,
        probe: backpaq_probe_sleeve,
        remove: backpaq_remove_sleeve,
};

int __init backpaq_init_module(void)
{
        h3600_sleeve_register_driver(&cf_driver);
        h3600_sleeve_register_driver(&pcmcia_driver);
        h3600_sleeve_register_driver(&backpaq_driver);
	return 0;
}

void backpaq_cleanup_module(void)
{
	printk(__FUNCTION__ ":\n");
        h3600_sleeve_unregister_driver(&cf_driver);
        h3600_sleeve_unregister_driver(&pcmcia_driver);
        h3600_sleeve_unregister_driver(&backpaq_driver);
}

module_init(backpaq_init_module);
module_exit(backpaq_cleanup_module);

