/*****************************************************************************/

/*
 *      usbh_h3600.c  --  iPAQ h3600 network driver
 *
 *	Modified from plusb.c network driver by C Flynn.
 *
 *      Copyright (C) 2000  Deti Fliegl (deti@fliegl.de)
 *
 *	
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *
 *  $Id: usbh_h3600.c,v 1.5 2000/10/27 07:34:47 flynn Exp $
 *
 */

/*****************************************************************************/

#include <linux/module.h>
#include <linux/socket.h>
#include <linux/miscdevice.h>
#include <linux/list.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>
#include <linux/delay.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
/* enable DEBUG to view output of dbg statements */
/* #define DEBUG 1 */
#include <linux/usb.h>

#include "usbh_h3600.h"

#define INTURB

/* --------------------------------------------------------------------- */

#define NRPLUSB 4
static int findEndPoints(struct usb_device *, plusb_t *);
static void dumpOctets(unsigned char *, unsigned char * ,unsigned int );

/*-------------------------------------------------------------------*/

static plusb_t plusb[NRPLUSB];
/* This is a dummy address given to all iPAQ clients ? */
static unsigned char gClientMacAddr[ETH_ALEN] = {0};

/* --------------------------------------------------------------------- */
static int plusb_add_buf_tail (plusb_t *s, struct list_head *dst, struct list_head *src)
{
	unsigned long flags;
	struct list_head *tmp;
	int ret = 0;

	spin_lock_irqsave (&s->lock, flags);

	if (list_empty (src)) {
		// no elements in source buffer
		ret = -1;
		goto err;
	}
	tmp = src->next;
	list_del (tmp);
	list_add_tail (tmp, dst);

  err:	spin_unlock_irqrestore (&s->lock, flags);
	return ret;
}
/*-------------------------------------------------------------------*/

static int plusb_my_bulk(plusb_t *s, int pipe, void *data, int size, int *actual_length)
{
	int ret;

	dbg("plusb_my_bulk: len:%d data %p",size,data);
#ifdef DEBUG
	dumpOctets( "TX:",(unsigned char *)data ,size);
#endif

	/* TODO what is best timeout? */
	ret=usb_bulk_msg(s->usbdev, pipe, data, size, actual_length, 1000);
	if(ret<0) {
		err("ERROR:plusb: usb_bulk_msg failed(%d)",ret);
	}
	
	if( ret == -EPIPE ) {
		err("CLEAR_FEATURE request to remove STALL condition.");
		if(usb_clear_halt(s->usbdev, usb_pipeendpoint(pipe)))
			err("ERROR:my_bulk:request failed");
		}

	dbg("plusb_my_bulk: finished act: %d", *actual_length);		
	return ret;
}

/* --------------------------------------------------------------------- */

static void plusb_bh(void *context)
{
	plusb_t *s=context;
	struct net_device_stats *stats=&s->net_stats;
	int ret=0;
	int actual_length;
	skb_list_t *skb_list;
	struct sk_buff *skb;
	unsigned char * data;
	unsigned int txlen;

	dbg("plusb_bh: i:%d",in_interrupt());

	while(!list_empty(&s->tx_skb_list)) {

#if 0
		/* TODO TODO what does this do?? */
		if(!(s->status&_PLUSB_TXOK))
		{
			printk("plusb_bh:!PLUSB_TXOK\n");
			break; 
		}
		
#endif
		skb_list = list_entry (s->tx_skb_list.next, skb_list_t, skb_list);
		if(!skb_list->state) {
			dbg("plusb_bh: not yet ready");
			schedule();
			continue;
		}

		skb=skb_list->skb;

		/* 	Discard the 12-octet MAC address then prepend the 
			2-octet USB LENGTH just before the Level 3 Pkt type(IP)
			(see PKT1 diagram before _rx_complete() )
		*/
		data = skb->data + MAC_OFFSET;
		txlen = skb->len - MAC_OFFSET;
		data[0] = (unsigned char) (txlen & 0x00ff);
		data[1] = (unsigned char) ((txlen & 0xff00) >> 8);

		/* TODO check writeendp is the correct parameter */
		ret=plusb_my_bulk(s, usb_sndbulkpipe(s->usbdev, s->writeendp),
				data, txlen, &actual_length);
	
#if 0
		/* TODO TODO why is a multiple of 64 a consideration? */
		if(ret || skb->len != actual_length ||!(skb->len%64))
#else
		if(ret || ( txlen != actual_length)  )
#endif
		{
		    err("plusb_bh:ERROR skb_>len=%d act_length=%d\n",
			skb->len,actual_length);

		    /* TODO TODO why is a NULL packet transmitted?
			   cant do this or the iPAQ will barf
			(complains about bad packet size)
		    plusb_my_bulk(s, usb_sndbulkpipe(s->usbdev, s->writeendp),
					NULL, 0, &actual_length);
		    */
		}

		if(!ret) {
			stats->tx_packets++;
			stats->tx_bytes+=txlen;
                }
		else {
			dbg("plusb_bh: unsuccessful call to plusb_my_bulk");
			stats->tx_errors++;
			stats->tx_aborted_errors++;
		}

		dbg("plusb_bh: dev_kfree_skb");

		dev_kfree_skb(skb);
		skb_list->state=0;
		plusb_add_buf_tail (s, &s->free_skb_list, &s->tx_skb_list);	
	}

	dbg("plusb_bh: finished");
	s->in_bh=0;
}

/* --------------------------------------------------------------------- */

static int plusb_net_xmit(struct sk_buff *skb, struct net_device *dev)
{
	plusb_t *s=dev->priv;
	skb_list_t *skb_list;
	int ret=NET_XMIT_SUCCESS;

	dbg("plusb_net_xmit: len:%d i:%d",skb->len,in_interrupt());
	if ( netif_queue_stopped(dev) )	
	{
		warn("NET_XMIT:WARNING:network has stopped\n");
		return -EBUSY;
	}

        if(!s->connected || list_empty(&s->free_skb_list)) {
		ret=NET_XMIT_CN;
		goto lab;
	}	

	plusb_add_buf_tail (s, &s->tx_skb_list, &s->free_skb_list);
	skb_list = list_entry (s->tx_skb_list.prev, skb_list_t, skb_list);
	skb_list->skb=skb;
	skb_list->state=1;

lab:
	if(s->in_bh)
		return ret;

	dbg("plusb_net_xmit: queue_task");

	s->in_bh=1;
	queue_task(&s->bh, &tq_scheduler);

	dbg("plusb_net_xmit: finished");
	return ret;

}

/*

	PKT1-Packet received from the USB
	+-------+-------+---------------------------------------+
	| rxlen	| type	| TCP/IP packet = rxlen - 4 octets	|	
	+-------+-------+---------------------------------------+
	    2	    2
	^
	|ptr

	PKT1b-PKT1 after being pushed to accomodate MAC address
	+---+----------+-------+-------------------------+
	| 2 |    10    | rxlen | type	| TCP/IP packet	 |
	+---+----------+-------+-------------------------+
	    ^				^
	    |skb->data			|16

	PKT2-Packet(skb)passed up to the IP layer (with 2 pad octets).
	+----+--------+-------+---------+-----------------------+
	| 2  |MAC_Src |MAC_Dst| type	| TCP/IP packet 	|
	+----+--------+-------+---------+-----------------------+
	  2	6	  6	  2
					^
	|< --------16 octets----------->|

*/

static void plusb_rx_complete(urb_t *purb)
{
	plusb_t *s=purb->context;
	struct net_device_stats *stats=&s->net_stats;

	dbg("plusb_rx_complete: status:%d length:%d",
		purb->status,purb->actual_length);

	/* TODO TODO do we really need this */
	if ( !netif_running(&s->net_dev))	
	{
		warn("RX_COMPLETE:WARNING network not running\n");
		return;
	}

	if(!s->connected)
	{
		err("plusb_rx_complete: PANIC not connected\n");
		return;
	}

	if( !purb->status)
	{
	    struct sk_buff *skb = NULL;
	    unsigned char *dst = purb->transfer_buffer;
	    int rxlen = purb->actual_length;

	    /* Is this the first fragment? - If so then extract the length */
	    if( s->pktlen == 0 )
	    {
		if( s->skb )
		{
		    err("plusb_rx_complete: PANIC skb not NULL %p\n",s->skb);
		    goto fail_exit;
		}

		/* TODO TODO I have assumed rxlen >=2 for this to work */
		s->pktlen = ( dst[0] & 0xff );
		s->pktlen += ( (dst[1] & 0xff ) << 8 );
		dbg("plusb_rx_complete: pktlen = %d",s->pktlen);

		if(s->pktlen > MAX_USB_LEN)
		{
		    err("ERROR: bad length %d\n",s->pktlen);
		    goto fail_exit;
		}

		/* CREATE A  NEW SOCKET BUFFER  - with room for MAC & stuff */
#if 0
		/* TODO TODO why cant we do this? */
		skb=dev_alloc_skb(s->pktlen + MAC_OFFSET + 16);
#else
		skb=dev_alloc_skb(1518);
#endif
		if(!skb)
		{
		    /* TODO how is err enabled? */
		    err("ERROR: dev_alloc_skb(%d)=NULL\n", rxlen);
		    goto fail_exit;
		}
		/*
		    Shift skb 12 octets to the right so that the TCP/IP
		    field aligns on a 16 octet boundary.
		    This results in PKT1b.
		    TODO magic number.
		*/
		dbg("reserving skb:");
		skb_reserve(skb, 12);

		/* It is now safe to save it in our device */
		s->skb = skb;
	    }
	    else
	    {
		/* Retrieve skb to append next fragment */
		skb=s->skb;
		if( skb == NULL)
		{
		    err("ERROR:packet fragment NULL skb\n");
		    goto fail_exit;
		}
	    }

	    /* Check we have enough room to store our fragment in skb */
	    if(skb_tailroom(skb) < rxlen)
	    {
		    err("ERROR: no room n skb %d\n",s->pktlen);
		    goto fail_exit;
	    }

	    /* Now copy our fragment to the skb */
	    dbg("copying %d octets to skb:",rxlen);
	    memcpy(skb_put(skb, rxlen), dst, rxlen);

	
	    /* Have we received all the fragments? */
	    if ( s->pktlen == skb->len )
	    {
		unsigned int idx;
		dbg("Done: %d bytes received:",s->pktlen);
		/* Now copy our MAC Header - first adjust skb->data to 
	        that shown in PKT1b above.
		To do this we push back 10 octets from the original skb->data.
		NOTE we copy over the 'rxlen' field.
		*/
		dbg("copying MAC address:");
		dst = skb_push(skb, (2*ETH_ALEN)-2);
		for (idx = 0; idx < ETH_ALEN; idx++)
		{
		  /* First the Source (this) address from the inbound packet*/
		  dst[idx] = s->net_dev.dev_addr[idx];
		  /* Next the destination (iPAQ) address */
		  dst[idx+ETH_ALEN] = gClientMacAddr[idx];
		}


		/* TODO pass up our skb - must be deallocated by TCP/IP? */
		dbg("passing skb %p upto TCP/IP:",skb);
#ifdef DEBUG
		dumpOctets("RXSKB:",(unsigned char *)skb->data,skb->len);
#endif
#if 1
		skb->dev=&s->net_dev;
		skb->protocol=eth_type_trans(skb, skb->dev);
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		stats->rx_packets++;
		stats->rx_bytes+=rxlen;
		netif_rx(skb);
#endif
		/* prepare for next packet TODO check skb is freed by netif_rx*/
		s->pktlen=0;
		s->skb=NULL;
	    }
	    else
	    {
		dbg("packet fragment:");
	    }

	    goto success_exit;

	}
	else
	{
	    err("ERROR: Bad USB status:%d \n",purb->status);
	    purb->status=0;
	    if( s->skb )
		dev_kfree_skb(s->skb);
	    s->skb=NULL;
	    s->pktlen=0;
	}

fail_exit:
	stats->rx_dropped++;
	if( s->skb )
	    dev_kfree_skb(s->skb);
	s->skb=NULL;
	s->pktlen=0;
success_exit:
	/* resubmit URB */
	purb->actual_length=0;
#ifndef INTURB
	if( usb_submit_urb(purb) < 0 )	/* rx_complete */
		err("rx_complete:PANIC:failed to resubmit URB\n");	
	else
		dbg("plusb_rx_complete done...\n");
#endif
}

/* --------------------------------------------------------------------- */

#if 0
static void plusb_int_complete(urb_t *purb)
{
	plusb_t *s=purb->context;
	s->status=((unsigned char*)purb->transfer_buffer)[0]&255;
#if 0
	if((s->status&0x3f)!=0x20) {
		warn("invalid device status %02X", s->status);
		return;
	}
#endif	
	dbg("WARNING plusb_int_complete: status:%d length:%d",
		purb->status,purb->actual_length);

	if(!s->connected)
		return;

	if(s->status&_PLUSB_RXD) {
		int ret;
		
		if(s->bulkurb->status) {
			err("plusb_int_complete: URB still in use");
			return;
		}
		
		ret=usb_submit_urb(s->bulkurb);
		if(ret && ret!=-EBUSY) {
			err("plusb_int_complete: usb_submit_urb failed");
		}
	}
		
	if(purb->status || s->status!=160)
		dbg("status: %p %d buf: %02X", purb->dev, purb->status, s->status);
}

#endif

/* --------------------------------------------------------------------- */

static void plusb_free_all(plusb_t *s)
{
	struct list_head *skb;
	skb_list_t *skb_list;
	
	info("plusb_free_all");
	run_task_queue(&tq_immediate);	

	if( !s )
	{
		err("plusb_free: NULL descriptor");
		return;
	}

#ifdef INTURB
	if(s->inturb) {
		info("unlink inturb");
		usb_unlink_urb(s->inturb);
	}

	if(s->inturb && s->inturb->transfer_buffer) {
		info("kfree inturb->transfer_buffer");
		kfree(s->inturb->transfer_buffer);
		s->inturb->transfer_buffer=NULL;
	}
	
	if(s->inturb) {
		info("free_urb inturb");
		usb_free_urb(s->inturb);
		s->inturb=NULL;
	}

#else
	if(s->bulkurb) {
		info("unlink bulkurb");
		usb_unlink_urb(s->bulkurb);
	}
	
	if(s->bulkurb && s->bulkurb->transfer_buffer) {
		info("kfree bulkurb->transfer_buffer");
		kfree(s->bulkurb->transfer_buffer);
		s->bulkurb->transfer_buffer=NULL;
	}
	if(s->bulkurb) {
		info("free_urb bulkurb");
		usb_free_urb(s->bulkurb);
		s->bulkurb=NULL;
	}
#endif
	
	while(!list_empty(&s->free_skb_list)) {
		skb=s->free_skb_list.next;
		list_del(skb);
		skb_list = list_entry (skb, skb_list_t, skb_list);
		kfree(skb_list);
	}

	while(!list_empty(&s->tx_skb_list)) {
		skb=s->tx_skb_list.next;
		list_del(skb);
		skb_list = list_entry (skb, skb_list_t, skb_list);
		kfree(skb_list);	
	}
	dbg("plusb_free_all: finished");	
}

/*-------------------------------------------------------------------*/

static int plusb_alloc(plusb_t *s)
{
	int i;
	skb_list_t *skb;
	unsigned char *rxbuff = NULL ;
	unsigned int pipe,rxlen;
	urb_t * purb;

	info("(0)plusb_alloc");
	
	for(i=0 ; i < _SKB_NUM ; i++) {
		skb=kmalloc(sizeof(skb_list_t), GFP_KERNEL);
		if(!skb) {
			err("ERROR:kmalloc for skb_list failed");
			goto reject;
		}
		memset(skb, 0, sizeof(skb_list_t));
		list_add(&skb->skb_list, &s->free_skb_list);
	}

	/* ALLOC URB */
	info("(1)URB allocation:");
	purb=usb_alloc_urb(0);
	if(!purb) {
		err("ERROR:alloc_urb failed");
		goto reject;
	}

#ifdef INTURB
	/* TODO check readendp is initialised before we calculate the pipe */
	s->inturb=purb;
	pipe = usb_rcvintpipe(s->usbdev, s->readendp);
	rxlen = usb_maxpacket(s->usbdev, pipe, usb_pipeout(pipe));
	/* TODO TODO rxbuff = (char *)__get_dma_pages(GFP_KERNEL, 0); */
	rxbuff=kmalloc(rxlen, GFP_KERNEL);
#else
	s->bulkurb=purb;
	pipe = usb_rcvbulkpipe(s->usbdev, s->readendp);
	rxlen = MAX_USB_LEN;
	rxbuff=kmalloc(rxlen, GFP_KERNEL);
#endif
	if(!rxbuff)
	{
		err("ERROR:RX: unable to allocate rxbuff");
		goto reject;
	}

	if( rxlen > MAX_USB_LEN )
	{
		err("ERROR:plusb_alloc: bad rxlen %d\n",rxlen);
		goto reject;
	}
	else
	{
		info("(2)rxlen = %d\n",rxlen);
	}

#ifdef INTURB
	FILL_INT_URB(s->inturb, s->usbdev, pipe, rxbuff, rxlen,
			plusb_rx_complete,s,s->read_interval);
	info("(3)inturb submission:");
#else
	FILL_BULK_URB(s->bulkurb, s->usbdev, pipe, rxbuff, rxlen,
			plusb_rx_complete,s);
	info("(3)bulkurb submission:");
#endif

	if(usb_submit_urb(purb) < 0 )	/* plusb_alloc */
	{
		err("ERROR:usb_submit_urb failed");
		goto reject;
	}
	
	info("(4)plusb_alloc: finished");
	
	return 0;

  reject:
  	err("ERROR:plusb_alloc: failed");
	
	plusb_free_all(s);
	return -ENOMEM;
}

/*-------------------------------------------------------------------*/

static int plusb_net_open(struct net_device *dev)
{
	plusb_t *s=dev->priv;
	
	dbg("plusb_net_open");
	
	if(plusb_alloc(s))
		return -ENOMEM;

	s->opened=1;
	s->skb=NULL;		/* TODO here or in probe? */

	netif_start_queue(dev);	

	MOD_INC_USE_COUNT;
	
	dbg("plusb_net_open: success");
	
	return 0;
	
}

/* --------------------------------------------------------------------- */

static int plusb_net_stop(struct net_device *dev)
{
	plusb_t *s=dev->priv;
	
	dbg("plusb_net_stop");	
	
	plusb_free_all(s);
	s->opened=0;
	netif_stop_queue(dev);	
	MOD_DEC_USE_COUNT;
	dbg("plusb_net_stop:finished");
	return 0;
}

/* --------------------------------------------------------------------- */

static struct net_device_stats *plusb_net_get_stats(struct net_device *dev)
{
	plusb_t *s=dev->priv;
	
	dbg("net_device_stats");
	
	return &s->net_stats;
}

/* --------------------------------------------------------------------- */

static plusb_t *plusb_find_struct (void)
{
	int u;

	for (u = 0; u < NRPLUSB; u++) {
		plusb_t *s = &plusb[u];
		if (!s->connected)
			return s;
	}
	return NULL;
}

/* --------------------------------------------------------------------- */

static void plusb_disconnect (struct usb_device *usbdev, void *ptr)
{
	plusb_t *s = ptr;

	info("plusb_disconnect");
	if ( netif_running( &s->net_dev) )
	    netif_stop_queue( &s->net_dev );	

	s->connected = 0;
	
	plusb_free_all(s);

	if(!s->opened && s->net_dev.name) {
		info("unregistering netdev: %s",s->net_dev.name);
		unregister_netdev(&s->net_dev);
		s->net_dev.name[0] = '\0';
	}
	
	info("plusb_disconnect: finished");
	MOD_DEC_USE_COUNT;
}

/* --------------------------------------------------------------------- */

/* Called by register_netdev() */

int plusb_net_init(struct net_device *dev)
{
	dbg("plusb_net_init");

	/* This is the MAC address of this interface.
		It is , in fact, the host address.
	*/
	dev->dev_addr[0] = 0x40;
	dev->dev_addr[1] = 0;
	dev->dev_addr[2] = 0;
	dev->dev_addr[3] = 0;
	dev->dev_addr[4] = 0xff;
	dev->dev_addr[5] = 1;

	/* This is the client (iPAQ) MAC address
		It seems that all iPAQs have the same MAC address.
	*/
	gClientMacAddr[0]= 0x40;
	gClientMacAddr[1]= 0x00;
	gClientMacAddr[2]= 0x00;
	gClientMacAddr[3]= 0x00;
	gClientMacAddr[4]= 0x00;
	gClientMacAddr[5]= 0x01;

	
	dev->open=plusb_net_open;
	dev->stop=plusb_net_stop;
	dev->hard_start_xmit=plusb_net_xmit;
	dev->get_stats	= plusb_net_get_stats;
	ether_setup(dev);
	dev->tx_queue_len = 0;
#if 0
	/*
		Do not include this flag unless it set on the iPAQ as well
		as ARP packets sent by the iPAQ will be ignored resulting
		in the iPAQ sending ARP packets indefinitely waiting on
		a response from the PC host that will never come.
	*/
	dev->flags = IFF_POINTOPOINT|IFF_NOARP;
#endif

#if 1
	/* CF keep IP_PNP (ipconfig.c) from trying to use this interface */
	dev->flags &= ~IFF_BROADCAST;	/* CLR this flag */
#endif

	
	dbg("plusb_net_init: finished");
	return 0;
}

/* --------------------------------------------------------------------- */

static void *plusb_probe (struct usb_device *usbdev, unsigned int ifnum)
{
	plusb_t *s;

	dbg("plusb: probe: vendor id 0x%x, device id 0x%x ifnum:%d",
	  usbdev->descriptor.idVendor, usbdev->descriptor.idProduct, ifnum);

	/* We don't handle multiple configurations */
	if (usbdev->descriptor.bNumConfigurations != 1)
	{
		warn("PANIC: multiple configuration detected\n");
		return NULL;
	}

	/* Check Vendor and Product ID */
	if (usbdev->descriptor.idVendor != ID_VENDOR ||
		usbdev->descriptor.idProduct != ID_PRODUCT)
	{
#if 0
		printk("usb_net_host: ignoring device %x %x\n",
			usb_dev->descriptor.idVendor,
			usb_dev->descriptor.idProduct);
#endif
		dbg("plusb: probe: device NOT found\n"); 	
		return NULL;
	}
	else
		dbg("plusb: probe: device found\n"); 	

	s = plusb_find_struct ();
	if (!s)
	{
		warn("PANIC: couldn't find free structure from pool\n");
		return NULL;
	}

#if 1
	if( findEndPoints(usbdev, s) )
	{
		dbg("Unable to find endpoints\n");
		return NULL;
	}
#endif
	s->usbdev = usbdev;

	if (usb_set_configuration (s->usbdev, usbdev->config[0].bConfigurationValue) < 0) {
		err("ERROR:set_configuration failed");
		return NULL;
	}

	/* TODO TODO is this required? */
	if (usb_set_interface (s->usbdev, 0, 0) < 0) {
		err("ERROR:set_interface failed");
		return NULL;
	}

	if(!s->net_dev.name[0]) {
	/* TODO TODO how do we map multiple names to multiple devices? */
		strcpy(s->net_dev.name, "usb0");
		s->net_dev.init=plusb_net_init;
		s->net_dev.priv=s;
		if(!register_netdev(&s->net_dev))
			info("registered: %s", s->net_dev.name);
		else {
			err("ERROR:register_netdev failed");
			s->net_dev.name[0] = '\0';
		}
	}
		
	s->connected = 1;
	s->skb=NULL;		/* TODO here or in _open? */

	if(s->opened) {
	    /* enter here if probe is called after network has been started */
		dbg("net device already allocated, restarting USB transfers");
		plusb_alloc(s);
	}

	info("bound to interface: %d dev: %p", ifnum, usbdev);
	MOD_INC_USE_COUNT;
	return s;
}
/* --------------------------------------------------------------------- */

static struct usb_driver plusb_driver =
{
	name: "plusb",
	probe: plusb_probe,
	disconnect: plusb_disconnect,
};

/* --------------------------------------------------------------------- */

static int __init plusb_init (void)
{
	unsigned u;
	dbg("plusb_init");
	
	/* initialize struct */
	for (u = 0; u < NRPLUSB; u++) {
		plusb_t *s = &plusb[u];
		memset (s, 0, sizeof (plusb_t));
		s->bh.routine = (void (*)(void *))plusb_bh;
		s->bh.data = s;
		INIT_LIST_HEAD (&s->tx_skb_list);
		INIT_LIST_HEAD (&s->free_skb_list);
		spin_lock_init (&s->lock);
	}

	/* register misc device */
	usb_register (&plusb_driver);

	dbg("plusb_init: driver registered");

	return 0;
}

/* --------------------------------------------------------------------- */

static void __exit plusb_cleanup (void)
{
	unsigned u;

	dbg("plusb_cleanup");
	for (u = 0; u < NRPLUSB; u++) {
		plusb_t *s = &plusb[u];
		if(s->net_dev.name[0]) {
			dbg("unregistering netdev: %s",s->net_dev.name);
			unregister_netdev(&s->net_dev);
		}
	}
	usb_deregister (&plusb_driver);
	dbg("plusb_cleanup: finished");
}

/* --------------------------------------------------------------------- */

MODULE_AUTHOR ("Deti Fliegl, deti@fliegl.de");
MODULE_DESCRIPTION ("PL-2302 USB Interface Driver for Linux (c)2000");


module_init (plusb_init);
module_exit (plusb_cleanup);

/* --------------------------------------------------------------------- */

/*
	-3- Find endpoints
	return 0 for success 
	initialises s->read_interval,s->writeendp,s->readendp
*/
static int findEndPoints(struct usb_device *usbdev, plusb_t *s)
{
    int cfgnum;
    int status = 0;

    for (cfgnum = 0; 
	 cfgnum < usbdev->config[0].interface[0].altsetting[0].bNumEndpoints;
	 cfgnum++)
    {
  	struct usb_endpoint_descriptor *ep;

	ep = &usbdev->config[0].interface[0].altsetting[0].endpoint[cfgnum];

	switch (ep->bEndpointAddress)
	{
	case 0x01 :	/* TODO remove magic numbers */
	    /* this is out write endpoint */
	    s->writeendp = ep->bEndpointAddress;
	    info("writeendp %02x\n", s->writeendp);
	    break;
	case 0x82 :
	    s->readendp = ep->bEndpointAddress;
	    s->read_interval = ep->bInterval;
	    info("readendp %02x interval %d\n", s->readendp,s->read_interval);
	    break;
	default:
	    err("ERROR: unable to get an endpoint address %02x\n",
		ep->bEndpointAddress);
	    status=cfgnum;
	    break;
	}
    }
    return status;
}

void dumpOctets(unsigned char *msg,unsigned char * ptr,unsigned int len)
{
	unsigned int i,max;
	if( !len || !ptr )
	{
	    err("dumpOctets: bad len %d or data %p\n",len,ptr);
	    return;
	}
	printk("%s[%d]",msg,len);
	max = ( len > 40) ? 40 : len;
	for(i=0; i < max; i++)
	    dbg("%02x:",*(ptr + i) );
	dbg("\n");
}
