
/* Overhauled routines for dealing with different mmap regions of flash */
/* $Id: map.h,v 1.18 2001/01/08 15:14:50 dwmw2 Exp $ */

#ifndef __LINUX_MTD_MAP_H__
#define __LINUX_MTD_MAP_H__

#include <linux/config.h>
#include <linux/types.h>
#include <linux/mtd/mtd.h>
#include <linux/malloc.h>

/* The map stuff is very simple. You fill in your struct map_info with
   a handful of routines for accessing the device, making sure they handle
   paging etc. correctly if your device needs it. Then you pass it off
   to a chip driver which deals with a mapped device - generally either
   do_cfi_probe() or do_ram_probe(), either of which will return a 
   struct mtd_info if they liked what they saw. At which point, you
   fill in the mtd->module with your own module address, and register 
   it.
   
   The mtd->priv field will point to the struct map_info, and any further
   private data required by the chip driver is linked from the 
   mtd->priv->fldrv_priv field. This allows the map driver to get at 
   the destructor function map->fldrv_destroy() when it's tired
   of living.
*/

struct map_info {
	char *name;
	unsigned long size;
	int buswidth; /* in octets */
	__u8 (*read8)(struct map_info *, unsigned long);
	__u16 (*read16)(struct map_info *, unsigned long);
	__u32 (*read32)(struct map_info *, unsigned long);  
	/* If it returned a 'long' I'd call it readl.
	 * It doesn't.
	 * I won't.
	 * dwmw2 */
	
	void (*copy_from)(struct map_info *, void *, unsigned long, ssize_t);
	void (*write8)(struct map_info *, __u8, unsigned long);
	void (*write16)(struct map_info *, __u16, unsigned long);
	void (*write32)(struct map_info *, __u32, unsigned long);
	void (*copy_to)(struct map_info *, unsigned long, const void *, ssize_t);

	void (*set_vpp)(struct map_info *, int);
	/* We put these two here rather than a single void *map_priv, 
	   because we want mappers to be able to have quickly-accessible
	   cache for the 'currently-mapped page' without the _extra_
	   redirection that would be necessary. If you need more than
	   two longs, turn the second into a pointer. dwmw2 */
	unsigned long map_priv_1;
	unsigned long map_priv_2;
	void *fldrv_priv;
	void (*fldrv_destroy)(struct mtd_info *);
	const char *im_name;
};

typedef struct mtd_info *map_probe_fn_t(struct map_info *);

/* 
 * Probe for the contents of a map device and make an MTD structure
 * if anything is recognised. Doesn't register it because the calling
 * map driver needs to set the 'module' field first.
 */

/* Define inline function for finding a probe function. This would be
 * _so_ much nicer if weak symbols actually worked. 
 */
#ifdef HAVE_INTER_MODULE
#define DECLARE_MAP_PROBE(funcname, modname) \
static inline struct mtd_info *do_##funcname(struct map_info *map) \
{ \
	map_probe_fn_t *probe_p;\
	struct mtd_info *mtd = NULL; \
\
	if ((probe_p = inter_module_get_request(modname, #funcname))) { \
		map->im_name = #funcname; \
		mtd = (*probe_p)(map); \
		} \
\
	return mtd; \
}
#else
/* Empty version for kernels without the inter_module_xxx stuff */
#define DECLARE_MAP_PROBE(funcname, modname) \
static inline struct mtd_info *do_##funcname (struct map_info *map) \
{ \
	return NULL; \
}
#endif

#ifdef CONFIG_MTD_CFI
#define do_cfi_probe(map) cfi_probe(map)
#else
DECLARE_MAP_PROBE(cfi_probe, "cfi_probe")
#endif

#ifdef CONFIG_MTD_ROM
#define do_map_rom_probe(map) map_rom_probe(map)
#else
DECLARE_MAP_PROBE(map_rom_probe, "map_rom")
#endif

#ifdef CONFIG_MTD_RAM
#define do_map_ram_probe(map) map_ram_probe(map)
#else
DECLARE_MAP_PROBE(map_ram_probe, "map_ram")
#endif

#ifdef CONFIG_MTD_JEDEC
#define do_jedec_probe(map) jedec_probe(map)
#else
DECLARE_MAP_PROBE(jedec_probe, "jedec")
#endif

#undef DECLARE_MAP_PROBE

/*
 * Destroy an MTD device which was created for a map device.
 * Make sure the MTD device is already unregistered before calling this
 */
static inline void map_destroy(struct mtd_info *mtd)
{
	struct map_info *map = mtd->priv;

	map->fldrv_destroy(mtd);
	if (map->im_name)
		inter_module_put(map->im_name);
	kfree(mtd);
}

#define ENABLE_VPP(map) do { if(map->set_vpp) map->set_vpp(map, 1); } while(0)
#define DISABLE_VPP(map) do { if(map->set_vpp) map->set_vpp(map, 0); } while(0)

#endif /* __LINUX_MTD_MAP_H__ */
