/*
 * Ethernet driver for the SA1100 USB client function
 * Copyright (c) 2001 by Nicolas Pitre
 *
 * This code was loosely inspired by the original initial ethernet test driver
 * Copyright (c) Compaq Computer Corporation, 1999
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This is still work in progress...
 * 
 * 19/02/2001 - Now we are compatible with generic usbnet driver. green@iXcelerator.com
 * 
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/timer.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <asm/unaligned.h>

#include "sa1100_usb.h"


#define ETHERNET_VENDOR_ID 0x49f
#define ETHERNET_PRODUCT_ID 0x505A

// Should be global, so that insmod can change these
int usb_rsize=64;
int usb_wsize=64;

/* FIXME - this should go into common include, once usbnet.c goes into 
   standart kernel */
// packets are always ethernet inside
// ... except they can be bigger (up to 64K with this framing)
#define MIN_PACKET      sizeof(struct ethhdr)
#define MAX_PACKET      32768

struct nc_header {		// packed:
	u16	hdr_len;		// sizeof nc_header (LE, all)
	u16	packet_len;		// payload size (including ethhdr)
	u16	packet_id;		// detects dropped packets
#define MIN_HEADER	6
} __attribute__((__packed__));

#define PAD_BYTE	((unsigned char)0xAC)
struct nc_trailer {
	u16	packet_id;
} __attribute__((__packed__));

#define FRAMED_SIZE(mtu) (sizeof (struct nc_header) \
				+ sizeof (struct ethhdr) \
				+ (mtu) \
				+ 1 \
				+ sizeof (struct nc_trailer))

#define MIN_FRAMED	FRAMED_SIZE(0)

static struct usbe_info_t {
  struct net_device *dev;
  char host_addr[ETH_ALEN];
  u16 packet_id;
  struct net_device_stats stats;
} usbe_info;

static char usb_eth_name[16] = "usbf";
static struct net_device usb_eth_device;
static struct sk_buff *cur_tx_skb, *next_tx_skb;
static struct sk_buff *cur_rx_skb, *next_rx_skb;
static volatile int terminating;

static int usb_change_mtu (struct net_device *net, int new_mtu)
{
	if (new_mtu <= sizeof (struct ethhdr) || new_mtu > MAX_PACKET)
		return -EINVAL;
	if (FRAMED_SIZE (new_mtu) > MAX_PACKET)
		return -EINVAL;
	// no second zero-length packet read wanted after mtu-sized packets
	net->mtu = new_mtu;
	return 0;
}

static struct sk_buff * 
usb_new_recv_skb(void)
{
	struct sk_buff *skb = dev_alloc_skb((12 + FRAMED_SIZE (usb_eth_device.mtu)));
	if (skb) {
		skb_reserve(skb, 12);
	}
	return skb;
}

static void 
usb_recv_callback(int flag, int size)
{
	struct sk_buff *skb;
	int pktlen, templen;
	struct nc_header	*header = NULL;
	struct nc_trailer	*trailer = NULL;
	
	if (terminating) 
		return;
	
	skb = cur_rx_skb;
	pktlen = 0;

	/* flag validation */
	if (flag == 0) {
		skb_put(skb, size);
		if (skb->len > MIN_FRAMED) {
			header = (struct nc_header *) skb->data;
			le16_to_cpus (&header->hdr_len);
			le16_to_cpus (&header->packet_len);
			if (header->hdr_len == MIN_HEADER ) {
				pktlen = header->packet_len;
			}
			if (pktlen < MIN_FRAMED || pktlen > FRAMED_SIZE(usb_eth_device.mtu)) {
				pktlen = 0;
				usbe_info.stats.rx_frame_errors++;
			}
		}
	} else if (flag == -EIO) {
		usbe_info.stats.rx_errors++;
	}

	templen = pktlen + header->hdr_len + sizeof (struct nc_trailer);
	templen += (templen & 0x01) ? 0:1;
	/* validate packet length */
	if (skb->len < templen ) {
		/* packet not complete yet */
		skb = NULL;
	}
	
	if (pktlen == 0) {
		goto error;
	}
	
	/* 
	 * At this point skb is non null if we have a complete packet.
	 * If so take a fresh skb right away and restart USB receive without
	 * further delays, then process the packet.  Otherwise resume USB
	 * receive on the current skb and exit.
	 */

	if (skb)
		cur_rx_skb = next_rx_skb;
	sa1100_usb_recv(cur_rx_skb->tail, skb_tailroom(cur_rx_skb), 
			usb_recv_callback);
	if (!skb)
		return;

	next_rx_skb = usb_new_recv_skb();
	if (!next_rx_skb) {
		/*
		 * We can't aford loosing buffer space...  
		 * So we drop the current packet and recycle its skb.
		 */
		printk("%s: can't allocate new skb\n", __FUNCTION__);
		usbe_info.stats.rx_dropped++;
		skb_trim(skb, 0);
		next_rx_skb = skb;
		return;
	}

	if (FRAMED_SIZE (header->packet_len) > MAX_PACKET) {
		usbe_info.stats.rx_frame_errors++;
		goto error;
	}
	skb_pull (skb, header->hdr_len);
	trailer = (struct nc_trailer *)
		(skb->data + skb->len - sizeof (struct nc_trailer));
	skb_trim (skb, skb->len - sizeof (struct nc_trailer));

	if ((header->packet_len & 0x01) == 0) {
		if (skb->data [header->packet_len] != PAD_BYTE) {
			usbe_info.stats.rx_frame_errors++;
			goto error;
		}
		skb_trim (skb, skb->len - 1);
	}
	if (skb->len != header->packet_len) {
		usbe_info.stats.rx_frame_errors++;
		goto error;
	}

	if (header->packet_id != get_unaligned (&trailer->packet_id)) {
		usbe_info.stats.rx_fifo_errors++;
		goto error;
	}

	if (skb->len) {
		int     status;
// FIXME: eth_copy_and_csum "small" packets to new SKB (small < ~200 bytes) ?

		skb->dev = &usb_eth_device;
		skb->protocol = eth_type_trans (skb, &usb_eth_device);
		usbe_info.stats.rx_packets++;
		usbe_info.stats.rx_bytes += skb->len;
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		status = netif_rx (skb);
	} else {
error:
		usbe_info.stats.rx_errors++;
		/* 
		 * Error due to bad frame, bad pktlen, etc.
		 * Recycle the current skb and reset USB reception.
		 */
		skb_trim(cur_rx_skb, 0);
		if (flag != -EINTR)
			sa1100_usb_recv_reset();
		sa1100_usb_recv(cur_rx_skb->tail, skb_tailroom(cur_rx_skb),
			usb_recv_callback);
	}
}


static void 
usb_send_callback(int flag, int size)
{
	struct net_device *dev = usbe_info.dev;
	struct net_device_stats *stats;
	struct sk_buff *skb=cur_tx_skb;
	int ret;

	if (terminating)
		return;

	stats = &usbe_info.stats;
	switch (flag) {
	    case 0:
		stats->tx_packets++;
		stats->tx_bytes += size;
		break;
	    case -EIO:
		stats->tx_errors++;
		break;
	    default:
		stats->tx_dropped++;
		break;
	}

	cur_tx_skb = next_tx_skb;
	next_tx_skb = NULL;
	dev_kfree_skb_irq(skb);
	if (!cur_tx_skb)
		return;
	
	dev->trans_start = jiffies;
	ret = sa1100_usb_send(cur_tx_skb->data, cur_tx_skb->len, usb_send_callback);
	if (ret) {
		/* If the USB core can't accept the packet, we drop it. */
		dev_kfree_skb_irq(cur_tx_skb);
		cur_tx_skb = NULL;
		usbe_info.stats.tx_carrier_errors++;
	}
	netif_wake_queue(dev);
}


static inline struct sk_buff *fixup_skb (struct sk_buff *skb, int flags)
{
	int			padlen;
	struct sk_buff		*skb2;

	padlen = ((skb->len + sizeof (struct nc_header)
			+ sizeof (struct nc_trailer)) & 0x01) ? 0 : 1;
	if (!skb_cloned (skb)) {
		int	headroom = skb_headroom (skb);
		int	tailroom = skb_tailroom (skb);

		if ((padlen + sizeof (struct nc_trailer)) <= tailroom
			    && sizeof (struct nc_header) <= headroom)
			return skb;
/*		if ((sizeof (struct nc_header) + padlen
					+ sizeof (struct nc_trailer)) <
				(headroom + tailroom)) {
			skb->data = memmove (skb->head
						+ sizeof (struct nc_header),
					    skb->data, skb->len);
			skb->tail = skb->data + skb->len;
			return skb;
		}*/
	}
	skb2 = skb_copy_expand (skb,
				sizeof (struct nc_header),
				sizeof (struct nc_trailer) + padlen,
				flags);
	dev_kfree_skb_any (skb);
	return skb2;
}

static int 
usb_eth_xmit(struct sk_buff *skb, struct net_device *dev)
{
	int ret;
	struct sk_buff  *skb2;
	long flags;
	int length = skb->len;
	struct nc_header *header = 0;
	struct nc_trailer *trailer = 0;
	
	if (next_tx_skb) {
		printk("%s: called with next_tx_skb != NULL\n", __FUNCTION__);
		return 1;
	}

	if (skb_shared (skb)) {
		skb2 = skb_unshare(skb, GFP_ATOMIC);
		if (!skb2) {
			usbe_info.stats.tx_dropped++;
			dev_kfree_skb(skb);
			return 1;
		}
		skb = skb2;
	}

	skb2 = fixup_skb (skb, GFP_ATOMIC);
	if (!skb2) {
		usbe_info.stats.tx_dropped++;
		dev_kfree_skb(skb);
		return 1;
	}
	skb = skb2;
	
	header = (struct nc_header *) skb_push (skb, sizeof (struct nc_header));
	header->hdr_len = cpu_to_le16 (sizeof (struct nc_header));
	header->packet_len = cpu_to_le16 (length);
	if (!((skb->len + sizeof (struct nc_trailer)) & 0x01))
		*skb_put (skb, 1) = PAD_BYTE;
	trailer = (struct nc_trailer *) skb_put (skb, sizeof (struct nc_trailer));
	header->packet_id = cpu_to_le16 (usbe_info.packet_id++);
	put_unaligned (header->packet_id, &trailer->packet_id);

	save_flags_cli(flags);
	if (cur_tx_skb) {
		next_tx_skb = skb;
		netif_stop_queue(dev);
	} else {
		cur_tx_skb = skb;
		dev->trans_start = jiffies;
		ret = sa1100_usb_send(skb->data, skb->len, usb_send_callback);
		if (ret) {
			/* If the USB core can't accept the packet, we drop it. */
			dev_kfree_skb(skb);
			cur_tx_skb = NULL;
			usbe_info.stats.tx_carrier_errors++;
		}
	}
	restore_flags(flags);
	return 0;
}

static void 
usb_xmit_timeout(struct net_device *dev )
{
	sa1100_usb_send_reset();
	dev->trans_start = jiffies;
	netif_wake_queue(dev);
}


static int 
usb_eth_open(struct net_device *dev)
{
	terminating = 0;
	cur_tx_skb = next_tx_skb = NULL;
	cur_rx_skb = usb_new_recv_skb();
	next_rx_skb = usb_new_recv_skb();
	if (!cur_rx_skb || !next_rx_skb) {
		printk("%s: can't allocate new skb\n", __FUNCTION__);
		if (cur_rx_skb)
			kfree_skb(cur_rx_skb);
		if (next_rx_skb)
			kfree_skb(next_rx_skb);
		return -ENOMEM;;
	}
	
	MOD_INC_USE_COUNT;
	sa1100_usb_recv(cur_rx_skb->tail, skb_tailroom(cur_rx_skb), 
			usb_recv_callback);
	return 0;
}

static int 
usb_eth_release(struct net_device *dev)
{
	terminating = 1;
	sa1100_usb_send_reset();
	sa1100_usb_recv_reset();
	if (cur_tx_skb)
		kfree_skb(cur_tx_skb);
	if (next_tx_skb)
		kfree_skb(next_tx_skb);
	if (cur_rx_skb)
		kfree_skb(cur_rx_skb);
	if (next_rx_skb)
		kfree_skb(next_rx_skb);
	MOD_DEC_USE_COUNT;
	return 0;
}

static struct net_device_stats *
usb_eth_stats(struct net_device *dev)
{
	struct usbe_info_t *priv =  (struct usbe_info_t*) dev->priv;
	struct net_device_stats *stats=NULL;

	if (priv)
		stats = &priv->stats;
	return stats;
}

static int 
usb_eth_probe(struct net_device *dev)
{
	/* 
	 * Assign the hardware address of the board: use 
	 * 40 00 00 00 00 XX, where XX is the USB address of the
	 * device
	 */
	dev->dev_addr[0] = 0x40;
	dev->dev_addr[1] = 0;
	dev->dev_addr[2] = 0;
	dev->dev_addr[3] = 0;
	dev->dev_addr[4] = 0;
	dev->dev_addr[5] = 1; 
   
	/* 
	 * we know the host mac address because it is derived from
	 * the USB address.
	 */
	usbe_info.host_addr[0] = 0x40;
	usbe_info.host_addr[1] = 0;
	usbe_info.host_addr[2] = 0;
	usbe_info.host_addr[3] = 0;
	usbe_info.host_addr[4] = 0xff;
	usbe_info.host_addr[5] = 1;

	dev->open = usb_eth_open;
	dev->change_mtu = usb_change_mtu;
	dev->stop = usb_eth_release;
	dev->hard_start_xmit = usb_eth_xmit;
	dev->get_stats = usb_eth_stats;
	dev->watchdog_timeo = 1*HZ;
	dev->tx_timeout = usb_xmit_timeout;
	dev->priv = &usbe_info;

	usbe_info.dev = dev;

	/* clear the statistics */
	memset(&usbe_info.stats, 0, sizeof(struct net_device_stats));

	ether_setup(dev);
	dev->flags &= ~IFF_MULTICAST;
	dev->flags &= ~IFF_BROADCAST;
	//dev->flags |= IFF_NOARP;

	return 0;
}

#ifdef MODULE
MODULE_PARM(usb_rsize, "1i");
MODULE_PARM_DESC(usb_rsize, "number of bytes in packets from host to sa1100");
MODULE_PARM(usb_wsize, "1i");
MODULE_PARM_DESC(usb_wsize, "number of bytes in packets from sa1100 to host");
#endif

static int __init
usb_eth_init(void)
{
	int rc; 

	strncpy(usb_eth_device.name, usb_eth_name, IFNAMSIZ);
	usb_eth_device.init = usb_eth_probe;
	if (register_netdev(&usb_eth_device) != 0)
		return -EIO;

	rc = sa1100_usb_open( "usb-eth" );
	if ( rc == 0 ) {
		 string_desc_t * pstr;
		 desc_t * pd = sa1100_usb_get_descriptor_ptr();

		 pd->b.ep1.wMaxPacketSize = make_word_c( usb_rsize );
		 pd->b.ep2.wMaxPacketSize = make_word_c( usb_wsize );
		 pd->dev.idVendor	  = ETHERNET_VENDOR_ID;
		 pd->dev.idProduct     = ETHERNET_PRODUCT_ID;
		 pstr = sa1100_usb_kmalloc_string_descriptor( "SA1100 USB NIC" );
		 if ( pstr ) {
			  sa1100_usb_set_string_descriptor( 1, pstr );
			  pd->dev.iProduct = 1;
		 }
		 rc = sa1100_usb_start();
	}
	return rc;
}

module_init(usb_eth_init);

static void __exit
usb_eth_cleanup(void)
{
	 string_desc_t * pstr;
	 sa1100_usb_stop();
	 sa1100_usb_close();
	 if ( (pstr = sa1100_usb_get_string_descriptor(1)) != NULL )
		  kfree( pstr );
	unregister_netdev(&usb_eth_device);
}

module_exit(usb_eth_cleanup);
