/*
 * $Id: cstm_cfi_jedec.c,v 
 *
 * Mapping of a custom board with both AMD CFI and JEDEC flash in partitions.
 * Config with both CFI and JEDEC device support.
 *
 * Basically physmap.c with the addition of partitions and 
 * an array of mapping info to accomodate more than one flash type per board.
 *
 * Copyright 2000 MontaVista Software Inc.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <linux/config.h>

__u8 cstm_cfi_jedec_read8(struct map_info *map, unsigned long ofs)
{
	return *(__u8 *)(map->map_priv_1 + ofs);
}

__u16 cstm_cfi_jedec_read16(struct map_info *map, unsigned long ofs)
{
	return *(__u16 *)(map->map_priv_1 + ofs);
}

__u32 cstm_cfi_jedec_read32(struct map_info *map, unsigned long ofs)
{
	return *(__u32 *)(map->map_priv_1 + ofs);
}

void cstm_cfi_jedec_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	memcpy_fromio(to, map->map_priv_1 + from, len);
}

void cstm_cfi_jedec_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	*(__u8 *)(map->map_priv_1 + adr) = d;
}

void cstm_cfi_jedec_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	*(__u16 *)(map->map_priv_1 + adr) = d;
}

void cstm_cfi_jedec_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	*(__u32 *)(map->map_priv_1 + adr) = d;
}

void cstm_cfi_jedec_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy_toio(map->map_priv_1 + to, from, len);
}

void cstm_cfi_jedec_set_vpp(struct map_info *map,int vpp)
{
  if (vpp) {
  }
  else {
  }
}

const struct map_info basic_cstm_cfi_jedec_map = {
	NULL,
	0,
	0,
	cstm_cfi_jedec_read8,
	cstm_cfi_jedec_read16,
	cstm_cfi_jedec_read32,
	cstm_cfi_jedec_copy_from,
	cstm_cfi_jedec_write8,
	cstm_cfi_jedec_write16,
	cstm_cfi_jedec_write32,
	cstm_cfi_jedec_copy_to,
        cstm_cfi_jedec_set_vpp,
	0,
	0
};

/* board and partition description */

#define MAX_PHYSMAP_PARTITIONS    8
struct cstm_cfi_jedec_info {
	char *name;
	unsigned long window_addr;
	unsigned long window_size;
	int buswidth;
	int num_partitions;
};

#define PHYSMAP_NUMBER  2  // number of cstm_cfi_jedec_info structs needed, one per contiguous flash type 
const struct cstm_cfi_jedec_info cstm_cfi_jedec_board_desc[PHYSMAP_NUMBER] = 
{
    {   // Am29DL323D in 2x16 configuration - CFI flash
        "big flash",                   // name
	0x40000000,      // window_addr
	0x00800000,      // window_size
        4,               // buswidth
	2,                             // num_partitions
    },
    {   // Am29W040B in 1x8 configuration - JEDEC flash
        "boot flash",    // name
	0xfff80000,      // window_addr
	0x00060000,      // window_size
        1,               // buswidth
	2,               // num_partitions
    },

};
static struct mtd_partition cstm_cfi_jedec_partitions[PHYSMAP_NUMBER][MAX_PHYSMAP_PARTITIONS] = {
{  // Am29DL323D in 2x16 configuration
	{
		name: "big flash 128k sectors",
		size: 0x007E0000, // 63 x (2 x 64k) byte sectors
		offset: 0,
	},
	{
		name: "big flash 16k sectors",
		size: 0x00020000, // 8  x (2 x 8k)  byte sectors
		offset: 0x007E0000,
	},
},
{  // Am29W040B in 1x8 configuration  
	{
		name: "boot environment",
		size: 0x00010000, // 1 x 64k byte sectors
		offset: 0
	},
	{
		name: "boot flash sectors",
		size: 0x00050000, // 5 x 64k byte sectors
		offset: 0x10000
	},
	// leaving last 2 sectors of flash unaccessable by this driver
}
};

struct map_info cstm_cfi_jedec_map[PHYSMAP_NUMBER];

#if LINUX_VERSION_CODE < 0x20212 && defined(MODULE)
#define init_cstm_cfi_jedec init_module
#define cleanup_cstm_cfi_jedec cleanup_module
#endif

int __init init_cstm_cfi_jedec(void)
{
	int i;
	int jedec;
        struct mtd_info *mymtd;
        struct mtd_partition *parts;

	/* Initialize mapping */
	for (i=0;i<PHYSMAP_NUMBER;i++) {
		printk(KERN_NOTICE "cstm_cfi_jedec flash device: %lx at %lx\n", cstm_cfi_jedec_board_desc[i].window_size, cstm_cfi_jedec_board_desc[i].window_addr);
                memcpy((char *)&cstm_cfi_jedec_map[i],(char *)&basic_cstm_cfi_jedec_map,sizeof(struct map_info));
		cstm_cfi_jedec_map[i].map_priv_1 = (unsigned long)ioremap(cstm_cfi_jedec_board_desc[i].window_addr, cstm_cfi_jedec_board_desc[i].window_size);
		if (!cstm_cfi_jedec_map[i].map_priv_1) {
			printk(KERN_WARNING "Failed to ioremap\n");
			return -EIO;
	        }
		cstm_cfi_jedec_map[i].name = cstm_cfi_jedec_board_desc[i].name;
		cstm_cfi_jedec_map[i].size = cstm_cfi_jedec_board_desc[i].window_size;
		cstm_cfi_jedec_map[i].buswidth = cstm_cfi_jedec_board_desc[i].buswidth;
		printk(KERN_NOTICE "cstm_cfi_jedec: ioremap is %x\n",(unsigned int)(cstm_cfi_jedec_map[i].map_priv_1));
	}

	for (i=0;i<PHYSMAP_NUMBER;i++) {
                parts = &cstm_cfi_jedec_partitions[i][0];
		jedec = 0;
		mymtd = (struct mtd_info *)do_cfi_probe(&cstm_cfi_jedec_map[i]);
		printk(KERN_NOTICE "phymap %d cfi_probe: mymtd is %x\n",i,(unsigned int)mymtd);
		if (!mymtd) {
			jedec = 1;
			mymtd = (struct mtd_info *)do_jedec_probe(&cstm_cfi_jedec_map[i]);
		        printk(KERN_NOTICE "cstm_cfi_jedec %d jedec: mymtd is %x\n",i,(unsigned int)mymtd);
		}
		if (mymtd) {
#ifdef MODULE
			mymtd->module = &__this_module;
#endif
	                cstm_cfi_jedec_map[i].map_priv_2 = (unsigned long)mymtd;
		        add_mtd_partitions(mymtd, parts, cstm_cfi_jedec_board_desc[i].num_partitions);
		}
		else
	           return -ENXIO;
	}
	return 0;
}

static void __exit cleanup_cstm_cfi_jedec(void)
{
	int i;
        struct mtd_info *mymtd;

	for (i=0;i<PHYSMAP_NUMBER;i++) {
	        mymtd = (struct mtd_info *)cstm_cfi_jedec_map[i].map_priv_2;
		if (mymtd) {
			del_mtd_partitions(mymtd);
			map_destroy(mymtd);
		}
		if (cstm_cfi_jedec_map[i].map_priv_1) {
			iounmap((void *)cstm_cfi_jedec_map[i].map_priv_1);
			cstm_cfi_jedec_map[i].map_priv_1 = 0;
		}
	}
}

module_init(init_cstm_cfi_jedec);
module_exit(cleanup_cstm_cfi_jedec);
