/*
 * linux/drivers/net/irda/sa1100_ir.c
 *
 * Infra-red driver for the StrongARM SA1100 embedded microprocessor
 *
 * (C) 2000 Russell King
 *
 *
 * 04 Feb 2001	green@iXcelerator.com - FIR support.
 *
 * TODO: get rid of data copying between skb & rx buffer, when using HSSP
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/slab.h>
#include <linux/rtnetlink.h>
#include <linux/interrupt.h>
#include <linux/pm.h>

#include <net/irda/irda.h>
#include <net/irda/irmod.h>
#include <net/irda/wrapper.h>
#include <net/irda/irda_device.h>

#include <asm/irq.h>
#include <asm/dma.h>
#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/system.h>
#include <linux/pci.h>

MODULE_AUTHOR("Russell King <rmk@arm.linux.org.uk>");
MODULE_DESCRIPTION("StrongARM SA1100 IrDA driver");

/*
 * Our netdevice.  There is only ever one of these.
 */
static struct net_device *netdev;

struct sa1100_irda {
	volatile unsigned char	utcr3;
	volatile unsigned char	hscr0;
	unsigned char		utcr4;
	unsigned char		using_hssr;
	unsigned char		power;
	unsigned char		open;
	unsigned char		unused[2];

	int			speed;
	int			newspeed;

	struct net_device_stats	stats;
	struct irlap_cb		*irlap;
	struct pm_dev		*pmdev;
	struct qos_info		qos;

	iobuff_t		tx_buff;
	iobuff_t		rx_buff;

};

static int rxdma, txdma;
static dma_addr_t dmaposrx, dmapostx;
static void sa1100_irda_rxdma_irq( void *buf_id, int size );
static void sa1100_irda_dma_receive(struct net_device *dev);

/*
 * Set the IrDA communications speed.
 */
static int sa1100_irda_set_speed(struct sa1100_irda *si, int speed)
{
	unsigned long flags;
	int brd, ret = -EINVAL;

	switch (speed) {
	case 9600:	case 19200:	case 38400:
	case 57600:	case 115200:
		brd = 3686400 / (16 * speed) - 1;

		save_flags(flags);
		cli();

		Ser2UTCR3 = 0;
		Ser2HSCR0 = si->hscr0 = HSCR0_UART;

		Ser2UTCR1 = brd >> 8;
		Ser2UTCR2 = brd;

		Ser2UTCR3 = si->utcr3 = UTCR3_RIE | UTCR3_RXE | UTCR3_TXE;
		Ser2HSCR2 = HSCR2_RcDataH | HSCR2_TrDataH;

#ifdef CONFIG_SA1100_ASSABET
		if (machine_is_assabet())
			BCR_clear(BCR_IRDA_FSEL);
#endif
#ifdef CONFIG_SA1100_BITSY
		if (machine_is_bitsy())
			clr_bitsy_egpio(EGPIO_BITSY_IR_FSEL);
#endif
#ifdef CONFIG_SA1100_YOPY
		if (machine_is_yopy()) 
			PPSR &= ~GPIO_IRDA_FIR;
#endif

		si->speed = speed;
		si->using_hssr = 0;

		restore_flags(flags);
		ret = 0;
		break;

	case 4000000:
		/* not currently supported */
		//printk("sa1100_ir: Enabling FIR\n");
#if 1
		save_flags(flags);
		cli();

		Ser2UTCR3 = si->utcr3 & ~(UTCR3_TXE | UTCR3_RXE);
		Ser2HSCR0 = si->hscr0 & ~(HSCR0_TXE | HSCR0_RXE);
		Ser2HSSR0 = 0xff; // clear all error conditions
		
		si->hscr0 = HSCR0_HSSP | HSCR0_TXE /*| HSCR0_LBM */;
		si->utcr3 = 0;

		Ser2HSCR0 = si->hscr0;
		Ser2UTCR3 = si->utcr3;

		si->speed = speed;
		si->using_hssr = 1;
		Ser2HSCR2 = HSCR2_RcDataL | HSCR2_TrDataH;

#ifdef CONFIG_SA1100_ASSABET
		if (machine_is_assabet())
			BCR_set(BCR_IRDA_FSEL);
#endif
#ifdef CONFIG_SA1100_BITSY
		if (machine_is_bitsy())
			set_bitsy_egpio(EGPIO_BITSY_IR_FSEL);
#endif
#ifdef CONFIG_SA1100_YOPY
		if (machine_is_yopy())
			PPSR |= GPIO_IRDA_FIR;
#endif

		restore_flags(flags);
#endif
		sa1100_irda_dma_receive(netdev);
		ret = 0;
		break;

	default:
		break;
	}
	return 0;
}

/*
 * Control the power state of the IrDA transmitter.
 * State:
 *  0 - off
 *  1 - short range, lowest power
 *  2 - medium range, medium power
 *  3 - maximum range, high power
 *
 * Currently, only assabet is known to support this.
 */
static int __sa1100_irda_set_power(struct sa1100_irda *si, int state)
{
	switch (state) {
	case 0:
#ifdef CONFIG_SA1100_ASSABET
		if (machine_is_assabet()) {
			BCR_clear(BCR_IRDA_MD1);
			BCR_set(BCR_IRDA_MD0);
		}
#endif
#ifdef CONFIG_SA1100_YOPY
		if (machine_is_yopy()) 
			PPSR |= GPIO_IRDA_POWER;
#endif
		break;

	case 1:
#ifdef CONFIG_SA1100_ASSABET
		if (machine_is_assabet()) {
			BCR_set(BCR_IRDA_MD1|BCR_IRDA_MD0);
		}
#endif
#ifdef CONFIG_SA1100_YOPY
		if (machine_is_yopy()) 
			PPSR &= ~GPIO_IRDA_POWER;
#endif
		break;

	case 2:
#ifdef CONFIG_SA1100_ASSABET
		if (machine_is_assabet()) {
			BCR_clear(BCR_IRDA_MD0);
			BCR_set(BCR_IRDA_MD1);
		}
#endif
#ifdef CONFIG_SA1100_YOPY
		if (machine_is_yopy()) 
			PPSR &= ~GPIO_IRDA_POWER;
#endif
		break;

	case 3:
#ifdef CONFIG_SA1100_ASSABET
		if (machine_is_assabet()) {
			BCR_clear(BCR_IRDA_MD1|BCR_IRDA_MD0);
		}
#endif
#ifdef CONFIG_SA1100_YOPY
		if (machine_is_yopy()) 
			PPSR &= ~GPIO_IRDA_POWER;
#endif
		break;
	}
	return 0;
}

static inline int sa1100_set_power(struct sa1100_irda *si, int state)
{
	int ret;

	ret = __sa1100_irda_set_power(si, state);
	if (ret == 0)
		si->power = state;

	return ret;
}

#ifdef CONFIG_PM
/*
 * Suspend the IrDA interface.
 */
static int sa1100_irda_suspend(struct net_device *dev, int state)
{
	struct sa1100_irda *si = dev->priv;

	if (si && si->open) {
		/*
		 * Stop the transmit queue
		 */
		netif_device_detach(dev);
		disable_irq(dev->irq);
		__sa1100_irda_set_power(si, 0);
	}

	return 0;
}

/*
 * Resume the IrDA interface.
 */
static int sa1100_irda_resume(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;

	if (si && si->open) {
		/*
		 * If we missed a speed change, initialise at the new speed
		 * directly.  It is debatable whether this is actually
		 * required, but in the interests of continuing from where
		 * we left off it is desireable.
		 */
		if (si->newspeed)
			si->speed = si->newspeed;

		Ser2UTCR3 = si->utcr3 = 0;
		Ser2HSCR0 = si->hscr0 = HSCR0_UART;
		Ser2UTCR4 = si->utcr4 = UTCR4_HPSIR | UTCR4_Z3_16Bit;
		Ser2UTCR0 = UTCR0_8BitData;
		Ser2UTSR0 = UTSR0_REB | UTSR0_RBB | UTSR0_RID;

		sa1100_irda_set_speed(si, si->speed);
		__sa1100_irda_set_power(si, si->power);
		enable_irq(dev->irq);

		/*
		 * This automatically wakes up the queue
		 */
		netif_device_attach(dev);
	}

	return 0;
}

static int sa1100_irda_pmproc(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	int ret;

	if (!dev->data)
		return -EINVAL;

	switch (rqst) {
	case PM_SUSPEND:
		ret = sa1100_irda_suspend((struct net_device *)dev->data,
					  (int)data);
		break;

	case PM_RESUME:
		ret = sa1100_irda_resume((struct net_device *)dev->data);
		break;

	default:
		ret = -EINVAL;
		break;
	}

	return ret;
}
#endif

static void sa1100_irda_dma_receive(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;

	//printk("sa1100_ir: Started FIR receive\n");
	// First empty receive FIFO
	Ser2HSCR0 = si->hscr0 & ~HSCR0_RXE;
	si->hscr0 |= HSCR0_RXE;
	Ser2HSCR0 = si->hscr0;
	si->rx_buff.data = si->rx_buff.head;

	// Here one would enable DMA and stuff
	sa1100_dma_flush_all(rxdma);
	if (dmaposrx)
		pci_unmap_single(NULL, dmaposrx, si->rx_buff.truesize, PCI_DMA_FROMDEVICE);
	dmaposrx = pci_map_single(NULL, si->rx_buff.data, si->rx_buff.truesize, PCI_DMA_FROMDEVICE);
	sa1100_dma_set_callback(rxdma, sa1100_irda_rxdma_irq);
	sa1100_dma_queue_buffer(rxdma, NULL, dmaposrx, si->rx_buff.truesize);
}

static void sa1100_irda_rxdma_irq( void *buf_id, int size )
{
	struct sa1100_irda *si = netdev->priv;
	// Strange, we should never get here.
	printk("sa1100_irda_rxdma_irq: Should never get here\n");
	si->stats.rx_over_errors++;
	sa1100_irda_dma_receive(netdev);
}

static void sa1100_irda_txdma_irq( void *buf_id, int size )
{
	struct sa1100_irda *si = netdev->priv;

	sa1100_irda_dma_receive(netdev);
	si->stats.tx_packets++;
	si->stats.tx_bytes+=size;
	sa1100_dma_set_callback(txdma, NULL);
	pci_unmap_single(NULL, dmapostx, size, PCI_DMA_TODEVICE);
	dmapostx=0;
	dev_kfree_skb_irq((struct sk_buff *)buf_id);
	netif_wake_queue(netdev);
}

static void sa1100_irda_dma_receive_complete(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;
	struct sk_buff *skb;
	dma_addr_t dma_addr;
	int len,bad=0;
	
	//printk("sa1100_ir: sa1100_irda_dma_receive_complete\n");
	sa1100_dma_set_callback(rxdma, NULL);
	sa1100_dma_stop(rxdma);
	sa1100_dma_get_current(rxdma, NULL, &dma_addr);
	// STOP DMA and obtain DMA count
	pci_unmap_single(NULL, dmaposrx, si->rx_buff.truesize, PCI_DMA_FROMDEVICE);
	len = dma_addr - dmaposrx;
	dmaposrx = 0;
	si->rx_buff.data += len;

	while (Ser2HSSR0 & HSSR0_EIF) {
		int bstat;

		bstat = Ser2HSSR1;

		if (bstat & HSSR1_ROR)
			si->stats.rx_fifo_errors++;
		if (bstat & HSSR1_CRE)
			si->stats.rx_crc_errors++;
		if ( bstat & ( HSSR1_ROR | HSSR1_CRE) )
			bad++;

		*(si->rx_buff.data++) = Ser2HSDR;
	}
	if (bad)
		return;
	
	Ser2HSCR0 = si->hscr0 = si->hscr0 & ~HSCR0_RXE;
	len = si->rx_buff.data - si->rx_buff.head;
//	printk("sa1100_irda_dma_receive_complete: %d\n",len);

	if (len) {
		skb = dev_alloc_skb(len+1);
		if (skb == NULL) {
			si->stats.rx_dropped++;
			return;
		}
		skb_reserve(skb, 1);
		skb_put(skb, len);
		memcpy(skb->data, si->rx_buff.data, len);
		skb->dev = dev;
		skb->mac.raw = skb->data;
		skb->protocol = htons(ETH_P_IRDA);
		netif_rx(skb);
		si->stats.rx_packets++;
		si->stats.rx_bytes += len;
	}

	return;
}


/*
 * HSSP format interrupt service routines.
 */
static void sa1100_irda_hssp_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;
	struct sa1100_irda *si = dev->priv;
	int status;

	status = Ser2HSSR0;
	//printk("sa1100_irda_hssp_irq: %x %x\n",status,Ser2HSSR1);

	/*
	 * Deal with any receive errors first.  The bytes in error may be
	 * the only bytes in the receive FIFO, so we do this first.
	 */
	while (status & HSSR0_EIF) {
		sa1100_irda_dma_receive_complete(dev);
		sa1100_irda_dma_receive(dev);
		status = Ser2HSSR0;
	}

	if (status & HSSR0_TUR) {
		// strange - why are we here?
		Ser2HSSR0 = HSSR0_TUR;
	}

	if (status & HSSR0_RAB) {
		Ser2HSSR0 = HSSR0_RAB;
		sa1100_irda_dma_receive_complete(dev);
		sa1100_irda_dma_receive(dev);
	}

	if (status & HSSR0_FRE) {
		si->stats.rx_frame_errors++;
		Ser2HSSR0 = HSSR0_FRE;
	}
}

/*
 * HP-SIR format interrupt service routines.
 */
static void sa1100_irda_hpsir_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;
	struct sa1100_irda *si = dev->priv;
	int status;

	status = Ser2UTSR0;

	/*
	 * Deal with any receive errors first.  The bytes in error may be
	 * the only bytes in the receive FIFO, so we do this first.
	 */
	while (status & UTSR0_EIF) {
		int bstat;

		bstat = Ser2UTSR1;

		if (bstat & UTSR1_FRE)
			si->stats.rx_frame_errors++;
		if (bstat & UTSR1_ROR)
			si->stats.rx_fifo_errors++;

		bstat = Ser2UTDR;
		status = Ser2UTSR0;
	}

	if (status & UTSR0_RFS) {
		/*
		 * There are at least 4 bytes in the FIFO.  Read 3 bytes
		 * and leave the rest to the block below.
		 */
		async_unwrap_char(dev, &si->stats, &si->rx_buff, Ser2UTDR);
		async_unwrap_char(dev, &si->stats, &si->rx_buff, Ser2UTDR);
		async_unwrap_char(dev, &si->stats, &si->rx_buff, Ser2UTDR);
	}

	if (status & (UTSR0_RFS | UTSR0_RID)) {
		/*
		 * Fifo contains more than 1 character.
		 */
		do {
			async_unwrap_char(dev, &si->stats, &si->rx_buff,
					  Ser2UTDR);
		} while (Ser2UTSR1 & UTSR1_RNE);

		dev->last_rx = jiffies;
	}

	if (status & UTSR0_TFS && si->tx_buff.len) {
		/*
		 * Transmitter FIFO is not full
		 */
		do {
			Ser2UTDR = *si->tx_buff.data++;
			si->tx_buff.len -= 1;
		} while (Ser2UTSR1 & UTSR1_TNF && si->tx_buff.len);

		if (si->tx_buff.len == 0) {
			si->stats.tx_packets++;
			si->stats.tx_bytes += si->tx_buff.data -
					      si->tx_buff.head;

			/*
			 * Still transmitting...
			 */
			si->utcr3 = UTCR3_TXE;
			Ser2UTCR3 = si->utcr3;

			/*
			 * We need to ensure that the transmitter has
			 * finished.  Reschedule ourselves for later.
			 * We may miss the other sides response, but
			 * they will retransmit.
			 */
			while (Ser2UTSR1 & UTSR1_TBY);

			/*
			 * Ok, we've finished transmitting.  Now enable
			 * the receiver.
			 */
			si->utcr3 = UTCR3_RIE | UTCR3_RXE | UTCR3_TXE;
			Ser2UTCR3 = si->utcr3;

			if (si->newspeed) {
				sa1100_irda_set_speed(si, si->newspeed);
				si->newspeed = 0;
			}

			/* I'm hungry! */
			netif_wake_queue(dev);
		}
	}

	/*
	 * We must clear certain bits.
	 */
	status &= (UTSR0_RID | UTSR0_RBB | UTSR0_REB);
	if (status)
		Ser2UTSR0 = status;
}

/* Generic irq routinue that chooses actual one on basis of format */
static void sa1100_irda_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct sa1100_irda *si = ((struct net_device *)dev_id)->priv;
	if ( si->using_hssr) {
		sa1100_irda_hssp_irq(irq, dev_id, regs);
	} else {
		sa1100_irda_hpsir_irq(irq, dev_id, regs);
	}
}

static int sa1100_irda_dma_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;

	if (dmapostx) {
		netif_stop_queue(dev);
		return 1;
	}
		
	si->tx_buff.data = si->tx_buff.head;

	if (skb->len > 2047) {
		printk("sa1100_ir: Trying to send too big packet, size=%d\n", skb->len);
		si->stats.tx_dropped++;
		return 1;
	}
	si->tx_buff.data=skb->data;
	dmapostx = pci_map_single(NULL, si->tx_buff.data, skb->len, PCI_DMA_TODEVICE);
	Ser2HSCR0 = si->hscr0 & ~HSCR0_RXE;
	sa1100_dma_set_callback(txdma, sa1100_irda_txdma_irq);
	sa1100_dma_queue_buffer(txdma, skb, dmapostx, skb->len);
	dev->trans_start = jiffies;

	return 0;
}

static int sa1100_irda_hard_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;
	int speed = irda_get_next_speed(skb);
	int ret = 0;

	/*
	 * If the speed has changed, schedule an update just
	 * after we've sent this packet.
	 */
	if (speed != si->speed && speed != -1)
		si->newspeed = speed;

	if (si->using_hssr == 0) {
		unsigned long flags;

		si->tx_buff.data = si->tx_buff.head;
		si->tx_buff.len  = async_wrap_skb(skb, si->tx_buff.data,
						  si->tx_buff.truesize);
		/*
		 * Set the transmit interrupt enable.
		 * This will fire off an interrupt immediately.
		 */
		save_flags(flags);
		cli();

		si->utcr3 = UTCR3_TIE | UTCR3_TXE;
		Ser2UTCR3 = si->utcr3;

		restore_flags(flags);
		dev->trans_start = jiffies;
		kfree_skb(skb);

	} else {
		ret = sa1100_irda_dma_xmit(skb, dev);
	}

	//printk("sa1100_ir: Queued TX packet for transmission %d\n",ret);

	return ret;
}

static int sa1100_irda_ioctl(struct net_device *dev, struct ifreq *ifreq,
			     int cmd)
{
	struct if_irda_req *rq = (struct if_irda_req *)ifreq;
	struct sa1100_irda *si = dev->priv;
	int ret = -EOPNOTSUPP;

	switch (cmd) {
	case SIOCSBANDWIDTH:
		if (in_irq() || capable(CAP_NET_ADMIN)) {
			/*
			 * We are unable to set the speed if the
			 * device is not running.
			 */
			if (si->open)
				ret = sa1100_irda_set_speed(si,
						rq->ifr_baudrate);
			else {
				printk("sa1100_irda_ioctl: SIOCSBANDWIDTH: !netif_running\n");
				ret = 0;
			}
		} else {
			printk("sa1100_irda_ioctl: SIOCSBANDWIDTH: !capable\n");
		}
		break;

	case SIOCSMEDIABUSY:
		ret = -EPERM;
		if (in_irq() || capable(CAP_NET_ADMIN)) {
			irda_device_set_media_busy(dev, TRUE);
			ret = 0;
		} else {
			printk("sa1100_irda_ioctl: SIOCSMEDIABUSY: !capable\n");
		}
		break;

	case SIOCGRECEIVING:
		rq->ifr_receiving = si->using_hssr
					? 0
					: si->rx_buff.state != OUTSIDE_FRAME;
		break;

	default:
		break;
	}
		
	return ret;
}

static struct net_device_stats *sa1100_irda_stats(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;

	return &si->stats;
}

static int sa1100_irda_startup(struct sa1100_irda *si)
{
	int ret;

#ifdef CONFIG_SA1100_YOPY
	if (machine_is_yopy()) {
		PPDR |= GPIO_IRDA_POWER | GPIO_IRDA_FIR;
		PPSR |= GPIO_IRDA_POWER | GPIO_IRDA_FIR;
		PSDR |= GPIO_IRDA_POWER | GPIO_IRDA_FIR;
		Ser2HSCR2 = HSCR2_TrDataH | HSCR2_RcDataL;
	}
#endif

	/*
	 * Enable HP-SIR modulation, and ensure that the port is disabled.
	 */
	Ser2UTCR3 = si->utcr3 = 0;
	Ser2HSCR0 = si->hscr0 = HSCR0_UART;
	Ser2UTCR4 = si->utcr4 = UTCR4_HPSIR | UTCR4_Z3_16Bit;
	Ser2UTCR0 = UTCR0_8BitData;

	/*
	 * Clear status register
	 */
	Ser2UTSR0 = UTSR0_REB | UTSR0_RBB | UTSR0_RID;

	ret = sa1100_irda_set_speed(si, si->speed);
	if (ret)
		return ret;

	si->open = 1;

	return 0;
}

static void sa1100_irda_shutdown(struct sa1100_irda *si)
{
	si->open = 0;

#ifdef CONFIG_SA1100_BITSY
	if (machine_is_bitsy())
		clr_bitsy_egpio(EGPIO_BITSY_IR_ON);
#endif

	/* Disable the port. */
	si->utcr3 = 0;
	si->hscr0 = 0;
	Ser2UTCR3 = 0;
	Ser2HSCR0 = 0;
}

static int sa1100_irda_start(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;
	int err;

	MOD_INC_USE_COUNT;

	si->speed = 9600;

	err = request_irq(dev->irq, sa1100_irda_irq, 0, dev->name, dev);
	if (err)
		goto out;

	/*
	 * The interrupt must remain disabled for now.
	 */
	disable_irq(dev->irq);

#ifdef CONFIG_SA1100_BITSY
	if (machine_is_bitsy())
		set_bitsy_egpio(EGPIO_BITSY_IR_ON);
#endif
	/*
	 * Setup the serial port for the specified speed.
	 */
	err = sa1100_irda_startup(si);
	if (err)
		goto out_irq;

	/*
	 * Open a new IrLAP layer instance.
	 */
	si->irlap = irlap_open(dev, &si->qos);
	err = -ENOMEM;
	if (!si->irlap)
		goto out_shutdown;

	/*
	 * Now enable the interrupt and start the queue
	 */
	sa1100_set_power(si, 1);
	enable_irq(dev->irq);
	netif_start_queue(dev);
	return 0;

out_shutdown:
	sa1100_irda_shutdown(si);
out_irq:
	free_irq(dev->irq, dev);
out:
	MOD_DEC_USE_COUNT;
	return err;
}

static int sa1100_irda_stop(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;

	disable_irq(dev->irq);

	/* Stop IrLAP */
	if (si->irlap) {
		irlap_close(si->irlap);
		si->irlap = NULL;
	}

	netif_stop_queue(dev);
	free_irq(dev->irq, dev);
	sa1100_set_power(si, 0);
	sa1100_irda_shutdown(si);

	MOD_DEC_USE_COUNT;

	return 0;
}

static int sa1100_irda_init_iobuf(iobuff_t *io, int size)
{
	io->head = kmalloc(size, GFP_KERNEL | GFP_DMA);
	if (io->head != NULL) {
		io->truesize = size;
		io->in_frame = FALSE;
		io->state    = OUTSIDE_FRAME;
		io->data     = io->head;
	}
	return io->head ? 0 : -ENOMEM;
}

static int sa1100_irda_net_init(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;
	int err = -ENOMEM;

	si = kmalloc(sizeof(struct sa1100_irda), GFP_KERNEL);
	if (!si)
		goto out;

	memset(si, 0, sizeof(*si));

	err = sa1100_irda_init_iobuf(&si->rx_buff, 14384);
	if (err)
		goto out;
	err = sa1100_irda_init_iobuf(&si->tx_buff, 4000);
	if (err)
		goto out_free_rx;

	dev->priv = si;
	dev->hard_start_xmit	= sa1100_irda_hard_xmit;
	dev->open		= sa1100_irda_start;
	dev->stop		= sa1100_irda_stop;
	dev->do_ioctl		= sa1100_irda_ioctl;
	dev->get_stats		= sa1100_irda_stats;

	irda_device_setup(dev);
	irda_init_max_qos_capabilies(&si->qos);

	/*
	 * We support original IRDA up to 115k2 and IrDA 1.2 FIR at 4 Mbps
	 * Min Turn Time set to 1ms or greater.
	 */
	si->qos.baud_rate.bits &= IR_9600|IR_19200|IR_38400|IR_57600|
				  IR_115200|IR_4000000;
	si->qos.min_turn_time.bits = 7;

	irda_qos_bits_to_value(&si->qos);

	/*
	 * Enable HP-SIR modulation for the beginning, and ensure that the port is disabled.
	 */
	Ser2UTCR3 = si->utcr3 = 0;
	Ser2UTCR4 = si->utcr4 = UTCR4_HPSIR | UTCR4_Z3_16Bit;
	Ser2HSCR0 = si->hscr0 = HSCR0_UART;

#ifdef CONFIG_PM
	/*
	 * Power-Management is optional.
	 */
	si->pmdev = pm_register(PM_SYS_DEV, PM_SYS_IRDA, sa1100_irda_pmproc);
	if (si->pmdev)
		si->pmdev->data = dev;
#endif

	return 0;

out_free_rx:
	kfree(si->rx_buff.head);
out:
	kfree(si);

	return err;
}

/*
 * Remove all traces of this driver module from the kernel, so we can't be
 * called.  Note that the device has already been stopped, so we don't have
 * to worry about that.
 */
static void sa1100_irda_net_uninit(struct net_device *dev)
{
	struct sa1100_irda *si = dev->priv;

	dev->hard_start_xmit	= NULL;
	dev->open		= NULL;
	dev->stop		= NULL;
	dev->do_ioctl		= NULL;
	dev->get_stats		= NULL;
	dev->priv		= NULL;

	pm_unregister(si->pmdev);

	if (dmaposrx)
		pci_unmap_single(NULL, dmaposrx, si->rx_buff.truesize, PCI_DMA_FROMDEVICE);

	kfree(si->tx_buff.head);
	kfree(si->rx_buff.head);
	kfree(si);
}

#ifdef MODULE
static
#endif
int __init sa1100_irda_init(void)
{
	struct net_device *dev;
	int err;

	err = sa1100_request_dma(&rxdma, "IrDA receive");
	if (err) {
		printk("sa1100_ir: unable to register rx dma\n");
		goto err_rx_dma;
	}
	err = sa1100_request_dma(&txdma, "IrDA transmit");
	if (err) {
		printk("sa1100_ir: unable to register tx dma\n");
		goto err_tx_dma;
	}

	rtnl_lock();
	dev = dev_alloc("irda%d", &err);
	if (dev) {
		dev->irq    = IRQ_Ser2ICP;
		dev->init   = sa1100_irda_net_init;
		dev->uninit = sa1100_irda_net_uninit;

		err = register_netdevice(dev);

		if (err)
			goto err_net_dev;
		else
			netdev = dev;
	}
	rtnl_unlock();
	sa1100_dma_set_device(rxdma, DMA_Ser2HSSPRd);
	sa1100_dma_set_device(txdma, DMA_Ser2HSSPWr);
	return 0;

err_net_dev:
	kfree(dev);
        sa1100_free_dma(txdma);
err_tx_dma:
        sa1100_free_dma(rxdma);
err_rx_dma:
	return err;
}

static void __exit sa1100_irda_exit(void)
{
	struct net_device *dev = netdev;

	netdev = NULL;
	if (dev) {
		rtnl_lock();
		unregister_netdevice(dev);
		rtnl_unlock();
	}

	/*
	 * We now know that the netdevice is no longer in use, and all
	 * references to our driver have been removed.  The only structure
	 * which may still be present is the netdevice, which will get
	 * cleaned up by net/core/dev.c
	 */

        sa1100_free_dma(txdma);
        sa1100_free_dma(rxdma);
}

#ifdef MODULE
module_init(sa1100_irda_init);
module_exit(sa1100_irda_exit);
#endif
