/*
 * drivers/pcmcia/sa1100_bitsy.c
 *
 * PCMCIA implementation routines for Bitsy
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/i2c.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/arch/pcmcia.h>
#include <asm/arch/h3600-sleeve.h>

#if defined(CONFIG_MERCURY_BACKPAQ) || defined(CONFIG_MERCURY_BACKPAQ_MODULE)
#include <asm/arch/backpaq.h>

struct mercury_backpaq_sysctl *mercury_backpaq_sysctl = ((struct mercury_backpaq_sysctl *)BACKPAQ_SYSCTL_BASE);

#endif /* CONFIG_MERCURY_BACKPAQ */

#ifdef CONFIG_H3600_SLEEVE
static enum sleeve_device sleeve_type = SLEEVE_NO_DEVICE;
#else
/* default to behavior of PCMCIA sleeve (also works for CF) */
static enum sleeve_device sleeve_type = SINGLE_PCMCIA_SLEEVE;
#endif

static struct pcmcia_init sa1100_bitsy_pcmcia_init;

static int bitsy_pcmcia_init(struct pcmcia_init *init)
{
        int irq, res;

        sa1100_bitsy_pcmcia_init = *init;
        switch (sleeve_type) {
        case SLEEVE_NO_DEVICE: 
                break;

        case MERCURY_BACKPAQ:
#if defined(CONFIG_MERCURY_BACKPAQ) || defined(CONFIG_MERCURY_BACKPAQ_MODULE)

                /* fall through */
#endif
        case SINGLE_COMPACTFLASH_SLEEVE:
        case SINGLE_PCMCIA_SLEEVE:

                /* Enable CF bus: */
                set_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
                clr_bitsy_egpio(EGPIO_BITSY_OPT_RESET);

                /* All those are inputs */
                GPDR &= ~(GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1 | GPIO_BITSY_PCMCIA_IRQ0| GPIO_BITSY_PCMCIA_IRQ1);

                /* Set transition detect */
                set_GPIO_IRQ_edge( GPIO_BITSY_PCMCIA_CD0 | GPIO_BITSY_PCMCIA_CD1, GPIO_BOTH_EDGES );
                set_GPIO_IRQ_edge( GPIO_BITSY_PCMCIA_IRQ0| GPIO_BITSY_PCMCIA_IRQ1, GPIO_FALLING_EDGE );

                /* Register interrupts */
                irq = IRQ_GPIO_BITSY_PCMCIA_CD0;
                res = request_irq( irq, init->handler, SA_INTERRUPT, "PCMCIA_CD0", NULL );
                if( res < 0 ) goto irq_err;
                irq = IRQ_GPIO_BITSY_PCMCIA_CD1;
                res = request_irq( irq, init->handler, SA_INTERRUPT, "PCMCIA_CD1", NULL );
                if( res < 0 ) goto irq_err;
        }

        return 2;

 irq_err:
        printk( KERN_ERR __FUNCTION__ ": Request for IRQ %u failed\n", irq );
        return -1;
}

static int bitsy_pcmcia_shutdown(void)
{
        printk(__FUNCTION__ "\n");
        switch (sleeve_type) {
        case MERCURY_BACKPAQ:
                /* fall through */
        case SINGLE_COMPACTFLASH_SLEEVE:
        case SINGLE_PCMCIA_SLEEVE:
                /* disable IRQs */
                free_irq( IRQ_GPIO_BITSY_PCMCIA_CD0, NULL );
                free_irq( IRQ_GPIO_BITSY_PCMCIA_CD1, NULL );
  
                /* Disable CF bus: */
                clr_bitsy_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
                set_bitsy_egpio(EGPIO_BITSY_OPT_RESET);
                break;
        case SLEEVE_NO_DEVICE:
        }

        return 0;
}

static int bitsy_pcmcia_socket_state(struct pcmcia_state_array *state_array)
{
        unsigned long levels;

        if(state_array->size<2) return -1;

        memset(state_array->state, 0, 
               (state_array->size)*sizeof(struct pcmcia_state));

        switch (sleeve_type) {
        case SINGLE_PCMCIA_SLEEVE:
        case SINGLE_COMPACTFLASH_SLEEVE: 
        case MERCURY_BACKPAQ: {
                levels=GPLR;

                state_array->state[0].detect=((levels & GPIO_BITSY_PCMCIA_CD0)==0)?1:0;
                state_array->state[0].ready=(levels & GPIO_BITSY_PCMCIA_IRQ0)?1:0;
                if (sleeve_type == MERCURY_BACKPAQ) {
#if defined(CONFIG_MERCURY_BACKPAQ) || defined(CONFIG_MERCURY_BACKPAQ_MODULE)
		  int bvd1 = ((mercury_backpaq_sysctl->socket[0] & BACKPAQ_SOCKET_BVD1) ? 1 : 0);
		  int bvd2 = ((mercury_backpaq_sysctl->socket[0] & BACKPAQ_SOCKET_BVD2) ? 2 : 0);
		  int vs = (((mercury_backpaq_sysctl->socket[0] & BACKPAQ_SOCKET_VS1) ? 1 : 0)
			    | ((mercury_backpaq_sysctl->socket[0] & BACKPAQ_SOCKET_VS2) ? 2 : 0));
		  printk(__FUNCTION__ ":%d &socket_status=%p socket_status=%x\n",
			 __LINE__, 
			 &mercury_backpaq_sysctl->socket[0], 
			 mercury_backpaq_sysctl->socket[0]);
                        state_array->state[0].bvd1 = bvd1;
                        state_array->state[0].bvd2 = bvd2;
                        state_array->state[0].wrprot = 0; 
                        state_array->state[0].vs_3v = (vs == 3) ? 0 : 1 ;
                        state_array->state[0].vs_Xv = 0 ;
#endif
                } else {
                        state_array->state[0].bvd1= 1;
                        state_array->state[0].bvd2= 1;
                        state_array->state[0].wrprot=0; /* Not available on Bitsy. */
                        state_array->state[0].vs_3v=0;
                        state_array->state[0].vs_Xv=0;
                }

                state_array->state[1].detect=((levels & GPIO_BITSY_PCMCIA_CD1)==0)?1:0;
                state_array->state[1].ready=(levels & GPIO_BITSY_PCMCIA_IRQ1)?1:0;
                if (sleeve_type == MERCURY_BACKPAQ) {
#if defined(CONFIG_MERCURY_BACKPAQ) || defined(CONFIG_MERCURY_BACKPAQ_MODULE)
		  int bvd1 = ((mercury_backpaq_sysctl->socket[1] & BACKPAQ_SOCKET_BVD1) ? 1 : 0);
		  int bvd2 = ((mercury_backpaq_sysctl->socket[1] & BACKPAQ_SOCKET_BVD2) ? 1 : 0);
		  int vs = (((mercury_backpaq_sysctl->socket[1] & BACKPAQ_SOCKET_VS1) ? 1 : 0)
			    | ((mercury_backpaq_sysctl->socket[1] & BACKPAQ_SOCKET_VS2) ? 2 : 0));

		  printk(__FUNCTION__ ":%d &socket_status=%p socket_status=%x\n",
			 __LINE__, 
			 &mercury_backpaq_sysctl->socket[1], 
			 mercury_backpaq_sysctl->socket[1]);
		        state_array->state[1].bvd1 = bvd1;
                        state_array->state[1].bvd2 = bvd2;
                        state_array->state[1].wrprot = 0;
                        state_array->state[1].vs_3v = (vs == 3) ? 0 : 1 ;;
			state_array->state[1].vs_Xv = 0;
#endif
                } else {
                        state_array->state[1].bvd1=1;
                        state_array->state[1].bvd2=1;
                        state_array->state[1].wrprot=0; /* Not available on Bitsy. */
                        state_array->state[1].vs_3v=0;
                        state_array->state[1].vs_Xv=0;
                }

        } break;

        case SLEEVE_NO_DEVICE:
        default:
                break;
        }
        return 1;
}

static int bitsy_pcmcia_get_irq_info(struct pcmcia_irq_info *info)
{

        switch (info->sock) {
        case 0:
                info->irq=IRQ_GPIO_BITSY_PCMCIA_IRQ0;
                break;
        case 1:
                info->irq=IRQ_GPIO_BITSY_PCMCIA_IRQ1;
                break;
        default:
                return -1;
        }
        return 0;
}

static int bitsy_pcmcia_configure_socket(const struct pcmcia_configure *configure)
{

        int sock = configure->sock;
        if(sock>1) return -1;

        if (sleeve_type != SLEEVE_NO_DEVICE)
                printk(__FUNCTION__ ": socket=%d vcc=%d vpp=%d reset=%d\n", 
                       sock, configure->vcc, configure->vpp, configure->reset);

        switch (sleeve_type) {
        case SINGLE_COMPACTFLASH_SLEEVE:
        case SINGLE_PCMCIA_SLEEVE: {
                unsigned long flags;

                save_flags_cli(flags);

                switch (configure->vcc) {
                case 0:
                        clr_bitsy_egpio(EGPIO_BITSY_OPT_ON);
                        break;

                case 50:
                case 33:
                        set_bitsy_egpio(EGPIO_BITSY_OPT_ON);
                        break;

                default:
                        printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
                               configure->vcc);
                        restore_flags(flags);
                        return -1;
                }

                if (configure->reset)
                        set_bitsy_egpio(EGPIO_BITSY_CARD_RESET);
                else
                        clr_bitsy_egpio(EGPIO_BITSY_CARD_RESET);
                restore_flags(flags);
        } break;

        case MERCURY_BACKPAQ: {
#if defined(CONFIG_MERCURY_BACKPAQ) || defined(CONFIG_MERCURY_BACKPAQ_MODULE)
                unsigned long flags;

		int shift = (sock ? 0 : 4);
	        int mask = 0xf << shift;
		int val = 0;

                save_flags_cli(flags);

                switch (configure->vcc) {
                case 0:
                        break;
                case 50:
			val |= (BACKPAQ_PCMCIA_A5VEN << shift);
                        break;
                case 33:
			val |= (BACKPAQ_PCMCIA_A3VEN << shift);
                        break;
                default:
                        printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
                               configure->vcc);
                        restore_flags(flags);
                        return -1;
                }

                switch (configure->vpp) {
                case 0:
                        break;
                case 50:
                        break;
                case 33:
                        break;
                default:
                        printk(KERN_ERR "%s(): unrecognized Vpp %u\n", __FUNCTION__,
                               configure->vpp);
                        restore_flags(flags);
                        return -1;
                }

                printk("     mercury_backpaq_sysctl->pcmcia_power=%x mask=%x val=%x vcc=%d\n", 
		       mercury_backpaq_sysctl->pcmcia_power, mask, val, configure->vcc);
                mercury_backpaq_sysctl->pcmcia_power &= ~(short) mask;
                mercury_backpaq_sysctl->pcmcia_power |= val;
                printk("  -> mercury_backpaq_sysctl->pcmcia_power=%x (should be %x)\n", 
                       mercury_backpaq_sysctl->pcmcia_power,
                       (mercury_backpaq_sysctl->pcmcia_power & ~(short) mask) | ( val << shift));
                

                if (configure->reset)
                        set_bitsy_egpio(EGPIO_BITSY_CARD_RESET);
                else
                        clr_bitsy_egpio(EGPIO_BITSY_CARD_RESET);
                restore_flags(flags);
#endif /* CONFIG_MERCURY_BACKPAQ */
        } break;

        case SLEEVE_NO_DEVICE: 
           printk("    no pcmcia sleeve\n");
        }

        /* silently ignore vpp and speaker enables. */

        if (sleeve_type != SLEEVE_NO_DEVICE)
          printk(__FUNCTION__ ": finished\n");

        return 0;
}

struct pcmcia_low_level bitsy_pcmcia_ops = { 
        bitsy_pcmcia_init,
        bitsy_pcmcia_shutdown,
        bitsy_pcmcia_socket_state,
        bitsy_pcmcia_get_irq_info,
        bitsy_pcmcia_configure_socket
};

void sa1100_bitsy_change_sleeves(enum sleeve_device new_sleeve_type) 
{
        if (sleeve_type == new_sleeve_type)
                return;
        if (sleeve_type != SLEEVE_NO_DEVICE)
                bitsy_pcmcia_shutdown();
        sleeve_type = new_sleeve_type;
        if (sa1100_bitsy_pcmcia_init.handler != NULL) {
                bitsy_pcmcia_init(&sa1100_bitsy_pcmcia_init);
        }
}
EXPORT_SYMBOL(sa1100_bitsy_change_sleeves);

