/*
 * JFFS2 -- Journalling Flash File System, Version 2.
 *
 * Copyright (C) 2001  Red Hat, Inc.
 *
 * Created by David Woodhouse <dwmw2@infradead.org>.
 *
 * The original JFFS, from which the design for JFFS2 was derived,
 * was designed and implemented by Axis Communications AB.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * $Id: compr_zlib.c,v 1.4 2001/03/01 22:34:28 dwmw2 Exp $
 *
 */

#include "zlib.h"

#define min(x,y) ((x)<(y)?(x):(y))
#ifdef __KERNEL__
#ifndef D1
#define D1(x) x
#endif
#include <linux/kernel.h>
#include <linux/slab.h>
static void *zalloc(void *opaque, unsigned nr, unsigned size)
{
	/* How much does it request? Should we use vmalloc? Or be dynamic? */
	return kmalloc(nr * size, GFP_KERNEL);
}

static void zfree(void *opaque, void *addr)
{
	kfree(addr);
}
#else
#ifndef D1
#define D1(x)
#endif
#define KERN_DEBUG
#define KERN_NOTICE
#define KERN_WARNING
#define printk printf
#include <stdio.h>
#include <asm/types.h>
#endif

	/* Plan: call deflate() with avail_in == *sourcelen, 
		avail_out = *dstlen - 12 and flush == Z_FINISH. 
		If it doesn't manage to finish,	call it again with
		avail_in == 0 and avail_out set to the remaining 12
		bytes for it to clean up. 
	   Q: Is 12 bytes sufficient?
	*/
#define STREAM_END_SPACE 12

int zlib_compress(unsigned char *data_in, unsigned char *cpage_out, 
		   __u32 *sourcelen, __u32 *dstlen)
{
	z_stream strm;
	int ret;

	if (*dstlen <= STREAM_END_SPACE)
		return -1;

#ifdef __KERNEL__
	strm.zalloc = zalloc;
	strm.zfree = zfree;
#else
	strm.zalloc = (void *)0;
	strm.zfree = (void *)0;
#endif

	if (Z_OK != deflateInit(&strm, 3)) {
		printk(KERN_WARNING "deflateInit failed\n");
		return -1;
	}
	strm.next_in = data_in;
	strm.total_in = 0;
	
	strm.next_out = cpage_out;
	strm.total_out = 0;

	while (strm.total_out < *dstlen - STREAM_END_SPACE && strm.total_in < *sourcelen) {
		strm.avail_out = *dstlen - (strm.total_out + STREAM_END_SPACE);
		strm.avail_in = min(*sourcelen-strm.total_in, strm.avail_out);
		D1(printk(KERN_DEBUG "calling deflate with avail_in %d, avail_out %d\n",
			  strm.avail_in, strm.avail_out));
		ret = deflate(&strm, Z_PARTIAL_FLUSH);
		D1(printk(KERN_DEBUG "deflate returned with avail_in %d, avail_out %d, total_in %ld, total_out %ld\n", 
			  strm.avail_in, strm.avail_out, strm.total_in, strm.total_out));
		if (ret != Z_OK) {
			D1(printk(KERN_DEBUG "deflate in loop returned %d\n", ret));
			deflateEnd(&strm);
			return -1;
		}
	}
	strm.avail_out += STREAM_END_SPACE;
	strm.avail_in = 0;
	ret = deflate(&strm, Z_FINISH);
	if (ret != Z_STREAM_END) {
		D1(printk(KERN_DEBUG "final deflate returned %d\n", ret));
		deflateEnd(&strm);
		return -1;
	}
	deflateEnd(&strm);

	D1(printk(KERN_DEBUG "zlib compressed %ld bytes into %ld\n", strm.total_in, strm.total_out));

	if (strm.total_out >= strm.total_in)
		return -1;


	*dstlen = strm.total_out;
	*sourcelen = strm.total_in;
	return 0;
}

void zlib_decompress(unsigned char *data_in, unsigned char *cpage_out,
		      __u32 srclen, __u32 destlen)
{
	z_stream strm;
	int ret;

#ifdef __KERNEL__
	strm.zalloc = zalloc;
	strm.zfree = zfree;
#else
	strm.zalloc = (void *)0;
	strm.zfree = (void *)0;
#endif

	if (Z_OK != inflateInit(&strm)) {
		printk(KERN_WARNING "inflateInit failed\n");
		return;
	}
	strm.next_in = data_in;
	strm.avail_in = srclen;
	strm.total_in = 0;
	
	strm.next_out = cpage_out;
	strm.avail_out = destlen;
	strm.total_out = 0;

	while((ret = inflate(&strm, Z_FINISH)) == Z_OK)
		;
	if (ret != Z_STREAM_END) {
		printk(KERN_NOTICE "inflate returned %d\n", ret);
	}
	inflateEnd(&strm);
}
