/*
 * JFFS2 -- Journalling Flash File System, Version 2.
 *
 * Copyright (C) 2001 Red Hat, Inc.
 *
 * Created by David Woodhouse <dwmw2@infradead.org>
 *
 * The original JFFS, from which the design for JFFS2 was derived,
 * was designed and implemented by Axis Communications AB.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * $Id: erase.c,v 1.15 2001/03/02 16:45:36 dwmw2 Exp $
 *
 */
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/mtd/mtd.h>
#include <linux/jffs2.h>
#include "nodelist.h"

struct erase_priv_struct {
	struct jffs2_eraseblock *jeb;
	struct jffs2_sb_info *c;
};
      
static void jffs2_erase_callback(struct erase_info *);
static void jffs2_free_all_node_refs(struct jffs2_sb_info *c, struct jffs2_eraseblock *jeb);

void jffs2_erase_block(struct jffs2_sb_info *c, struct jffs2_eraseblock *jeb)
{
	struct erase_info *instr;
	int ret;

	instr = kmalloc(sizeof(struct erase_info) + sizeof(struct erase_priv_struct), GFP_KERNEL);
	if (!instr) {
		printk(KERN_WARNING "kmalloc for struct erase_info in jffs2_erase_block failed. Refiling block for later\n");
		spin_lock_bh(&c->erase_completion_lock);
		list_del(&jeb->list);
		list_add(&jeb->list, &c->erase_pending_list);
		c->erasing_size -= c->sector_size;
		spin_unlock_bh(&c->erase_completion_lock);
		return;
	}

	memset(instr, 0, sizeof(*instr));

	instr->mtd = c->mtd;
	instr->addr = jeb->offset;
	instr->len = c->sector_size;
	instr->callback = jffs2_erase_callback;
	instr->priv = (unsigned long)(&instr[1]);
	
	((struct erase_priv_struct *)instr->priv)->jeb = jeb;
	((struct erase_priv_struct *)instr->priv)->c = c;

	ret = c->mtd->erase(c->mtd, instr);
	if (!ret) {
		return;
	}
	if (ret == -ENOMEM || ret == -EAGAIN) {
		/* Erase failed immediately. Refile it on the list */
		D1(printk(KERN_DEBUG "Erase at 0x%08x failed: %d. Refiling on erase_pending_list\n", jeb->offset, ret));
		spin_lock_bh(&c->erase_completion_lock);
		list_del(&jeb->list);
		list_add(&jeb->list, &c->erase_pending_list);
		c->erasing_size -= c->sector_size;
		spin_unlock_bh(&c->erase_completion_lock);
		kfree(instr);
		return;
	}

	printk(KERN_WARNING "Erase at 0x%08x failed immediately: %d\n", jeb->offset, ret);
	spin_lock_bh(&c->erase_completion_lock);
	list_del(&jeb->list);
	list_add(&jeb->list, &c->bad_list);
	c->nr_erasing_blocks--;
	c->bad_size += c->sector_size;
	c->erasing_size -= c->sector_size;
	spin_unlock_bh(&c->erase_completion_lock);
	wake_up(&c->erase_wait);
	kfree(instr);
}

void jffs2_erase_pending_blocks(struct jffs2_sb_info *c)
{
	struct jffs2_eraseblock *jeb;

	spin_lock_bh(&c->erase_completion_lock);
	while (!list_empty(&c->erase_pending_list)) {

		jeb = list_entry(c->erase_pending_list.next, struct jffs2_eraseblock, list);

		D1(printk(KERN_DEBUG "Starting erase of pending block 0x%08x\n", jeb->offset));

		list_del(&jeb->list);
		c->erasing_size += c->sector_size;
		c->free_size -= jeb->free_size;
		c->used_size -= jeb->used_size;
		c->dirty_size -= jeb->dirty_size;
		jeb->used_size = jeb->dirty_size = jeb->free_size = 0;
		jffs2_free_all_node_refs(c, jeb);
		list_add(&jeb->list, &c->erasing_list);
		spin_unlock_bh(&c->erase_completion_lock);
		
		jffs2_erase_block(c, jeb);
		/* Be nice */
		if (current->need_resched)
			schedule();
		spin_lock_bh(&c->erase_completion_lock);
	}
	spin_unlock_bh(&c->erase_completion_lock);
	D1(printk(KERN_DEBUG "jffs2_erase_pending_blocks completed\n"));
}


static void jffs2_erase_callback(struct erase_info *instr)
{
	struct erase_priv_struct *priv = (void *)instr->priv;

	if(instr->state != MTD_ERASE_DONE) {
		printk(KERN_WARNING "Erase at 0x%08lx finished, but state != MTD_ERASE_DONE. State is 0x%x instead.\n", instr->addr, instr->state);
		spin_lock(&priv->c->erase_completion_lock);
		priv->c->erasing_size -= priv->c->sector_size;
		priv->c->bad_size += priv->c->sector_size;
		list_del(&priv->jeb->list);
		list_add(&priv->jeb->list, &priv->c->bad_list);
		priv->c->nr_erasing_blocks--;
		spin_unlock(&priv->c->erase_completion_lock);
	} else {
		D1(printk(KERN_DEBUG "Erase completed successfully at 0x%08lx\n", instr->addr));
		spin_lock(&priv->c->erase_completion_lock);
		list_del(&priv->jeb->list);
		priv->c->erasing_size -= priv->c->sector_size;
		priv->c->free_size += priv->c->sector_size;
		priv->jeb->free_size = priv->c->sector_size;
		priv->jeb->used_size = priv->jeb->dirty_size = 0;

		list_add_tail(&priv->jeb->list, &priv->c->free_list);
		priv->c->nr_erasing_blocks--;
		priv->c->nr_free_blocks++;
		spin_unlock(&priv->c->erase_completion_lock);
	}	
	wake_up(&priv->c->erase_wait);
	kfree(instr);
}

/* Hmmm. Maybe we should accept the extra space it takes and make
   this a standard doubly-linked list? */
static inline void jffs2_remove_node_ref_from_ino_list(struct jffs2_sb_info *sbinfo, struct jffs2_raw_node_ref *ref)
{
	struct jffs2_inode_cache *ic;
	struct jffs2_raw_node_ref **prev, *this;
	D2(int c=0);

	this = ref;
	while(this->next_in_ino)
		 this = this->next_in_ino;

	ic = (struct jffs2_inode_cache *)this;

	D1(printk(KERN_DEBUG "Removing node at phys 0x%08x from ino #%u\n", ref->flash_offset &~3, ic->ino));

	prev = &ic->nodes;
	if (!*prev) {
		printk(KERN_WARNING "Eep. ic->nodes == NULL.\n");
		return;
	}
	while (*prev != ref) {
		if (!(*prev)->next_in_ino) {
		        printk(KERN_WARNING "Eep. node at phys 0x%08x, mem %p. next_in_ino is NULL.\n", (*prev)->flash_offset &~3, 
			       *prev);
			return;
		}
		prev = &(*prev)->next_in_ino;
	}
	*prev = ref->next_in_ino;
	this = ic->nodes;
	D2(printk(KERN_DEBUG "After remove_node_ref_from_ino_list: \n" KERN_DEBUG);
	while(this) {
		printk( "0x%08x(%d)->", this->flash_offset & ~3, this->flash_offset &3);
		if (++c == 5) {
			printk("\n" KERN_DEBUG);
			c=0;
		}
		this = this->next_in_ino;
	}
	printk("\n"););
	if (ic->nodes == (void *)ic) {
		D1(printk(KERN_DEBUG "inocache for ino #%u is all gone now. Freeing\n", ic->ino));
		jffs2_del_ino_cache(sbinfo, ic);
		jffs2_free_inode_cache(ic);
	}
}

static void jffs2_free_all_node_refs(struct jffs2_sb_info *c, struct jffs2_eraseblock *jeb)
{
	struct jffs2_raw_node_ref *ref;
	D1(printk(KERN_DEBUG "Freeing all node refs for eraseblock offset 0x%08x\n", jeb->offset));
	while(jeb->first_node) {
		ref = jeb->first_node;
		jeb->first_node = ref->next_phys;
		
		/* Remove from the inode-list */
		jffs2_remove_node_ref_from_ino_list(c, ref);
		jffs2_free_raw_node_ref(ref);
	}
	jeb->last_node = NULL;
}

void jffs2_erase_pending_trigger(struct jffs2_sb_info *c)
{
	OFNI_BS_2SFFJ(c)->s_dirt = 1;
}
