/*
 * JFFS2 -- Journalling Flash File System, Version 2.
 *
 * Copyright (C) 2001 Red Hat, Inc.
 *
 * Created by David Woodhouse <dwmw2@infradead.org>
 *
 * The original JFFS, from which the design for JFFS2 was derived,
 * was designed and implemented by Axis Communications AB.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * $Id: symlink.c,v 1.3 2001/03/01 22:34:28 dwmw2 Exp $
 *
 */


#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/jffs2.h>
#include "nodelist.h"

int jffs2_readlink(struct dentry *dentry, char *buffer, int buflen);
int jffs2_follow_link(struct dentry *dentry, struct nameidata *nd);

struct inode_operations jffs2_symlink_inode_operations =
{	
	readlink:	jffs2_readlink,
	follow_link:	jffs2_follow_link
};

static char *jffs2_getlink(struct dentry *dentry)
{
	struct jffs2_inode_info *f = JFFS2_INODE_INFO(dentry->d_inode);
	char *buf;
	int ret;

	if (!f->metadata) {
		printk(KERN_NOTICE "No metadata for symlink inode #%lu\n", dentry->d_inode->i_ino);
		return ERR_PTR(-EINVAL);
	}
	buf = kmalloc(f->metadata->size+1, GFP_USER);
	if (!buf)
		return ERR_PTR(-ENOMEM);
	buf[f->metadata->size]=0;

	ret = jffs2_read_dnode(JFFS2_SB_INFO(dentry->d_inode->i_sb), f->metadata, buf, 0, f->metadata->size);
	if (ret) {
		kfree(buf);
		return ERR_PTR(ret);
	}
	return buf;

}
int jffs2_readlink(struct dentry *dentry, char *buffer, int buflen)
{
	unsigned char *kbuf;
	int ret;

	kbuf = jffs2_getlink(dentry);
	if (IS_ERR(kbuf))
		return PTR_ERR(kbuf);

	ret = vfs_readlink(dentry, buffer, buflen, kbuf);
	kfree(kbuf);
	return ret;
}

int jffs2_follow_link(struct dentry *dentry, struct nameidata *nd)
{
	unsigned char *buf;
	int ret;

	buf = jffs2_getlink(dentry);

	if (IS_ERR(buf))
		return PTR_ERR(buf);

	ret = vfs_follow_link(nd, buf);
	kfree(buf);
	return ret;
}
