/*
 * JFFS2 -- Journalling Flash File System, Linux implementation.
 *
 * Copyright (C) 2001 Red Hat, Inc.
 *
 * Created by David Woodhouse <dwmw2@infradead.org>
 *
 * The original JFFS, from which the design for JFFS2 was derived,
 * was designed and implemented by Axis Communications AB.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * $Id: jffs2.h,v 1.14 2001/02/28 10:06:41 dwmw2 Exp $
 *
 */

#ifndef __LINUX_JFFS2_H__
#define __LINUX_JFFS2_H__

#include <asm/types.h>
#define JFFS2_SUPER_MAGIC 0x72b6

/* Values we may expect to find in the 'magic' field */
#define JFFS2_MAGIC_BITMASK 0x1984
#define KSAMTIB_CIGAM_2SFFJ 0x4981 /* For detecting wrong-endian fs */
#define JFFS2_EMPTY_BITMASK 0xffff
#define JFFS2_DIRTY_BITMASK 0x0000

/* We only allow a single char for length, and 0xFF is empty flash so
   we don't want it confused with a real length. Hence max 254.
*/
#define JFFS2_MAX_NAME_LEN 254

/* How small can we sensibly write nodes? */
#define JFFS2_MIN_DATA_LEN 128

#define JFFS2_COMPR_NONE	0x00
#define JFFS2_COMPR_ZERO	0x01
#define JFFS2_COMPR_RTIME	0x02
#define JFFS2_COMPR_RUBINMIPS	0x03
#define JFFS2_COMPR_COPY	0x04
#define JFFS2_COMPR_DYNRUBIN	0x05
#define JFFS2_COMPR_ZLIB	0x06
/* Compatibility flags. */
#define JFFS2_COMPAT_MASK 0xc000      /* What do to if an unknown nodetype is found */
#define JFFS2_NODE_ACCURATE 0x2000
/* INCOMPAT: Fail to mount the filesystem */
#define JFFS2_FEATURE_INCOMPAT 0xc000 | JFFS2_NODE_ACCURATE
/* ROCOMPAT: Mount read-only */
#define JFFS2_FEATURE_ROCOMPAT 0x8000 | JFFS2_NODE_ACCURATE
/* RWCOMPAT_COPY: Mount read/write, and copy the node when it's GC'd */
#define JFFS2_FEATURE_RWCOMPAT_COPY 0x4000 | JFFS2_NODE_ACCURATE
/* RWCOMPAT_DELETE: Mount read/write, and delete the node when it's GC'd */
#define JFFS2_FEATURE_RWCOMPAT_DELETE 0x0000 | JFFS2_NODE_ACCURATE

#define JFFS2_NODETYPE_DIRENT (JFFS2_FEATURE_INCOMPAT | 1)
#define JFFS2_NODETYPE_INODE (JFFS2_FEATURE_INCOMPAT | 2)
// Maybe later...
//#define JFFS2_NODETYPE_CHECKPOINT (JFFS2_FEATURE_RWCOMPAT_DELETE | 3)
//#define JFFS2_NODETYPE_OPTIONS (JFFS2_FEATURE_RWCOMPAT_COPY | 4)

/* Same as the non_ECC versions, but with extra space for real 
 * ECC instead of just the checksum. For use on NAND flash 
 */
//#define JFFS2_NODETYPE_DIRENT_ECC (JFFS2_FEATURE_INCOMPAT | 5)
//#define JFFS2_NODETYPE_INODE_ECC (JFFS2_FEATURE_INCOMPAT | 6)

struct jffs2_unknown_node
{
	/* All start like this */
	__u16 magic;
	__u16 nodetype;
	__u32 totlen; /* So we can skip over nodes we don't grok */
};

struct jffs2_raw_dirent
{
	__u16 magic;
	__u16 nodetype;	/* == JFFS_NODETYPE_DIRENT */
	__u32 totlen;
	__u32 pino;
	__u32 version;
	__u32 ino; /* == zero for unlink */
	__u32 mctime;
	__u8 nsize;
	__u8 type;
	__u8 unused[2];
	__u32 node_crc;
	__u32 name_crc;
	__u8 name[0];
};

/* The JFFS2 raw inode structure: Used for storage on physical media.  */
/* The uid, gid, atime, mtime and ctime members could be longer, but 
   are left like this for space efficiency. If and when people decide
   they really need them extended, it's simple enough to add support for
   a new type of raw node.
*/
struct jffs2_raw_inode
{
	__u16 magic;      /* A constant magic number.  */
	__u16 nodetype;   /* == JFFS_NODETYPE_INODE */
	__u32 totlen;     /* Total length of this node (inc data, etc.) */
	__u32 ino;        /* Inode number.  */
	__u32 version;    /* Version number.  */
	__u32 mode;       /* The file's type or mode.  */
	__u16 uid;        /* The file's owner.  */
	__u16 gid;        /* The file's group.  */
	__u32 isize;      /* Total resultant size of this inode (used for truncations)  */
	__u32 atime;      /* Last access time.  */
	__u32 mtime;      /* Last modification time.  */
	__u32 ctime;      /* Change time.  */
	__u32 offset;     /* Where to begin to write.  */
	__u32 csize;      /* (Compressed) data size */
	__u32 dsize;	  /* Size of the node's data. (after decompression) */
	__u8 compr;       /* Compression algorithm used */
	__u8 unused[3];
	__u32 data_crc;    /* CRC for the (compressed) data.  */
	__u32 node_crc;    /* CRC for the raw inode (excluding data)  */
//	__u8 data[dsize];
};

union jffs2_node_union {
	struct jffs2_raw_inode i;
	struct jffs2_raw_dirent d;
	struct jffs2_unknown_node u;
};

#endif /* __LINUX_JFFS2_H__ */
