/*
 *	Destination option handling code
 *
 *	Authors:
 *	Sami Kivisaari		<skivisaa@cc.hut.fi>	
 *
 *	$Id: dstopts.c,v 1.7 2000/10/17 08:39:33 antti Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *	TODO: creation functions could check if the option fits into
 *	      the header 
 */


#define __NO_VERSION__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/types.h>
#include <linux/ipv6.h>
#include <linux/slab.h>
#include <linux/in6.h>

#include <net/ipv6.h>

#include "mipv6.h"
#include "dstopts.h"
#include "debug.h"

/*
 *
 *   Returns how many bytes of padding must be appended before (sub)option
 *   (alignment requirements can be found from the mobile-IPv6 draft, sect5)
 *
 */
static __inline__ int mipv6_calculate_option_pad(__u8 type, int offset)
{
	switch(type) {

	case MIPV6_TLV_BINDUPDATE:		/* 4n + 2 */
		return (2 - (offset)) & 3;
		
	case MIPV6_TLV_BINDACK:  		/* 4n + 3 */
		return (3 - (offset)) & 3;
		
	case MIPV6_SUBOPT_UNIQUEID:		/* 2n     */
		return (offset) & 1;
		
	case MIPV6_TLV_HOMEADDR:		/* 8n + 6 */
	case MIPV6_SUBOPT_ALTERNATE_COA:	
		return (6 - (offset)) & 7;
	case MIPV6_TLV_BINDRQ:
	case MIPV6_SUBOPT_PAD1:			/* no pad */
	case MIPV6_SUBOPT_PADN:
		return 0;
		
	default:
		DEBUG((DBG_ERROR, "invalid option type 0x%x", type));
		return 0;
	}
}


/*
 * Add padding before a tlv-encoded option. Can also be used to
 * padding suboptions (pad1 and padN have the same code).
 */
static __inline__ void mipv6_tlv_pad(__u8 *padbuf, int pad)
{
	int i;

	if(pad<=0) return;

	if(pad==1) {
		padbuf[0] = MIPV6_SUBOPT_PAD1;
	} else {
		padbuf[0] = MIPV6_SUBOPT_PADN;
		padbuf[1] = pad-2;
		for(i=2; i<pad; i++) padbuf[i] = 0;
	}	
}


static int mipv6_parse_subopts(
	__u8 *opt, int offset,
	struct mipv6_subopt_info *sinfo)
{
	int optlen = opt[1] + 2;
	struct mipv6_subopt_unique_id *ui;
	struct mipv6_subopt_alternate_coa *ac;

	DEBUG_FUNC();

	sinfo->f.flags = 0;

	while(offset < optlen) {
		switch(opt[offset]) {

		case MIPV6_SUBOPT_PAD1:
			offset++;
			break;

		case MIPV6_SUBOPT_PADN:
			offset += opt[offset + 1] + 2;
			break;

		case MIPV6_SUBOPT_UNIQUEID:		
			ui = (struct mipv6_subopt_unique_id *)(opt+offset);			

			if(ui->length !=
			   sizeof(struct mipv6_subopt_unique_id) - 2)
				goto fail;
			
			if(sinfo->fso_uid)
				DEBUG((DBG_WARNING, "UID-suboption already exists"));
			
			sinfo->fso_uid = 1;
			sinfo->uid = ntohs(ui->unique_id);
			offset += ui->length + 2;
			break;

		case MIPV6_SUBOPT_ALTERNATE_COA:	
			ac = (struct mipv6_subopt_alternate_coa *)&opt[offset];

			if(ac->length !=
			   sizeof(struct mipv6_subopt_alternate_coa)-2)
				goto fail;
			
			if(sinfo->fso_alt_coa)
				DEBUG((DBG_WARNING, "ACOA-suboption already exists"));
			
			sinfo->fso_alt_coa = 1;
			ipv6_addr_copy(&sinfo->alt_coa, &ac->addr);
			offset += ac->length + 2;
			break;

		default:
			/* unrecognized suboption */
			DEBUG((DBG_WARNING, "unrecognized suboption identifier"));
			
			goto fail;
		}
	}
	
	/* check if succeeded */
	if(offset == optlen) return 0;
	
fail:
	DEBUG((DBG_WARNING, "malformed suboption field"));

	/* failed! */
	return 1;
}


/*
 *  Create a binding update destination option
 * 
 *   0                   1                   2                   3
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *                                  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                  |  Option Type  | Option Length |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |A|H|R|D|Reservd| Prefix Length |        Sequence Number        |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                            Lifetime                           |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |   Sub-Options...
 *  +-+-+-+-+-+-+-+-+-+-+-+-
 */
int mipv6_create_bindupdate(
	__u8 *opt, int offset, __u8 **ch, __u8 flags,
	__u8 plength, __u16 sequence, __u32 lifetime,
	struct mipv6_subopt_info *sinfo)
{
	int pad, new_offset;
	struct mipv6_dstopt_bindupdate *bu;

	DEBUG_FUNC();

	/* do padding */
	pad = mipv6_calculate_option_pad(MIPV6_TLV_BINDUPDATE, offset);
	mipv6_tlv_pad(opt + offset, pad);

	bu = (struct mipv6_dstopt_bindupdate *)(opt + offset + pad);

	/* fill in the binding update */
	bu->type = MIPV6_TLV_BINDUPDATE;
	bu->length = sizeof(struct mipv6_dstopt_bindupdate) - 2;
	bu->flags = flags;
	bu->prefixlen = plength;

	bu->seq = htons(sequence);
  	bu->lifetime = htonl(lifetime);

	new_offset = offset + pad + sizeof(struct mipv6_dstopt_bindupdate);

	if (sinfo && sinfo->fso_flags != 0) {
		int so_off;
		__u8 so_len = 0;
		if (sinfo->fso_uid) {
			so_off = mipv6_add_unique_id(opt, new_offset, 
						     &so_len, sinfo->uid);
			if (so_off > 0) {
				new_offset = so_off;
				bu->length += so_len;
			} else {
				DEBUG((DBG_WARNING,
				       "Could not add Unique ID sub-option"));
			}
		}
		if (sinfo->fso_alt_coa) {
			so_off = mipv6_add_alternate_coa(opt, new_offset, 
							 &so_len, &sinfo->alt_coa);
			if (so_off > 0) {
				new_offset = so_off;
				bu->length += so_len;
			} else {
				DEBUG((DBG_WARNING,
				       "Could not add Alternate CoA sub-option"));
			}
		}
	}

	/* save reference to length field */
	*ch = &bu->length;

	return new_offset;
}

int mipv6_parse_bindupdate(
	__u8 *opt, __u8 *flags,
	__u8 *plength, __u16 *sequence, __u32 *lifetime,
	struct mipv6_subopt_info *sinfo)
{
	int optsize = sizeof(struct mipv6_dstopt_bindupdate);
	struct mipv6_dstopt_bindupdate *bu =
		(struct mipv6_dstopt_bindupdate *)opt;

	DEBUG_FUNC();

	if(bu->length + 2 < optsize) return -1;

	*flags = bu->flags;
	*plength = bu->prefixlen;
	*sequence = ntohs(bu->seq);
	*lifetime = ntohl(bu->lifetime);

	return mipv6_parse_subopts(opt, optsize, sinfo);
}



/*
 *  0                   1                   2                   3
 *  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *                                                  +-+-+-+-+-+-+-+-+
 *                                                  |  Option Type  |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  | Option Length |    Status     |        Sequence Number        |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                            Lifetime                           |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                            Refresh                            |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *
 */
int mipv6_create_bindack(
	__u8 *opt, int offset, __u8 **ch,
	__u8 status, __u16 sequence, __u32 lifetime, __u32 refresh)
{
	int pad;
	struct mipv6_dstopt_bindack *ba;

	DEBUG_FUNC();

	/* do padding */
	pad = mipv6_calculate_option_pad(MIPV6_TLV_BINDACK, offset);
	mipv6_tlv_pad(opt + offset, pad);

	ba = (struct mipv6_dstopt_bindack *)(opt + offset + pad);

	/* fill in the binding acknowledgement */
	ba->type = MIPV6_TLV_BINDACK;
	ba->length = sizeof(struct mipv6_dstopt_bindack) - 2;
	ba->status = status;
	ba->seq = htons(sequence);
  	ba->lifetime = htonl(lifetime);
	ba->refresh = htonl(refresh);

	/* save reference to length field */
	*ch = &ba->length;

	return offset + pad + sizeof(struct mipv6_dstopt_bindack);
}

int mipv6_parse_bindack(
	__u8 *opt,
	__u8 *status, __u16 *sequence, __u32 *lifetime, __u32 *refresh,
	struct mipv6_subopt_info *sinfo)
{
	int optsize = sizeof(struct mipv6_dstopt_bindack);
	struct mipv6_dstopt_bindack *ba = (struct mipv6_dstopt_bindack *)opt;

	DEBUG_FUNC();

	if(ba->length + 2 < optsize) return -1;

	*status = ba->status;
	*sequence = ntohs(ba->seq);
	*lifetime = ntohl(ba->lifetime);
	*refresh = ntohl(ba->refresh);

	return mipv6_parse_subopts(opt, optsize, sinfo);
}


/*
 *   0                   1
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
 *  |  Option Type  | Option Length |   Sub-Options...
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-
 *
 */

int mipv6_create_bindrq(
	__u8 *opt, int offset, __u8 **ch)
{
	int pad, new_offset;
	struct mipv6_dstopt_bindrq *ho;
	struct mipv6_subopt_info *sinfo = NULL;

	DEBUG_FUNC();

	/* do padding */
	pad = mipv6_calculate_option_pad(MIPV6_TLV_BINDRQ, offset);
	mipv6_tlv_pad(opt + offset, pad);

	ho = (struct mipv6_dstopt_bindrq *)(opt + offset + pad);

	/* fill in the binding request */
	ho->type = MIPV6_TLV_BINDRQ;
	ho->length = sizeof(struct mipv6_dstopt_bindrq) - 2;

	new_offset = offset + pad + sizeof(struct mipv6_dstopt_bindrq);

	if (sinfo && sinfo->fso_flags != 0) {
		int so_off;
		__u8 so_len = 0;
		if (sinfo->fso_uid) {
			so_off = mipv6_add_unique_id(opt, new_offset, 
						     &so_len, sinfo->uid);
			if (so_off > 0) {
				new_offset = so_off;
				ho->length += so_len;
			} else {
				DEBUG((DBG_WARNING,
				       "Could not add Unique ID sub-option"));
			}
		}
	}

	/* save reference to length field */
	*ch = &ho->length;

	return new_offset;
}

int mipv6_parse_bindrq(
	__u8 *opt,
	struct mipv6_subopt_info *sinfo)
{
	int optsize = sizeof(struct mipv6_dstopt_bindrq);
	struct mipv6_dstopt_bindrq *br = (struct mipv6_dstopt_bindrq *)opt;

	DEBUG_FUNC();

	if(br->length + 2 < optsize) return -1;

	return mipv6_parse_subopts(opt, optsize, sinfo);
}


/*
 *   0                   1                   2                   3
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *                                  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                  |  Option Type  | Option Length |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                                                               |
 *  +                                                               +
 *  |                                                               |
 *  +                          Home Address                         +
 *  |                                                               |
 *  +                                                               +
 *  |                                                               |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |   Sub-Options...
 *  +-+-+-+-+-+-+-+-+-+-+-+-
 */

/* The option is created but the original source address in ipv6 header is left
 * intact. The source address will be changed from home address to CoA  
 * after the checksum has been calculated in getfrag
 */
int mipv6_create_home_addr(
	__u8 *opt, int offset, __u8 **ch,  struct in6_addr *addr)
{
	int pad;
	struct mipv6_dstopt_homeaddr *ho;

	DEBUG_FUNC();

	/* do padding */
	pad = mipv6_calculate_option_pad(MIPV6_TLV_HOMEADDR, offset);
	mipv6_tlv_pad(opt + offset, pad);

	ho = (struct mipv6_dstopt_homeaddr *)(opt + offset + pad);

	/* fill in the binding home address option */
	ho->type = MIPV6_TLV_HOMEADDR;
	ho->length = sizeof(struct mipv6_dstopt_homeaddr) - 2;
	
	ipv6_addr_copy(&ho->addr, addr); 

	/* save reference to length field */
	*ch = &ho->length;

	return offset + pad + sizeof(struct mipv6_dstopt_homeaddr);
}

int mipv6_parse_homeaddr(
	__u8 *opt,
	struct in6_addr *addr,
	struct mipv6_subopt_info *sinfo)
{
	int optsize = sizeof(struct mipv6_dstopt_homeaddr);
	struct mipv6_dstopt_homeaddr *hd = (struct mipv6_dstopt_homeaddr *)opt;

	DEBUG_FUNC();

	if(hd->length + 2 < optsize) return -1;

	ipv6_addr_copy(addr, &hd->addr);

	return mipv6_parse_subopts(opt, optsize, sinfo);
}


/*
 *  Unique Identifier Sub-Option   (alignment requirement: 2n)
 *
 *   0                   1                   2                   3
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |       2       |       2       |       Unique Identifier       |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */

int mipv6_add_unique_id(__u8 *option, int offset, __u8 *ol,  __u16 uid)
{
	int pad, suboptlen;
	struct mipv6_subopt_unique_id *ui;

	DEBUG_FUNC();

	pad = mipv6_calculate_option_pad(MIPV6_SUBOPT_UNIQUEID, offset); 
	suboptlen = sizeof(struct mipv6_subopt_unique_id);

	if(*ol + suboptlen + pad < 0x100) {
		mipv6_tlv_pad(option + offset, pad);

		ui = (struct mipv6_subopt_unique_id *)(option + offset + pad);

		ui->type = MIPV6_SUBOPT_UNIQUEID;
		ui->length = suboptlen - 2;
		ui->unique_id = htons(uid);

		*ol += pad + suboptlen;

		return offset + pad + suboptlen;
	} else {
		/* error, suboption does not fit into option */
		return -1;
	}
}


/*
 * Alternate Care-of Address Sub-Option   (alignment requirement: 8n+6) 
 *
 *   0                   1                   2                   3
 *   0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
 *                                  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *                                  |       4       |       16      |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 *  |                                                               |
 *  |                  Alternate Care-of Addresses                  |
 *  |                                                               |
 *  |                                                               |
 *  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */

int mipv6_add_alternate_coa(__u8 *option, int offset, __u8 *ol, struct in6_addr *addr)
{
	int pad, suboptlen;
	struct mipv6_subopt_alternate_coa *ac;

	DEBUG_FUNC();

	pad = mipv6_calculate_option_pad(MIPV6_SUBOPT_ALTERNATE_COA, offset); 
	suboptlen = sizeof(struct in6_addr) + 2;
	
	if(*ol + suboptlen + pad < 0x100) {
		mipv6_tlv_pad(option + offset, pad);

		ac = (struct mipv6_subopt_alternate_coa *)(option + offset + pad);

		ac->type = MIPV6_SUBOPT_ALTERNATE_COA;
		ac->length = suboptlen - 2;
		ipv6_addr_copy(&ac->addr, addr);

		*ol += pad + suboptlen;

		return offset + pad + suboptlen;
	} else {
		/* error, suboption does not fit into option */
		return -1;
	}
}

/* 
 * Function to finalize a destination option header that has the
 * necessary options included. Checks that header is not too long
 * and header length is multiple of eight.
 * 
 * arguments: hdr    = pointer to dstopt-header
 *            offset = index of next to last byte of header
 */
int mipv6_finalize_dstopt_header(__u8 *hdr, int offset)
{
	struct ipv6_opt_hdr *dhdr = (struct ipv6_opt_hdr *)hdr;

	DEBUG_FUNC();

        if((offset < 2) || (offset > 2048)) {
                DEBUG((DBG_ERROR,
                       "invalid destination option header length (%d)", offset));

                return offset;
        }

	if(offset & 0x7) {
		/*  total dstopt len is not 8n, pad the rest with zero  */
		mipv6_tlv_pad(&hdr[offset], (-offset)&0x7);

		offset = offset + ((-offset)&0x7);
	}

	dhdr->hdrlen = (offset - 1) >> 3;

	return offset;
}

/*
 * Appends a route segment to type 0 routing header, if the
 * function is supplied with NULL routing header, a new one
 * is created.
 *
 * NOTE!: The caller is responsible of arranging the freeing of old rthdr
 */
struct ipv6_rt_hdr * mipv6_append_rt_header(
	struct ipv6_rt_hdr *rthdr, struct in6_addr *addr)
{
	struct rt0_hdr *nhdr, *hdr = (struct rt0_hdr *)rthdr; 

        DEBUG_FUNC(); 

	if (ipv6_addr_type(addr) == IPV6_ADDR_MULTICAST)
		return rthdr;

	if(hdr == NULL) {
		nhdr = (struct rt0_hdr *)
			kmalloc(sizeof(struct rt0_hdr) +
				sizeof(struct in6_addr), GFP_ATOMIC);
		if(!nhdr) return NULL;

		nhdr->rt_hdr.hdrlen = 2;	/* route segments = 1    */
		nhdr->rt_hdr.type = 0;		/* type 0 routing header */ 
		nhdr->rt_hdr.segments_left = 1;	/* route segments = 1    */
		nhdr->bitmap = 0;
	} else {
		if(hdr->rt_hdr.type != 0) {
			DEBUG((DBG_ERROR, "not type 0 routing header"));
			return rthdr;
		}

		if(hdr->rt_hdr.hdrlen != (hdr->rt_hdr.segments_left << 1)) {
			DEBUG((DBG_ERROR,
			       "hdrlen and segments_left fields do not match!"));

			return rthdr;
		}

		nhdr = (struct rt0_hdr *)
			kmalloc(ipv6_optlen(&hdr->rt_hdr) +
				sizeof(struct in6_addr), GFP_ATOMIC);

		if(!nhdr) return rthdr;

		/*  addresses from old routing header  */
		memcpy(&nhdr->addr[0], &hdr->addr[0],
		       sizeof(struct in6_addr) * hdr->rt_hdr.segments_left);

		nhdr->rt_hdr.hdrlen = hdr->rt_hdr.hdrlen + 2;
		nhdr->rt_hdr.segments_left = hdr->rt_hdr.segments_left + 1;
		nhdr->rt_hdr.type = 0;
	        nhdr->bitmap = 0;
	}

	/*  copy the last route segment into header  */
	ipv6_addr_copy(&nhdr->addr[(nhdr->rt_hdr.hdrlen>>1) - 1], addr);

	return (struct ipv6_rt_hdr *)nhdr;
}


/*
 * Prints suboption info block in cleartext, for debugging purposes
 */
void mipv6_print_subopt_info(struct mipv6_subopt_info *sinfo)
{
	if(sinfo->f.flags == 0) return;

	if(sinfo->fso_uid)
		DEBUG((DBG_INFO,
		       "UID suboption: (UID=0x%04x)", sinfo->uid));
	if(sinfo->fso_alt_coa) {
		DEBUG((DBG_INFO,
		       "ACOA suboption: ACOA="));
		debug_print_addr(DBG_INFO, &sinfo->alt_coa);
	}

}




