/*
 *      Movement Detection Module
 *
 *      Authors:
 *      Henrik Petander                <lpetande@cc.hut.fi>
 *
 *      $Id: mdetect.c,v 1.26 2000/10/19 14:05:00 henkku Exp $
 *
 *      Changes:
 * 
 *      Nanno Langstrat : Fixed locking
 *
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *      mdetect.c:
 *       
 *      Handles the L3 movement detection of mobile node and also changing of 
 *      its routes.  
 *  
 *      TODO: Should locking use write_lock_irqsave and restore instead of 
 *            spin_lock_irqsave?
 *      In contrast with MIPV6 draft v12:
 *             - Reception of tunneled router advertisements is not 
 *              implemented,  the functionality should be added to either 
 *              mn or mdetect
 *             
 */



#define __NO_VERSION__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/autoconf.h>

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in6.h>
#include <linux/route.h>
#include <linux/init.h>

#include <linux/if_arp.h>
#include <linux/ipv6.h>
#include <linux/icmpv6.h>
#include <linux/init.h>

#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif

#include <net/sock.h>
#include <net/snmp.h>

#include <net/ipv6.h>
#include <net/protocol.h>
#include <net/ndisc.h>
#include <net/ip6_route.h>
#include <net/addrconf.h>
#include <net/icmp.h>
#include <net/neighbour.h>
#include <net/dst.h>

#include "mipv6.h"
#include "mdetect.h"
#include "mn.h"
#include "debug.h"

#ifdef CONFIG_SYSCTL
#include "sysctl.h"
#endif /* CONFIG_SYSCTL */

#define START 0
#define CONTINUE 1

extern void ndisc_send_na(struct net_device *dev, struct neighbour *neigh,
			  struct in6_addr *daddr, struct in6_addr *solicited_addr,
			  int router, int solicited, int override, int inc_opt);
extern int icmpv6_rcv(struct sk_buff *skb, unsigned long len);
/* dad could also be RESPECT_DAD for duplicate address detection of new care-of addresses */
static int isathome = 1, dad=0,mdetect_is_initialized=0;
spinlock_t router_lock=SPIN_LOCK_UNLOCKED;
int rs_interval=HZ/2, mdet_mech=EAGER_CELL_SWITCHING; /* Only one choice */

#ifdef CONFIG_SYSCTL

static char mipv6_mn_home_address[36];

static void mipv6_mdetect_sysctl_register(void);
static void mipv6_mdetect_sysctl_unregister(void);

static int mipv6_home_address_sysctl_handler(ctl_table *ctl,
					     int write,
					     struct file * filp,
					     void *buffer,
					     size_t *lenp);
static int mipv6_mdetect_mech_sysctl_handler(ctl_table *ctl,
					     int write,
					     struct file * filp,
					     void *buffer,
					     size_t *lenp);

static struct ctl_table_header *mipv6_mdetect_sysctl_header;


static struct mipv6_mn_sysctl_table
{
	struct ctl_table_header *sysctl_header;
	ctl_table mipv6_vars[3];
	ctl_table mipv6_mobility_table[2];
	ctl_table mipv6_proto_table[2];
	ctl_table mipv6_root_table[2];
} mipv6_mdetect_sysctl = {
	NULL,

        {{NET_IPV6_MOBILITY_HOME_ADDRESS, "home_address",
	  &mipv6_mn_home_address, 36 * sizeof(char),
	  0644, NULL, &mipv6_home_address_sysctl_handler},
	 {NET_IPV6_MOBILITY_MDETECT_MECHANISM, "mdetect_mechanism",
	  &mdet_mech, sizeof(int), 0644, NULL,
	  &mipv6_mdetect_mech_sysctl_handler},

	 {0}},

	{{NET_IPV6_MOBILITY, "mobility", NULL, 0, 0555, 
	  mipv6_mdetect_sysctl.mipv6_vars}, {0}},
	{{NET_IPV6, "ipv6", NULL, 0, 0555, 
	  mipv6_mdetect_sysctl.mipv6_mobility_table}, {0}},
	{{CTL_NET, "net", NULL, 0, 0555, 
	  mipv6_mdetect_sysctl.mipv6_proto_table},{0}}
};

#endif /* CONFIG_SYSCTL */

int mipv6_is_care_of_address(struct in6_addr *addr);
static void router_state(unsigned long foo);


#define LOOPB {{{0, 0, 0, 1}}}

static struct router *curr_router=NULL, *next_router=NULL;
static struct in6_addr home_address={{{0,0,0,0}}}, prev_router=LOOPB;

struct prefix_info home_address_pinfo;

static struct timer_list r_timer = { function: router_state };

/*  Sends router solicitations to all valid devices 
 *  source  = link local address (of sending interface)
 *  dstaddr = all routers multicast address
 *  Solicitations are sent at an exponentially decreasing rate
 * TODO: send solicitation first at a normal rate (from ipv6) and after that 
 * use the exponentially increasing intervals
 */
static void rs_send(int exp)
{
	struct net_device *dev;
	struct in6_addr raddr, lladdr;
	struct inet6_dev *in6_dev=NULL;
	static int next_rs, ival=HZ/2;

	if(jiffies/HZ<next_rs)
		return;
	ipv6_addr_all_routers(&raddr);
	DEBUG_FUNC();
	rtnl_lock(); 
	/*  Send router solicitations to all interfaces  */
	for (dev = dev_base; dev; dev = dev->next) {
		if ((dev->flags & IFF_UP) && dev->type == ARPHRD_ETHER) {
			DEBUG((DBG_INFO, "Sending RS to device %s", (int)dev->name));
			ipv6_get_lladdr(dev, &lladdr);
			ndisc_send_rs(dev, &lladdr, &raddr);
			in6_dev = in6_dev_get(dev);

			in6_dev->if_flags |= IF_RS_SENT;
			in6_dev_put(in6_dev);

		}

	}
	rtnl_unlock();
	if (exp==START)
		ival=HZ/2;
	else
		ival=ival*2;

	next_rs=jiffies+ival;		
}
/*
 * Extracts mobility related options from ras 
 */
static __u32 process_ra(struct sk_buff *skb, struct in6_addr *raddr, 
			int *plen, __u32 *ival, unsigned long *lifet)
{
	int optlen;
	int global_address;
	struct prefix_info *pinfo = NULL;
	struct ra_msg *ra = (struct ra_msg *) skb->h.raw;
	
	__u8 * opt = (__u8 *)(ra + 1);
	DEBUG_FUNC();
	*lifet = ntohs(ra->icmph.icmp6_rt_lifetime)*HZ;
	optlen = (skb->tail - skb->h.raw) - sizeof(struct ra_msg);
	
	while (optlen > 0) {
		int len = (opt[1] << 3);
		if(len == 0) return 1;
		
		if (opt[0] == ND_OPT_PREFIX_INFO) {
			pinfo = (struct prefix_info *) opt;
			if (len < sizeof(struct prefix_info)) return -1;
			ipv6_addr_copy(raddr,&pinfo->prefix);
			*plen = pinfo->prefix_len;
			DEBUG((DBG_DATADUMP, "the length of the prefix is %d",*plen));
			global_address = 0/*(int)pinfo->router_address*/; 
		}

		if (opt[0] == ND_OPT_RTR_ADV_INTERVAL) {
			*ival = ntohl(*(__u32 *)(opt+4))*HZ / 1000;
			DEBUG((DBG_INFO, "received router interval option with interval : %d ",*ival/HZ));
		}

		optlen -= len;
		opt += len;
	}

	return (int) ra->icmph.icmp6_dataun.u_nd_ra.home_agent;
}

/*
 * Decides what to do based on the new prefix.Keeps track of CoAs
 * and deletes the old CoA.  Also notifies mn of movement.  
 */ 
static int handle_prefix(struct router *old, struct router *new)
{

	struct in6_addr mcdest;
	extern struct in6_addr home_agent;
	DEBUG_FUNC();
	if (old!=NULL){

		if (mipv6_prefix_compare(&new->raddr,
					&old->raddr, new->pfix_len)) {

			DEBUG((DBG_WARNING,"Router adv from same router\n new:"));
			debug_print_addr(DBG_WARNING, &new->raddr);
			DEBUG((DBG_WARNING,"old:"));
			debug_print_addr(DBG_WARNING, &old->raddr);
			return 0; /* Same network */
		}
	}
      
	/* prefix same as in home network ? */

	if (mipv6_prefix_compare(&new->raddr,&home_address_pinfo.prefix,
				 home_address_pinfo.prefix_len )) {

	  	DEBUG((DBG_INFO,"Router adv from home router"));
		if (!isathome) {
			struct in6_addr mcaddr;
			isathome = 1;
			/* Back in home subnet */
			ipv6_addr_all_nodes(&mcdest);  
			/* A neighbor advertisement is sent to all 
			 *  nodes multicast address 
			 * to redirect datagrams to MN's home address to 
			 * mobile node's HW-address, otherwise 
			 * they would be sent to HA's hw-address 
			 */


			addrconf_addr_solict_mult_new(&home_agent, &mcaddr);
			ndisc_send_ns(new->dev, NULL, 
				      &home_agent, &mcaddr, NULL);
			/* Movement event  */
			mipv6_mn_returned_home(&prev_router, 
					       home_address_pinfo.prefix_len);
			ndisc_send_na(new->dev, NULL, &mcdest, 
				      &home_address, 0, 0, 1, 1);
			
		}
		
		/* For routing purposes */		
		return 0;
	} else {
		struct in6_addr coa;
		DEBUG((DBG_INFO,"Router adv from new foreign router"));
		mipv6_get_care_of_address(NULL,&coa);
		/* RADV prefix NOT the same as in home network (in foreign net) */
		isathome = 0;
		
		/* Movement event  */
		mipv6_mobile_node_moved(&coa,&prev_router, new->pfix_len);
		return 0;
	}
	return 1;
}

/*
 *  Function that determines whether given rt6_info should be destroyed
 *  (negative => destroy rt6_info,
 *   zero or positive => do nothing)
 */
static int mn_route_cleaner(struct rt6_info *rt, void *arg)
{
	int type;
	DEBUG_FUNC();
	
	DEBUG((DBG_INFO, "clean route ?: via dev=\"%s\"", rt->rt6i_dev->name));

	/*   Do not delete routes to local addresses or 
	 *   to multicast addresses, since we need them to get 
	 *   router advertisements etc. Multicast addresses are more tricky, 
	 *   but we don't delete them in any case 
	 */ 
	type=ipv6_addr_type(&rt->rt6i_dst.addr);

	if ((type&IPV6_ADDR_MULTICAST || type&IPV6_ADDR_LINKLOCAL)
	   || rt->rt6i_dev==&loopback_dev 
	   || !ipv6_addr_cmp(&rt->rt6i_gateway,&next_router->ll_addr) 
	   || mipv6_prefix_compare(&rt->rt6i_dst.addr, &next_router->raddr, 64))
	  return 0;

	DEBUG((DBG_DATADUMP, "deleting route to: "));
	debug_print_addr(DBG_DATADUMP, &rt->rt6i_dst.addr);

	/*   delete all others */
	return -1;
}


/* Deletes old routes 
 */
static __inline void delete_routes(void)
{
	struct in6_addr homeaddr;
	int plength;

	DEBUG_FUNC();

	/* Routing table is locked to ensure that nobody uses its */  
	write_lock_bh(&rt6_lock);
	DEBUG((DBG_INFO,"mipv6: Purging routes"));
	mipv6_get_home_addr(NULL, &homeaddr, &plength);
	/*  TODO: Does not prune, should it?  */
	fib6_clean_tree(&ip6_routing_table, 
			mn_route_cleaner, 0, &homeaddr);
	write_unlock_bh(&rt6_lock);

}

/*  Changes the router, called from ndisc.c after handling the router advertisement
 */

void mipv6_change_router(void)
{
	struct router *tmp;
	unsigned long flags;
	DEBUG_FUNC(); 
	if (curr_router!=NULL && 
	    !ipv6_addr_cmp(&curr_router->ll_addr,&next_router->ll_addr)){
		DEBUG((DBG_DATADUMP,"Trying to handoff from:"));
                debug_print_addr(DBG_DATADUMP,&curr_router->ll_addr);
		DEBUG((DBG_DATADUMP,"Trying to handoff to:"));
                debug_print_addr(DBG_DATADUMP,&next_router->ll_addr);
		
		
		return;
        }

	if (!mdetect_is_initialized){
		DEBUG((DBG_WARNING,
		       "We don't change router before initialization"));
		return;
	}

	delete_routes();
	spin_lock_irqsave(&router_lock , flags);
	/* The global address of the router is copied to prev_router
	 * 
	 */
	if (curr_router!=NULL){
		if(curr_router->is_ha&&curr_router->glob_addr)
			ipv6_addr_copy(&prev_router,&curr_router->raddr);
	}
	else
		ipv6_addr_set(&prev_router,0,0,0,0);
	tmp=curr_router;
	curr_router=next_router;
	spin_unlock_irqrestore(&router_lock, flags);
	handle_prefix(tmp,curr_router);
}

/*
 * advertisement interval-based movement detection
 */
static void router_state(unsigned long foo)
{
	unsigned long flags;
	spin_lock_irqsave(&router_lock, flags);

	if (curr_router==NULL){
		DEBUG((DBG_DATADUMP,"Sending RS"));
		rs_send(CONTINUE); 
		mod_timer(&r_timer,jiffies + HZ/2);
		spin_unlock_irqrestore(&router_lock, flags);
		return;
	}
	
	switch (curr_router->state) {
		
	case ROUTER_REACHABLE:
		/* if radvd doesn't support interval option handoffs will 
		 * be based on the expiry of lifetime 
		 */
		
		if (curr_router->last_ra_rcvd + curr_router->interval >= 
		    jiffies || (curr_router->interval==0 && 
				   (curr_router->last_ra_rcvd + 
				    curr_router->lifetime >= jiffies) )) {
			break;

		}
		else {
			curr_router->state = NOT_REACHABLE;
			rs_send(START);
			DEBUG((DBG_INFO, 
			       "Last ra rcvd: %d, interval: %d, jiffies/HZ: %d",
			       curr_router->last_ra_rcvd,curr_router->interval, 
			       jiffies ));
			DEBUG((DBG_INFO,
			       "Current router changed state to NOT_REACHABLE"));
		}
		break;
	
	case NOT_REACHABLE:
		DEBUG((DBG_DATADUMP,
		       "Current router state: Router NOT reachable"));
		if(curr_router->last_ra_rcvd+curr_router->lifetime <= jiffies)
			
			rs_send(CONTINUE);
	}
	spin_unlock_irqrestore(&router_lock, flags);
	mod_timer(&r_timer,jiffies + HZ/10);
	
}
/* Deletes old coas, assumes they are in EUI-64 format */ 
void coa_del(struct net_device *dev, struct in6_addr *prefix)
{
	int err;
	struct in6_ifreq coa_del;
	struct in6_addr coa;
	__u32 coa_pfix=64;
	DEBUG_FUNC();
	/* We get the suffix  and combine it with the 64 bit prefix*/
	if (prefix==NULL || dev==NULL)
		return;
	ipv6_get_lladdr(dev, &coa);
	coa.s6_addr32[0]=prefix->s6_addr32[0];
	coa.s6_addr32[1]=prefix->s6_addr32[1];
	if (ipv6_chk_addr(&coa, dev) != 0 &&
	    ipv6_addr_cmp(&coa,&home_address)){
		ipv6_addr_copy(&coa_del.ifr6_addr,&coa);
		coa_del.ifr6_prefixlen= coa_pfix; /* EUI-64 CoAs */
		coa_del.ifr6_ifindex=dev->ifindex;
		err=addrconf_del_ifaddr((void *) &coa_del);
		if (err){
			DEBUG((DBG_CRITICAL, "error: %d Couldn't delete old care-of address:", 
			       err)); 
			debug_print_addr(DBG_CRITICAL, &coa);
			DEBUG((DBG_CRITICAL, 
			       "It must be deleted manually after unloading the module"));
		}
	}
}
/*  Defined in ndisc.c of IPv6 module */
extern void ndisc_send_na(
	struct net_device *dev, struct neighbour *neigh,
	struct in6_addr *daddr, struct in6_addr *solicited_addr,
	int router, int solicited, int override, int inc_opt);

int mipv6_is_at_home(void)
{
	return isathome;
}

/*
 * Returns the current care-of address, assumes addresses in EUI-64 format
 * TODO: Change the first parameter to dev
 */
void mipv6_get_care_of_address(
	struct in6_addr *homeaddr, struct in6_addr *coaddr)
{
	unsigned long flags;

	DEBUG_FUNC();

	if (curr_router==NULL || isathome){
		ipv6_addr_copy(coaddr, &home_address);
		return;
	}
	spin_lock_irqsave(&router_lock, flags);
	ipv6_get_lladdr(curr_router->dev, coaddr);
	coaddr->s6_addr32[0]=curr_router->raddr.s6_addr32[0];
	coaddr->s6_addr32[1]=curr_router->raddr.s6_addr32[1];
	if (dad==RESPECT_DAD && ipv6_chk_addr(coaddr, curr_router->dev) == 0){ 
		/* address check failure probably due to dad wait*/
	
		DEBUG((DBG_WARNING, 
		       "care-of address not valid, using home address instead"));
		ipv6_addr_copy(coaddr, &home_address);
	}
	spin_unlock_irqrestore(&router_lock, flags);

}

/* returns previous care of address */
void mipv6_get_old_care_of_address(
        struct in6_addr *homeaddr,struct in6_addr *prevcoa)

{
	unsigned long flags;
        spin_lock_irqsave(&router_lock, flags);
        ipv6_get_lladdr(curr_router->dev, prevcoa);
        prevcoa->s6_addr32[0]=prev_router.s6_addr32[0];
        prevcoa->s6_addr32[1]=prev_router.s6_addr32[1];
        spin_unlock_irqrestore(&router_lock, flags);

}

/*
 * Returns the home address
 * TODO: Change the ifp to dev
 */
void mipv6_get_home_addr(
	struct inet6_ifaddr *ifp,
	struct in6_addr *home_addr,
	int *prefix_len)
{
	DEBUG_FUNC();
	
	*prefix_len = home_address_pinfo.prefix_len;
	ipv6_addr_copy(home_addr, &home_address);
/* TODO: Make the check use a real device*/
/*	ipv6_chk_addr(&home_address, home_device);*/
	DEBUG((DBG_DATADUMP, "srcaddr: "));
	debug_print_addr(DBG_DATADUMP, home_addr);
}




/* Called from ndisc.c's router_discovery to determine whether to
 * change the current default router and pass the ra to the default  router 
 * adding mechanism of ndisc.c. return value 1 passes ra and 0 doesn't.  
 */
int mipv6_ra_rcv(struct sk_buff *skb)
{
	unsigned long flags;
	struct in6_addr *saddr = &skb->nh.ipv6h->saddr;
	DEBUG_FUNC();
	
	if (mdetect_is_initialized) {
		DEBUG((DBG_DATADUMP, "Received a RA from:"));
		debug_print_addr(DBG_DATADUMP, saddr);
 
		/* Whether from current router */
		spin_lock_irqsave(&router_lock, flags);
		if (curr_router!=NULL && 
		   !ipv6_addr_cmp(&skb->nh.ipv6h->saddr, 
				  &curr_router->ll_addr)){
			curr_router->last_ra_rcvd=jiffies;
			curr_router->state=ROUTER_REACHABLE;
			DEBUG((DBG_DATADUMP,
			       "Received a ra from current router"));
			spin_unlock_irqrestore(&router_lock, flags);
			return 1; 
		}
		/* Another router */
		else {
			int plen=0;
			__u32 ival=0;
			int is_ha=0;
			unsigned long lifet=0;
			struct in6_addr r_glob;
			struct router *tmp=NULL;
			
			/* Whether the router exists, TODO: check lifetime of ra */
			tmp = iterate_routers(saddr, curr_router);
			if (tmp == NULL){
				int glob=1;
				is_ha=process_ra(skb, &r_glob, &plen, &ival,&lifet);
				/* Insert router to list first */      	
				tmp = new_router(saddr, 
						  &r_glob,skb->dev, plen, 
						 ival, is_ha,glob,lifet);
				if (tmp == NULL){
					DEBUG((0,
					       "Failed to add new router to list"));
					spin_unlock_irqrestore(&router_lock, 
							       flags);
					return 0;
				}

				
				tmp->last_ra_rcvd=jiffies;
				tmp->state=ROUTER_REACHABLE;
				
				if(curr_router == NULL && rt6_get_dflt_router(saddr,skb->dev)){
                                  next_router = tmp;
                                  DEBUG((1,"Startup"));
				  spin_unlock_irqrestore(&router_lock,
							 flags);
                                  return 1;
                                }

				if (mdet_mech==EAGER_CELL_SWITCHING) 
				 { 
					 DEBUG((DBG_INFO,"Changing router to previously unknown router:"));
					 next_router=tmp;
					 neigh_ifdown(&nd_tbl, 
						      next_router->dev);
					 spin_unlock_irqrestore(&router_lock ,
								flags);
					 return 1;
					 
				 }
			}
			else {
				/* TODO: check if ra ival is still the same, 
				 *  also lifetime should be checked 
				 */
				tmp->last_ra_rcvd = jiffies;
				tmp->state = ROUTER_REACHABLE;
				DEBUG((DBG_INFO,"Updating entry for router"));
				
				if (curr_router == NULL || 
				   curr_router->state == NOT_REACHABLE){
					DEBUG((DBG_INFO,
					       "Changing def. router to a new router, since old is not reachable"));
					next_router = tmp;
					neigh_ifdown(&nd_tbl, 
						     next_router->dev);

					spin_unlock_irqrestore(&router_lock, 
							       flags);
					return 1;
				}
				spin_unlock(&router_lock);
				return 0; 
			}
		}
		spin_unlock_irqrestore(&router_lock, flags);
	}
	return 0;
}

#ifdef CONFIG_SYSCTL
static void mipv6_mdetect_sysctl_register(void)
{
	mipv6_mdetect_sysctl_header =
		register_sysctl_table(mipv6_mdetect_sysctl.mipv6_root_table, 0);
}

static void mipv6_mdetect_sysctl_unregister(void)
{
	unregister_sysctl_table(mipv6_mdetect_sysctl_header);
}

static int mipv6_mdetect_mech_sysctl_handler(
	ctl_table *ctl,
	int write,
	struct file * filp,
	void *buffer,
	size_t *lenp)
{
	char *p;
	p = ctl->data;

	if (write) {
		proc_dointvec(ctl, write, filp, buffer, lenp);
		if (mdet_mech < 0)
			mdet_mech = EAGER_CELL_SWITCHING; /* default */

		/* Add new movement detection mechanism below */
		switch (mdet_mech) 
		{
		case EAGER_CELL_SWITCHING: break;
		}
	} 
	else {
		sprintf(p, "%d", mdet_mech);
	}
	
	return 0;
}

static int mipv6_home_address_sysctl_handler(
	ctl_table *ctl,
	int write,
	struct file * filp,
	void *buffer,
	size_t *lenp)
{
	int i, itmp[4], iptmp[4], ret, plen, mask;
	char *p, *pc;
	char tmp[9];

	p = ctl->data;

	if (!write) {
		memset(p, 0, 36);
		sprintf(p, "%08x%08x%08x%08x", 
			(int)ntohl(home_address.s6_addr32[0]),
			(int)ntohl(home_address.s6_addr32[1]),
			(int)ntohl(home_address.s6_addr32[2]),
			(int)ntohl(home_address.s6_addr32[3]));
	}

	ret = proc_dostring(ctl, write, filp, buffer, lenp);

	if (write) {
		for (i = 0; i < 4; i++) {
			memset(tmp, 0, 9);
			strncpy(tmp, p, 8);
			pc = tmp;
			itmp[i] = simple_strtoul(tmp, &pc, 16);
			if (*pc) return 0; /* could not convert to number*/
			p+=8;
		}
		p+=1;
		memset(tmp, 0, 3);
		strncpy(tmp, p, 2);
		pc = tmp;
		plen = simple_strtoul(tmp, &pc, 16);
		if (*pc) return 0;

		home_address_pinfo.prefix_len = plen;

		for (i = 0; i < 4; i++) {
			mask = (~0);
			if (plen < 32 && plen > 0)
				mask <<= plen;
			if (plen <= 0)
				mask = 0;
			iptmp[i] = itmp[i] & mask;
			plen -= 32;
		}

		ipv6_addr_set(&home_address_pinfo.prefix,
			      (int)htonl(iptmp[0]), (int)htonl(iptmp[1]),
			      (int)htonl(iptmp[2]), (int)htonl(iptmp[3]));

		ipv6_addr_set(&home_address, (int)htonl(itmp[0]),
			      (int)htonl(itmp[1]), (int)htonl(itmp[2]),
			      (int)htonl(itmp[3]));

		DEBUG((DBG_INFO, "Setting home address to:"));
		debug_print_addr(DBG_INFO, &home_address);
		DEBUG((DBG_INFO, "Home address prefix (plen=%d:)",
		       home_address_pinfo.prefix_len));
		debug_print_addr(DBG_INFO, &home_address_pinfo.prefix);
		mdetect_is_initialized=1;
		/* We add the timer only after mdetect is configured */
		r_timer.expires = jiffies + HZ;
		add_timer(&r_timer);
	}

	return ret;
}

#endif /* CONFIG_SYSCTL */


int __init mipv6_initialize_mdetect(void)
{

	DEBUG_FUNC();

#ifdef CONFIG_SYSCTL
	mipv6_mdetect_sysctl_register();
#endif	
	init_timer(&r_timer);
	
	return 0;
}

int __exit mipv6_shutdown_mdetect()
{
	unsigned long flags;
	DEBUG_FUNC();
	
#ifdef CONFIG_SYSCTL
	mipv6_mdetect_sysctl_unregister();
#endif	

	del_timer(&r_timer);
	/* Free the memory allocated by router list */
	spin_lock_irqsave(&router_lock, flags);
	list_free();
	spin_unlock_irqrestore(&router_lock, flags);

	return 0;
}

