/*
 *      Mobile IPv6 main module
 *
 *      Authors:
 *      Sami Kivisaari          <skivisaa@cc.hut.fi>
 *      Antti Tuominen          <ajtuomin@tml.hut.fi>
 *
 *      $Id: mipv6.c,v 1.21 2000/10/17 08:44:48 antti Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#include <linux/kernel.h>   /* We're doing kernel work */
#include <linux/module.h>   /* Specifically, a module */

#include <linux/autoconf.h>
#include <linux/netfilter.h>
#include <linux/netfilter_ipv6.h>
#include <linux/ipsec.h>
#include <net/mipglue.h>
#include <net/addrconf.h>
#include <net/ip6_route.h>
#include <net/checksum.h>

#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif /* CONFIG_SYSCTL */

int mipv6_is_ha = 0;
int mipv6_is_mn = 0;
int mipv6_debug = 1;

#if defined(MODULE) && LINUX_VERSION_CODE > 0x20115
MODULE_AUTHOR("MIPL Team");
MODULE_DESCRIPTION("Mobile IPv6");
MODULE_PARM(mipv6_is_ha, "i");
MODULE_PARM(mipv6_is_mn, "i");
MODULE_PARM(mipv6_debug, "i");
#endif

#include "mipv6.h"
#include "tunnel.h"
#include "ha.h"
#include "mn.h"
#include "mdetect.h"
#include "procrcv.h"
#include "stats.h"
#include "sendopts.h"
#include "bul.h"
#include "debug.h"
#include "bcache.h"
#include "dstopts.h"
#include "ah.h"
#ifdef CONFIG_SYSCTL
#include "sysctl.h"
#endif /* CONFIG_SYSCTL */

#define MIPV6_BUL_SIZE 32
#define MIPV6_BCACHE_SIZE 32

#define SUCC 1
#define TRUE 1
#define FALSE 0

/*
 * Perform prefix comparison bitwise
 */
int mipv6_prefix_compare(struct in6_addr *addr,
			 struct in6_addr *prefix, unsigned int nprefix)
{
	int i;

	if (nprefix > 128)
		return FALSE;

	for (i = 0; nprefix > 0; nprefix -= 32, i++) {
		if (nprefix >= 32) {
			if (addr->s6_addr32[i] != prefix->s6_addr32[i])
				return FALSE;
		} else {
			if (((addr->s6_addr32[i] ^ prefix->s6_addr32[i]) &
			     ((~0) << (32 - nprefix))) != 0)
				return FALSE;
			return TRUE;
		}
	}

	return TRUE;
}

/*
 * Perform suffix comparison bitwise
 */
int mipv6_suffix_compare(struct in6_addr *addr,
			 struct in6_addr *suffix, unsigned int nsuffix)
{
	int i;

	if (nsuffix > 128)
		return FALSE;

	for (i = 3; nsuffix > 0; nsuffix -= 32, i--) {
		if (nsuffix >= 32) {
			if (addr->s6_addr32[i] != suffix->s6_addr32[i])
				return FALSE;
		} else {
			if (((addr->s6_addr32[i] ^ suffix->s6_addr32[i]) &
			     ((~0) << nsuffix)) != 0)
				return FALSE;
			return TRUE;
		}
	}

	return TRUE;
}

/* could make mods atomic, don't know if its worth it */
void mipv6_finalize_modify_xmit(void **alloclist)
{
	int i;

	DEBUG_FUNC();
	for (i = 0; alloclist[i] != NULL; i++)
		kfree(alloclist[i]);
}



int mipv6_create_ah(struct ipv6_txoptions *opt)
{

	struct mipv6_ah *ah = kmalloc(sizeof(struct mipv6_ah), GFP_ATOMIC);
	if (ah == NULL) {
		DEBUG((0, "Couldn't allocate memory for AH"));
		return -1;
	}
	DEBUG_FUNC();

	/* fill in the AH */
	ah->ah_hl = sizeof(struct mipv6_ah) / 4 - 2;
	DEBUG((DBG_DATADUMP, "ah length: %d", ah->ah_hl));
	ah->ah_rv = 0;
	ah->ah_spi = 0;
	ah->ah_rpl = 0;
	memset(ah->ah_data, 0, AHHMAC_HASHLEN);
	opt->auth = (struct ipv6_opt_hdr *) ah;
	return 0;
}

struct ipv6_txoptions *mipv6_modify_xmit_packets(struct sock *sk,
						 struct sk_buff *skb,
						 struct ipv6_txoptions
						 *opts, struct flowi *fl,
						 struct dst_entry **dst,
						 void *allocptrs[])
{
	struct ipv6_rt_hdr *oldrthdr = NULL, *newrthdr = NULL;
	struct ipv6_opt_hdr *newdo0hdr = NULL, *olddo0hdr = NULL,
		*newdo1hdr = NULL, *olddo1hdr = NULL;
	struct ipv6_txoptions *newopts = NULL;
	struct mipv6_bcache_entry bc_entry;
	struct in6_addr tmpaddr, *saddr, *daddr;
	int nalloc = 0, changed_rt_hdr = FALSE, changed_dstopts0_hdr =
		FALSE, changed_dstopts1_hdr = FALSE, add_ha = 0;

	DEBUG_FUNC();

	/*
	 * we have to be prepared to the fact that saddr might not be present,
	 * if that is the case, we acquire saddr just as kernel does.
	 */
	saddr = fl ? fl->fl6_src : NULL;
	daddr = fl ? fl->fl6_dst : NULL;

	if (daddr == NULL)
		return opts;
	if (saddr == NULL) {
		/*
		 * TODO!: we might end up having wrong saddr here.
		 * Kernel does this bit differently.
		 */
		int err = ipv6_get_saddr(NULL, daddr, &tmpaddr);
		if (err)
			return opts;
		else
			saddr = &tmpaddr;
	}

	debug_print_addr(DBG_INFO, saddr);
	debug_print_addr(DBG_INFO, daddr);

	/* Check old routing header and destination options headers */
	oldrthdr = (opts != NULL) ? opts->srcrt : NULL;
	olddo0hdr = (opts != NULL) ? opts->dst0opt : NULL;
	olddo1hdr = (opts != NULL) ? opts->dst1opt : NULL;
	if (mipv6_bcache_get(daddr, &bc_entry) == 0) {
		DEBUG((DBG_INFO, "Binding exists. Adding routing header"));

		/*
		 * Append care-of-address to routing header (original
		 * destination address is home address, the first
		 * source route segment gets put to the destination
		 * address and the home address gets to the last
		 * segment of source route (just as it should)) 
		 */
		newrthdr = mipv6_append_rt_header(oldrthdr, &bc_entry.coa);

		/*
		 * reroute output (we have to do this in case of TCP
                 * segment) 
		 */
		if (dst) {
			struct in6_addr *tmp = fl->fl6_dst;
			fl->fl6_dst = &bc_entry.coa;

			dst_release(*dst);
			*dst = ip6_route_output(sk, fl);
			if (skb)
				skb->dst = *dst;
			fl->fl6_dst = tmp;

			DEBUG((DBG_INFO, "Rerouted outgoing packet"));
		}

		changed_rt_hdr = (newrthdr != NULL)
			&& (newrthdr != oldrthdr);
		if (changed_rt_hdr)
			allocptrs[nalloc++] = newrthdr;
		else {
			DEBUG((DBG_ERROR,
			       "Could not add routing header (hdr=0x%p)",
			       newrthdr));
			/* TODO: Should drop the packet */
		}
	}

	/* Add destination options */
	add_ha = !mipv6_is_at_home();
	newdo1hdr = mipv6_add_dst1opts(saddr, daddr, olddo1hdr, &add_ha);
	/* Only home address option is inserted to first dst opt header */
	if (mipv6_is_mn && add_ha)
		newdo0hdr = mipv6_add_dst0opts(saddr, olddo0hdr, add_ha);

	changed_dstopts0_hdr = ((newdo0hdr != NULL)
				&& (newdo0hdr != olddo0hdr));
	changed_dstopts1_hdr = ((newdo1hdr != NULL)
				&& (newdo1hdr != olddo1hdr));

	if (changed_dstopts0_hdr)
		allocptrs[nalloc++] = newdo0hdr;

	if (changed_dstopts1_hdr)
		allocptrs[nalloc++] = newdo1hdr;
	else {
		DEBUG((DBG_DATADUMP,
		       "Destination options not piggybacked. (ohdr=0x%p, nhdr=0x%p)",
		       olddo1hdr, newdo1hdr));
	}

	/*
	 * If any headers were changed, reallocate new ipv6_txoptions and
	 * update it to match new headers
	 */
	if (changed_rt_hdr || changed_dstopts0_hdr || changed_dstopts1_hdr) {
		DEBUG((DBG_INFO, "Rebuilding ipv6_txoptions"));

		newopts = (struct ipv6_txoptions *)
			kmalloc(sizeof(struct ipv6_txoptions), GFP_ATOMIC);

		allocptrs[nalloc++] = newopts;

		if (newopts == NULL) {
			DEBUG((DBG_ERROR,
			       "Error: failed to allocate ipv6_txoptions!"));
			return opts;
		}

		if (opts == NULL) {
			memset(newopts, 0, sizeof(struct ipv6_txoptions));
			newopts->tot_len = sizeof(struct ipv6_txoptions);
		} else
			memcpy(newopts, opts,
			       sizeof(struct ipv6_txoptions));

		if (changed_rt_hdr) {
			newopts->srcrt = newrthdr;
			newopts->opt_nflen += ipv6_optlen(newopts->srcrt) -
				((oldrthdr != NULL) ? 
				 ipv6_optlen(oldrthdr) : 0);
		}
		if (changed_dstopts0_hdr) {
			newopts->dst0opt = newdo0hdr;
			newopts->opt_nflen += ipv6_optlen(newopts->dst0opt) -
				((olddo0hdr != NULL) ? 
				 ipv6_optlen(olddo0hdr) : 0);
		}
		if (changed_dstopts1_hdr) {
			newopts->dst1opt = newdo1hdr;
			newopts->opt_flen += ipv6_optlen(newopts->dst1opt) -
				((olddo1hdr != NULL) ? 
				 ipv6_optlen(olddo0hdr) : 0);
			/* We add ah only when sending a ba or bu */
			/*
			if (mipv6_create_ah(newopts) >= 0) {
				allocptrs[nalloc++] = newopts->auth;  
				newopts->opt_flen += sizeof(struct mipv6_ah);
				if (skb) skb->nfmark = SND_AUTH;
				else DEBUG((0, "skb was NULL"));
			}
			*/
		}


		opts = newopts;
	}

	allocptrs[nalloc] = NULL;

	return opts;
}


/*
 * Required because we can only modify addresses after the packet is
 * constructed.  We otherwise mess with higher level protocol
 * pseudoheaders. With strict protocol layering life would be SO much
 * easier!  
 */
static unsigned int modify_xmit_addrs(unsigned int hooknum,
				      struct sk_buff **skb,
				      const struct net_device *in,
				      const struct net_device *out,
				      int (*okfn) (struct sk_buff *))
{
	struct ipv6hdr *hdr = (*skb) ? (*skb)->nh.ipv6h : NULL;

	if (hdr == NULL)
		return NF_ACCEPT;

	DEBUG((DBG_INFO,
	       "Replace source address (home-address => care-of-address)"));
	debug_print_addr(DBG_INFO, &hdr->saddr);
	mipv6_get_care_of_address(&hdr->saddr, &hdr->saddr);
	debug_print_addr(DBG_INFO, &hdr->saddr);

	return NF_ACCEPT;
}

/* Destination Unreachable ICMP error message handler */

static int mipv6_icmpv6_dest_unreach(struct sk_buff *skb)
{
	struct icmp6hdr *phdr = (struct icmp6hdr *) skb->h.raw;
	int left = (skb->tail - (unsigned char *) (phdr + 1))
	    - sizeof(struct ipv6hdr);
	struct ipv6hdr *hdr = (struct ipv6hdr *) (phdr + 1);
	struct ipv6_opt_hdr *ehdr;
	struct ipv6_rt_hdr *rthdr = NULL;
	struct in6_addr *addr;
	int hdrlen, nexthdr = hdr->nexthdr;

	DEBUG_FUNC();

	ehdr = (struct ipv6_opt_hdr *) (hdr + 1);

	while (nexthdr != NEXTHDR_ROUTING) {
		hdrlen = ipv6_optlen(ehdr);
		if (hdrlen > left)
			return 0;
		if (!(nexthdr == NEXTHDR_HOP || nexthdr == NEXTHDR_DEST))
			return 0;
		nexthdr = ehdr->nexthdr;
		ehdr = (struct ipv6_opt_hdr *) ((u8 *) ehdr + hdrlen);
		left -= hdrlen;
	}

	if (nexthdr == NEXTHDR_ROUTING) {
		if (ipv6_optlen(ehdr) > left)
			return 0;
		rthdr = (struct ipv6_rt_hdr *) ehdr;
		if (rthdr->segments_left != 1)
			return 0;
	}

	if (rthdr == NULL) {
		DEBUG((DBG_ERROR, "null pointer in rthdr"));
		return 0;
	}

	addr = (struct in6_addr *) ((u32 *) rthdr + 2);
	if (mipv6_bcache_exists(addr) >= 0) {
		if (!mipv6_bcache_delete(addr, CACHE_ENTRY)) {
			DEBUG((DBG_INFO, "Deleted bcache entry (reason: "
			       "dest unreachable)"));
			debug_print_addr(DBG_INFO, addr);
		}
	}
	return 0;
}

/* BUL callback */

static int bul_entry_expired(struct mipv6_bul_entry *bulentry)
{
	DEBUG((DBG_INFO, "bul entry 0x%x lifetime expired, deleting entry",
	       (int) bulentry));
	return 1;
}

/* Parameter Problem ICMP error message handler */

static int mipv6_icmpv6_paramprob(struct sk_buff *skb)
{
	struct icmp6hdr *phdr = (struct icmp6hdr *) skb->h.raw;
	struct in6_addr *saddr = skb ? &skb->nh.ipv6h->saddr : NULL;
	struct ipv6hdr *hdr = (struct ipv6hdr *) (phdr + 1);
	int ulen = (skb->tail - (unsigned char *) (phdr + 1));

	int errptr;
	__u8 *off_octet;

	DEBUG_FUNC();

	/* We only handle code 2 messages. */
	if (phdr->icmp6_code != ICMPV6_UNK_OPTION)
		return 0;

	/* Find offending octet in the original packet. */
	errptr = ntohl(phdr->icmp6_pointer);

	/* There is not enough of the original packet left to figure
	 * out what went wrong. Bail out. */
	if (ulen <= errptr)
		return 0;

	off_octet = ((__u8 *) hdr + errptr);
	DEBUG((DBG_INFO, "Parameter problem: offending octet %d [0x%2x]",
	       errptr, *off_octet));

	/* If CN did not understand Binding Update, set BUL entry to
	 * SEND_NOMORE so no further BUs are sumbitted to this CN. */
	if (*off_octet == MIPV6_TLV_BINDUPDATE) {
		struct mipv6_bul_entry *bulentry = mipv6_bul_get(saddr);

		if (bulentry) {
			bulentry->state = SEND_NOMORE;
			bulentry->callback = bul_entry_expired;
			bulentry->callback_time = jiffies +
				DUMB_CN_BU_LIFETIME * HZ;
			mipv6_bul_put(bulentry);
			DEBUG((DBG_INFO, "BUL entry set to SEND_NOMORE"));
		}
	}
	/* If CN did not understand Home Address Option, we log an
	 * error and discard the error message. */
	else if (*off_octet == MIPV6_TLV_HOMEADDR) {
		DEBUG((DBG_WARNING, "Correspondent node does not "
		       "implement Home Address Option receipt."));
		return 1;
	}

	return 0;
}

/* Home Agent Address Discovery Request/Reply ICMP message handler */
/* TODO: The Mobility Support draft (12) defines two new ICMP types
 * called Home Agent Address Discovery Request Message and Home Agent
 * Address Discovery Reply Message. However, type numbers for these
 * are still to be assigned by IANA. We should anyway do handlers for
 * them. For testing we can assign some (non conflicting) dummy values
 * for the types since only MIPv6 capable hosts send them. Names might
 * be changed later too. Definitions are in mipv6.h.  */
static int mipv6_ha_addr_disc(struct sk_buff *skb)
{
	struct icmp6hdr *phdr = (struct icmp6hdr *) skb->h.raw;
	struct in6_addr *address;
	__u8 type = phdr->icmp6_type;
	__u16 identifier;
	int ulen = (skb->tail - (unsigned char *) ((__u32 *) phdr + 3));

	DEBUG_FUNC();

	/* Invalid packet checks. */
	if (phdr->icmp6_type == MIPV6_HA_ADDR_DISC_REQUEST &&
	    ulen < sizeof(struct in6_addr)) return 0;

	if (phdr->icmp6_type == MIPV6_HA_ADDR_DISC_REPLY &&
	    ulen % sizeof(struct in6_addr) != 0)
		return 0;

	if (phdr->icmp6_code != 0)
		return 0;

	identifier = ntohs(phdr->icmp6_identifier);
	address = (struct in6_addr *) ((__u32 *) phdr + 3);

	DEBUG((DBG_INFO, "DHAD not implemented yet."));

	/* TODO: real functionality */
	if (type == MIPV6_HA_ADDR_DISC_REQUEST) {
		if (mipv6_is_ha) {
			/* Initiate DHAD */
			/* request(identifier, address) */
		}
	} else {
		if (mipv6_is_mn) {
			/* int n_addr = ulen / sizeof(struct in6_addr); */
			/* Conclude DHAD */
			/* reply(identifier, address, n_addr) */
		}
	}

	return 1;
}

/* Receives ICMPv6 messages and passes to corresponding handlers */

static int mipv6_icmpv6_rcv(struct sk_buff *skb, unsigned long len)
{
	int type, ret = 0;
	struct in6_addr *saddr = &skb->nh.ipv6h->saddr;
	struct in6_addr *daddr = &skb->nh.ipv6h->daddr;
	struct icmp6hdr *hdr =
	    (struct icmp6hdr *) (skb ? skb->h.raw : NULL);

	DEBUG_FUNC();

	if (len < sizeof(struct icmp6hdr)) {
		kfree_skb(skb);
		return 0;
	}

	if (skb->ip_summed)
		if (csum_ipv6_magic(saddr, daddr, len,
				    IPPROTO_ICMPV6, skb->csum)) {
			kfree_skb(skb);
			return 0;
		}

	type = hdr->icmp6_type;

	/* TODO: Test all handlers extensively! */
	switch (type) {
	case ICMPV6_DEST_UNREACH:
		ret = mipv6_icmpv6_dest_unreach(skb);
		break;

	case ICMPV6_PARAMPROB:
		if (mipv6_is_mn)
			ret = mipv6_icmpv6_paramprob(skb);
		break;

	case NDISC_ROUTER_ADVERTISEMENT:
		/* TODO: check why blocking does not work */
		/*      do_block=!mipv6_ra_rcv(skb); */
		break;

	case MIPV6_HA_ADDR_DISC_REQUEST:
	case MIPV6_HA_ADDR_DISC_REPLY:
		ret = mipv6_ha_addr_disc(skb);
		break;

	default:
	}

	kfree_skb(skb);
	return 0;
}


/* We set a netfilter hook so that we can modify outgoing packet's
 * source addresses 
 */
#define HUGE_NEGATIVE (~((~(unsigned int)0) >> 1))

struct nf_hook_ops addr_modify_hook_ops = {
	{NULL, NULL},		/* List head, no predecessor, no successor */
	modify_xmit_addrs,
	PF_INET6,
	NF_IP6_LOCAL_OUT,
	HUGE_NEGATIVE		/* Should be of EXTREMELY high priority since we
				 * do not want to mess with IPSec (possibly
				 * implemented as packet filter)
				 */
};


void mipv6_get_saddr_hook(struct inet6_ifaddr *ifp,
			  struct in6_addr *homeaddr)
{
	int dummy;
	/*  TODO!: conditional stuff if the address scope is e.g. link-local  */
	mipv6_get_home_addr(ifp, homeaddr, &dummy);
}


#ifdef CONFIG_SYSCTL
/* Sysctl table */
ctl_table mipv6_mobility_table[] = {
	{NET_IPV6_MOBILITY_DEBUG, "debuglevel",
	 &mipv6_debug, sizeof(int), 0644, NULL,
	 &proc_dointvec},
	{0}
};
#endif				/* CONFIG_SYSCTL */

static struct inet6_protocol mipv6_icmpv6_protocol = {
	mipv6_icmpv6_rcv,	/* handler              */
	NULL,			/* error control        */
	NULL,			/* next                 */
	IPPROTO_ICMPV6,		/* protocol ID          */
	0,			/* copy                 */
	NULL,			/* data                 */
	"MIPv6 ICMPv6"		/* name                 */
};

/*  Initialize the module  */
int __init init_module(void)
{
	printk(KERN_INFO "Initializing MIPL mobile IPv6\n");
	printk(KERN_INFO "Nodeconfig:  ha=%d mn=%d\n", !!mipv6_is_ha,
	       !!mipv6_is_mn);
	printk(KERN_INFO "Debug-level: %d\n", mipv6_debug);

	if (mipv6_is_ha && mipv6_is_mn) {
		printk(KERN_ERR "** Fatal error: Invalid nodeconfig! **\n");
		return 1;
	}

	/*  Initialize data structures  */
	mipv6_initialize_bcache(MIPV6_BCACHE_SIZE);

	mipv6_initialize_stats();

	if (mipv6_is_ha || mipv6_is_mn)
		mipv6_initialize_tunnel();

	if (mipv6_is_ha)
		mipv6_initialize_ha();
	if (mipv6_is_mn) {
		mipv6_initialize_bul(MIPV6_BUL_SIZE);
		mipv6_initialize_mn();
		mipv6_initialize_mdetect();

		/* COA to home transformation hook */
		MIPV6_SETCALL(mipv6_get_home_address, mipv6_get_saddr_hook);
		/* router advertisement processing and clearing of neighbor table */
		MIPV6_SETCALL(mipv6_ra_rcv, mipv6_ra_rcv);
		/* Actual HO, deletes also old routes after the addition of new ones in ndisc */
		MIPV6_SETCALL(mipv6_change_router, mipv6_change_router);
		/* Set packet modification hook (source addresses) */
		nf_register_hook(&addr_modify_hook_ops);
	}

	mipv6_initialize_sendopts();
	mipv6_initialize_procrcv();


	/* Register our ICMPv6 handler */
	inet6_add_protocol(&mipv6_icmpv6_protocol);

	/* Set hook on outgoing packets */
	MIPV6_SETCALL(mipv6_finalize_modify_xmit, mipv6_finalize_modify_xmit);
	MIPV6_SETCALL(mipv6_modify_xmit_packets, mipv6_modify_xmit_packets);

#if defined(MODULE) && defined(CONFIG_SYSCTL)
	/* register sysctl table */
	mipv6_sysctl_register();
#endif

	return 0;
}

/*  Cleanup module  */
void cleanup_module(void)
{
	DEBUG_FUNC();

#if defined(MODULE) && defined(CONFIG_SYSCTL)
	/* unregister sysctl table */
	mipv6_sysctl_unregister();
#endif

	/* Unregister our ICMPv6 handler */
	inet6_del_protocol(&mipv6_icmpv6_protocol);

	/*  Invalidate all custom kernel hooks  */
	mipv6_invalidate_calls();
	mipv6_shutdown_procrcv();

	if (mipv6_is_mn) {
		nf_unregister_hook(&addr_modify_hook_ops);
		mipv6_shutdown_mdetect();
		mipv6_shutdown_mn();
	}

	mipv6_shutdown_tunnel();
	mipv6_shutdown_sendopts();

	if (mipv6_is_ha)
		mipv6_shutdown_ha();

	mipv6_shutdown_stats();
	mipv6_shutdown_bcache();
	mipv6_shutdown_bul();
}
