/*
 *      Mobile-node functionality
 *
 *      Authors:
 *      Sami Kivisaari          <skivisaa@cc.hut.fi>
 *
 *      $Id: mn.c,v 1.12.2.1 2000/10/25 11:20:28 henkku Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */



#define __NO_VERSION__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/sched.h>
#include <linux/autoconf.h>
#include <linux/ipv6.h>
#include <linux/net.h>
#include <linux/init.h>

#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif

#include <net/ipv6.h>
#include <net/addrconf.h>
#include <net/ndisc.h>

#include "mipv6.h"
#include "mdetect.h"
#include "mipv6.h"
#include "bul.h"
#include "sendopts.h"
#include "ha.h"
#include "debug.h"
#include "mn.h"
#ifdef CONFIG_SYSCTL
#include "sysctl.h"
#endif

struct in6_addr home_agent;

static struct in6_addr addrany=ADDRANY;


#define SYSCTL_IPV6_ADDRESS_SIZE 36
static char mipv6_mn_home_agent_address[SYSCTL_IPV6_ADDRESS_SIZE];

static int mipv6_cn_bu_send(struct mipv6_bul_entry *entry);
static int mipv6_cn_bu_send_dereg(struct mipv6_bul_entry *entry);


#ifdef CONFIG_SYSCTL

static void mipv6_mn_sysctl_register(void);
static void mipv6_mn_sysctl_unregister(void);
static int mipv6_home_agent_address_sysctl_handler(
	ctl_table *ctl,
	int write,
	struct file * filp,
	void *buffer,
	size_t *lenp);

static struct ctl_table_header *mipv6_mn_sysctl_header;

static struct mipv6_mn_sysctl_table
{
	struct ctl_table_header *sysctl_header;
	ctl_table mipv6_vars[2];
	ctl_table mipv6_mobility_table[2];
	ctl_table mipv6_proto_table[2];
	ctl_table mipv6_root_table[2];
} mipv6_mn_sysctl = {
	NULL,

	{{NET_IPV6_MOBILITY_HOME_AGENT_ADDRESS, "home_agent_address",
	  &mipv6_mn_home_agent_address, SYSCTL_IPV6_ADDRESS_SIZE * sizeof(char),
	  0644, NULL, &mipv6_home_agent_address_sysctl_handler},

	 {0}},

	{{NET_IPV6_MOBILITY, "mobility", NULL, 0, 0555, mipv6_mn_sysctl.mipv6_vars}, {0}},
	{{NET_IPV6, "ipv6", NULL, 0, 0555, mipv6_mn_sysctl.mipv6_mobility_table}, {0}},
	{{CTL_NET, "net", NULL, 0, 0555, mipv6_mn_sysctl.mipv6_proto_table},{0}}
};

static void mipv6_mn_sysctl_register(void)
{
	mipv6_mn_sysctl_header =
		register_sysctl_table(mipv6_mn_sysctl.mipv6_root_table, 0);
}

static void mipv6_mn_sysctl_unregister(void)
{
	unregister_sysctl_table(mipv6_mn_sysctl_header);
}

static int mipv6_home_agent_address_sysctl_handler(
	ctl_table *ctl,
	int write,
	struct file * filp,
	void *buffer,
	size_t *lenp)
{
	int i, itmp[4], ret;
	char *p, *pc;
	char tmp[9];

	p = ctl->data;

	if (!write) {
		memset(p, 0, 36);
		sprintf(p, "%08x%08x%08x%08x", (int)ntohl(home_agent.s6_addr32[0]),
			(int)ntohl(home_agent.s6_addr32[1]),
			(int)ntohl(home_agent.s6_addr32[2]),
			(int)ntohl(home_agent.s6_addr32[3]));
	}

	ret = proc_dostring(ctl, write, filp, buffer, lenp);

	if (write) {
		for (i = 0; i < 4; i++) {
			memset(tmp, 0, 9);
			strncpy(tmp, p, 8);
			pc = tmp;
			itmp[i] = simple_strtoul(tmp, &pc, 16);
			if (*pc) return 0; /* could not convert to number*/
			p+=8;
		}

		ipv6_addr_set(&home_agent, (int)htonl(itmp[0]),
			      (int)htonl(itmp[1]), (int)htonl(itmp[2]),
			      (int)htonl(itmp[3]));
	}
	
	return ret;
}

#endif /* CONFIG_SYSCTL */

#define TRUE 1


void mipv6_tunneled_packet_received(struct sk_buff *skb,
				    struct ipv6hdr *inner, 
				    struct ipv6hdr *outer)
{
	struct mipv6_skbcb_data *cbdata;
	struct mipv6_bul_entry *bulentry = mipv6_bul_get(&inner->saddr);
	struct in6_addr tmp;
	int plen;
	
	DEBUG_FUNC();

	mipv6_get_home_addr(NULL,&tmp,&plen);
	/* We don't send bus in response to all tunneled packets */
	if (ipv6_addr_cmp(&tmp,&inner->daddr)) {
		DEBUG((DBG_INFO, 
		       "Received a tunneld packet NOT sent originally to home address"));
		return;
	}
	
	if (bulentry) {
		int state = bulentry->state;
		mipv6_bul_put(bulentry);
		if (state == SEND_NOMORE)
			return;
	}
	
	DEBUG((DBG_INFO, "Sending BU to correspondent node"));


	if (inner->nexthdr != IPPROTO_DSTOPTS){
		DEBUG((DBG_INFO, "Received tunneled packet without dst_opts"));
		/*  max wait 1000 ms  before sending an empty packet with BU */
		mipv6_send_upd_option(&inner->daddr,&inner->saddr,
				      CN_BU_DELAY, 
				      INITIAL_BINDACK_TIMEOUT,
				      MAX_BINDACK_TIMEOUT , 1, 
				      MIPV6_BU_F_ACK, 0, 
				      CN_BU_DEF_LIFETIME, NULL);
	}
	cbdata = MIPV6_SKBCB_DATA(skb);
	cbdata->cookie = MIPV6_SKBCB_COOKIE;
	cbdata->tunneled_packet = TRUE;

}

void mipv6_check_tunneled_packet(struct sk_buff *skb)
{
	struct mipv6_skbcb_data *cbdata = MIPV6_SKBCB_DATA(skb);

	DEBUG_FUNC();

	if(cbdata->cookie == MIPV6_SKBCB_COOKIE) {
		DEBUG((DBG_INFO, "skb contains MIPV6_SKBCB_COOKIE"));

		if(cbdata->tunneled_packet) {
			DEBUG((DBG_INFO, "packet was tunneled. Sending BU to CN"));
			debug_print_addr(DBG_INFO, &skb->nh.ipv6h->saddr);
			debug_print_addr(DBG_INFO, &skb->nh.ipv6h->daddr);
			mipv6_send_upd_option(
				&skb->nh.ipv6h->daddr,
				&skb->nh.ipv6h->saddr,
				CN_BU_DELAY, INITIAL_BINDACK_TIMEOUT,
				MAX_BINDACK_TIMEOUT , 1, 
				MIPV6_BU_F_ACK, 64, CN_BU_DEF_LIFETIME,
				NULL);
		}
	}
}



/*  We have moved to a foreign subnet, send home registrations
 *
 *  TODO!: If we have multiple interfaces then it is possible that only one
 *         of them moved.  This function should take as argument the home
 *         address that moved.
 */
int mipv6_mobile_node_moved(struct in6_addr *coa, struct in6_addr *prev_router, int plen)
{
	struct in6_addr homeaddr;
	int plength;

	DEBUG_FUNC();
	DEBUG((DBG_INFO, "Sending a BU to HA with new Care of address : "));
	debug_print_addr(DBG_INFO, coa);
	DEBUG((DBG_INFO, "and with home address "));
	mipv6_get_home_addr(NULL, &homeaddr, &plength);
	debug_print_addr(DBG_INFO, &homeaddr);

        /*  Binding update for HA  */
	mipv6_send_upd_option(
		&homeaddr, &home_agent, HA_BU_DELAY, INITIAL_BINDACK_TIMEOUT, 
		MAX_BINDACK_TIMEOUT, 1,
		MIPV6_BU_F_HOME | MIPV6_BU_F_ACK, plen,
		HA_BU_DEF_LIFETIME, NULL);

	/* BU to previous router */
       	mipv6_bu_to_prev_router(prev_router,plen);

        /*  A bu is sent to all CNs in the binding update list  */ 
	bul_iterate(mipv6_cn_bu_send);  
	return 0;
}

	
/*  We have returned home, send home deregistrations
 *
 *  TODO!: If we have multiple interfaces then it is possible that only one
 *         of them returned home.  This function should take take as argument
 *         the home address that returned home.
 */
int mipv6_mn_returned_home(struct in6_addr *prev_router, int plen)
{
	struct in6_addr homeaddr;
	int plength;

	DEBUG_FUNC();
	DEBUG((DBG_INFO,
	       "Returned home. Sending a BU to HA with home address"));

	mipv6_get_home_addr(NULL, &homeaddr, &plength);
	mipv6_send_upd_option(
		&homeaddr, &home_agent, 
		HA_BU_DELAY, INITIAL_BINDACK_TIMEOUT, 
		MAX_BINDACK_TIMEOUT, 1,
		MIPV6_BU_F_HOME | MIPV6_BU_F_ACK, 
		plen, 0, NULL);

        /* BU to previous router */
		mipv6_bu_to_prev_router(prev_router,plen);
	
        /* A bu is sent to all CNs in the binding update list */ 
	bul_iterate(mipv6_cn_bu_send_dereg);

	return 0;
}

/* Sends BU to previous router */
void mipv6_bu_to_prev_router(struct in6_addr *prev_router,int plen)
{

	struct in6_addr homeaddr,prevcoa;
	int plength;
 
        mipv6_get_home_addr(NULL, &homeaddr, &plength);
	
  if(ipv6_addr_cmp(&home_agent, prev_router) && ipv6_addr_cmp(&addrany, prev_router)
        && prev_router!=NULL){

	mipv6_get_old_care_of_address(NULL,&prevcoa);

	DEBUG((DBG_INFO, "Sending a BU to former router with prevcoa: "));
        debug_print_addr(DBG_INFO, &prevcoa);
        DEBUG((DBG_INFO, "and with prev_router:"));
        debug_print_addr(DBG_INFO, prev_router);
    
        mipv6_send_upd_option(
                &prevcoa, prev_router, HA_BU_DELAY, INITIAL_BINDACK_TIMEOUT,
                MAX_BINDACK_TIMEOUT, 1,
                MIPV6_BU_F_HOME, plen, ROUTER_BU_DEF_LIFETIME, NULL);

		}
}  

static int mipv6_cn_bu_send(struct mipv6_bul_entry *entry)
{
	struct in6_addr homeaddr;
	int plength;

	if(entry->flags & MIPV6_BU_F_HOME) return OP_SUCC;

	mipv6_get_home_addr(NULL, &homeaddr, &plength);

	/*
	 * A lifetime of 30 seconds is used and a wait of 1 sec before using
	 * an empty packet.
	 */
	DEBUG((DBG_INFO,
	       "Sending REGISTRATION to CN:"));
	debug_print_addr(DBG_INFO, &entry->cn_addr);

	mipv6_send_upd_option(
		&homeaddr, &entry->cn_addr, CN_BU_DELAY, INITIAL_BINDACK_TIMEOUT, MAX_BINDACK_TIMEOUT, 1,
		MIPV6_BU_F_ACK, plength, CN_BU_DEF_LIFETIME, NULL);

	return OP_SUCC;
}

static int mipv6_cn_bu_send_dereg(struct mipv6_bul_entry *entry)
{
	struct in6_addr homeaddr;
	int plength;

	if(entry->flags & MIPV6_BU_F_HOME) return OP_SUCC;

	mipv6_get_home_addr(NULL, &homeaddr, &plength);

	DEBUG((DBG_INFO, "Sending DEREGISTRATION to CN:"));
	debug_print_addr(DBG_INFO, &entry->cn_addr);
	mipv6_send_upd_option(
		&homeaddr, &entry->cn_addr, CN_BU_DELAY, INITIAL_BINDACK_TIMEOUT, MAX_BINDACK_TIMEOUT, 1,
		MIPV6_BU_F_ACK, plength, 0, NULL);

	return OP_SUCC;
}

int mipv6_mn_get_homeagent(struct in6_addr *ha_addr)
{
	if (ipv6_addr_type(&home_agent) != IPV6_ADDR_LOOPBACK) {
		ipv6_addr_copy(ha_addr, &home_agent);
		return 0;
	}

	return -1;
}

int __init mipv6_initialize_mn(void)
{
	DEBUG_FUNC();
	
#ifdef CONFIG_SYSCTL
	mipv6_mn_sysctl_register();
#endif

	/* Set initial home agent address to loopback ::1 until we
	 * receive the real home agent from sysctl.
	 */
	ipv6_addr_set(&home_agent, 0, 0, 0, (int)htonl(1));
	return 0;
}

void mipv6_shutdown_mn(void)
{

#ifdef CONFIG_SYSCTL
	mipv6_mn_sysctl_unregister();
#endif
	DEBUG_FUNC();
}

