/*
 *      Mobile IPv6-related Destination Options processing
 *
 *      Authors:
 *      Toni Nykanen <tpnykane@cc.hut.fi>
 *
 *      Marko Myllynen <myllynen@lut.fi> - Prefix List, DAD
 *
 *      $Id: procrcv.c,v 1.22.2.1 2000/10/27 10:43:12 mm Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */


/*
  sub-options:
  BU:
    Unique Identifier Sub-Option
    Alternate Care-of Address Sub-Option

  BA:

  BR:
    Unique Identifier Sub-Option

  HA:
*/

#define __NO_VERSION__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/types.h>

#include <linux/in6.h>
#include <linux/spinlock.h>
#include <linux/skbuff.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/addrconf.h>
#include <net/mipglue.h>

#include "dstopts.h"
#include "mipv6.h"

#include "bcache.h"
#include "bul.h"
#include "sendopts.h"
#include "ha.h"
#include "access.h"
#include "stats.h"
#include "mn.h"

#include "debug.h"


/* module ID for this source module */
static char module_id[] = "mipv6/procrcv";

#define DISCARD -1
#define FALSE 0
#define TRUE 1



#define BU_PTR 0
#define BA_PTR 1
#define BR_PTR 2
#define HADDR_PTR 3
#define SKB_PTR 4
#define MAX_PTRS 5


static void * ptrs[MAX_PTRS] = { NULL };
static rwlock_t destopt_lock = RW_LOCK_UNLOCKED;
static unsigned long destopt_lock_flags;

/*
 * these print the contents of several destination options.
 */
void dump_bu(__u8 opt, __u8 ack, __u8 home, __u8 router, __u8 dad,
	     __u8 plength, __u16 sequence, __u32 lifetime,
	     struct mipv6_subopt_info sinfo)
{
	DEBUG((DBG_INFO, "Binding Update Destination Option:"));

	DEBUG((DBG_INFO, "Option type: %d", opt));
	if (ack)
		DEBUG((DBG_INFO, "Binding ack requested."));
	if (home)
		DEBUG((DBG_INFO, "Home registration bit is set."));
	if (router)
		DEBUG((DBG_INFO, "Router bit is set."));
	if (dad)
		DEBUG((DBG_INFO, "DAD bit is set."));
	DEBUG((DBG_INFO, "Prefix length: %d", plength));
	DEBUG((DBG_INFO, "Sequence number: %d", sequence));
	DEBUG((DBG_INFO, "Lifetime: %d", lifetime));

	if (sinfo.fso_flags != 0) {
		DEBUG((DBG_INFO, "BU contains the following sub-options:"));
		mipv6_print_subopt_info(&sinfo);
	} else {
		DEBUG((DBG_INFO, "BU has no sub-options"));
	}
}

void dump_ba(__u8 opt, __u8 status, __u16 sequence, __u32 lifetime,
	     __u32 refresh, struct mipv6_subopt_info sinfo)
{
	DEBUG((DBG_INFO, "Binding Ack Destination Option:"));

	DEBUG((DBG_INFO, "Option type: %d", opt));
	DEBUG((DBG_INFO, "Status: %d", status));
	DEBUG((DBG_INFO, "Sequence number: %d", sequence));
	DEBUG((DBG_INFO, "Lifetime: %d", lifetime));

	if (sinfo.fso_flags != 0) {
		DEBUG((DBG_INFO, "BA contains the following sub-options:"));
		mipv6_print_subopt_info(&sinfo);
	} else {
		DEBUG((DBG_INFO, "BA has no sub-options"));
	}
}

void dump_br(__u8 opt, struct mipv6_subopt_info sinfo)
{
	DEBUG((DBG_INFO, "Binding Req Destination Option:"));
	DEBUG((DBG_INFO, "Option type: %d", opt));

	if (sinfo.fso_flags != 0) {
		DEBUG((DBG_INFO, "BR contains the following sub-options:"));
		mipv6_print_subopt_info(&sinfo);
	} else {
		DEBUG((DBG_INFO, "BR has no sub-options"));
	}
}

/*
 * lock the processing of Destination Options of the packet, for the
 * following reason:
 *
 * (DRAFT11, 8.1)
 *  "Packets sent by a mobile node while away from home generally include
 *   a Home Address option.  When any node receives a packet containing
 *   a Home Address option, it MUST process the option in a manner
 *   consistent with copying the Home Address field from the Home Address
 *   option into the IPv6 header, replacing the original value of the
 *   Source Address field there.  However, any actual modifications to
 *   the Source Address field in the packet's IPv6 header MUST not be
 *   performed until after all processing of other options contained in
 *   this same Destination Options extension header is completed."
 *
 * And, since the IPv6 code processes Destination Options separately,
 * the actual processing must be delayed, and executed in sequence without
 * any intervention by another thread/process/whatever.
 *
 * The other way to achieve this would require some additional buffer for
 * the options inside the sk_buff, in such a place that in case the buffer
 * get copied (cow'ed for example), the pointers get copied also. Then,
 * no locking would be needed..
 */

/* TODO (?): remove the comments and test... */
static void lock_procrcv(void)
{
	DEBUG_FUNC();
	read_lock_irqsave(&destopt_lock, destopt_lock_flags);
}


static void unlock_procrcv(void)
{
	DEBUG_FUNC();
	read_unlock_irqrestore(&destopt_lock, destopt_lock_flags);
}


/*
 * initialize pointers to NULL.
 */
static void initialize_ptrs(void)
{
	DEBUG_FUNC();

	memset(ptrs, 0, sizeof(ptrs));
}


/* only called for home agent. */
static int mipv6_is_authorized(struct in6_addr* haddr)
{
	DEBUG((DBG_WARNING, "%s: mipv6_is_authorized: %d", module_id,
	       mipv6_is_allowed_home_addr(mipv6_mobile_node_acl, haddr)));

	return mipv6_is_allowed_home_addr(mipv6_mobile_node_acl, haddr);
}

/* This is 64 since we have defined CONFIG_IPV6_EUI64 and CONFIG_IPV6_NO_PB */
#define PREFIX_LENGTH 64

static struct inet6_ifaddr *is_on_link_ipv6_address(struct in6_addr* mn_haddr)
{
	struct rt6_info *rt;
	struct inet6_dev *in6_dev = NULL;
	struct inet6_ifaddr *ifp = NULL;

	rt = rt6_lookup(mn_haddr, NULL, 0, 0);

	if (rt) {
		in6_dev = in6_dev_get(rt->rt6i_dev);
		if (in6_dev)
			ifp = in6_dev->addr_list;
	}

	while (ifp != NULL) {
		spin_lock_bh(&ifp->lock);
		if (mipv6_prefix_compare(&ifp->addr, mn_haddr,
					PREFIX_LENGTH) == TRUE &&
			!(ifp->flags&IFA_F_TENTATIVE)) {
				spin_unlock_bh(&ifp->lock);
				DEBUG((DBG_INFO, "%s: Home Addr Opt: on-link",
					module_id));
				in6_dev_put(in6_dev);
				return ifp;
		}
		spin_unlock_bh(&ifp->lock);
		ifp = ifp->if_next;
	}

	DEBUG((DBG_WARNING, "%s: Home Addr Opt NOT on-link", module_id));
	if (in6_dev)
		in6_dev_put(in6_dev);
	return ifp;
}

static int dad_check(void)
{
	/* TODO. see ipv6/addrconf.c */

	DEBUG((DBG_WARNING, "%s: TODO::: DAD", module_id));
	return -1;
}

/*
 * Lifetime checks. ifp->valid_lft >= ifp->prefered_lft always (see addrconf.c)
 * Returned value is in seconds.
 */

static __u32 get_min_lifetime(struct inet6_ifaddr *ifp, __u32 lifetime)
{
	__u32 rem_lifetime;
	unsigned long now = jiffies;

	if (ifp->valid_lft == 0) {
		rem_lifetime = lifetime;
	} else {
		__u32 valid_lft_left =
			ifp->valid_lft - time_after(now, ifp->tstamp) / HZ;
		rem_lifetime = (valid_lft_left > lifetime)
		? lifetime : valid_lft_left;
	}

	if (ifp->prefered_lft != 0) {
		__u32 prefered_lft_left =
			ifp->prefered_lft - time_after(now, ifp->tstamp) / HZ;
		rem_lifetime = (prefered_lft_left > rem_lifetime)
			? rem_lifetime : prefered_lft_left;
	}

	return rem_lifetime;
}

/*
 * only called by mipv6_dstopt_process_bu. could be inlined.
 *
 * performs the actual processing of a binding update the purpose
 * of which is to add a binding.
 */
static void mipv6_bu_add(struct in6_addr *saddr, struct in6_addr *daddr,
			 struct in6_addr *haddr,
			 struct in6_addr *coa,
			 int ack, int home, int router, int dad, int plength,
			 __u16 sequence, __u32 lifetime)
{
	__u8 ba_status = SUCCESS;
	__u32 ba_lifetime = 0;
	__u32 ba_refresh = 0;
	__u32 rem_lifetime = 0;
	__u64 maxdelay = 0;

	DEBUG_FUNC();

	if (home == 0) {
		/* Not "home registration", use 0 as plength */
		plength = 0;
		if (mipv6_bcache_add(daddr, haddr, coa, lifetime, plength,
				     sequence, router, CACHE_ENTRY) != 0) {
			DEBUG((DBG_ERROR, "%s: mipv6_bu_add: binding failed.",
			       module_id));
			ba_status = INSUFFICIENT_RESOURCES;
		} else
			DEBUG((DBG_INFO, "%s: mipv6_bu_add: binding succeeded",
			       module_id));

		DEBUG((DBG_DATADUMP, "%s: mipv6_bu_add: home_addr:", module_id));
		debug_print_addr(DBG_DATADUMP, haddr);
		DEBUG((DBG_DATADUMP, "%s: mipv6_add: coa:", module_id));
		debug_print_addr(DBG_DATADUMP, coa);
		DEBUG((DBG_DATADUMP, "%s:mipv6_bu_add:lifet: %d, plen: %d, seq: %d",
		       module_id, lifetime, plength, sequence));

	} else if ((!mipv6_is_ha) && (home)) {
		ack = TRUE;
		ba_status = HOME_REGISTRATION_NOT_SUPPORTED;
	} else if (mipv6_is_ha) {
		struct inet6_ifaddr *ifp = NULL;
		/*
		  ADMINISTRATIVELY_PROHIBITED could be set..
		*/

		ifp = is_on_link_ipv6_address(haddr);

		if (mipv6_is_authorized(haddr) != TRUE || ifp == NULL) {
			ack = TRUE;
			ba_status = NOT_HOME_SUBNET;
		} else if (plength != 0 && plength != PREFIX_LENGTH) {
			ack = TRUE;
			ba_status = INCORRECT_INTERFACE_ID_LEN;
		} else {
			int send_nd = FALSE;
				  
			if (dad != 0 && ack != 0) {
				if (dad_check() != 0) {
					ba_status = DUPLICATE_ADDR_DETECT_FAIL;
					goto out;
				}
			}

			rem_lifetime = get_min_lifetime(ifp, lifetime);

			if (plength != 0) {
				DEBUG((DBG_INFO, "%s: mipv6_bu_add:plength != 0",
					module_id));
				ifp = ifp->idev->addr_list;
				while (ifp != NULL) {
					rem_lifetime = get_min_lifetime(ifp,
								rem_lifetime);
					ifp = ifp->if_next;
				}
			}

			rem_lifetime = mipv6_lifetime_check(rem_lifetime);

			if(mipv6_bcache_exists(haddr) != HOME_REGISTRATION)
				send_nd = TRUE;

			if (send_nd) {
				if (mipv6_proxy_nd(haddr, plength,
						   router) == 0)
					DEBUG((DBG_INFO,"%s:bu_add:proxy_nd succ",
					       module_id));
				else
					DEBUG((DBG_WARNING,"%s:bu_add:proxy_nd fail",
					       module_id));
			}
			if (mipv6_bcache_add(daddr, haddr, coa, rem_lifetime,
					     plength, sequence, router,
					     HOME_REGISTRATION) != 0) {
				DEBUG((DBG_WARNING, "%s:bu_add: home reg failed.",
				       module_id));

				/* ack SHOULD be set (says draft) */
				ack = TRUE;
				ba_status = INSUFFICIENT_RESOURCES;
			} else {
				DEBUG((DBG_INFO, "%s:bu_add: home reg succeeded.",
				       module_id));
				ba_lifetime = rem_lifetime;

				/*
				  TODO (?): Lifetime could be less than this:
				  (in fact, if the cache is not crash proof,
				  the value really really ought to be less!!)
				*/
				ba_refresh = rem_lifetime;
			}

			DEBUG((DBG_DATADUMP, "%s:bu_add: home_addr:", module_id));
			debug_print_addr(DBG_DATADUMP, haddr);
			DEBUG((DBG_DATADUMP, "%s: mipv6_add: coa:", module_id));
			debug_print_addr(DBG_DATADUMP, coa);
			DEBUG((DBG_DATADUMP, "%s:bu_add: lifet:%d, plen:%d, seq:%d",
			       module_id, rem_lifetime, plength, sequence));
		}
	}

out:
	if (ack) {
		DEBUG((DBG_INFO, "%s:mipv6_bu_add: sending ack (code=%d)",
		       module_id, ba_status));
                mipv6_send_ack_option(daddr, haddr, maxdelay, ba_status,
				      sequence, ba_lifetime, ba_refresh);
	}
}


/*
 * only called by mipv6_dstopt_process_bu. could well be inlined.
 *
 * performs the actual processing of a binding update the purpose
 * of which is to delete a binding.
 *
 */
static void mipv6_bu_delete(struct in6_addr *saddr, struct in6_addr *daddr,
			    struct in6_addr *haddr, struct in6_addr *coa,
			    int ack, int home, int router, int plength,
			    __u16 sequence, __u32 lifetime)
{
	__u8 ba_status = SUCCESS;
	__u32 ba_lifetime = 0;
	__u32 ba_refresh = 0;
	__u64 maxdelay = 0;

	DEBUG_FUNC();

	if (home == 0) {
		/* Primary Care-of Address Registration */

                if (mipv6_bcache_exists(haddr) == CACHE_ENTRY) {

			if (mipv6_bcache_delete(haddr, CACHE_ENTRY) != 0)
				DEBUG((DBG_ERROR, "%s:bu_delete: delete failed.",
				       module_id));
			else DEBUG((DBG_INFO, "%s:bu_delete: delete succeeded.",
				    module_id));

			DEBUG((DBG_DATADUMP, "%s: mipv6_bu_delete: home_addr:",
			       module_id));
			debug_print_addr(DBG_DATADUMP, haddr);
			DEBUG((DBG_DATADUMP, "%s: mipv6_delete: coa:", module_id));
			debug_print_addr(DBG_DATADUMP, coa);
			DEBUG((DBG_DATADUMP, "%s:bu_del: lifet:%d, plen:%d, seq:%d",
			       module_id, lifetime, plength, sequence));

		} else {
			DEBUG((DBG_WARNING, "%s:bu_delete: entry is not in cache",
			       module_id));
			ba_status = REASON_UNSPECIFIED;
		}

        } else if (mipv6_is_ha && (home)) {
		/* Primary Care-of Address Deregistration */

		if (mipv6_bcache_exists(haddr) == HOME_REGISTRATION) {

			/* remove proxy_nd */
			if (mipv6_proxy_nd_rem(haddr, plength, router) == 0)
				DEBUG((DBG_INFO, "%s:bu_delete:proxy_nd succ.",
				       module_id));
			else
				DEBUG((DBG_WARNING, "%s:bu_delete: proxy_nd failed.",
      				       module_id));

		      	if (mipv6_bcache_delete(haddr,
						HOME_REGISTRATION) != 0)
				DEBUG((DBG_ERROR, "%s: bu_delete: delete failed.",
				       module_id));
			else
				DEBUG((DBG_INFO, "%s: bu_delete: delete succ.",
				       module_id));
		       
			ba_lifetime = 0;
			ba_refresh = 0;

			DEBUG((DBG_DATADUMP, "%s: bu_delete: home_addr:", module_id));
			debug_print_addr(DBG_DATADUMP, haddr);
			DEBUG((DBG_DATADUMP, "%s: mipv6_delete: coa:", module_id));
			debug_print_addr(DBG_DATADUMP, coa);
			DEBUG((DBG_DATADUMP, "%s:bu_delete:lifet:%d,plen:%d,seq:%d",
			       module_id, lifetime, plength, sequence));

		} else {
			ack = TRUE;
			ba_status = NOT_HA_FOR_MN;
		}
	} else {
		/*
		  TODO (?):node is not a Home Agent, but the sender believes so.
		  The draft doesn't tell if, when _deleting_, sender should
		  be informed with a code 132 (Home Reg not supported.)
		*/

		ba_status = REASON_UNSPECIFIED;
	}

	if (ack) {
                mipv6_send_ack_option(daddr, saddr, maxdelay, ba_status,
				      sequence, ba_lifetime, ba_refresh);
	}
}


/*
 * called by mipv6_finalize_dstopt_rcv(), in case
 * a reference was saved before.
 *
 * at this stage it is known that the packet contains a valid
 * Home Address Option.
 */
static int mipv6_dstopt_process_bu(struct in6_addr *saddr,
				   struct in6_addr *daddr,
				   struct in6_addr *haddr)
{
	__u8 flags;
	__u8 ack, home, router, dad, plength;
	__u16 sequence;
	__u32 lifetime;
	struct mipv6_subopt_info sinfo;
	struct in6_addr coa;

	DEBUG_FUNC();

	if (!ptrs[BU_PTR]) {
		DEBUG((DBG_INFO, "mipv6_dstopt_process_bu: BU does not exist."));
		return 0;
	}

	if (mipv6_parse_bindupdate((__u8 *)ptrs[BU_PTR], &flags,
				   &plength, &sequence, &lifetime,
				   &sinfo) != 0) {
		DEBUG((DBG_ERROR, "%s:dstopt_process_bu: couldn't parse bu:0x%x.",
		       (unsigned int) ( *(__u8 *)ptrs[BU_PTR]), module_id));

		return DISCARD;
	}

	ack = !!(flags & MIPV6_BU_F_ACK);
	home = !!(flags & MIPV6_BU_F_HOME);
	router = !!(flags & MIPV6_BU_F_ROUTER);
	dad = !!(flags & MIPV6_BU_F_DAD);

	/* dump the contents of the binding update. */
	dump_bu(*(__u8 *)ptrs[BU_PTR], ack, home, router, dad, plength,
		sequence, lifetime, sinfo);

	/* care-of address: */
	if (sinfo.fso_alt_coa) ipv6_addr_copy(&coa, &sinfo.alt_coa);
	else ipv6_addr_copy(&coa, saddr);

	if ((lifetime != 0) && (ipv6_addr_cmp(&coa, haddr) != 0)) {
		DEBUG((DBG_INFO, "%s:dstopt_process_bu: calling bu_add.",
		       module_id));
		mipv6_bu_add(saddr, daddr, haddr, &coa, ack, home, router, dad,
			     plength, sequence, lifetime);

	} else if ((lifetime == 0) || (ipv6_addr_cmp(&coa, haddr) == 0)) {
		DEBUG((DBG_INFO, "%s:dstopt_process_bu: calling bu_delete.",
		       module_id));
		mipv6_bu_delete(saddr, daddr, haddr, &coa, ack, home, router,
				plength, sequence, lifetime);
	}

	return 0;
}


/*
 * called by mipv6_finalize_dstopt_rcv(), in case
 * a reference was saved before.
 */
static int mipv6_dstopt_process_br(struct in6_addr *daddr)
{
	struct mipv6_subopt_info sinfo;
	int lifetime = 0;
	struct mipv6_bul_entry *entry;
	struct in6_addr home_agent;

	DEBUG_FUNC();

	if (ptrs[BR_PTR]) {
		debug_print_buffer(DBG_INFO, ptrs[BR_PTR], 10);

		if (mipv6_parse_bindrq((__u8 *)ptrs[BR_PTR], &sinfo) != 0) {
			DEBUG((DBG_ERROR,"%s: couldn't parse br :0x%x.",
			       module_id,
			       (unsigned int) ( *(__u8 *)ptrs[BR_PTR])));

			return DISCARD;
		}

		if (mipv6_mn_get_homeagent(&home_agent)) {
			DEBUG((DBG_ERROR, "%s: no home agent found.",
			       module_id));
			return -1; /* TODO (?): should do something else? */
		}
		entry = (struct mipv6_bul_entry *)mipv6_bul_get(&home_agent);

		lifetime = (entry->expire - jiffies) / HZ < 
			CN_BU_DEF_LIFETIME ? 
			(entry->expire - jiffies) / HZ : 
			CN_BU_DEF_LIFETIME;

		if (mipv6_send_upd_option(&entry->home_addr, daddr, CN_BU_DELAY,
					  INITIAL_BINDACK_TIMEOUT, MAX_BINDACK_TIMEOUT,
					  0, MIPV6_BU_F_ACK, 0, lifetime,
					  &sinfo))
			DEBUG((DBG_ERROR, "%s: BU send failed.", module_id));

		/* dump the contents of the Binding Request. */
		dump_br(*(__u8 *)ptrs[BR_PTR], sinfo);
	}
	return 0;
}


/*
 * called by mipv6_finalize_dstopt_rcv(), in case
 * a reference was saved before.
 *
 * If an invalid BA is encountered, not the whole packet will
 * be discarded.
 */
static int mipv6_dstopt_process_ba(struct in6_addr* coa)
{
	__u8 status;
	__u16 sequence;
	__u32 lifetime, refresh;
	struct mipv6_subopt_info sinfo;

	DEBUG_FUNC();

	if (ptrs[BA_PTR]) {
		if (mipv6_parse_bindack((__u8 *)ptrs[BA_PTR], &status,
					&sequence, &lifetime, &refresh,
					&sinfo) != 0) {

			DEBUG((DBG_ERROR, "%s:process_ba:couldn't parse ba:0x%x.",
			       module_id,
			       (unsigned int) ( *(__u8 *)ptrs[BA_PTR])));
			return 0;
		}
		DEBUG((DBG_INFO, "%s: process_ba: calling mipv6_ack_rcvd",
		       module_id));

		if (status >= REASON_UNSPECIFIED) {
			DEBUG((DBG_WARNING, "%s:ack status indicates error: %d",
			       module_id, status));
			mipv6_ack_rcvd(coa, sequence,
				       lifetime, refresh,
				       STATUS_REMOVE);
		} else
			mipv6_ack_rcvd(coa, sequence,
				       lifetime, refresh,
				       STATUS_UPDATE);

		/* dump the contents of the Binding Acknowledgment. */
		dump_ba(*(__u8 *)ptrs[BA_PTR], status, sequence, lifetime,
			refresh, sinfo);
	}
	return 0;
}

/*
 * Called as the processing of Destination Options begins.
 */
void mipv6_initialize_dstopt_rcv(struct sk_buff *skb)
{
	DEBUG_FUNC();

	initialize_ptrs();
	ptrs[SKB_PTR] = (void *)skb;
	lock_procrcv();
}


/*
 * These save the pointer to the Destination Option being processed.
 */
int mipv6_update_bindupdate_rcv_state(struct sk_buff *skb, __u8 * ptr)
{
	DEBUG_FUNC();

	MIPV6_INC_STATS(n_bu_rcvd);
	ptrs[BU_PTR] = (void *)ptr;

	return TRUE;
}

int mipv6_update_bindack_rcv_state(struct sk_buff *skb, __u8 * ptr)
{
	DEBUG_FUNC();

        MIPV6_INC_STATS(n_ba_rcvd);
 	ptrs[BA_PTR] = (void *)ptr;

	return TRUE;
}

int mipv6_update_bindrq_rcv_state(struct sk_buff *skb, __u8 * ptr)
{
	DEBUG_FUNC();

        MIPV6_INC_STATS(n_br_rcvd); 
	ptrs[BR_PTR] = (void *)ptr;

	return TRUE;
}

int mipv6_update_homeaddr_rcv_state(struct sk_buff *skb, __u8 * ptr)
{
	DEBUG_FUNC();

        MIPV6_INC_STATS(n_ha_rcvd);
	ptrs[HADDR_PTR] = (void *)ptr;

	return TRUE;
}


/*
 * Performs the processing of the destination options saved before.
 *
 * Home Address Option,
 * Binding Update Option,
 * Binding Acknowledgements Option, and
 * Binding Request Option will be processed, and
 * corresponding actions taken, based on the configuration of
 * the node, and the contents of the Destination Option
 *
 */
int mipv6_finalize_dstopt_rcv(int process)
{
	int ret = DISCARD;
	struct in6_addr home_addr; /* initialization?? */

	struct mipv6_subopt_info haddr_sinfo;
	struct sk_buff *skb;
	struct in6_addr *saddr;
	struct in6_addr *daddr;

	DEBUG_FUNC();

	/* process is 0, if there wasn't any destination option. 
        if (!process) {
		
		DEBUG((DBG_INFO, "%s:finalize_dstopt_rcv: nothing to process.",
		       module_id));

		unlock_procrcv();
		return 0;
		}*/	

	/* TODO (?):actually, fetching of these parameters could well be
	   postponed until the actual need. optimization, if you like. */
	skb = (struct sk_buff *)ptrs[SKB_PTR];
	saddr = &(skb->nh.ipv6h->saddr);
	daddr = &(skb->nh.ipv6h->daddr);

	/* MN may receive BR's */
        if (mipv6_is_mn && (mipv6_dstopt_process_br(saddr)))
		;
	/* MN may receive BA's */
        if (mipv6_is_mn && (mipv6_dstopt_process_ba(saddr)))
		;

        /* If there is no Home Address Option, there MUST NOT
	   be Binding Update Option either. */
	if (!ptrs[HADDR_PTR]) {
		if (ptrs[BU_PTR]) {
			DEBUG((DBG_ERROR, "%s:finalize: no Home Address Option"
			       " with BU. discard.", module_id));
			unlock_procrcv();
			return DISCARD;
		}
		unlock_procrcv();
		return 0;
	}

	if (mipv6_parse_homeaddr((__u8 *)ptrs[HADDR_PTR], &home_addr,
				 &haddr_sinfo) != 0) {

		DEBUG((DBG_WARNING, "%s:finalize_dstopt_rcv: invalid home addr opt.",
		       module_id));

		ret = DISCARD;

	} else { /* we have a valid Home Address Option. */

		/* process BU. */
		/* Return nonzero if an invalid Binding Update Option. */

		if (mipv6_dstopt_process_bu(saddr, daddr, &home_addr) != 0) {

			DEBUG((DBG_ERROR, "%s:invalid BU option.", module_id));
			ret = DISCARD;

		} else { /* everything's fine! */

			/* change the addresses */
			DEBUG((DBG_INFO, "%s:finalize_dstopt_rcv: replacing source"
			       " address of the packet.",
			       module_id));
			ipv6_addr_copy(saddr, &home_addr);
			ret = 0;
		}
        }

	unlock_procrcv();
	DEBUG((DBG_INFO, "%s: mipv6_finalize_dstopt_rcv: that's it.", module_id));

	/*  let mn.c check if packet was tunneled and send a BU to CN  */
	mipv6_check_tunneled_packet(skb);

	return ret;
}

void mipv6_initialize_procrcv(void)
{
        DEBUG_FUNC();

        /*  Set kernel hooks  */
        MIPV6_SETCALL(mipv6_initialize_dstopt_rcv,
                      mipv6_initialize_dstopt_rcv);
        MIPV6_SETCALL(mipv6_handle_bindupdate,
                      mipv6_update_bindupdate_rcv_state);
        MIPV6_SETCALL(mipv6_handle_bindack, mipv6_update_bindack_rcv_state);
        MIPV6_SETCALL(mipv6_handle_bindrq, mipv6_update_bindrq_rcv_state);
        MIPV6_SETCALL(mipv6_handle_homeaddr, mipv6_update_homeaddr_rcv_state);
        MIPV6_SETCALL(mipv6_finalize_dstopt_rcv, mipv6_finalize_dstopt_rcv);
}

/* these are, in fact, already nullified, thanks to a call to */
/* mipv6_invalidate_calls. TODO: A safe order in cleaning up would be nice.. */
void mipv6_shutdown_procrcv(void)
{
        DEBUG_FUNC();

        /*  Remove kernel hooks  */
        MIPV6_SETCALL(mipv6_initialize_dstopt_rcv, NULL);
        MIPV6_SETCALL(mipv6_handle_bindupdate, NULL);
        MIPV6_SETCALL(mipv6_handle_bindack, NULL);
        MIPV6_SETCALL(mipv6_handle_bindrq, NULL);
        MIPV6_SETCALL(mipv6_handle_homeaddr, NULL);
        MIPV6_SETCALL(mipv6_finalize_dstopt_rcv, NULL);
}
