/*
 *      Router List Operations
 *
 *      Authors:
 *      Henrik Petander                <lpetande@cc.hut.fi>
 *     
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *      $Id: router.c,v 1.11 2000/10/19 14:05:22 henkku Exp $
 *
 *      Contains functions for handling the default router list, which 
 *      movement detection uses
 *      for avoiding loops etc. 
 * 
 */

#include <linux/kernel.h>
#include <linux/version.h>

#include <linux/autoconf.h>

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/in6.h>
#include <linux/route.h>
#include <linux/init.h>

#include <linux/if_arp.h>
#include <linux/ipv6.h>
#include <linux/icmpv6.h>
#include <linux/init.h>

#include <net/sock.h>
#include <net/snmp.h>

#include <net/ipv6.h>
#include <net/protocol.h>
#include <net/ndisc.h>
#include <net/ip6_route.h>
#include <net/addrconf.h>
#include "mdetect.h"
#include "debug.h"
#include "mipv6.h"
struct router *head=NULL;
/* searches for a specific router or any router that is reachable, 
 * if address is NULL. Also deletes obsolete routers.
 */
struct router *iterate_routers(struct in6_addr *search_addr, struct router *curr_router){

	struct router *curr=NULL,*prev=NULL, *tmp=NULL;
	DEBUG_FUNC();
	prev=head;

	for(curr=head;curr!=NULL;){
		if((curr->state==NOT_REACHABLE &&  
		   curr->last_ra_rcvd < jiffies - R_TIME_OUT - 
		   curr->interval && curr->interval!=0) || 
		   curr->last_ra_rcvd < jiffies - curr->lifetime  ){ 
			/* Clean the list of routers not reachable in the last 30 secs or 
			   whose lifetime has expired */
			if(curr_router==curr){
				/* TODO: stop sending bus at this point and resume normal opertaion 
				   only after obtaining a new default router */
				/*curr_router=NULL;*/
				DEBUG((DBG_INFO,
				       "curr_router not reachable in %ds", R_TIME_OUT));
				prev=curr;
				curr=curr->next;
			}
			else{
				tmp=curr;
				/*coa_del(tmp->dev,&tmp->raddr);*/
				if(curr!=head)
					prev->next=curr->next;
				else
					head=curr->next;
				curr=curr->next;
				DEBUG((DBG_INFO, "Iterate: deleting router:"));
				
				debug_print_addr(DBG_INFO, &tmp->ll_addr);
				kfree((void *)tmp); 
			}
		}
		else{
			
			DEBUG((DBG_DATADUMP,
			       "Iterate: curr->last_ra_rcvd %d curr->state %d, from:", 
			       curr->last_ra_rcvd, curr->state));
			debug_print_addr(3, &curr->ll_addr);
			if(search_addr!=NULL){
				if(!ipv6_addr_cmp(search_addr, &curr->ll_addr)){
					DEBUG((DBG_INFO,
					       "Iterate routers found a match"));
					return curr;
				}
			}
			else
				if(curr->state==ROUTER_REACHABLE){
					if(curr->last_ra_rcvd + curr->interval < jiffies)
						curr->state=NOT_REACHABLE;
					else
						return curr;
				}
			prev=curr;
			curr=curr->next;
		}
	}

return NULL;
}

/* 
 * Adds router as the first item on the list
 */
void head_add(struct router *new) {
	DEBUG_FUNC();
	new->next=head;
	head=new;
}

/*
 * creates a new router 
 */
struct router *new_router(struct in6_addr *ll_addr,struct in6_addr *raddr,struct net_device *dev, int plen, __u32 ival,int is_ha,int glob, unsigned long lifet ){
	struct router *rptr;
	DEBUG_FUNC();
	rptr = (struct router *) kmalloc(sizeof(struct router), GFP_ATOMIC);
	if(rptr==NULL){
		DEBUG((0, "mipv6 mdetect: couldn't allocate memory for new router"));
		return NULL;
	}
	if(ll_addr==NULL || raddr==NULL || dev==NULL){
		DEBUG((0,"Received NULL arguments"));
		return NULL;
	}
	ipv6_addr_copy(&rptr->ll_addr,ll_addr);
	ipv6_addr_copy(&rptr->raddr,raddr);

	rptr->dev=dev;
	rptr->pfix_len=plen;
	rptr->state=ROUTER_REACHABLE;
	rptr->last_ra_rcvd=jiffies;
	rptr->interval=ival;
	rptr->glob_addr=glob;
	rptr->is_ha=is_ha;
	rptr->lifetime=lifet;
	head_add(rptr);
	debug_print_addr(DBG_INFO, &rptr->ll_addr);
	return rptr;
}

/* Cleans up the list */
void list_free(void){
	struct router *tmp, *curr;
	DEBUG_FUNC();
	DEBUG((DBG_INFO, "Freeing the router list"));
	for(curr=head;curr!=NULL;){
		tmp=curr;
		/*	coa_del(tmp->dev,&tmp->raddr);*/
		curr=curr->next;
		debug_print_addr(DBG_INFO, &tmp->ll_addr);
		kfree((void *)tmp); 
	}
}








