/*
 *      Option sending and piggybacking module
 *
 *      Authors:
 *      Niklas Kmpe                <nhkampe@cc.hut.fi>
 *
 *      $Id: sendopts.c,v 1.9 2000/10/11 10:46:40 antti Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#define __NO_VERSION__
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/ipv6.h>
#include <linux/net.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <net/ipv6.h>

#include "mipv6.h"
#include "sendopts.h"
#include "dstopts.h"
#include "bul.h"
#include "stats.h"
#include "mdetect.h"
#include "debug.h"

#define OPT_UPD 0		/* Possible values for opt_type */
#define OPT_RQ 1
#define OPT_ACK 2

#define MAX_UPD_OPT_SIZE 0x30	/* Assumed upper limits for sizes of */
#define MAX_RQ_OPT_SIZE 0x30	/* destination options (size includes */
#define MAX_ACK_OPT_SIZE 0x30   /* padding + T + L + V [+ subopts]) */
#define MAX_HOMEADDR_OPT_SIZE 0x30

#define MAX_OPT_QUEUE_SIZE 64	/* Max send queue size (entries) */
#define HASH_TABLE_SIZE 32	/* Send queue hash table size */
#define MAX_OPT_HDR_SIZE 0x200	/* Maximum size destination options
				   header is allowed to grow to when
				   piggy-backing options */

#define RESEND_PIGGYBACK_MAXDELAY 500	/* milliseconds */

/* struct that desribes an entry in the send queue: */

struct opt_queue_entry {
	struct opt_queue_entry	*next_chain,	/* For hash table chaining */
				*prev_chain,
				*next_list,	/* For linked list and free_list */
		                *prev_list;	/* For linked list */
	unsigned int		hashkey;

	unsigned long		maxdelay_expires;  /* jiffies value when to
						      forget about piggy-backing
						      and send option in a
						      packet of its own */
	struct in6_addr	saddr, daddr;		   /* source and destination
						      addresses of option */
	/* Option data: */
	int			opt_type;	/* OPT_UPD, OPT_ACK or OPT_RQ */
	__u32			lifetime;	/* for BUs and BAs */
	__u32			refresh;	/* for BAs */
	__u16			sequence;	/* for BUs and BAs */
	__u8			exp;		/* for BUs */
	__u8			plength;	/* for BUs */
	__u8			status;		/* for BAs */
	__u8			flags;		/* for BUs */
	/* Sub-options */
	struct mipv6_subopt_info sinfo;
};

/* Array of option queue entries.
   The actual queue is referenced in two ways: as a hash table with the
   hash key computed from the option's destination address, and as a
   linked list sorted by max send delay expiration time. */
static struct opt_queue_entry *opt_queue_array;

/* Singly linked list of free option queue entries. */
static struct opt_queue_entry *free_list;

/* Hash table of option queue entries. Collisions are handled by chaining
   the entries as a doubly linked list using struct fields next_chain and
   prev_chain. */
static struct opt_queue_entry *opt_queue_hashtable[HASH_TABLE_SIZE];

/* Pointer to head of a doubly linked list of option queue entries in
   expiration order. List is linked using struct fields next_list and
   prev_list. */ 
static struct opt_queue_entry *opt_queue_list = NULL;

/* Timer which is set to call timer_handler() when the entry at the head
   of the queue has reached its maximum send delay time. */
static struct timer_list *timer = NULL;

/*
 *
 * Binding update sequence number assignment
 *
 */

static spinlock_t seqnum_lock = SPIN_LOCK_UNLOCKED;
static volatile __u16 next_sequence_number = 0;

static __inline__ __u16 get_new_sequence_number(void)
{
	__u16 seqnum;

	spin_lock(&seqnum_lock);
	seqnum = next_sequence_number++;
	spin_unlock(&seqnum_lock);
	return seqnum;
}

/*
 *
 * Option send queue management functions
 *
 */

/* Locking functions for exclusive access to the queue.
 * Queue should be locked
 *  - if its contents are examined or changed
 *  - if a queue entry is dereferenced by a pointer, the queue should
 *    be locked until the pointer is no longer dereferenced
 */

static spinlock_t queue_lock = SPIN_LOCK_UNLOCKED;

static __inline__ void lock_queue(void)
{
	DEBUG_FUNC();
	spin_lock(&queue_lock);
}

static __inline__ void unlock_queue(void)
{
	DEBUG_FUNC();
	spin_unlock(&queue_lock);
}


static __inline__ unsigned int calc_hash_key(struct in6_addr *daddr)
{
	return (daddr->s6_addr32[2] ^ daddr->s6_addr32[3]) % HASH_TABLE_SIZE;
}


/* Get an unused option queue entry from the free list.
 * Returns NULL if option queue has reached its maximum size.
 */
static struct opt_queue_entry *get_free_opt_queue_entry(void)
{
	struct opt_queue_entry *entry;

	if (free_list) {
		entry = free_list;
		free_list = entry->next_list;
		return entry;
	} else
		return NULL;
}


/* Add an option queue entry to the option queue.
 * Also reschedules the timer if needed.
 */
static void add_to_opt_queue(struct opt_queue_entry *entry)
{
	struct opt_queue_entry *chainhead, *listelem, *prevlistelem;

	/* Add to hash table, at the head of the hash chain. */

	entry->hashkey = calc_hash_key(&entry->daddr);
	chainhead = opt_queue_hashtable[entry->hashkey];
	entry->next_chain = chainhead;
	entry->prev_chain = NULL;
	opt_queue_hashtable[entry->hashkey] = entry;
	if (chainhead)
		chainhead->prev_chain = entry;

	/* Add to linked list, sorted by max send delay expiration time. */

	if (opt_queue_list == NULL) {
		/* Add to empty list */
		opt_queue_list = entry;
		entry->next_list = NULL;
		entry->prev_list = NULL;
		mod_timer(timer, entry->maxdelay_expires);
	} else {
		listelem = opt_queue_list;
		prevlistelem = NULL;
		while (listelem) {
			if (listelem->maxdelay_expires > entry->maxdelay_expires) {
				/* Add in front of found entry */
				listelem->prev_list = entry;
				if (prevlistelem)
					prevlistelem->next_list = entry;
				else {
					opt_queue_list = entry;
					mod_timer(timer, entry->maxdelay_expires);
				}
				entry->next_list = listelem;
				entry->prev_list = prevlistelem;
				break;	/* while */
			}
			prevlistelem = listelem;
			listelem = listelem->next_list;
		}
		if (listelem == NULL) {
			/* Add to tail of list */
			prevlistelem->next_list = entry;
			entry->next_list = NULL;
			entry->prev_list = prevlistelem;
		}
	}
}

/* Remove an entry from the option queue.
 * Also reschedules the timer if needed.
 */
static void remove_from_opt_queue(struct opt_queue_entry *entry)
{
	/* Remove from hash chain */
	
	if (entry->prev_chain)
		entry->prev_chain->next_chain = entry->next_chain;
	else
		opt_queue_hashtable[entry->hashkey] = entry->next_chain;

	if (entry->next_chain)
		entry->next_chain->prev_chain = entry->prev_chain;


	/* Remove from linked list */

	if (entry->prev_list)
		entry->prev_list->next_list = entry->next_list;
	else
		opt_queue_list = entry->next_list;

	if (entry->next_list)
		entry->next_list->prev_list = entry->prev_list;


	/* Add to free list */

	entry->next_list = free_list;
	free_list = entry;

	/* Reschedule timer */

	if (opt_queue_list) {
		if (opt_queue_list->maxdelay_expires > jiffies)
			mod_timer(timer, opt_queue_list->maxdelay_expires);
		else {
			DEBUG((DBG_WARNING, "attempted to schedule sendopts timer with a historical jiffies count!"));
			mod_timer(timer, jiffies + HZ);
		}
	} else
		del_timer(timer);
}


/* Gets first entry in queue with the given source and destination
 * addresses. Next entry can be got with get_next_opt_queue_entry().
 */
static __inline__ struct opt_queue_entry *get_first_opt_queue_entry(
	struct in6_addr *saddr, struct in6_addr *daddr)
{
	struct opt_queue_entry *entry;

	entry = opt_queue_hashtable[calc_hash_key(daddr)];
	while (entry) {
		if (ipv6_addr_cmp(daddr, &entry->daddr) == 0 &&
		    ipv6_addr_cmp(saddr, &entry->saddr) == 0)
			return entry;
		entry = entry->next_chain;
	}

	return NULL;
}


/* Returns next entry with given source and destination addresses.
 * current_entry = previous entry got with get_first_xxx or get_next_xxx.
 */
static __inline__ struct opt_queue_entry *get_next_opt_queue_entry(
	struct in6_addr *saddr, struct in6_addr *daddr,
	struct opt_queue_entry *current_entry)
{
	current_entry = current_entry->next_chain;
	while (current_entry) {
		if (ipv6_addr_cmp(daddr, &current_entry->daddr) == 0 &&
		    ipv6_addr_cmp(saddr, &current_entry->saddr) == 0)
			return current_entry;
		current_entry = current_entry->next_chain;
	}

	return NULL;
}

/*
 *
 * Functions for sending an empty packet to which any queued
 * destination options will be added
 *
 */

static struct socket *dstopts_socket = NULL;
extern struct net_proto_family inet6_family_ops;

static int alloc_dstopts_socket(void)
{
	struct net_proto_family *ops = &inet6_family_ops;
	struct sock *sk;
	int err;

	dstopts_socket = (struct socket *) sock_alloc();
	if (dstopts_socket == NULL) {
		DEBUG((DBG_CRITICAL, 
		       "Failed to create the IPv6 destination options socket."));
		return -1;
	}
	dstopts_socket->inode->i_uid = 0;
	dstopts_socket->inode->i_gid = 0;
	dstopts_socket->type = SOCK_RAW;

	if ((err = ops->create(dstopts_socket, NEXTHDR_NONE)) < 0) {
		DEBUG((DBG_CRITICAL,
		       "Failed to initialize the IPv6 destination options socket (err %d).",
		       err));
		sock_release(dstopts_socket);
		dstopts_socket = NULL; /* for safety */
		return err;
	}

	sk = dstopts_socket->sk;
	sk->allocation = GFP_ATOMIC;
	sk->net_pinfo.af_inet6.hop_limit = 254;
	sk->net_pinfo.af_inet6.mc_loop = 0;
	sk->prot->unhash(sk);

	/* To disable the use of dst_cache, 
	 *  which slows down the sending of BUs 
	 */
	sk->dst_cache=NULL; 
	return 0;
}

static void dealloc_dstopts_socket(void)
{
	if (dstopts_socket) sock_release(dstopts_socket);
	dstopts_socket = NULL; /* For safety. */
}

static int dstopts_getfrag(
	const void *data, struct in6_addr *addr,
	char *buff, unsigned int offset, unsigned int len)
{
	ASSERT(len == 0);
	return 0;
}

/* Send an empty packet to a given destination. This packet will
 * be looped back to mipv6_modify_xmit_packets which will add any
 * options queued for sending to the destination address.
 */
static int send_empty_packet(
	struct in6_addr *saddr,
	struct in6_addr *daddr)
{
	struct flowi fl;
	struct sock *sk = dstopts_socket->sk;

	fl.proto = NEXTHDR_NONE;
        fl.fl6_dst = daddr;
        fl.fl6_src = saddr;
	fl.fl6_flowlabel = 0;
	fl.oif = sk->bound_dev_if;
	fl.uli_u.data = 0;

	ip6_build_xmit(sk, dstopts_getfrag, NULL, &fl, 0, NULL, 255, MSG_DONTWAIT);
	
	return 0;
}	

/*
 *
 * Miscellaneous functions
 *
 */

/* Add a destination option to a destination options header.
 * Also updates binding update list if adding a binding update option.
 * Returns next free offset in header on success or -1 if
 * option could not fit in header.
 * offset = offset in header at which to add option
 * hdrsize = size of memory block allocated for header
 */
static int add_dstopt(
	struct ipv6_opt_hdr *hdr,
	struct opt_queue_entry *entry, int offset, int hdrsize)
{
	__u8 *opt, *optsize;
	int nextoffset;
	struct mipv6_bul_entry *bulentry;

	opt = (__u8 *) hdr;

	switch(entry->opt_type) {
		case OPT_RQ:
			if (offset + MAX_RQ_OPT_SIZE > hdrsize)
				return -1;
			nextoffset = mipv6_create_bindrq(opt, offset, &optsize);
			break;
		case OPT_ACK:
			if (offset + MAX_ACK_OPT_SIZE > hdrsize)
				return -1;
			nextoffset = mipv6_create_bindack(
				opt, offset,
				&optsize, entry->status, entry->sequence,
				entry->lifetime, entry->refresh);
			break;
		case OPT_UPD:
			if (offset + MAX_UPD_OPT_SIZE > hdrsize)
				return -1;
			nextoffset = mipv6_create_bindupdate(
				opt, offset,
				&optsize, entry->flags, entry->plength,
				entry->sequence, entry->lifetime,
				&entry->sinfo);
	
			/* Change state in BU list from NOT_SENT to
			   SEND_NOMORE if this is a BU not requiring
			   acknowledgement. */
			if (mipv6_is_mn && !(entry->flags & MIPV6_BU_F_ACK)) {
				bulentry = mipv6_bul_get(&entry->daddr);
				if (bulentry) {
					bulentry->state = SEND_NOMORE;
					mipv6_bul_put(bulentry);
				}
			}
			break;
		default:
			/* Invalid option type - entry has been corrupted */
			DEBUG((DBG_ERROR, "sendopts.c: corrupted queue entry in add_dstopt()"));
			nextoffset = offset;
	}

	if (nextoffset > hdrsize)
		/* Serious: wrote data past end of memory block allocated
		   for header. */
		DEBUG((DBG_ERROR, "sendopts.c: add_dstopt() option buffer overrun - memory corrupted!!!"));

	hdr->hdrlen = (nextoffset-1)/8;

	return nextoffset;
}


/* This function will take the first option from the queue (the one whose
 * maximum piggybacking waiting time will expire first) and send it.
 */
static int send_oldest_queued_opt(void)
{
	struct in6_addr saddr, daddr;

	lock_queue();

	if (opt_queue_list == NULL) {
		unlock_queue();
		return -1;
	}

	/* Just send an empty packet. The packet will be looped back
	 * to mipv6_modify_xmit_packets() which will add the destination
	 * options. */

	ipv6_addr_copy(&saddr, &opt_queue_list->saddr);
	ipv6_addr_copy(&daddr, &opt_queue_list->daddr);

	unlock_queue();

	send_empty_packet(&saddr, &daddr);

	return 0;
}

/*
 *
 * Timer related functions
 *
 */

/* The timer handler function
 */
static void timer_handler(unsigned long data)
{
	struct in6_addr saddr, daddr;
	int sent_something = 0;

	/* Pick all expired entries from head of queue. */
	while (1) {
		lock_queue();
		if (opt_queue_list && opt_queue_list->maxdelay_expires <= jiffies) {
			/* Send option now. Release lock before sending because
			 * mipv6_modify_xmit_packets will need to acquire it. */
			DEBUG((DBG_INFO,
			       "sendopts.c timer_handler() sending queued option in empty packet"));
			sent_something = 1;
			ipv6_addr_copy(&saddr, &opt_queue_list->saddr);
			ipv6_addr_copy(&daddr, &opt_queue_list->daddr);
			unlock_queue();
			send_empty_packet(&saddr, &daddr);
		} else {
			/* No more options to send now. */
			unlock_queue();
			break;
		}
	}

	if (!sent_something)
		DEBUG((DBG_INFO, "sendopts timer called but found no work to do"));
}


/*
 *
 * Callback handlers for binding update list
 *
 */

/* Return value 0 means keep entry, non-zero means discard entry. */


/* Callback for BUs not requiring acknowledgement
 */
static int bul_expired(struct mipv6_bul_entry *bulentry)
{
	/* Lifetime expired, delete entry. */
	DEBUG((DBG_INFO, "bul entry 0x%x lifetime expired, deleting entry", (int) bulentry));
	return 1;
}


/* Callback for BUs requiring acknowledgement with constant resending
 * scheme */
static int bul_resend_const(struct mipv6_bul_entry *bulentry)
{
	struct opt_queue_entry *entry;
	unsigned long now = jiffies;
	__u32 lifetime;
 
	DEBUG((DBG_INFO, "bul_resend_const(0x%x) resending binding update", (int) bulentry));

	/* If BU lifetime has expired, stop resending BU.
	 * Exception: If sending a de-registration, do not care about the
	 * lifetime value, as de-registrations are normally sent with
	 * a zero lifetime value. De-registrations are identified by coa ==
	 * home_addr.
	 */
	lifetime = (bulentry->expire > now) ? (bulentry->expire - now)/HZ : 0;
	if (lifetime == 0 && ipv6_addr_cmp(&bulentry->coa, &bulentry->home_addr)) {
		DEBUG((DBG_WARNING, "Binding lifetime expired while BU "
			"not yet acknowledged (bul_resend_const)"));
		return -1;
	}

	lock_queue();
	entry = get_free_opt_queue_entry();
	unlock_queue();
	if (entry == NULL) {
		/* Queue full, try again later.
		 * Don't try to flush queue because BUL is kept locked
		 * while this function is called and trying to
		 * send something from here may cause synchronization
		 * problems.
		 */
		DEBUG((DBG_INFO, "queue was full, trying again after 1 second"));
		bulentry->callback_time = now + HZ;
		return 0;
	}

	/* Build binding update and queue for sending */

	entry->opt_type = OPT_UPD;
	ipv6_addr_copy(&entry->saddr, &bulentry->home_addr);
	ipv6_addr_copy(&entry->daddr, &bulentry->cn_addr);
	entry->exp = 0;
	entry->flags = bulentry->flags;
	entry->plength = bulentry->prefix;
	bulentry->seq = get_new_sequence_number();
	entry->sequence = bulentry->seq;
	entry->lifetime = lifetime;

	entry->maxdelay_expires = now + RESEND_PIGGYBACK_MAXDELAY*HZ/1000;
	lock_queue();
	add_to_opt_queue(entry);
	unlock_queue();

	/* Schedule next retransmission */
	
	if (--bulentry->maxdelay > 0) {
		bulentry->callback_time = now + bulentry->delay*HZ;
		return 0;
	} else {
		DEBUG((DBG_INFO, "max send count reached, deleting"
		       "bul entry (0x%x)", (int) bulentry));
		return 1;
	}
}


/* Callback for BUs requiring acknowledgement with exponential resending
 * scheme */
static int bul_resend_exp(struct mipv6_bul_entry *bulentry)
{
	struct opt_queue_entry *entry;
	unsigned long now = jiffies;
	__u32 lifetime;
	
	DEBUG((DBG_INFO, "bul_resend_exp(0x%x) resending bu", (int) bulentry));

	/* If BU lifetime has expired, stop resending BU.
	 * Exception: If sending a de-registration, do not care about the
	 * lifetime value, as de-registrations are normally sent with
	 * a zero lifetime value. De-registrations are identified by coa ==
	 * home_addr.
	 */
	lifetime = (bulentry->expire > now) ? (bulentry->expire - now)/HZ : 0;
	if (lifetime == 0 && ipv6_addr_cmp(&bulentry->coa, &bulentry->home_addr)) {
		DEBUG((DBG_WARNING, "Binding lifetime expired while BU "
			"not yet acknowledged (bul_resend_exp)"));
		return -1;
	}

	lock_queue();
	entry = get_free_opt_queue_entry();
	unlock_queue();
	if (entry == NULL) {
		/* Queue full, try again later.
		 * Don't try to flush queue because BUL is kept locked
		 * while this function is called and trying to
		 * send something from here may cause synchronization
		 * problems.
		 */
		DEBUG((DBG_INFO, "queue was full, trying again after 1 second"));
		bulentry->callback_time = now + HZ;
		return 0;
	}

	/* Build binding update and queue for sending */

	entry->opt_type = OPT_UPD;
	ipv6_addr_copy(&entry->saddr, &bulentry->home_addr);
	ipv6_addr_copy(&entry->daddr, &bulentry->cn_addr);
	entry->exp = 1;
	entry->flags = bulentry->flags;
	entry->plength = bulentry->prefix;
	entry->sequence = bulentry->seq;
	entry->lifetime = lifetime;

	entry->maxdelay_expires = now + RESEND_PIGGYBACK_MAXDELAY*HZ/1000;
	lock_queue();
	add_to_opt_queue(entry);
	unlock_queue();

	/* Schedule next retransmission */

	if (bulentry->delay < bulentry->maxdelay) {
		bulentry->callback_time = now + bulentry->delay*HZ;
		bulentry->delay = 2*bulentry->delay;
		return 0;
	} else {
		DEBUG((DBG_INFO, "max ack wait time reached, deleting bul entry (0x%x)", (int) bulentry));
		return 1;
	}
}

/* Callback for sending a registration refresh BU
 */
static int bul_refresh(struct mipv6_bul_entry *bulentry)
{
	struct opt_queue_entry *entry;
	unsigned long now = jiffies;
	
	/* Refresh interval passed, send new BU */
	DEBUG((DBG_INFO, "bul entry 0x%x refresh interval passed, sending new BU", (int) bulentry));

	lock_queue();
	entry = get_free_opt_queue_entry();
	unlock_queue();
	if (entry == NULL) {
		/* Queue full, try again later.
		 * Don't try to flush queue because BUL is kept locked
		 * while this function is called and trying to
		 * send something from here may cause synchronization
		 * problems.
		 */
		DEBUG((DBG_INFO, "queue was full, trying again after 1 second"));
		bulentry->callback_time = now + HZ;
		if (bulentry->callback_time >= bulentry->expire)
			bulentry->expire += HZ;
		return 0;
	}

	/* Build binding update */

	/* Assumptions done here:
	 * 1) use exponential backoff
	 * 2) use 5 * remaining lifetime as lifetime
	 * 3) use INITIAL_BINDACK_TIMEOUT as initial acknowledgement
	 *    timeout and MAX_BINDACK_TIMEOUT as max timeout
	 * TODO: The system should be improved so that these assumptions
	 * would not need to be done but rather specific values
	 * would be available.
	 */

	entry->opt_type = OPT_UPD;
	ipv6_addr_copy(&entry->saddr, &bulentry->home_addr);
	ipv6_addr_copy(&entry->daddr, &bulentry->cn_addr);
	entry->exp = 1;
	entry->flags = bulentry->flags;
	entry->plength = bulentry->prefix;
	bulentry->seq = get_new_sequence_number();
	entry->sequence = bulentry->seq;
	entry->lifetime = (bulentry->expire - now)/HZ*5;
	bulentry->expire = now + entry->lifetime*HZ;
	if (bulentry->expire <= now) {		/* Sanity check */
		DEBUG((DBG_WARNING, "bul entry lifetime caused jiffies wrap-around - setting lifetime to 30 secs"));
		entry->lifetime = 30;
		bulentry->expire = now + 30*HZ;
	}

	/* Set up retransmission */

	bulentry->state = RESEND_EXP;
	bulentry->callback = bul_resend_exp;
	bulentry->callback_time = now + INITIAL_BINDACK_TIMEOUT*HZ;
	bulentry->delay = INITIAL_BINDACK_TIMEOUT;
	bulentry->maxdelay = MAX_BINDACK_TIMEOUT;

	/* Queue for sending */

	entry->maxdelay_expires = now + RESEND_PIGGYBACK_MAXDELAY*HZ/1000;
	lock_queue();
	add_to_opt_queue(entry);
	unlock_queue();

	MIPV6_INC_STATS(n_bu_sent);

	return 0;
}



/*
 *
 * Module interface functions
 *
 */

struct ipv6_opt_hdr *mipv6_add_dst0opts(
	struct in6_addr *saddr,
	struct ipv6_opt_hdr *hdr,
	int append_ha_opt)
{
	struct ipv6_opt_hdr *newhdr;
	int hdrsize = -1; /* Size of memory block allocated for dest opts
			     header, -1 if not known */
	int offset = 0;	  /* Offset at which to add next dest option */
	int nextoffset;

	/* Check if need to add any options: options queued for sending to
	   the given daddr or the homeaddr option.
	 */
	DEBUG_FUNC();

	if (hdr == NULL) {
		/* Add destination options header to packet */
		hdr =  (struct ipv6_opt_hdr *) kmalloc(MAX_OPT_HDR_SIZE, GFP_ATOMIC);
		if (hdr == NULL)
			goto quit;
		hdr->nexthdr = 0; /* does this need to be set?? */
		hdr->hdrlen = 0;  /* and this? */
		hdrsize = MAX_OPT_HDR_SIZE;
		offset = 2;
	}
	if (hdrsize == -1) {
		/* Reallocate destination options header */
		if (ipv6_optlen(hdr) >= MAX_OPT_HDR_SIZE)
			goto quit;
		newhdr = (struct ipv6_opt_hdr *) kmalloc(MAX_OPT_HDR_SIZE, GFP_ATOMIC);
		if (newhdr == NULL)
			goto quit;
		memcpy(newhdr, hdr, ipv6_optlen(hdr));
		hdr = newhdr;
		hdrsize = MAX_OPT_HDR_SIZE;
		offset = ipv6_optlen(hdr);
	}


	/* Add home address option (if required). */

	if (append_ha_opt) {
		__u8 *dummy;
		MIPV6_INC_STATS(n_ha_sent);
		if (offset + MAX_HOMEADDR_OPT_SIZE > hdrsize)
			/* Does not fit in header. */
			goto quit;
		nextoffset = mipv6_create_home_addr((__u8 *) hdr, offset, &dummy, saddr);

		/* Verify that option did fit in header. */
		if (nextoffset > hdrsize)
			DEBUG((DBG_ERROR, "destination options header overrun when adding home address option - memory corrupted!!!"));
		if (nextoffset >= 0)
			offset = nextoffset;
	}

 quit: 


	if (hdrsize != -1) {
		offset = mipv6_finalize_dstopt_header((__u8 *)hdr, offset);
		if (offset > hdrsize)
			DEBUG((DBG_ERROR, "destination options header overrun when finalizing header"));
	}
	
	return hdr;
}
struct ipv6_opt_hdr *mipv6_add_dst1opts(
	struct in6_addr *saddr,
	struct in6_addr *daddr,
	struct ipv6_opt_hdr *hdr,
	int *append_ha_opt)
{
	struct opt_queue_entry *entry, *nextentry;
	struct ipv6_opt_hdr *newhdr;
	int hdrsize = -1; /* Size of memory block allocated for dest opts
			     header, -1 if not known */
	int offset = 0;	  /* Offset at which to add next dest option */
	int nextoffset;

	/* Check if need to add any options: options queued for sending to
	   the given daddr or the homeaddr option.
	 */
	DEBUG_FUNC();
	lock_queue();

	entry = get_first_opt_queue_entry(saddr, daddr);
	if (entry == NULL)
		goto quit;

	/* Yes, need to add one or more options.
	   - If packet does not have a destination options header, create one
	     of size MAX_OPT_HDR_SIZE
	   - If packet already has a destination options header, reallocate
	     header with size MAX_OPT_HDR_SIZE
	   - Then stuff as many destination options queued for this
	     destination that will fit into the header
	*/

	if (hdr == NULL) {
		/* Add destination options header to packet */
		hdr =  (struct ipv6_opt_hdr *) kmalloc(MAX_OPT_HDR_SIZE, GFP_ATOMIC);
		if (hdr == NULL)
			goto quit;
		hdr->nexthdr = 0; /* does this need to be set?? */
		hdr->hdrlen = 0;  /* and this? */
		hdrsize = MAX_OPT_HDR_SIZE;
		offset = 2;
	}
	if (hdrsize == -1) {
		/* Reallocate destination options header */
		if (ipv6_optlen(hdr) >= MAX_OPT_HDR_SIZE)
			goto quit;
		newhdr = (struct ipv6_opt_hdr *) kmalloc(MAX_OPT_HDR_SIZE, GFP_ATOMIC);
		if (newhdr == NULL)
			goto quit;
		memcpy(newhdr, hdr, ipv6_optlen(hdr));
		hdr = newhdr;
		hdrsize = MAX_OPT_HDR_SIZE;
		offset = ipv6_optlen(hdr);
	}

	/* Add any options queued for sending to the given destination
	 * address.
	 */

	while (entry) {
		nextentry = get_next_opt_queue_entry(saddr, daddr, entry);
		if (entry->opt_type == OPT_UPD) /* Add home address option always when sending BUs! */
			*append_ha_opt = 1;
		if ((nextoffset = add_dstopt(hdr, entry, offset, hdrsize)) >= 0) {
			remove_from_opt_queue(entry);
			offset = nextoffset;
		} else
			/* Option did not fit into header */
			goto quit;		
		entry = nextentry;
	}


quit:
	unlock_queue();

	if (hdrsize != -1) {
		offset = mipv6_finalize_dstopt_header((__u8 *)hdr, offset);
		if (offset > hdrsize)
			DEBUG((DBG_ERROR, "destination options header overrun when finalizing header"));
	}
	
	return hdr;
}

int mipv6_send_rq_option(
	struct in6_addr *saddr,
	struct in6_addr *daddr,
	long maxdelay)
{
	struct opt_queue_entry *entry;
	
	MIPV6_INC_STATS(n_br_sent);

	/* If queue full, get some free space by sending away an option from
	   the head of the queue. */

	lock_queue();
	entry = get_free_opt_queue_entry();
	unlock_queue();
	if (entry == NULL) {
		send_oldest_queued_opt();
		/* Now there SHOULD be room on the queue. */
		lock_queue();
		entry = get_free_opt_queue_entry();
		unlock_queue();
		if (entry == NULL) {
			DEBUG((DBG_ERROR, "sendopts queue full and failed to send "
			       "first option - lost option (mipv6_send_rq_option)"));
			return -1;
		}
	}

	entry->opt_type = OPT_RQ;
	ipv6_addr_copy(&entry->saddr, saddr);
	ipv6_addr_copy(&entry->daddr, daddr);

	entry->maxdelay_expires = (maxdelay != 0) ? jiffies + maxdelay*HZ/1000 : jiffies + HZ;
	lock_queue();
	add_to_opt_queue(entry);
	unlock_queue();

	if (maxdelay == 0)
		send_empty_packet(saddr, daddr);

	return 0;
}


int mipv6_send_ack_option(
	struct in6_addr *saddr, struct in6_addr *daddr,
	long maxdelay,	__u8 status, __u16 sequence, __u32 lifetime,
	__u32 refresh)
{
	struct opt_queue_entry *entry;
	
	MIPV6_INC_STATS(n_ba_sent);

	/* If queue full, get some free space by sending away an option from
	   the head of the queue. */

	lock_queue();
	entry = get_free_opt_queue_entry();
	unlock_queue();
	if (entry == NULL) {
		send_oldest_queued_opt();
		/* Now there SHOULD be room on the queue. */
		lock_queue();
		entry = get_free_opt_queue_entry();
		unlock_queue();
		if (entry == NULL) {
			DEBUG((DBG_ERROR, "sendopts queue full and failed to send "
			       "first option - lost option (mipv6_send_ack_option)"));
			return -1;
		}
	}

	entry->opt_type = OPT_ACK;
	ipv6_addr_copy(&entry->saddr, saddr);
	ipv6_addr_copy(&entry->daddr, daddr);
	entry->status = status;
	entry->sequence = sequence;
	entry->lifetime = lifetime;
	entry->refresh = refresh;

	entry->maxdelay_expires = (maxdelay != 0) ? jiffies + maxdelay*HZ/1000 : jiffies + HZ;
	lock_queue();
	add_to_opt_queue(entry);
	unlock_queue();

	if (maxdelay == 0)
		send_empty_packet(saddr, daddr);

	return 0;
}


int mipv6_send_upd_option(
	struct in6_addr *saddr, struct in6_addr *daddr,
	long maxdelay, __u32 initdelay, __u32 maxackdelay,
	__u8 exp, __u8 flags, __u8 plength, __u32 lifetime,
	struct mipv6_subopt_info *sinfo)
{
	struct opt_queue_entry *entry;
	__u8 state;
	int (*callback)(struct mipv6_bul_entry *);
	__u32 callback_time;
	struct in6_addr coa;

	MIPV6_INC_STATS(n_bu_sent);

	/* If queue full, get some free space by sending away an option from
	   the head of the queue. */

	lock_queue();
	entry = get_free_opt_queue_entry();
	unlock_queue();
	if (entry == NULL) {
		send_oldest_queued_opt();
		/* Now there SHOULD be room on the queue. */
		lock_queue();
		entry = get_free_opt_queue_entry();
		unlock_queue();
		if (entry == NULL) {
			DEBUG((DBG_ERROR,
			       "sendopts queue full and failed to send first option - "
			       "lost option (mipv6_send_upd_option)"));
			return -1;
		}
	}

	entry->opt_type = OPT_UPD;
	ipv6_addr_copy(&entry->saddr, saddr);
	ipv6_addr_copy(&entry->daddr, daddr);
	entry->exp = exp;
	entry->flags = flags;
	entry->plength = plength;
	entry->sequence = get_new_sequence_number();
	entry->lifetime = lifetime;

	if (sinfo && sinfo->fso_flags != 0)
		memcpy(&entry->sinfo, sinfo, 
		       sizeof(struct mipv6_subopt_info));
	else
		memset(&entry->sinfo, 0, 
		       sizeof(struct mipv6_subopt_info));
	
	/* Add to binding update list */
	
	if (mipv6_is_mn) {
		if (entry->flags & MIPV6_BU_F_ACK) {
			if (exp) {
				/* Send using exponential backoff */
				state = RESEND_EXP;
				callback = bul_resend_exp;
				callback_time = initdelay;
			} else {
				/* Send using linear backoff */
				state = RESEND_CONST;
				callback = bul_resend_const;
				callback_time = initdelay;
			}
		} else {
			/* No acknowledgement/resending required */
			state = NOT_SENT;
			callback = bul_expired;
			callback_time = lifetime;
		}

		mipv6_get_care_of_address(saddr, &coa);

		mipv6_bul_add(daddr, saddr, &coa, lifetime, entry->sequence,
			      plength, flags, callback, callback_time, state,
			      initdelay, maxackdelay);
	} else
		DEBUG((DBG_ERROR, "mipv6_send_upd_option() called on non-mobile node configuration"));

	entry->maxdelay_expires = (maxdelay != 0) ? jiffies + maxdelay*HZ/1000 : jiffies + HZ;

	lock_queue();
	add_to_opt_queue(entry);
	unlock_queue();

	if (maxdelay == 0)
		send_empty_packet(saddr, daddr);

	return 0;
}


int mipv6_ack_rcvd(struct in6_addr *cnaddr, __u16 sequence,
	__u32 lifetime, __u32 refresh, int status)
{
	struct mipv6_bul_entry *bulentry;
	unsigned long now = jiffies;

	if (status != STATUS_UPDATE && status != STATUS_REMOVE)
		DEBUG((DBG_ERROR, "mipv6_ack_rcvd() called with invalid status code"));

	DEBUG((DBG_INFO, "binding ack received with sequence number 0x%x, status: %d ",
	       (int) sequence, status));

	if (status == STATUS_REMOVE) {
		DEBUG((DBG_INFO, "- NACK - deleting bul entry"));
		return mipv6_bul_delete(cnaddr);
	}

	/* Find corresponding entry in binding update list */
	
	if ((bulentry = mipv6_bul_get(cnaddr))) {
		/* Check that sequence numbers match */
		if (sequence == bulentry->seq) {
			bulentry->state = SEND_NOMORE;
			if (bulentry->flags & MIPV6_BU_F_HOME && lifetime != 0) {
				/* For home registrations: schedule a refresh binding update.
				 * Use the refresh interval given by home agent or 80%
				 * of lifetime, whichever is less.
				 *
				 * Adjust binding lifetime if 'granted' lifetime
				 * (lifetime value in received binding acknowledgement)
				 * is shorter than 'requested' lifetime (lifetime
				 * value sent in corresponding binding update).
				 */
				if (lifetime >= (bulentry->expire - now)/HZ)
					lifetime = (bulentry->expire - now)/HZ;
				if (refresh >= lifetime || refresh == 0)
					refresh = 4*lifetime/5;
				DEBUG((DBG_INFO, "mipv6_ack_rcvd: setting callback for expiration of"
					" a Home Registration: lifetime:%d, refresh:%d",
				       lifetime, refresh));
				bulentry->callback = bul_refresh;
				bulentry->callback_time = now + refresh*HZ;
				bulentry->expire = now + lifetime*HZ;
			} else {
				bulentry->callback = bul_expired;
				bulentry->callback_time = bulentry->expire;
			}
			mipv6_bul_put(bulentry);
			DEBUG((DBG_INFO, "- accepted"));
			return 0;
		}

		DEBUG((DBG_INFO, "- discarded, seq number mismatch"));
		return -1;
	}

	DEBUG((DBG_INFO, "- discarded, no entry in bul matches BA source address"));
	return -1;
}


/*
 *
 * Module initialization & deinitialization functions
 *
 */

int __init mipv6_initialize_sendopts(void)
{
	int i;
	struct opt_queue_entry *entry;

	/* Allocate option queue entries and put them to free list. */

	opt_queue_array = (struct opt_queue_entry *)
		kmalloc(MAX_OPT_QUEUE_SIZE * sizeof(struct opt_queue_entry),
			GFP_KERNEL);

	if (opt_queue_array == NULL) {
		mipv6_shutdown_sendopts();
		return -1;
	}

	entry = opt_queue_array;
	free_list = NULL;
	for (i = 0; i < MAX_OPT_QUEUE_SIZE; i++) {
		entry->next_list = free_list;
		free_list = entry++;
	}

	/* Allocate socket for sending destination options */

	if (alloc_dstopts_socket() != 0) {
		mipv6_shutdown_sendopts();
		return -1;
	}

	/* Init some variables */

	for (i = 0; i < HASH_TABLE_SIZE; i++)
		opt_queue_hashtable[i] = NULL;

	/* Allocate timer */

	timer = (struct timer_list *) kmalloc(sizeof(struct timer_list), GFP_KERNEL);
	if (timer == NULL) {
		mipv6_shutdown_sendopts();
		return -1;
	}

	init_timer(timer);
	timer->function = timer_handler;
	timer->expires = jiffies + HZ;

	return 0;
}

void mipv6_shutdown_sendopts(void)
{
	/* Deallocate timer */

	if (timer) {
		del_timer(timer);
		kfree(timer);
	}
	
	/* Deallocate socket for sending destination options */

	dealloc_dstopts_socket();

	/* Deallocate option queue entries */
	
	if (opt_queue_array)
		kfree(opt_queue_array);
}
