/*
 *	IPv6-IPv6 tunneling module
 *
 *	Authors:
 *	Sami Kivisaari		<skivisaa@cc.hut.fi>
 *
 *	$Id: tunnel.c,v 1.4 2000/10/11 10:46:40 antti Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#define __NO_VERSION__
#include <linux/module.h>
#include <linux/version.h> 
#include <linux/kernel.h>

#include <linux/net.h>
#include <linux/skbuff.h>
#include <linux/ipv6.h>
#include <linux/net.h>
#include <linux/netdevice.h>
#include <linux/init.h>
#include <linux/autoconf.h>
#include <net/protocol.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/dst.h>
#include <net/addrconf.h>


#include "mn.h"
#include "stats.h"
#include "debug.h"

static struct socket *tunnel_socket = NULL; 

/*
 * fragmentation call-back function, used by ipv6_build_xmit
 */
static int mipv6_tunnel_getfrag(
	const void *data, struct in6_addr *addr,
	char *buff, unsigned int offset, unsigned int len)
{
	DEBUG((DBG_DATADUMP, 
	       "mipv6_get_frag(0x%p, 0x%p, 0x%p, 0x%x, 0x%x)",
	       data, addr, buff, offset, len));

	debug_print_buffer(DBG_DATADUMP, data, len);

	memcpy(buff, data + offset, len);

	return 0;
}


#define TUNNEL_OVERHEAD (sizeof(struct ipv6hdr) + 128)
int mipv6_encapsulate_ipv6(
	struct sk_buff *skb,
	struct in6_addr *saddr,
	struct in6_addr *daddr)
{
	struct flowi fl;
	struct sock *sk = tunnel_socket->sk;
	struct in6_addr source;
	struct dst_entry *dst;
	int err = 0;

 	DEBUG((DBG_INFO,
	       "mipv6_encapsulate_ipv6(0x%p, %x:%x:%x:%x)",
	       skb,
	       (int)ntohl(daddr->s6_addr32[0]), (int)ntohl(daddr->s6_addr32[1]),
	       (int)ntohl(daddr->s6_addr32[2]), (int)ntohl(daddr->s6_addr32[3])));

	fl.proto = NEXTHDR_IPV6;
	fl.fl6_flowlabel = 0;
	fl.oif = sk->bound_dev_if;
	fl.uli_u.data = 0;
	fl.fl6_dst = daddr;
	fl.fl6_src = (saddr == NULL) ? &source : saddr;

	dst = ip6_route_output(sk, &fl);
	if((err = dst->error) != 0) {
		DEBUG((DBG_ERROR, "No route to tunnel destination."));
		goto out;
	}
	/* drop packet if hop limit is 0 */
	if (skb->nh.ipv6h->hop_limit==0){
		kfree_skb(skb);
		err = 0;
		goto out;
	}

	/* Hop limit of inner header is decremented */
	skb->nh.ipv6h->hop_limit-=1;
	/*  send ICMPv6 packet too big if tunneled data exceeds tunnel MTU */
	err = (dst->pmtu - TUNNEL_OVERHEAD) < (skb->tail - skb->data);
	if(err) {
		DEBUG((DBG_ERROR, "Tunneled packet too big. (%d > %d)",
		       skb->tail - skb->data, dst->pmtu - TUNNEL_OVERHEAD));
		icmpv6_send(
			skb, ICMPV6_PKT_TOOBIG, 0,
			dst->pmtu - TUNNEL_OVERHEAD, skb->dev);
		goto out;
	}

	/*  get source address if one was not supplied                     */
	if(saddr == NULL) {
		if((err = ipv6_get_saddr(dst, daddr, fl.fl6_src)) != 0) {
			DEBUG((DBG_ERROR, "Could not get source address"));
			goto out;
		}
	}
		  
	ip6_build_xmit(sk, mipv6_tunnel_getfrag, skb->data,
		       &fl, skb->tail - skb->data, NULL, 255, MSG_DONTWAIT);

 out:
	dst_release(dst);
	return err;
}


/*
 * protocol handler for ipv6 inside ipv6
 */
static int mipv6_ipv6_tunnel_handler(
	struct sk_buff *skb,
	unsigned long len)
{
	struct packet_type ipv6_packet_type;
	
	/* we need to do the following:
	     a) we modify the packet so that the transport layer
	        protocol (ipv6 in this case)
                pointer is copied back to network layer protocol pointer
	     b) check if tunnel endpoint is local address, if not then discard
	     c) feed the inner ipv6-packet back to ipv6_rcv which should 
	        process it as any normal packet
	     d) notify mobile node that a tunneled packet was received
	*/

	DEBUG((DBG_INFO,
	       "mipv6_ipv6_tunnel_handler(skb=0x%p len=%d)", skb, (int)len));

	/*  Event for mn.c  */
	mipv6_tunneled_packet_received(
		skb,
		(struct ipv6hdr *)skb->h.raw,
		(struct ipv6hdr *)skb->nh.raw);

	DEBUG((DBG_DATADUMP, "outer ipv6 header dump:"));
	debug_print_buffer(DBG_DATADUMP, skb->nh.raw, skb->h.raw - skb->nh.raw);

	DEBUG((DBG_DATADUMP, "inner ipv6 header dump:"));
	debug_print_buffer(DBG_DATADUMP, skb->h.raw, skb->tail - skb->h.raw);

	
	/* set new network-layer header pointer to inner ipv6-packet
	 * skb->h is not valid since we are passing the packet from
	 * data-link layer to network layer and thus can not know
	 * anything about higher protocol layers
	 */
	ASSERT(skb->h.raw > skb->nh.raw);
	skb->mac.raw =skb->nh.raw;
	skb->nh.raw=skb_pull(skb, skb->h.raw - skb->data);
	skb->pkt_type = PACKET_HOST;
	skb->protocol = __constant_htons(ETH_P_IPV6);
        skb->ip_summed = CHECKSUM_NONE;
	dst_release(skb->dst);
	skb->dst = NULL;
 
	MIPV6_INC_STATS(n_decapsulations);

	/* pass the packet to ipv6-module */
	return ipv6_rcv(skb, skb->dev, &ipv6_packet_type);
}

#define IPPROTO_IPV6 NEXTHDR_IPV6

/*
 * New INET6 protocol definition, tunneled IPv6 packets
 */
static struct inet6_protocol mipv6_ipv6_tunnel = {
	mipv6_ipv6_tunnel_handler,
	NULL,
	NULL,
	IPPROTO_IPV6,
	0,
	NULL,
        "IPv6-IPv6 Encapsulation"
};

extern struct net_proto_family inet6_family_ops;

static int mipv6_initialize_tunnel_socket(void)
{
	struct net_proto_family *ops = &inet6_family_ops;
	struct sock *sk;
	int err;

	DEBUG_FUNC();

	tunnel_socket = (struct socket *)sock_alloc();
	if (tunnel_socket == NULL) {
		DEBUG((DBG_CRITICAL,
		       "Failed to create the IPv6 tunneling control socket."));
		return -1;
	}
	tunnel_socket->inode->i_uid = 0;
	tunnel_socket->inode->i_gid = 0;
	tunnel_socket->type = SOCK_RAW;

	if ((err = ops->create(tunnel_socket, NEXTHDR_IPV6)) < 0) {
		DEBUG((DBG_CRITICAL,
		       "Failed to initialize IPv6 tunneling control socket (err %d).",
		       err));
		sock_release(tunnel_socket);
		tunnel_socket = NULL; /* for safety */
		return err;
	}

	sk = tunnel_socket->sk;
	sk->allocation = GFP_ATOMIC;
	sk->net_pinfo.af_inet6.hop_limit = 254;
	sk->net_pinfo.af_inet6.mc_loop = 0;
	sk->prot->unhash(sk);
	
	return 0;
}


static void mipv6_terminate_tunnel_socket(void)
{
	DEBUG_FUNC();

	if(tunnel_socket) sock_release(tunnel_socket);
	tunnel_socket = NULL; /* For safety. */
}


int __init mipv6_initialize_tunnel(void)
{
	int err;

	DEBUG_FUNC();

	/* create control socket for sending */
	err = mipv6_initialize_tunnel_socket();
	if(err) return err;	 

	/* register a tunneled packets as a new inet6 protocol */
	inet6_add_protocol(&mipv6_ipv6_tunnel);
	return 0;
}


void mipv6_shutdown_tunnel(void)
{
	DEBUG_FUNC();

	mipv6_terminate_tunnel_socket();

	/* unregister a tunneled packets protocol */
	inet6_del_protocol(&mipv6_ipv6_tunnel);
}





















