/*
 *      Movement Detection Module
 *
 *      Authors:
 *      Henrik Petander                <lpetande@cc.hut.fi>
 *
 *      $Id: mdetect.c,v 1.50 2001/02/23 18:25:38 antti Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 *      Handles the L3 movement detection of mobile node and also
 *      changing of its routes.
 *  
 *      TODO: Should locking use write_lock_irqsave and restore
 *            instead of spin_lock_irqsave?
 *      TODO: Reception of tunneled router advertisements is not
 *            implemented, the functionality should be added to either 
 *            mn or mdetect
 */

/*
 *	Changes:
 *
 *	Nanno Langstraat	:	Locking fixes
 */

#include <linux/autoconf.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/if_arp.h>
#include <linux/icmpv6.h>
#ifdef CONFIG_SYSCTL
#include <linux/sysctl.h>
#endif
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/addrconf.h>

#include "mipv6.h"
#include "mdetect.h"
#include "dhaad.h"
#include "mn.h"
#include "debug.h"

#ifdef CONFIG_SYSCTL
#include "sysctl.h"
#endif /* CONFIG_SYSCTL */

#define START 0
#define CONTINUE 1

#define SUBNET_UNDETERMINED 0
#define SUBNET_HOME 1
#define SUBNET_FOREIGN 2

/* dad could also be RESPECT_DAD for duplicate address detection of
   new care-of addresses */
static int dad = 0;

/* Only one choice, nothing else implemented */
int mdet_mech = EAGER_CELL_SWITCHING; 

static int location = SUBNET_UNDETERMINED;
static int mdetect_is_initialized = 0;
spinlock_t router_lock = SPIN_LOCK_UNLOCKED;
int rs_interval = HZ / 2;


#ifdef CONFIG_SYSCTL

static char mipv6_mn_home_address[36];

static void mipv6_mdetect_sysctl_register(void);
static void mipv6_mdetect_sysctl_unregister(void);

static int mipv6_home_address_sysctl_handler(
	ctl_table *ctl, int write, struct file * filp,
	void *buffer, size_t *lenp);
static int mipv6_mdetect_mech_sysctl_handler(
	ctl_table *ctl, int write, struct file * filp,
	void *buffer, size_t *lenp);

static struct ctl_table_header *mipv6_mdetect_sysctl_header;

static struct mipv6_mn_sysctl_table
{
	struct ctl_table_header *sysctl_header;
	ctl_table mipv6_vars[3];
	ctl_table mipv6_mobility_table[2];
	ctl_table mipv6_proto_table[2];
	ctl_table mipv6_root_table[2];
} mipv6_mdetect_sysctl = {
	NULL,

        {{NET_IPV6_MOBILITY_HOME_ADDRESS, "home_address",
	  &mipv6_mn_home_address, 36 * sizeof(char),
	  0644, NULL, &mipv6_home_address_sysctl_handler},
	 {NET_IPV6_MOBILITY_MDETECT_MECHANISM, "mdetect_mechanism",
	  &mdet_mech, sizeof(int), 0644, NULL,
	  &mipv6_mdetect_mech_sysctl_handler},

	 {0}},

	{{NET_IPV6_MOBILITY, "mobility", NULL, 0, 0555, 
	  mipv6_mdetect_sysctl.mipv6_vars}, {0}},
	{{NET_IPV6, "ipv6", NULL, 0, 0555, 
	  mipv6_mdetect_sysctl.mipv6_mobility_table}, {0}},
	{{CTL_NET, "net", NULL, 0, 0555, 
	  mipv6_mdetect_sysctl.mipv6_proto_table},{0}}
};

#endif /* CONFIG_SYSCTL */

int mipv6_is_care_of_address(struct in6_addr *addr);
static void router_state(unsigned long foo);

#define LOOPB {{{0, 0, 0, 1}}}
#define ADDRANY {{{0, 0, 0, 0}}}

static struct in6_addr addr_any = ADDRANY;
static struct router *curr_router = NULL, *next_router = NULL;
static struct timer_list r_timer = { function: router_state };

/* Sends router solicitations to all valid devices 
 * source  = link local address (of sending interface)
 * dstaddr = all routers multicast address
 * Solicitations are sent at an exponentially decreasing rate
 *
 * TODO: send solicitation first at a normal rate (from ipv6) and
 *       after that use the exponentially increasing intervals 
 */
static void rs_send(int exp)
{
	struct net_device *dev;
	struct in6_addr raddr, lladdr;
	struct inet6_dev *in6_dev = NULL;
	static int next_rs, ival = HZ / 2;

	DEBUG_FUNC();

	if (jiffies / HZ < next_rs)
		return;

	ipv6_addr_all_routers(&raddr);
	rtnl_lock(); 
	/*  Send router solicitations to all interfaces  */
	for (dev = dev_base; dev; dev = dev->next) {
		if ((dev->flags & IFF_UP) && dev->type == ARPHRD_ETHER) {
			DEBUG((DBG_INFO, "Sending RS to device %s", 
			       (int)dev->name));
			ipv6_get_lladdr(dev, &lladdr);
			ndisc_send_rs(dev, &lladdr, &raddr);
			in6_dev = in6_dev_get(dev);

			in6_dev->if_flags |= IF_RS_SENT;
			in6_dev_put(in6_dev);

		}

	}
	rtnl_unlock();
	if (exp == START)
		ival = HZ / 2;
	else
		ival = ival * 2;

	next_rs = jiffies + ival;		
}

/*
 * Decides what to do based on the new prefix.  Keeps track of CoAs
 * and deletes the old CoA.  Also notifies mn of movement.  
 */
static int handle_prefix(struct router *old, struct router *new)
{
	struct neighbour *neigh;
	struct in6_addr *prev;
	struct in6_addr home_addr;
	int home_plen = 0;

	DEBUG_FUNC();

	if (!new) {
		DEBUG((DBG_ERROR,"handle_prefix: new router was NULL"));
		return -1;
	}
	if (old != NULL) {
		if (mipv6_prefix_compare(&new->raddr,
					&old->raddr, new->pfix_len)) {

			DEBUG((DBG_WARNING, "Router adv from same router"));
			DEBUG((DBG_WARNING, " new: %x:%x:%x:%x:%x:%x:%x:%x",
			       NIPV6ADDR(&new->raddr)));
			DEBUG((DBG_WARNING, " old: %x:%x:%x:%x:%x:%x:%x:%x", 
			       NIPV6ADDR(&old->raddr)));
			return 0; /* Same network */
		}
	}

	/* To enable BU sending to previous router, set the previous
	 * router's address to correct one if it exists, implements HA
	 * and we know its global address, otherwise to addr_any 
	 */
	if (old != NULL && (old->flags & ND_RA_FLAG_HA) && old->glob_addr)
		prev = &old->raddr;
        else
                prev = &addr_any;
	
	if ((home_plen = mipv6_mn_get_homeaddr(&home_addr)) < 0)
		return -1;

	if (home_plen == new->pfix_len &&
	    mipv6_prefix_compare(&new->raddr, &home_addr, home_plen)) {

		DEBUG((DBG_INFO, "Router adv from home router"));
		if (location == SUBNET_FOREIGN) {
			struct net_device *newdev;
			/* Back in home subnet */

			/* A neighbor advertisement is sent to all
			 * nodes multicast address to redirect
			 * datagrams to MN's home address to mobile
			 * node's HW-address, otherwise they would be
			 * sent to HA's hw-address 
			 */
			newdev = dev_get_by_index(new->ifindex);
			if (newdev == NULL) {
				DEBUG((DBG_WARNING, "Device not present"));
				return -1;
			}
			neigh = __neigh_lookup(&nd_tbl, &new->ll_addr, newdev, 1);
			if (neigh) {
				neigh_update(neigh, (u8 *)&new->link_addr, NUD_VALID, 1, 1);
				neigh_release(neigh);
			}

#if 0
			/* This does not work at present.  Further
                         * investigation needed or else this should be
                         * dumped.  See draft section 10.20.
			 */
			if (!nc_ok) {
				struct in6_addr unspec_addr = ADDRANY;
				/* 
				 * if we have no neighbour entry send
				 * a nsol with unspec source address 
				 */
				DEBUG((DBG_INFO, "HA LL address: ndisc"));
				ipv6_addr_set(&unspec_addr, 0, 0, 0, 0);
				neigh = neigh_create(&nd_tbl, &home, newdev);
				neigh_release(neigh);
				ndisc_send_ns(newdev, NULL, &home, 
					      &home, &unspec_addr);
			}
#endif
			/* Movement event  */
			mipv6_mn_returned_home(prev, home_plen);
			dev_put(newdev);
		}
		location = SUBNET_HOME;
		
		/* For routing purposes */		
		return 0;
	} else {
		struct in6_addr coa;
		DEBUG((DBG_INFO,"Router adv from new foreign router"));
		/* RADV prefix NOT the same as in home network (in
                   foreign net) */
		location = SUBNET_FOREIGN;
		mipv6_get_care_of_address(NULL, &coa);
		
		/* Movement event  */
		mipv6_mobile_node_moved(&coa, prev, new->pfix_len);
		return 0;
	}
	return 1;
}

/*
 * Function that determines whether given rt6_info should be destroyed
 * (negative => destroy rt6_info, zero or positive => do nothing) 
 */
static int mn_route_cleaner(struct rt6_info *rt, void *arg)
{
	int type;
	DEBUG_FUNC();

	if (!rt || !next_router){
		DEBUG((DBG_ERROR, "mn_route_cleaner: rt or next_router NULL"));
		return 0;
	}
	DEBUG((DBG_INFO, "clean route ?: via dev=\"%s\"", rt->rt6i_dev->name));

	/* Do not delete routes to local addresses or to multicast
	 * addresses, since we need them to get router advertisements
	 * etc. Multicast addresses are more tricky, but we don't
	 * delete them in any case 
	 */
	type = ipv6_addr_type(&rt->rt6i_dst.addr);

	if ((type & IPV6_ADDR_MULTICAST || type & IPV6_ADDR_LINKLOCAL)
	   || rt->rt6i_dev == &loopback_dev 
	   || !ipv6_addr_cmp(&rt->rt6i_gateway, &next_router->ll_addr) 
	   || mipv6_prefix_compare(&rt->rt6i_dst.addr, &next_router->raddr, 64))
	  return 0;

	DEBUG((DBG_DATADUMP, "deleting route to: %x:%x:%x:%x:%x:%x:%x:%x",
	       NIPV6ADDR(&rt->rt6i_dst.addr)));

	/*   delete all others */
	return -1;
}


/* 
 * Deletes old routes 
 */
static __inline void delete_routes(void)
{
	struct in6_addr homeaddr;
	int plength;

	DEBUG_FUNC();

	/* Routing table is locked to ensure that nobody uses its */  
	write_lock_bh(&rt6_lock);
	DEBUG((DBG_INFO, "mipv6: Purging routes"));
	plength = mipv6_mn_get_homeaddr(&homeaddr);
	/*  TODO: Does not prune, should it?  */
	fib6_clean_tree(&ip6_routing_table, 
			mn_route_cleaner, 0, &homeaddr);
	write_unlock_bh(&rt6_lock);

}

/* 
 * Changes the router, called from ndisc.c after handling the router
 * advertisement.
 */

void mipv6_change_router(void)
{
	struct router *tmp;
	unsigned long flags;

	DEBUG_FUNC(); 

	if (next_router==NULL) {
		DEBUG((DBG_DATADUMP,"mipv6_change_router: next_router NULL"));
		return;
	}

	if (curr_router != NULL && 
	    !ipv6_addr_cmp(&curr_router->ll_addr, &next_router->ll_addr)) {
		DEBUG((DBG_DATADUMP,"Trying to handoff from: "
		       "%x:%x:%x:%x:%x:%x:%x:%x",
		       NIPV6ADDR(&curr_router->ll_addr)));
		DEBUG((DBG_DATADUMP,"Trying to handoff to: "
		       "%x:%x:%x:%x:%x:%x:%x:%x",
		       NIPV6ADDR(&next_router->ll_addr)));
		next_router = NULL; /* Let's not leave dangling pointers */
		return;
        }

	if (!mdetect_is_initialized) {
		DEBUG((DBG_WARNING,
		       "We don't change router before initialization"));
		return;
	}

	delete_routes();
	spin_lock_irqsave(&router_lock, flags);


	tmp = curr_router;
	curr_router = next_router;
	next_router = NULL; 
	if (tmp)
		curr_router->prev_router = tmp;
	
	spin_unlock_irqrestore(&router_lock, flags);
	
	handle_prefix(tmp, curr_router);
	mipv6_get_care_of_address(NULL, &curr_router->CoA);
} 

/*
 * advertisement interval-based movement detection
 */
static void router_state(unsigned long foo)
{
	unsigned long flags;

	spin_lock_irqsave(&router_lock, flags);

	if (curr_router == NULL){
		DEBUG((DBG_DATADUMP, "Sending RS"));
		rs_send(CONTINUE); 
		mod_timer(&r_timer, jiffies + HZ / 2);
		spin_unlock_irqrestore(&router_lock, flags);
		return;
	}
	
	switch (curr_router->state) {
		
	case ROUTER_REACHABLE:
		/* if radvd doesn't support interval option handoffs will 
		 * be based on the expiry of lifetime 
		 */
		if (curr_router->last_ra_rcvd  >= (jiffies - curr_router->interval) ||
		    (curr_router->interval == 0 &&  curr_router->last_ra_rcvd >= 
		     (jiffies - curr_router->lifetime) )) {

			/* check if maximum RADV time elapsed */
			 if( curr_router->last_ra_rcvd < (jiffies - MAX_RADV_INTERVAL)){
				   curr_router->state=NOT_REACHABLE; 
				   DEBUG((DBG_INFO,
					  "MAX_RADV elapsed, current router changed state to NOT_REACHABLE"));
			 }
			break; 

		} else {
			curr_router->state = RADV_MISSED;
			DEBUG((DBG_INFO, "Current router changed state to RADV_MISSED"));
		}
		break;
	case RADV_MISSED:
		if (curr_router->interval!=0
		    && curr_router->last_ra_rcvd < (jiffies - (curr_router->interval*2)) ){
			curr_router->state=NOT_REACHABLE;  
			rs_send(START);
			DEBUG((DBG_INFO,
			       "Current router changed state to NOT_REACHABLE"));
		}
		break;
	case NOT_REACHABLE:
		DEBUG((DBG_DATADUMP,
		       "Current router state: Router NOT reachable"));
		if (curr_router->last_ra_rcvd + curr_router->lifetime <= jiffies)
			rs_send(CONTINUE);
		break;
	}
	spin_unlock_irqrestore(&router_lock, flags);
	mod_timer(&r_timer, jiffies + HZ / 10);
	
}
/* Deletes old coas, assumes they are in EUI-64 format */ 
void coa_del(struct net_device *dev, struct in6_addr *prefix)
{
	int err;
	struct in6_ifreq coa_del;
	struct in6_addr coa, home_addr;
	__u32 coa_pfix = 64;
	DEBUG_FUNC();
	/* We get the suffix  and combine it with the 64 bit prefix*/
	if (prefix == NULL || dev == NULL)
		return;
	ipv6_get_lladdr(dev, &coa);
	coa.s6_addr32[0] = prefix->s6_addr32[0];
	coa.s6_addr32[1] = prefix->s6_addr32[1];

	mipv6_mn_get_homeaddr(&home_addr);
	if (ipv6_chk_addr(&coa, dev) != 0 &&
	    ipv6_addr_cmp(&coa, &home_addr)) {
		ipv6_addr_copy(&coa_del.ifr6_addr, &coa);
		coa_del.ifr6_prefixlen = coa_pfix; /* EUI-64 CoAs */
		coa_del.ifr6_ifindex = dev->ifindex;
		err = addrconf_del_ifaddr((void *) &coa_del);
		if (err) {
			DEBUG((DBG_CRITICAL, "error: %d Couldn't delete old "
			       "care-of address: %x:%x:%x:%x:%x:%x:%x:%x", 
			       err, NIPV6ADDR(&coa)));
			DEBUG((DBG_CRITICAL, "It must be deleted manually "
			       "after unloading the module"));
		}
	}
}

int mipv6_mn_subnet_foreign(void)
{
	return (location == SUBNET_FOREIGN);
}

int mipv6_mn_subnet_unknown(void)
{
	return (location == SUBNET_UNDETERMINED);
}

/*
 * Returns the current care-of address, assumes addresses in EUI-64 format
 * TODO: Change the first parameter to dev
 */
void mipv6_get_care_of_address(
	struct in6_addr *homeaddr, struct in6_addr *coaddr)
{
	unsigned long flags;
	struct net_device *currdev;

	DEBUG_FUNC();

	if (curr_router == NULL || location == SUBNET_HOME) {
		mipv6_mn_get_homeaddr(coaddr);
		return;
	}
	if ((currdev = dev_get_by_index(curr_router->ifindex)) == NULL) {
		DEBUG((DBG_WARNING, "Device is not present"));
		return;
	}
	spin_lock_irqsave(&router_lock, flags);
	ipv6_get_lladdr(currdev, coaddr);
	coaddr->s6_addr32[0] = curr_router->raddr.s6_addr32[0];
	coaddr->s6_addr32[1] = curr_router->raddr.s6_addr32[1];

	if (dad == RESPECT_DAD && ipv6_chk_addr(coaddr, currdev) == 0) { 
		/* address check failure probably due to dad wait*/
		DEBUG((DBG_WARNING, "care-of address not valid, using home "
		       "address instead"));
		mipv6_mn_get_homeaddr(coaddr);
	}
	spin_unlock_irqrestore(&router_lock, flags);
	dev_put(currdev);
}

/* returns previous care of address */
void mipv6_get_old_care_of_address(
        struct in6_addr *homeaddr, struct in6_addr *prevcoa)

{
	unsigned long flags;

        spin_lock_irqsave(&router_lock, flags);
	if (curr_router!=NULL && curr_router->prev_router!=NULL)
		ipv6_addr_copy(prevcoa, &curr_router->prev_router->CoA);
	else {
		ipv6_addr_copy(prevcoa, &addr_any);
		DEBUG((DBG_ERROR,"No previous router, giving addr_any as prevcoa"));
	}
	spin_unlock_irqrestore(&router_lock, flags);
}

/* Called from mipv6_ra_rcv to determine whether to change the current
 * default router and pass the ra to the default router adding
 * mechanism of ndisc.c. return value 1 passes ra and 0 doesn't. 
 */
int mipv6_router_event(struct router *nrt)
{
	unsigned long flags;
	struct in6_addr home_prefix;
	int home_plen = 0;

	DEBUG_FUNC();
	
	if (!mdetect_is_initialized)
		return 0;

	DEBUG((DBG_DATADUMP, "Received a RA from: "
	       "%x:%x:%x:%x:%x:%x:%x:%x", NIPV6ADDR(&nrt->ll_addr)));

	/* Whether from current router */
	spin_lock_irqsave(&router_lock, flags);
	if (curr_router != NULL && 
	    !ipv6_addr_cmp(&nrt->ll_addr, &curr_router->ll_addr)) {
		curr_router->last_ra_rcvd = jiffies;
		curr_router->state = ROUTER_REACHABLE;
		DEBUG((DBG_DATADUMP,
		       "Received a ra from current router"));
		spin_unlock_irqrestore(&router_lock, flags);
		return 1; 
	} else {		/* Another router */
		struct router *tmp = NULL;
		struct net_device *next_router_dev;
		/* Whether the router exists
		 * TODO: check lifetime of ra */
		tmp = iterate_routers(&nrt->ll_addr, &curr_router);
		if (tmp == NULL) {
			struct net_device *nrtdev;
			/* Insert router to list first */      	
			tmp = new_router(nrt);

			if (tmp == NULL) {
				DEBUG((DBG_ERROR, "Failed to add new router to list"));
				spin_unlock_irqrestore(&router_lock, flags);
				return 0;
			}
				
			tmp->last_ra_rcvd = jiffies;
			tmp->state = ROUTER_REACHABLE;
				
			if ((nrtdev = dev_get_by_index(nrt->ifindex)) == NULL) {
				DEBUG((DBG_WARNING, "Device is not present"));
				return 1;
			}
			if (curr_router == NULL && rt6_get_dflt_router(&nrt->ll_addr, nrtdev)) {
				dev_put(nrtdev);
				next_router = tmp;
				DEBUG((DBG_INFO, "Startup"));
				spin_unlock_irqrestore(&router_lock, flags);
				return 1;
			}
			dev_put(nrtdev);

			if (mdet_mech == EAGER_CELL_SWITCHING) { 
				DEBUG((DBG_INFO, "Changing router to previously unknown router:"));
				next_router = tmp;
				if ((next_router_dev = dev_get_by_index(next_router->ifindex)) != NULL) {
					neigh_ifdown(&nd_tbl, next_router_dev);
					dev_put(next_router_dev);
				} else {
					DEBUG((DBG_WARNING, "Device is not present"));
				}
				spin_unlock_irqrestore(&router_lock, flags);
				return 1;
			}
		} else {
			/* TODO: check if ra ival is still the same, 
			 *       also lifetime should be checked, 
			 *       also: should neigh_ifdown really be called here ? 
			 */
			tmp->last_ra_rcvd = jiffies;
			tmp->state = ROUTER_REACHABLE;
			DEBUG((DBG_INFO, "Updating entry for router"));

			home_plen = mipv6_mn_get_homeaddr(&home_prefix);
			/* Insert policy handler here, e.g., preferred interfaces & 
			 *  networks, etc.
			 */
			if (curr_router == NULL || 
			    curr_router->state != ROUTER_REACHABLE ||  
			    (tmp->pfix_len == home_plen &&
			     mipv6_prefix_compare(&tmp->raddr, &home_prefix,
						  home_plen))) {

				DEBUG((DBG_INFO,
				       "Changing def. router to a new router, since old is not reachable"));
				next_router = tmp;
				if ((next_router_dev = dev_get_by_index(next_router->ifindex)) != NULL) {
					neigh_ifdown(&nd_tbl, next_router_dev);
					dev_put(next_router_dev);
				} else {
					DEBUG((DBG_WARNING, "Device is not present"));
				}
				spin_unlock_irqrestore(&router_lock, flags);
				return 1;
			}
			spin_unlock_irqrestore(&router_lock, flags);
			return 0; 
		}
	}
	spin_unlock_irqrestore(&router_lock, flags);
	return 0;
}

#ifdef CONFIG_SYSCTL
static void mipv6_mdetect_sysctl_register(void)
{
	mipv6_mdetect_sysctl_header =
		register_sysctl_table(mipv6_mdetect_sysctl.mipv6_root_table, 0);
}

static void mipv6_mdetect_sysctl_unregister(void)
{
	unregister_sysctl_table(mipv6_mdetect_sysctl_header);
}

static int mipv6_mdetect_mech_sysctl_handler(
	ctl_table *ctl, int write, struct file * filp,
	void *buffer, size_t *lenp)
{
	char *p;
	p = ctl->data;

	if (write) {
		proc_dointvec(ctl, write, filp, buffer, lenp);
		if (mdet_mech < 0)
			mdet_mech = EAGER_CELL_SWITCHING; /* default */

		/* Add new movement detection mechanism below */
		switch (mdet_mech) 
		{
		case EAGER_CELL_SWITCHING: break;
		}
	} 
	else {
		sprintf(p, "%d", mdet_mech);
		proc_dointvec(ctl, write, filp, buffer, lenp);
	}
	
	return 0;
}

static int mipv6_home_address_sysctl_handler(
	ctl_table *ctl, int write, struct file * filp,
	void *buffer, size_t *lenp)
{
	struct in6_addr home_addr, home_prefix;
	int i, itmp[4], iptmp[4], ret, plen, mask, tplen;
	char *p, *pc;
	char tmp[9];

	p = ctl->data;

	if (!write) {
		memset(p, 0, 36);
		mipv6_mn_get_homeaddr(&home_addr);
		sprintf(p, "%04x%04x%04x%04x%04x%04x%04x%04x", 
			NIPV6ADDR(&home_addr));
	}

	ret = proc_dostring(ctl, write, filp, buffer, lenp);

	if (write) {
		if (strlen(p) < 35) {
			DEBUG((DBG_ERROR, "Invalid home address value %s", p));
			return 0;
		}
		for (i = 0; i < 4; i++) {
			memset(tmp, 0, 9);
			strncpy(tmp, p, 8);
			pc = tmp;
			itmp[i] = simple_strtoul(tmp, &pc, 16);
			if (*pc) return 0; /* could not convert to number*/
			p+=8;
		}
		p += 1;
		memset(tmp, 0, 3);
		strncpy(tmp, p, 2);
		pc = tmp;
		plen = simple_strtoul(tmp, &pc, 16);
		if (*pc) return 0;

		tplen = plen;
		for (i = 0; i < 4; i++) {
			mask = (~0);
			if (tplen < 32 && tplen > 0)
				mask <<= tplen;
			if (tplen <= 0)
				mask = 0;
			iptmp[i] = itmp[i] & mask;
			tplen -= 32;
		}

		ipv6_addr_set(&home_prefix,
			      (int)htonl(iptmp[0]), (int)htonl(iptmp[1]),
			      (int)htonl(iptmp[2]), (int)htonl(iptmp[3]));

		ipv6_addr_set(&home_addr, (int)htonl(itmp[0]),
			      (int)htonl(itmp[1]), (int)htonl(itmp[2]),
			      (int)htonl(itmp[3]));

		mipv6_mn_set_info(&home_addr, plen, 0);

		DEBUG((DBG_INFO, "Setting home address to: "
		       "%x:%x:%x:%x:%x:%x:%x:%x", NIPV6ADDR(&home_addr)));
		DEBUG((DBG_INFO, "Home address prefix (plen=%d:) "
		       "%x:%x:%x:%x:%x:%x:%x:%x", plen,
		       NIPV6ADDR(&home_prefix)));
		if(!mdetect_is_initialized) {
			mdetect_is_initialized = 1;
			/* We add the timer only after mdetect is configured */		
			r_timer.expires = jiffies + HZ;
			add_timer(&r_timer);
		}
	}
	return ret;
}

#endif /* CONFIG_SYSCTL */


int __init mipv6_initialize_mdetect(void)
{

	DEBUG_FUNC();

#ifdef CONFIG_SYSCTL
	mipv6_mdetect_sysctl_register();
#endif	
	init_timer(&r_timer);
	
	return 0;
}

int __exit mipv6_shutdown_mdetect()
{
	unsigned long flags;
	DEBUG_FUNC();
	
#ifdef CONFIG_SYSCTL
	mipv6_mdetect_sysctl_unregister();
#endif	

	del_timer(&r_timer);
	/* Free the memory allocated by router list */
	spin_lock_irqsave(&router_lock, flags);
	list_free(&curr_router);
	spin_unlock_irqrestore(&router_lock, flags);

	return 0;
}

