/*
 *
 * Definitions for Samsung HAMCOMP Asic Companion chip
 *
 * Copyright 2003 Hewlett-Packard Company.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Jamey Hicks
 *
 * Names and values copied from the H2200 Asic Companion specifications:
 * http://www.handhelds.org/platforms/hp/ipaq-h22xx/
 * 
 * History:
 * 
 * See http://cvs.handhelds.org/cgi-bin/viewcvs.cgi/linux/kernel26/include/asm-arm/hardware/ipaq-hamcop.h
 * for the history of changes.
 */

#ifndef _INCLUDE_IPAQ_HAMCOP_H_ 
#define _INCLUDE_IPAQ_HAMCOP_H_

/******* Macros ********/

#define SHAMCOP_REGISTER(_b,s,x,y)					\
     (*((volatile s *) (_b + (_SHAMCOP_ ## x ## _ ## y))))

#define GET_SHAMCOP_BIT(x,y,b) ((SHAMCOP_ ## x ## _ ## y)&(_SHAMCOP_ ## x ## _ ## y ## _ ## b))

#define GET_SHAMCOP_GPIO(_b, gpx, gpio)					\
	((SHAMCOP_GPIO_ ## gpx ## _DAT(_b) & SHAMCOP_GPIO_ ## gpx ## _ ## gpio) \
	 >> _SHAMCOP_GPIO_ ## gpx ## _ ## gpio)

#define SET_SHAMCOP_GPIO(_b, gpx, gpio, setp)	\
do { \
if (setp) \
	SHAMCOP_GPIO_ ## gpx ## _DAT(_b) |= SHAMCOP_GPIO_ ## gpx ## _ ## gpio; \
else \
	SHAMCOP_GPIO_ ## gpx ## _DAT(_b) &= ~(SHAMCOP_GPIO_ ## gpx ## _ ## gpio); \
} while (0)

#define SET_SHAMCOP_GPIO_N(_b, gpx, gpio, setp)	\
do { \
if (setp) \
	SHAMCOP_GPIO_ ## gpx ## _DAT(_b) &= ~(SHAMCOP_GPIO_ ## gpx ## _ ## gpio); \
else \
	SHAMCOP_GPIO_ ## gpx ## _DAT(_b) |= SHAMCOP_GPIO_ ## gpx ## _ ## gpio; \
} while (0)

/******* General stuff *********/

#define _SHAMCOP_SRAM_Base		0x00000
#define SHAMCOP_SRAM_SIZE		16384

/* CPU Interface Control */

#define _SHAMCOP_CIF_Base		0x04000
#define _SHAMCOP_CIF_IFMODE		0x0000
#define _SHAMCOP_CIF_BOOTMODE		0x0002

/******* Interrupt controller ********/

#define _SHAMCOP_IC_Base		0x04800

#define _SHAMCOP_IC_SRCPND		0x00000
#define _SHAMCOP_IC_SRCPND0             0x00000
#define _SHAMCOP_IC_SRCPND1             0x00004
#define _SHAMCOP_IC_INTMSK		0x00008
#define _SHAMCOP_IC_INTMSK0             0x00008
#define _SHAMCOP_IC_INTMSK1             0x0000c
#define _SHAMCOP_IC_PRIORITY		0x00010
#define _SHAMCOP_IC_PRIORITY0		0x00010
#define _SHAMCOP_IC_PRIORITY1		0x00014
#define _SHAMCOP_IC_INTPND		0x00018
#define _SHAMCOP_IC_INTPND0		0x00018
#define _SHAMCOP_IC_INTPND1		0x0001c
#define _SHAMCOP_IC_DREQ		0x00020

#define SHAMCOP_IC_SRCPND0(_b)         SHAMCOP_REGISTER(_b, u16, IC, SRCPND0 ) /* raw pending interrupts */
#define SHAMCOP_IC_SRCPND1(_b)         SHAMCOP_REGISTER(_b, u16, IC, SRCPND1 ) /* raw pending interrupts */
#define SHAMCOP_IC_SRCPND(_b)          (SHAMCOP_IC_SRCPND0(_b) | (SHAMCOP_IC_SRCPND1(_b) << 16))       /* only use this for reads */

#define SHAMCOP_IC_INTMSK0(_b)         SHAMCOP_REGISTER(_b, u16, IC, INTMSK0 ) /* interrupt mask */
#define SHAMCOP_IC_INTMSK1(_b)         SHAMCOP_REGISTER(_b, u16, IC, INTMSK1 ) /* interrupt mask */
#define SHAMCOP_IC_INTMSK(_b)          (SHAMCOP_IC_INTMSK0(_b) | (SHAMCOP_IC_INTMSK1(_b) << 16))       /* only use this for reads */

#define SHAMCOP_IC_PRIORITY0(_b)               SHAMCOP_REGISTER(_b, u16, IC, PRIORITY0 ) /* interrupt priority control */
#define SHAMCOP_IC_PRIORITY1(_b)               SHAMCOP_REGISTER(_b, u16, IC, PRIORITY1 ) /* interrupt priority control */

#define SHAMCOP_IC_INTPND0(_b)         SHAMCOP_REGISTER(_b, u16, IC, INTPND0 ) /* pending interrupts, accounting for mask */
#define SHAMCOP_IC_INTPND1(_b)         SHAMCOP_REGISTER(_b, u16, IC, INTPND1 ) /* pending interrupts, accounting for mask */
#define SHAMCOP_IC_INTPND(_b)          (SHAMCOP_IC_INTPND0(_b) | (SHAMCOP_IC_INTPND1(_b) << 16))               /* only use this for reads */

#define SHAMCOP_IC_DREQ(_b)            SHAMCOP_REGISTER(_b, u8, IC, DREQ )   /* DREQ control */

#define _SHAMCOP_IC_INT_NFIF  		25       
#define _SHAMCOP_IC_INT_KEYPAD  	24       
#define _SHAMCOP_IC_INT_WDT  		23       
#define _SHAMCOP_IC_INT_ADC  		22       
#define _SHAMCOP_IC_INT_ADCTS		21
#define _SHAMCOP_IC_INT_SDHC		20
#define _SHAMCOP_IC_INT_ONEWIRE		19
#define _SHAMCOP_IC_INT_ALARM		18
#define _SHAMCOP_IC_INT_TICK		17
#define _SHAMCOP_IC_INT_DMA		16
#define _SHAMCOP_IC_INT_EAR_IN		15
#define _SHAMCOP_IC_INT_GPB6		14
#define _SHAMCOP_IC_INT_GPB5		13
#define _SHAMCOP_IC_INT_GPB4		12
#define _SHAMCOP_IC_INT_GPB3		11
#define _SHAMCOP_IC_INT_JOYPAD_ACTION	10 
#define _SHAMCOP_IC_INT_JOYPAD_RIGHT	 9
#define _SHAMCOP_IC_INT_JOYPAD_LEFT	 8
#define _SHAMCOP_IC_INT_BATT_FLT	 7
#define _SHAMCOP_IC_INT_JOYPAD_DOWN	 6
#define _SHAMCOP_IC_INT_JOYPAD_UP	 5
#define _SHAMCOP_IC_INT_APPBUTTON4	 4
#define _SHAMCOP_IC_INT_APPBUTTON3	 3
#define _SHAMCOP_IC_INT_APPBUTTON2	 2
#define _SHAMCOP_IC_INT_APPBUTTON1	 1
#define _SHAMCOP_IC_INT_RSTBUTTON	 0

#define IRQ_SHAMCOP_BUTTON_MASK		((1 << _SHAMCOP_IC_INT_APPBUTTON1)|(1 << _SHAMCOP_IC_INT_APPBUTTON2)\
				   	|(1 << _SHAMCOP_IC_INT_APPBUTTON3)|(1 << _SHAMCOP_IC_INT_APPBUTTON4))

#define IRQ_SHAMCOP_FCD_MASK    	(1 << _SHAMCOP_IC_INT_FCD)
#define IRQ_SHAMCOP_ONEWIRE_MASK	(1 << _SHAMCOP_IC_INT_ONEWIRE)
#define IRQ_SHAMCOP_UERR_MASK		((1 << _SHAMCOP_IC_INT_UERR0)|(1 << _SHAMCOP_IC_INT_UERR1))
#define IRQ_SHAMCOP_DMA_MASK		((1 << _SHAMCOP_IC_INT_DMA0)|(1 << _SHAMCOP_IC_INT_DMA1))
#define IRQ_SHAMCOP_SD_MASK		((1 << _SHAMCOP_IC_INT_SD)|(1 << _SHAMCOP_IC_INT_SD_WAKEUP))
#define IRQ_SHAMCOP_URXD_MASK		((1 << _SHAMCOP_IC_INT_URXD0)|(1 << _SHAMCOP_IC_INT_URXD1))
#define IRQ_SHAMCOP_USBH_MASK		(1 << _SHAMCOP_IC_INT_USBH)
#define IRQ_SHAMCOP_ACCEL_MASK		(1 << _SHAMCOP_IC_INT_ACCEL)
#define IRQ_SHAMCOP_WAKEUP_MASK		((1 << _SHAMCOP_IC_INT_WAKEUP2)|(1 << _SHAMCOP_IC_INT_WAKEUP3))
#define IRQ_SHAMCOP_UTXD_MASK		((1 << _SHAMCOP_IC_INT_UTXD0)|(1 << _SHAMCOP_IC_INT_UTXD1))
#define IRQ_SHAMCOP_ADCTS_MASK		(1 << _SHAMCOP_IC_INT_ADCTS)
#define IRQ_SHAMCOP_ADC_MASK		(1 << _SHAMCOP_IC_INT_ADC)
#define IRQ_SHAMCOP_GPIO_MASK		(1 << _SHAMCOP_IC_INT_GPIO)

/********* DMA **********/

/* DMAC: old SAMCOP setting: #define _SHAMCOP_DMAC_Base                0x04400 */

#define _SHAMCOP_DMA_Base		0x04400
#define _SHAMCOP_DMA_DISRC0		0x0000
#define _SHAMCOP_DMA_DISRC1		0x0004
#define _SHAMCOP_DMA_DIDST0		0x0008
#define _SHAMCOP_DMA_DIDST1		0x000c
#define _SHAMCOP_DMA_DCON0		0x0010
#define _SHAMCOP_DMA_DCON1		0x0014
#define _SHAMCOP_DMA_DSTAT0		0x0018
#define _SHAMCOP_DMA_DSTAT1		0x001c
#define _SHAMCOP_DMA_DCSRC		0x0020
#define _SHAMCOP_DMA_DCDST		0x0024
#define _SHAMCOP_DMA_DMASKTRIG		0x0028

/* FIXME: Still need to declare DMA registers according to specifications */

/********* Clock and Power Management **********/

#define _SHAMCOP_CPM_Base		0x04c00
#define _SHAMCOP_CPM_LockTime		0x00000 /* defaults to 0x0fff */
#define _SHAMCOP_CPM_PllControl		0x00004 /* defaults to 0x2820 */
#define _SHAMCOP_CPM_ClockControl	0x00008 /* defaults to 0x03fe */
#define _SHAMCOP_CPM_ClockSleep		0x0000c

#define SHAMCOP_CPM_LockTime(_b)	SHAMCOP_REGISTER( _b, u16, CPM, LockTime )
#define SHAMCOP_CPM_PllControl(_b)	SHAMCOP_REGISTER( _b, u16, CPM, PllControl )
#define SHAMCOP_CPM_ClockControl(_b)	SHAMCOP_REGISTER( _b, u16, CPM, ClockControl )
#define SHAMCOP_CPM_ClockSleep(_b)	SHAMCOP_REGISTER( _b, u8, CPM, ClockSleep )

/* upll lock time count value for uclk, ltime > 150us */

#define SHAMCOP_CPM_LOCKTIME_MASK    ((1 << 11) -1) 
#define SHAMCOP_CPM_PLLCON_MDIV_SHIFT 8 /* main divider control */
#define SHAMCOP_CPM_PLLCON_MDIV_WIDTH 8
#define SHAMCOP_CPM_PLLCON_PDIV_SHIFT 2 /* pre-divider control */
#define SHAMCOP_CPM_PLLCON_PDIV_WIDTH 6
#define SHAMCOP_CPM_PLLCON_SDIV_SHIFT 0 /* post divider control */
#define SHAMCOP_CPM_PLLCON_SDIV_WIDTH 2

#define SHAMCOP_CPM_CLKCON_GPIO_CLKEN	(1 << 9)
#define SHAMCOP_CPM_CLKCON_ADC_CLKEN	(1 << 8)
#define SHAMCOP_CPM_CLKCON_1WIRE_CLKEN	(1 << 7)
#define SHAMCOP_CPM_CLKCON_SD_CLKEN	(1 << 6)
#define SHAMCOP_CPM_CLKCON_RTC_CLKEN	(1 << 5)
#define SHAMCOP_CPM_CLKCON_LED_CLKEN	(1 << 4)
#define SHAMCOP_CPM_CLKCON_NAND_CLKEN	(1 << 3)
#define SHAMCOP_CPM_CLKCON_DMAC_CLKEN	(1 << 2)
#define SHAMCOP_CPM_CLKCON_KEYPAD_CLKEN	(1 << 1)
#define SHAMCOP_CPM_CLKCON_PLLON_CLKEN	(1 << 0)

#define SHAMCOP_CPM_CLKSLEEP_XTCOM	(1 << 7)
#define SHAMCOP_CPM_CLKSLEEP_UCLK_ON	(1 << 3)
#define SHAMCOP_CPM_CLKSLEEP_CLKSEL	(1 << 1)
#define SHAMCOP_CPM_CLKSLEEP_SLEEP	(1 << 0)


/********* NAND Flash Controller **********/

#define _SHAMCOP_NF_Base		0x05000

#define _SHAMCOP_NF_CONF0		0x0000
#define _SHAMCOP_NF_CONT0		0x0008
#define _SHAMCOP_NF_CONT1		0x000c
#define _SHAMCOP_NF_CMMD 		0x0010
#define _SHAMCOP_NF_ADDR0		0x0014
#define _SHAMCOP_NF_DATA 		0x001c
#define _SHAMCOP_NF_ECCL0		0x0020
#define _SHAMCOP_NF_ECCL1		0x0024
#define _SHAMCOP_NF_ECCH0		0x0028
#define _SHAMCOP_NF_ECCH1		0x002c
#define _SHAMCOP_NF_STAT0		0x0030 /* reserved */
#define _SHAMCOP_NF_STAT1		0x0034
#define _SHAMCOP_NF_SBLK0		0x0038
#define _SHAMCOP_NF_SBLK1		0x003c
#define _SHAMCOP_NF_EBLK0		0x0040
#define _SHAMCOP_NF_EBLK1 		0x0044

#define SHAMCOP_NF_CONF0(_b)		SHAMCOP_REGISTER( _b, u16, NF, CONF0 )
#define SHAMCOP_NF_CONT0(_b)		SHAMCOP_REGISTER( _b, u16, NF, CONT0 )
#define SHAMCOP_NF_CONT1(_b)		SHAMCOP_REGISTER( _b, u16, NF, CONT1 )
#define SHAMCOP_NF_CMMD(_b) 		SHAMCOP_REGISTER( _b, u16, NF, CMMD )
#define SHAMCOP_NF_ADDR0(_b)		SHAMCOP_REGISTER( _b, u16, NF, ADDR0 )
#define SHAMCOP_NF_DATA(_b) 		SHAMCOP_REGISTER( _b, u16, NF, DATA )
#define SHAMCOP_NF_ECCL0(_b)		SHAMCOP_REGISTER( _b, u16, NF, ECCL0 )
#define SHAMCOP_NF_ECCL1(_b)		SHAMCOP_REGISTER( _b, u16, NF, ECCL1 )
#define SHAMCOP_NF_ECCH0(_b)		SHAMCOP_REGISTER( _b, u16, NF, ECCH0 )
#define SHAMCOP_NF_ECCH1(_b)		SHAMCOP_REGISTER( _b, u16, NF, ECCH1 )
#define SHAMCOP_NF_STAT0(_b)		SHAMCOP_REGISTER( _b, u16, NF, STAT0 )
#define SHAMCOP_NF_STAT1(_b)		SHAMCOP_REGISTER( _b, u16, NF, STAT1 )
#define SHAMCOP_NF_SBLK0(_b)		SHAMCOP_REGISTER( _b, u16, NF, SBLK0 )
#define SHAMCOP_NF_SBLK1(_b)		SHAMCOP_REGISTER( _b, u16, NF, SBLK1 )
#define SHAMCOP_NF_EBLK0(_b)		SHAMCOP_REGISTER( _b, u16, NF, EBLK0 )
#define SHAMCOP_NF_EBLK1(_b)		SHAMCOP_REGISTER( _b, u16, NF, EBLK1 )

#define SHAMCOP_NF_CONF0_TACLS            0x7000
#define SHAMCOP_NF_CONF0_TACLS_0          0x0000
#define SHAMCOP_NF_CONF0_TACLS_1          0x1000
#define SHAMCOP_NF_CONF0_TACLS_2          0x2000
#define SHAMCOP_NF_CONF0_TACLS_3          0x3000
#define SHAMCOP_NF_CONF0_TACLS_4          0x4000
#define SHAMCOP_NF_CONF0_TACLS_5          0x5000
#define SHAMCOP_NF_CONF0_TACLS_6          0x6000
#define SHAMCOP_NF_CONF0_TACLS_7          0x7000
#define SHAMCOP_NF_CONF0_TWRPH0           0x0700
#define SHAMCOP_NF_CONF0_TWRPH0_0         0x0000
#define SHAMCOP_NF_CONF0_TWRPH0_1         0x0100
#define SHAMCOP_NF_CONF0_TWRPH0_2         0x0200
#define SHAMCOP_NF_CONF0_TWRPH0_3         0x0300
#define SHAMCOP_NF_CONF0_TWRPH0_4         0x0400
#define SHAMCOP_NF_CONF0_TWRPH0_5         0x0500
#define SHAMCOP_NF_CONF0_TWRPH0_6         0x0600
#define SHAMCOP_NF_CONF0_TWRPH0_7         0x0700
#define SHAMCOP_NF_CONF0_TWRPH1           0x0070
#define SHAMCOP_NF_CONF0_TWRPH1_0         0x0000
#define SHAMCOP_NF_CONF0_TWRPH1_1         0x0010
#define SHAMCOP_NF_CONF0_TWRPH1_2         0x0020
#define SHAMCOP_NF_CONF0_TWRPH1_3         0x0030
#define SHAMCOP_NF_CONF0_TWRPH1_4         0x0040
#define SHAMCOP_NF_CONF0_TWRPH1_5         0x0050
#define SHAMCOP_NF_CONF0_TWRPH1_6         0x0060
#define SHAMCOP_NF_CONF0_TWRPH1_7         0x0070
#define SHAMCOP_NF_CONF0_HW_NCE           0x0008
#define SHAMCOP_NF_CONF0_BUSWIDTH         0x0004  /* read-only */
#define SHAMCOP_NF_CONF0_PAGESIZE         0x0002  /* read-only */
#define SHAMCOP_NF_CONF0_ADDRCYCLE        0x0001  /* read-only */

#define SHAMCOP_NF_CONT0_WP_EN            (1 << 12)
#define SHAMCOP_NF_CONT0_INITECC          (1 << 11)
#define SHAMCOP_NF_CONT0_NFNCE            (1 << 10)
#define SHAMCOP_NF_CONT0_LOCKTIGHT        (1 << 3)
#define SHAMCOP_NF_CONT0_LOCK             (1 << 2)
#define SHAMCOP_NF_CONT0_MODE             (3 << 0)
#define SHAMCOP_NF_CONT0_MODE_DISABLE     (0 << 0)
#define SHAMCOP_NF_CONT0_MODE_NFMODE      (3 << 0)

#define SHAMCOP_NF_CONT1_ENBILLEGALACCINT 0x0010
#define SHAMCOP_NF_CONT1_RNBTRANSMODE     0x0001
#define SHAMCOP_NF_STAT1_ILLEGALACCESS    0x0040
#define SHAMCOP_NF_STAT1_RNB_TRANSDETECT  0x0008
#define SHAMCOP_NF_STAT1_FLASH_NCE        0x0004  /* read-only */
#define SHAMCOP_NF_STAT1_FLASH_RNB        0x0001  /* read-only */


/********* ADC **********/

#define _SHAMCOP_ADC_Base                  0x07800
#define _SHAMCOP_ADC_Control               0x00000 /* resets to 0x3fc4 */
#define _SHAMCOP_ADC_TouchScreenControl    0x00004 /* resets to 0x58 */
#define _SHAMCOP_ADC_Delay                 0x00008 /* initializes to 0xff */
#define _SHAMCOP_ADC_Data0                 0x0000c /* xp conversion data register */
#define _SHAMCOP_ADC_Data1                 0x00010 /* yp conversion data register */

#define SHAMCOP_ADC_Control(_b)			SHAMCOP_REGISTER( _b,  u16, ADC, Control )
#define SHAMCOP_ADC_TouchScreenControl(_b)	SHAMCOP_REGISTER( _b,  u16, ADC, TouchScreenControl )
#define SHAMCOP_ADC_Delay(_b)			SHAMCOP_REGISTER( _b, u16, ADC, Delay )
#define SHAMCOP_ADC_Data0(_b)			SHAMCOP_REGISTER( _b, u16, ADC, Data0 )
#define SHAMCOP_ADC_Data1(_b)			SHAMCOP_REGISTER( _b, u16, ADC, Data1 )

#define SHAMCOP_ADC_CONTROL_FORCE_START		(1 << 0) /* enables conversion start, bit cleared on start */
#define SHAMCOP_ADC_CONTROL_READ_START		(1 << 1) /* enable start by read operation */
#define SHAMCOP_ADC_CONTROL_STANDBY		(1 << 2)

#define SHAMCOP_ADC_CONTROL_SEL_AIN0		(0 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_AIN1		(1 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_AIN2		(2 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_AIN3		(3 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_YM		(4 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_YP		(5 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_XM		(6 << 3)
#define SHAMCOP_ADC_CONTROL_SEL_XP		(7 << 3)

#define SHAMCOP_ADC_CONTROL_PRESCALER_SHIFT	6  
#define SHAMCOP_ADC_CONTROL_PRESCALER_WIDTH	8	  
#define SHAMCOP_ADC_CONTROL_PRESCALER_ENABLE	(1 << 14)
#define SHAMCOP_ADC_CONTROL_CONVERSION_END	(1 << 15) /* read only */ 

/* Touch screen measurement mode */

#define SHAMCOP_ADC_TSC_XYPST_NONE		(0 << 0)
#define SHAMCOP_ADC_TSC_XYPST_XPOS		(1 << 0)
#define SHAMCOP_ADC_TSC_XYPST_YPOS		(2 << 0)
#define SHAMCOP_ADC_TSC_XYPST_INTERRUPT		(3 << 0)
#define SHAMCOP_ADC_TSC_AUTO_MODE		(1 << 2) /* enable auto mode */
#define SHAMCOP_ADC_TSC_PULL_UP			(1 << 3) /* xp pullup disable */
#define SHAMCOP_ADC_TSC_XP_SEN			(1 << 4) /* xp output driver disable */
#define SHAMCOP_ADC_TSC_XM_SEN			(1 << 5) /* xm output driver disable */
#define SHAMCOP_ADC_TSC_YP_SEN			(1 << 6) /* yp output driver disable */
#define SHAMCOP_ADC_TSC_YM_SEN			(1 << 7) /* ym output driver disable */

#define SHAMCOP_ADC_DATA0_XPDATA		((1 << 10) - 1)
#define SHAMCOP_ADC_DATA0_XYPST_NONE		(0 << 12)
#define SHAMCOP_ADC_DATA0_XYPST_XPOS		(1 << 12)
#define SHAMCOP_ADC_DATA0_XYPST_YPOS		(2 << 12)
#define SHAMCOP_ADC_DATA0_XYPST_INTERRUPT	(3 << 12)
#define SHAMCOP_ADC_DATA0_AUTO_MODE		(1 << 14)
#define SHAMCOP_ADC_DATA0_PEN_UP		(1 << 15) /* stylus up */ 

/********* Watchdog Timer **********/

#define _SHAMCOP_WT_Base		0x06000
#define _SHAMCOP_WT_Control		0x00000
#define _SHAMCOP_WT_Data		0x00004
#define _SHAMCOP_WT_Count		0x00008

#define SHAMCOP_WT_Control(_b)		SHAMCOP_REGISTER( _b, u16, WT, Control )
#define SHAMCOP_WT_Data(_b)		SHAMCOP_REGISTER( _b, u16, WT, Data )
#define SHAMCOP_WT_Count(_b)		SHAMCOP_REGISTER( _b, u16, WT, Count )

/********* UART **********/

/* No UART available in HAMCOP */
/* #define _SHAMCOP_UART_Base                 0x90000 */

/********* MISC **********/
/* #define _SHAMCOP_MISC_Base                 0xa0000 */

/********* OWM **********/

#define _SHAMCOP_OWM_Base                   0x06800
#define _SHAMCOP_OWM_Command                0x0000    /* R/W 4 bits command register */
#define _SHAMCOP_OWM_Data                   0x0004    /* R/W 8 bits, transmit / receive buffer */
#define _SHAMCOP_OWM_Interrupt              0x0008    /* R/W Command register */
#define _SHAMCOP_OWM_InterruptEnable        0x000c    /* R/W Command register */
#define _SHAMCOP_OWM_ClockDivisor           0x0010    /* R/W 5 bits of divisor and pre-scale */

#define SHAMCOP_OWM_Command(_b)            SHAMCOP_REGISTER( _b, u8, OWM, Command )
#define SHAMCOP_OWM_Data(_b)               SHAMCOP_REGISTER( _b, u8, OWM, Data )
#define SHAMCOP_OWM_Interrupt(_b)          SHAMCOP_REGISTER( _b, u8, OWM, Interrupt )
#define SHAMCOP_OWM_InterruptEnable(_b)    SHAMCOP_REGISTER( _b, u8, OWM, InterruptEnable )
#define SHAMCOP_OWM_ClockDivisor(_b)       SHAMCOP_REGISTER( _b, u8, OWM, ClockDivisor )

#define OWM_CMD_ONE_WIRE_RESET ( 1 << 0 )    /* Set to force reset on 1-wire bus */
#define OWM_CMD_SRA            ( 1 << 1 )    /* Set to switch to Search ROM accelerator mode */
#define OWM_CMD_DQ_OUTPUT      ( 1 << 2 )    /* Write only - forces bus low */
#define OWM_CMD_DQ_INPUT       ( 1 << 3 )    /* Read only - reflects state of bus */

#define OWM_INT_PD             ( 1 << 0 )    /* Presence detect */
#define OWM_INT_PDR            ( 1 << 1 )    /* Presence detect result */
#define OWM_INT_TBE            ( 1 << 2 )    /* Transmit buffer empty */
#define OWM_INT_TSRE           ( 1 << 3 )    /* Transmit shift register empty */
#define OWM_INT_RBF            ( 1 << 4 )    /* Receive buffer full */
#define OWM_INT_RSBF           ( 1 << 5 )    /* Receive shift buffer full */

#define OWM_INTEN_EPD          ( 1 << 0 )    /* Enable presence detect interrupt */
#define OWM_INTEN_IAS          ( 1 << 1 )    /* INTR active state */
#define OWM_INTEN_ETBE         ( 1 << 2 )    /* Enable transmit buffer empty interrupt */
#define OWM_INTEN_ETMT         ( 1 << 3 )    /* Enable transmit shift register empty interrupt */
#define OWM_INTEN_ERBF         ( 1 << 4 )    /* Enable receive buffer full interrupt */
#define OWM_INTEN_ERSBF        ( 1 << 5 )    /* Enable receive shift buffer full interrupt */

/********* Secure Digital Interface **********/

#define _SHAMCOP_SDI_Base		0x06c00

/********* GPIO **********/

#define _SHAMCOP_GPIO_Base		0x07000

#define _SHAMCOP_GPIO_GPA_CON		0x0000 /* reset value is 0x8000 */
#define _SHAMCOP_GPIO_GPA_DAT		0x0004 /* gpio A data register, reset value is 0x00000000 */
#define _SHAMCOP_GPIO_GPA_PUP		0x0008 /* pull up disable register, reset value is 0x00080000 */

#define SHAMCOP_GPIO_GPA_CON(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA_CON )
#define SHAMCOP_GPIO_GPA_DAT(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA_DAT ) /* gpio A data register */
#define SHAMCOP_GPIO_GPA_PUP(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA_PUP ) /* pull up disable register */

#define _SHAMCOP_GPIO_GPB_CON		0x0010
#define _SHAMCOP_GPIO_GPB_DAT		0x0014 
#define _SHAMCOP_GPIO_GPB_PUP		0x0018

#define SHAMCOP_GPIO_GPB_CON(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB_CON )
#define SHAMCOP_GPIO_GPB_DAT(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB_DAT ) /* gpio B data register */
#define SHAMCOP_GPIO_GPB_PUP(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB_PUP ) /* gpio B pull up disable register */

#define _SHAMCOP_GPIO_GPC_CON		0x0020
#define _SHAMCOP_GPIO_GPC_DAT		0x0024 
#define _SHAMCOP_GPIO_GPC_PUP		0x0028

#define SHAMCOP_GPIO_GPC_CON(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPC_CON )
#define SHAMCOP_GPIO_GPC_DAT(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPC_DAT ) /* gpio C data register */
#define SHAMCOP_GPIO_GPC_PUP(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPC_PUP ) /* gpio C pull up disable register */

#define _SHAMCOP_GPIO_GPD_CON		0x0030
#define _SHAMCOP_GPIO_GPD_DAT		0x0034 
#define _SHAMCOP_GPIO_GPD_PUP		0x0038

#define SHAMCOP_GPIO_GPD_CON(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPD_CON )
#define SHAMCOP_GPIO_GPD_DAT(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPD_DAT ) /* gpio D data register */
#define SHAMCOP_GPIO_GPD_PUP(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPD_PUP ) /* gpio D pull up disable register */

	
#define _SHAMCOP_GPIO_CLKOUTCON		0x0040
#define _SHAMCOP_GPIO_GPAINT0		0x0050
#define _SHAMCOP_GPIO_GPAINT1		0x0054
#define _SHAMCOP_GPIO_GPBINT0		0x0058
#define _SHAMCOP_GPIO_GPBINT1		0x005c
#define _SHAMCOP_GPIO_GPA0FLT0		0x0060
#define _SHAMCOP_GPIO_GPA0FLT1		0x0064
#define _SHAMCOP_GPIO_GPA1FLT0		0x0068
#define _SHAMCOP_GPIO_GPA1FLT1		0x006c
#define _SHAMCOP_GPIO_GPA2FLT0		0x0070
#define _SHAMCOP_GPIO_GPA2FLT1		0x0074
#define _SHAMCOP_GPIO_GPA3FLT0		0x0078
#define _SHAMCOP_GPIO_GPA3FLT1		0x007c
#define _SHAMCOP_GPIO_GPA4FLT0		0x0080
#define _SHAMCOP_GPIO_GPA4FLT1		0x0084
#define _SHAMCOP_GPIO_GPA5FLT0		0x0088
#define _SHAMCOP_GPIO_GPA5FLT1		0x008c
#define _SHAMCOP_GPIO_GPA6FLT0		0x0090
#define _SHAMCOP_GPIO_GPA6FLT1		0x0094
#define _SHAMCOP_GPIO_GPA7FLT0		0x0098
#define _SHAMCOP_GPIO_GPA7FLT1		0x009c
#define _SHAMCOP_GPIO_GPB0FLT0		0x00a0
#define _SHAMCOP_GPIO_GPB0FLT1		0x00a4
#define _SHAMCOP_GPIO_GPB1FLT0		0x00a8
#define _SHAMCOP_GPIO_GPB1FLT1		0x00ac
#define _SHAMCOP_GPIO_GPB2FLT0		0x00b0
#define _SHAMCOP_GPIO_GPB2FLT1		0x00b4
#define _SHAMCOP_GPIO_GPB3FLT0		0x00b8
#define _SHAMCOP_GPIO_GPB3FLT1		0x00bc
#define _SHAMCOP_GPIO_GPB4FLT0		0x00c0
#define _SHAMCOP_GPIO_GPB4FLT1		0x00c4
#define _SHAMCOP_GPIO_GPB5FLT0		0x00c8
#define _SHAMCOP_GPIO_GPB5FLT1		0x00cc
#define _SHAMCOP_GPIO_GPB6FLT0		0x00d0
#define _SHAMCOP_GPIO_GPB6FLT1		0x00d4
#define _SHAMCOP_GPIO_GPB7FLT0		0x00d8
#define _SHAMCOP_GPIO_GPB7FLT1		0x00dc

#define _SHAMCOP_GPIO_MON		0x00e0
#define _SHAMCOP_GPIO_SPUCR		0x00e4

#define SHAMCOP_GPIO_CLKOUTCON(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, CLKOUTCON )
#define SHAMCOP_GPIO_GPAINT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPAINT0 ) /* gpio interrupt config 1 */
#define SHAMCOP_GPIO_GPAINT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPAINT1 ) /* gpio interrupt config 1 */
#define SHAMCOP_GPIO_GPBINT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPBINT0 ) /* gpio interrupt config 1 */
#define SHAMCOP_GPIO_GPBINT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPBINT1 ) /* gpio interrupt config 1 */
#define SHAMCOP_GPIO_GPA0FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA0FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA0FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA0FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA1FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA1FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA1FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA1FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA2FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA2FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA2FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA2FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA3FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA3FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA3FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA3FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA4FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA4FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA4FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA4FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA5FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA5FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA5FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA5FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA6FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA6FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA6FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA6FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA7FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA7FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPA7FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPA7FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB0FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB0FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB0FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB0FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB1FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB1FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB1FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB1FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB2FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB2FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB2FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB2FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB3FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB3FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB3FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB3FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB4FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB4FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB4FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB4FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB5FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB5FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB5FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB5FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB6FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB6FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB6FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB6FLT1 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB7FLT0(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB7FLT0 ) /* gpio filter config 1 */
#define SHAMCOP_GPIO_GPB7FLT1(_b)	SHAMCOP_REGISTER( _b, u16, GPIO, GPB7FLT1 ) /* gpio filter config 1 */

#define SHAMCOP_GPIO_MON(_b)		SHAMCOP_REGISTER( _b, u16, GPIO, MON )
#define SHAMCOP_GPIO_SPUCR(_b)		SHAMCOP_REGISTER( _b, u8, GPIO, SPUCR )

/********* RTC **********/

#define _SHAMCOP_RTC_Base            	0x06440

#define _SHAMCOP_ASIC_RTC_RTCCON	0x0000
#define _SHAMCOP_ASIC_RTC_TICNT		0x0004
#define _SHAMCOP_ASIC_RTC_RTCALM	0x0010
#define _SHAMCOP_ASIC_RTC_ALMSEC	0x0014
#define _SHAMCOP_ASIC_RTC_ALMMIN	0x0018
#define _SHAMCOP_ASIC_RTC_ALMHOUR	0x001c
#define _SHAMCOP_ASIC_RTC_ALMDAY	0x0020
#define _SHAMCOP_ASIC_RTC_ALMMON	0x0024
#define _SHAMCOP_ASIC_RTC_ALMYEAR	0x0028
#define _SHAMCOP_ASIC_RTC_RTCRST	0x002c
#define _SHAMCOP_ASIC_RTC_BCDSEC	0x0030
#define _SHAMCOP_ASIC_RTC_BCDMIN	0x0034
#define _SHAMCOP_ASIC_RTC_BCDHOUR	0x0038
#define _SHAMCOP_ASIC_RTC_BCDDAY	0x003c
#define _SHAMCOP_ASIC_RTC_BCDDATE	0x0040
#define _SHAMCOP_ASIC_RTC_BCDMON	0x0044
#define _SHAMCOP_ASIC_RTC_BCDYEAR	0x0048

#define SHAMCOP_RTC_RTCON(_b)		SHAMCOP_REGISTER( _b, u8, RTC, RTCON)   /* reset value: 0x0000 */
#define SHAMCOP_RTC_TICNT(_b)		SHAMCOP_REGISTER( _b, u8, RTC, TICNT)   /* reset value: 0x0000 */
#define SHAMCOP_RTC_RTCALM(_b)		SHAMCOP_REGISTER( _b, u8, RTC, RTCALM)  /* reset value: 0x0000 */
#define SHAMCOP_RTC_ALMSEC(_b)		SHAMCOP_REGISTER( _b, u8, RTC, ALMSEC)  /* reset value: 0x0000 */
#define SHAMCOP_RTC_ALMMIN(_b)		SHAMCOP_REGISTER( _b, u8, RTC, ALMMIN)  /* reset value: 0x0000 */
#define SHAMCOP_RTC_ALMHOUR(_b)		SHAMCOP_REGISTER( _b, u8, RTC, ALMHOUR) /* reset value: 0x0000 */
#define SHAMCOP_RTC_ALMDAY(_b)		SHAMCOP_REGISTER( _b, u8, RTC, ALMDAY)  /* reset value: 0x0001 */
#define SHAMCOP_RTC_ALMMON(_b)		SHAMCOP_REGISTER( _b, u8, RTC, ALMMON)  /* reset value: 0x0001 */
#define SHAMCOP_RTC_ALMYEAR(_b)		SHAMCOP_REGISTER( _b, u8, RTC, ALMYEAR) /* reset value: 0x0000 */
#define SHAMCOP_RTC_RTCRST(_b)		SHAMCOP_REGISTER( _b, u8, RTC, RTCRST)  /* reset value: 0x0000 */
#define SHAMCOP_RTC_BCDSEC(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDSEC)  /* reset value: undefined */
#define SHAMCOP_RTC_BCDMIN(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDMIN)  /* reset value: undefined */
#define SHAMCOP_RTC_BCDDATE(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDDATE) /* reset value: undefined */
#define SHAMCOP_RTC_BCDHOUR(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDHOUR) /* reset value: undefined */
#define SHAMCOP_RTC_BCDDAY(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDDAY)  /* reset value: undefined */
#define SHAMCOP_RTC_BCDMON(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDMON)  /* reset value: undefined */
#define SHAMCOP_RTC_BCDYEAR(_b)		SHAMCOP_REGISTER( _b, u8, RTC, BCDYEAR) /* reset value: undefined */


/********* LED **********/

#define	SHAMCOP_NUMBER_OF_LEDS		5

#define _SHAMCOP_LED_Base		0x07200
#define _SHAMCOP_LED_LED0CON0		0x00000
#define _SHAMCOP_LED_LED0CON1		0x00004
#define _SHAMCOP_LED_LED1CON0          0x00008
#define _SHAMCOP_LED_LED1CON1          0x0000C
#define _SHAMCOP_LED_LED2CON0          0x00010
#define _SHAMCOP_LED_LED2CON1          0x00014
#define _SHAMCOP_LED_LED3CON0          0x00018
#define _SHAMCOP_LED_LED3CON1          0x0001C
#define _SHAMCOP_LED_LED4CON0          0x00020
#define _SHAMCOP_LED_LED4CON1          0x00024


#define _SHAMCOP_LED_LEDPS0		0x00028
#define _SHAMCOP_LED_LEDPS1		0x0002c

#define SHAMCOP_LEDN_CON0(_b,_n)	SHAMCOP_REGISTER( _b,  u16, LED, LED0CON0 + (8 * (_n)))
#define SHAMCOP_LEDN_CON1(_b,_n)	SHAMCOP_REGISTER( _b,  u16, LED, LED0CON1 + (8 * (_n)))

#define SHAMCOP_LED_LEDPS0(_b)		SHAMCOP_REGISTER( _b,  u16, LED, LEDPS0 )
#define SHAMCOP_LED_LEDPS1(_b)		SHAMCOP_REGISTER( _b,  u16, LED, LEDPS1 )

/********* Keypad **********/

#define _SHAMCOP_KEYPAD_Base		0x07400

#define _SHAMCOP_KEYPAD_DAT		0x00000
#define _SHAMCOP_KEYPAD_INTC		0x00004
#define _SHAMCOP_KEYPAD_FLT0		0x00008
#define _SHAMCOP_KEYPAD_FLT1		0x0000C

#define SHAMCOP_KEYPAD_DAT(_b)		SHAMCOP_REGISTER( _b, u8, KEYPAD, DAT) /* reset value: 0x10 */
#define SHAMCOP_KEYPAD_INTC(_b)		SHAMCOP_REGISTER( _b, u8, KEYPAD, INTC) /* reset value: 0x00 */
#define SHAMCOP_KEYPAD_FLT0(_b)		SHAMCOP_REGISTER( _b, u16, KEYPAD, FLT0) /* reset value: 0x0000 */
#define SHAMCOP_KEYPAD_FLT1(_b)		SHAMCOP_REGISTER( _b, u16, KEYPAD, FLT1) /* reset value: 0x0000 */

#define SHAMCOP_KEYPAD_DAT_DAT		(0xf << 0)
#define SHAMCOP_KEYPAD_DAT_VALID	(1 << 4)
#define SHAMCOP_KEYPAD_DAT_CLEAR	(1 << 5)
#define SHAMCOP_KEYPAD_DAT_EN		(1 << 6)

#define SHAMCOP_KEYPAD_INTC_INTLV	(7 << 0)
#define SHAMCOP_KEYPAD_INTC_INTLV_LOW	(0 << 0)
#define SHAMCOP_KEYPAD_INTC_INTLV_HIGH	(1 << 0)
#define SHAMCOP_KEYPAD_INTC_INTLV_RE	(2 << 0)
#define SHAMCOP_KEYPAD_INTC_INTLV_FE	(4 << 0)
#define SHAMCOP_KEYPAD_INTC_INTLV_BOTH_EDGES	(6 << 0)
#define SHAMCOP_KEYPAD_INTC_INTEN	(1 << 3)

#define SHAMCOP_KEYPAD_FLT0_SELCLK	(1 << 0)
#define SHAMCOP_KEYPAD_FLT0_FILEN	(1 << 1)

#define SHAMCOP_KEYPAD_FLT1_WIDTH	(0x3fff << 0)


/********* IRQ *************/

#define SHAMCOP_MAP_SIZE		0x100000

#define SHAMCOP_IC_IRQ_START		0
#define SHAMCOP_IC_IRQ_COUNT		26

#define _IRQ_SHAMCOP_RSTBUTTON	 	(0)
#define _IRQ_SHAMCOP_APPBUTTON1	 	(1)
#define _IRQ_SHAMCOP_APPBUTTON2	 	(2)
#define _IRQ_SHAMCOP_APPBUTTON3	 	(3)
#define _IRQ_SHAMCOP_APPBUTTON4	 	(4)
#define _IRQ_SHAMCOP_JOYPAD_UP		(5)
#define _IRQ_SHAMCOP_JOYPAD_DOWN	(6)
#define _IRQ_SHAMCOP_BATT_FLT		(7)
#define _IRQ_SHAMCOP_JOYPAD_LEFT	(8)
#define _IRQ_SHAMCOP_JOYPAD_RIGHT	(9)
#define _IRQ_SHAMCOP_JOYPAD_ACTION	(10) 
#define _IRQ_SHAMCOP_GPB3		(11)
#define _IRQ_SHAMCOP_GPB4		(12)
#define _IRQ_SHAMCOP_GPB5		(13)
#define _IRQ_SHAMCOP_GPB6		(14)
#define _IRQ_SHAMCOP_EAR_IN		(15)
#define _IRQ_SHAMCOP_DMA		(16)
#define _IRQ_SHAMCOP_TICK		(17)
#define _IRQ_SHAMCOP_ALARM		(18)
#define _IRQ_SHAMCOP_ONEWIRE		(19)
#define _IRQ_SHAMCOP_SDHC		(20)
#define _IRQ_SHAMCOP_ADCTS		(21)
#define _IRQ_SHAMCOP_ADC  		(22)      
#define _IRQ_SHAMCOP_WDT  		(23)       
#define _IRQ_SHAMCOP_KEYPAD  		(24)       
#define _IRQ_SHAMCOP_NFIF  		(25)       

#define SHAMCOP_IRQ_COUNT		SHAMCOP_IC_IRQ_COUNT

struct ipaq_shamcop_platform_data
{
	u16 clocksleep;
	u16 pllcontrol;
};

#endif
