#include "bootldr.h"
#include "btpci.h"
#include "btflash.h"
#include "btusb.h"
#include "heap.h"
#include "xmodem.h"
#include "lcd.h"
#include "mmu.h"

#ifdef __linux__
typedef unsigned long u32;
typedef unsigned short u16;
typedef unsigned char u8;
#include <asm-arm/setup.h>
#endif
#include "sa1100.h"
#include "bsdsum.h"
#include "architecture.h"
typedef unsigned long u_int32_t;

/*
 *  sa1100fb_disable_lcd_controller():
 *    	Disables LCD controller by and enables LDD interrupt.
 * The controller_state is not changed until the LDD interrupt is
 * received to indicate the current
 * frame has completed.  Platform specific hardware disabling is also included.
 */
static void sa1100fb_disable_lcd_controller(void)
{
    putstr("Disabling LCD controller\n\r");
    
    lcd_light(0, 0);
    
    LCSR = 0;	/* Clear LCD Status Register */
    LCCR0 &= ~(LCCR0_LDM);	/* Enable LCD Disable Done Interrupt */
    LCCR0 &= ~(LCCR0_LEN);	/* Disable LCD Controller */
}

unsigned short*	cur_pal;
int		cur_bpp = LCD_BPP;

void
patch_palette_with_bpp(
    unsigned short  *palp,
    int		    bpp)
{
    *palp &= 0xCFFF; 	           
    *palp |= SA1100_PALETTE_MODE_VAL(bpp);
}

/*
 *  sa1100fb_enable_lcd_controller():
 *    	Enables LCD controller.  If the controller is already enabled, it is first disabled.
 *      This forces all changes to the LCD controller registers to be done when the 
 *      controller is disabled.  Platform specific hardware enabling is also included.
 */
static void
sa1100fb_enable_lcd_controller(
    lcd_params_t*   params)
{
    u_long	flags;
    
    sa1100fb_disable_lcd_controller();
    
    putstr("Enabling LCD controller\n\r"); let_uart_drain0();

    /* make sure the GPIO pins are carrying LCD data */
    SA1100_GPIO_GAFR_WRITE(SA1100_GPIO_GPDR_LCD_BITS);/* alt function */
    SA1100_GPIO_GPDR_WRITE(SA1100_GPIO_GPDR_LCD_BITS);/* outputs */
    
    /* Make sure the mode bits are present in the first palette entry */
    cur_pal = params->v_palette_base;
    cur_bpp = params->bits_per_pixel;
    patch_palette_with_bpp(cur_pal, cur_bpp);
    
    /* Sequence from 11.7.10 */
    LCCR3 = params->lccr3;
    LCCR2 = params->lccr2;
    LCCR1 = params->lccr1;
    LCCR0 = params->lccr0 & ~LCCR0_LEN;
    DBAR1 = (Address)vaddr_to_paddr((unsigned long)(params->v_palette_base));

    /* not used w/our panel */    
    DBAR2 = (Address)vaddr_to_paddr((unsigned long)(params->v_palette_base+0x80));
    
    putstr("before EN\n\r"); let_uart_drain0();
    LCCR0 |= LCCR0_LEN;
    putstr("after EN\n\r"); let_uart_drain0();
    
    set_egpio(EGPIO_BITSY_LCD_ON | EGPIO_BITSY_LCD_PCI |
	      EGPIO_BITSY_LCD_5V_ON | EGPIO_BITSY_LVDD_ON);
    putstr("after EGPIO \n\r"); let_uart_drain0();
    putLabeledWord("DBAR1=", DBAR1);
    putLabeledWord("LCCR0=", LCCR0);
    putLabeledWord("LCCR1=", LCCR1);
    putLabeledWord("LCCR2=", LCCR2);
    putLabeledWord("LCCR3=", LCCR3);
    putLabeledWord("params=0x", (unsigned long)params);

    lcd_light(1, 5);
}

/* extra at end for possible DMA overrun */
unsigned long lcd_frame_buffer[LCD_FB_MAX() + 16];

void
setup_solid_palette(
    int	palval)
{
    unsigned short* palp = (unsigned short*)lcd_frame_buffer;
    int		    i;
    
    for (i = 0; i < SA1100_NUM_PALETTE_ENTRIES(cur_bpp); i++) {
	*palp++ = palval;
    }
    patch_palette_with_bpp(cur_pal, cur_bpp);
}

void
setup_mono4_palette(
    int	start,
    int	end,
    int	incr)
{
    unsigned short* palp = (unsigned short*)lcd_frame_buffer;
    int		    i;
    
    for (i = start; i < end; i += incr) {
	*palp++ = i;
    }
    patch_palette_with_bpp(cur_pal, 4);
}

#if 0

void
setup_spectral_palette(
    void)
{
    /* this is quite wrong !!!! */
    unsigned short* palp = (unsigned short*)lcd_frame_buffer;
    int		i;
    unsigned short  palval = 0;
    int		num_pal_entries = SA1100_NUM_PALETTE_ENTRIES(cur_bpp);
    int		max_pal_colors = SA1100_MAX_PALETTE_COLORS();
    int		color_inc = max_pal_colors/num_pal_entries;

    if (color_inc <= 0)
	color_inc = 1;

    putLabeledWord("color_inc: 0x", color_inc);
    
    for (i = 0; i < num_pal_entries; i++) {
	*palp++ = palval;
	palval += color_inc;
    }
    patch_palette_with_bpp(cur_pal, cur_bpp);
}
#endif

unsigned short primaries[] =
{
    0xF000,			/* red */
    0x0780,			/* green */
    0x001e,			/* blue */
};
    
void
lcd_setup_default_image(
    int	params_id)
{
    int	    x, y;
    int	    pixel;
    unsigned short*   bufp = (unsigned short*)LCD_FB_IMAGE(lcd_frame_buffer,
							   cur_bpp);
    putstr("setup def img\n\r"); let_uart_drain0();
    
    /*  setup_spectral_palette(); */
    if (params_id == LCDP_MONO) {
	putstr("setup mono4 palette\n\r");
	setup_mono4_palette(0, 16, 1);
    }

    putstr("pal done"); let_uart_drain0();
    
    pixel = 0;
    for (y = 0; y < LCD_YRES; y++) {
	for (x = 0; x < LCD_XRES; x++) {
	    *bufp++ = pixel;
	}
	if ((y % (LCD_YRES/3)) == 0)
	    pixel = primaries[y / (LCD_YRES/3)];
    }

    putstr("img done\n\r"); let_uart_drain0();
}

    
lcd_params_t lcd_color_params = {
    LCCR0_LEN + LCCR0_Color + LCCR0_Sngl + LCCR0_Act +
      LCCR0_LtlEnd + LCCR0_LDM + LCCR0_BAM + LCCR0_ERM + 
      LCCR0_DMADel(0),
    
    LCCR1_DisWdth( LCD_XRES ) +
      LCCR1_HorSnchWdth( 4 ) +
      LCCR1_BegLnDel( 0xC ) +
      LCCR1_EndLnDel( 0x11 ),
    
    LCCR2_DisHght( LCD_YRES + 1 ) +
      LCCR2_VrtSnchWdth( 3 )+
      LCCR2_BegFrmDel( 10 ) +
      LCCR2_EndFrmDel( 1 ),
    
    /* PCD */ 0x10
      | /* ACB */ 0
      | /* API */ 0
      | LCCR3_VrtSnchL
      | LCCR3_HorSnchL,

    (unsigned short*)lcd_frame_buffer,
    LCD_BPP
};

lcd_params_t lcd_mono_params = {
    LCCR0_LEN + LCCR0_Mono + LCCR0_4PixMono + LCCR0_Sngl +
    LCCR0_Pas + LCCR0_LtlEnd + LCCR0_LDM + LCCR0_BAM +
    LCCR0_ERM + LCCR0_DMADel(0),

    LCCR1_DisWdth( LCD_XRES ) +
    LCCR1_HorSnchWdth( 26 ) +
    LCCR1_BegLnDel( 0x4 ) +
    LCCR1_EndLnDel( 0x4 ),

    LCCR2_DisHght( LCD_YRES + 1 ) +
    LCCR2_VrtSnchWdth( 41 )+
    LCCR2_BegFrmDel( 1 ) +
    LCCR2_EndFrmDel( 1 ),
    
    /* PCD */ 0x28
    | /* ACB */ 0
    | /* API */ 0
    | LCCR3_VrtSnchH
    | LCCR3_HorSnchH
    | LCCR3_PixRsEdg
    | LCCR3_OutEnH,

    (unsigned short*)lcd_frame_buffer,
    LCD_MONO_BPP
};

    

/*
 * must match the values of the LCDP_ enums
 */
lcd_params_t*	params_list[] =
{
    &lcd_color_params,		/* color LCD panel */
    &lcd_mono_params		/* mono LCD panel */
};

lcd_params_t*
lcd_get_params(
    int		    params_id,
    lcd_params_t*   params)
{
    if (params_id >= LCDP_CUSTOM) {
	if (params_id > LCDP_CUSTOM) {
	    putLabeledWord("illegal params_id: 0x", params_id);
	    return (NULL);
	}
    }
    else
	params = params_list[params_id];

    return (params);
}


void
lcd_default_init(
    int		    params_id,
    lcd_params_t*   params)
{
    cur_pal = lcd_color_params.v_palette_base;
    lcd_setup_default_image(params_id);
    params = lcd_get_params(params_id, params);
    if (params == NULL) {
	putstr("lcd_ddefault_init: params is NULL\n\r");
	return;
    }
	
    sa1100fb_enable_lcd_controller(params);
}

void
lcd_display_bitmap(
    char*	    bitmap_in,
    size_t	    len,
    int		    params_id,
    lcd_params_t*   params)
{
    int	    pal_size = SA1100_PALETTE_MEM_SIZE(params->bits_per_pixel);
    int	    num_pixels;
    int	    pixel;
    unsigned short* bufp = (unsigned short*)LCD_FB_IMAGE(lcd_frame_buffer,
							 cur_bpp);
    unsigned short* bmp = (unsigned short*)bitmap_in;

    params = lcd_get_params(params_id, params);
    if (params == NULL) {
	putstr("lcd_display_bitmap: params is NULL\n\r");
	return;
    }
	
    if (len > LCD_FB_MAX() - pal_size)
	len = LCD_FB_MAX() - pal_size;

    sa1100fb_disable_lcd_controller();

    /*
     * LCD's pixel format:
     * 
     *  1111 11
     *  5432 1098 7654 3210
     *  rrrr 0ggg g00b bbb0
     */

    for (num_pixels = LCD_NUM_PIXELS(); num_pixels; num_pixels--) {
	pixel = *bmp++;
	*bufp++ = ((pixel&0xf00)<<4) |
	    ((pixel&0x0f0)<<3) |
	    ((pixel&0x00f)<<1);
    }

    sa1100fb_enable_lcd_controller(params);
}

void
lcd_off(
    void)
{
    clr_egpio(EGPIO_BITSY_LCD_ON | EGPIO_BITSY_LCD_PCI |
	      EGPIO_BITSY_LCD_5V_ON | EGPIO_BITSY_LVDD_ON);
    sa1100fb_disable_lcd_controller();
}

    
void
lcd_light(
    int	on_off,
    int	level)
{
    char    level_buf[] = {0x02, 0x01, level};
    
    if (on_off)
	auxm_send_cmd(0x0d, 3, level_buf);
    else {
	level_buf[1] = 0;
	auxm_send_cmd(0x0d, 3, level_buf);
    }
}

void
lcd_fill(
    int	color,
    int	color_inc)
{
    unsigned short* bufp = (unsigned short*)LCD_FB_IMAGE(lcd_frame_buffer,
							 cur_bpp);
    int	    num_pixels;

    putLabeledWord("fb: 0x", (unsigned long)lcd_frame_buffer);
    putLabeledWord("bufp: 0x", (unsigned long)bufp);
    
    for (num_pixels = LCD_NUM_PIXELS(); num_pixels; num_pixels--) {
	*bufp++ = color;
	color += color_inc;
    }
}
    
char*
lcd_get_image_buffer(
    void)
{
    return (char*)LCD_FB_IMAGE(lcd_frame_buffer, cur_bpp);
}

size_t
lcd_get_image_buffer_len(
    void)
{
    return (LCD_FB_MAX());
}


