/* 
 * Copyright (C) 2000 Jeff Dike (jdike@karaya.com)
 * Copyright (C) 2001 Ridgerun,Inc (glonnon@ridgerun.com)
 * Licensed under the GPL
 */

#include <unistd.h>
#include <errno.h>
#include <sched.h>
#include <signal.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/fcntl.h>
#include <string.h>
#include <sys/mman.h>
#include "user_util.h"
#include "user.h"
#include "ubd_user.h"

extern void panic(char *fmt, ...);

struct cow_header {
	int magic;
	int version;
	char backing_file[256];
	time_t mtime;
	__u64 size;
	int sectorsize;
};

#define COW_MAGIC 0x4f4f4f4d  /* mooo */
#define COW_VERSION 1

static void sizes(__u64 size, int sectorsize, int *bitmap_len_out,
		  int *data_offset_out)
{
	*bitmap_len_out = (size + sectorsize - 1) / (8 * sectorsize);

	*data_offset_out = sizeof(struct cow_header) + *bitmap_len_out;
	*data_offset_out = (*data_offset_out + sectorsize - 1) / sectorsize;
	*data_offset_out *= sectorsize;
}

int open_ubd_file(char *file, int *openflags, char **backing_file_out, 
		  int *bitmap_len_out, int *data_offset_out)
{
        struct cow_header header;
	struct stat64 buf;
        int fd, err, mode = 0644;

	if(backing_file_out != NULL) *backing_file_out = NULL;
        if((fd = open64(file, *openflags, mode)) < 0){
                if(((*openflags & O_ACCMODE) != O_RDWR) || 
                   ((errno != EROFS) && (errno != EACCES))) return(-errno);
		*openflags &= ~O_ACCMODE;
                *openflags |= O_RDONLY;
                if((fd = open64(file, *openflags, mode)) < 0) return(-errno);
        }
	if(backing_file_out == NULL) return(fd);
	err = -EINVAL;
	if(read(fd, &header, sizeof(header)) != sizeof(header)){
		printk("Failed to check COW header in \"%s\", errno = %d\n",
		       file, errno);
		goto error;
	}
	if(header.magic != COW_MAGIC){
  		*backing_file_out = NULL;
  		return(fd);
  	}

  	err = stat64(header.backing_file, &buf);
  	if(err) goto error;

  	err = -EINVAL;
  	if(buf.st_size != header.size){
		printk("Size mismatch (%ld vs %ld) of COW header vs backing "
		       "file\n", buf.st_size, header.size);
		goto error;
	}
	if(buf.st_mtime != header.mtime){
		printk("mtime mismatch (%ld vs %ld) of COW header vs backing "
		       "file\n", buf.st_mtime, header.mtime);
		goto error;
	}

	*backing_file_out = um_kmalloc(strlen(header.backing_file) + 1);
	if(*backing_file_out == NULL){
		close(fd);
		return(-ENOMEM);
	}
	strcpy(*backing_file_out, header.backing_file);

	sizes(header.size, header.sectorsize, bitmap_len_out, data_offset_out);

        return(fd);
 error:
	close(fd);
	return(err);
}

int read_cow_bitmap(int fd, void *buf, int len)
{
	int err;

	err = lseek64(fd, sizeof(struct cow_header), SEEK_SET);
	if(err != sizeof(struct cow_header)) return(-errno);
	err = read(fd, buf, len);
	if(err < 0) return(-errno);
	return(0);
}

int create_cow_file(char *cow_file, char *backing_file, int sectorsize, 
		    int *bitmap_len_out, int *data_offset_out)
{
        struct cow_header header;
	struct stat64 buf;
	__u64 blocks;
	long zero;
	int err, fd, i, flags;

	flags = O_RDWR | O_CREAT;
	fd = open_ubd_file(cow_file, &flags, NULL, NULL, NULL);
	if(fd < 0) return(fd);

	header.magic = COW_MAGIC;
	header.version = COW_VERSION;

	err = -EINVAL;
	if(strlen(backing_file) > sizeof(header.backing_file) - 1){
		printk("Backing file name \"%s\" is too long - names are "
		       "limited to %d characters\n", backing_file, 
		       sizeof(header.backing_file) - 1);
		goto error;
	}
	strcpy(header.backing_file, backing_file);

	err = stat64(header.backing_file, &buf);
	if(err < 0) goto error;

	header.mtime = buf.st_mtime;
	header.size = buf.st_size;
	header.sectorsize = sectorsize;

	err = write(fd, &header, sizeof(header));
	if(err != sizeof(header)) goto error;

	blocks = (header.size + sectorsize - 1) / sectorsize;
	blocks = (blocks + sizeof(long) * 8 - 1) / (sizeof(long) * 8);
	zero = 0;
	for(i = 0; i < blocks; i++){
		err = write(fd, &zero, sizeof(zero));
		if(err != sizeof(zero)) goto error;
	}

	sizes(header.size, header.sectorsize, bitmap_len_out, data_offset_out);

	return(fd);

 error:
	close(fd);
	return(err);
}

int read_ubd_fs(int fd, void *buffer, int len)
{
	return(read(fd, buffer, len));
}

int write_ubd_fs(int fd, char *buffer, int len)
{
	return(write(fd, buffer, len));
}

int ubd_is_dir(char *file)
{
	struct stat64 buf;

	if(stat64(file, &buf) < 0) return(0);
	return(S_ISDIR(buf.st_mode));
}

unsigned long last_offset = -1;

int do_io(struct io_thread_req *req)
{
	char *buf;
	unsigned long len;
	int n, off, nsectors, start, end, bit;

	nsectors = req->length / req->sectorsize;
	start = 0;
	do {
		bit = ubd_test_bit(start, &req->sector_mask);
		end = start;
		while((end < nsectors) && 
		      (ubd_test_bit(end, &req->sector_mask) == bit))
			end++;

		if(end != nsectors)
			printk("end != nsectors\n");
		off = req->offset + req->offsets[bit];
		len = (end - start) * req->sectorsize;
		buf = &req->buffer[start * req->sectorsize];
		last_offset = off;

		if(lseek64(req->fds[bit], off, SEEK_SET) != off){
			printk("do_io - lseek failed : errno = %d\n",
			       errno);
			return(-1);
		}
		if(req->read){
			n = read(req->fds[bit], buf, len);
			if(n <= 0){
				printk("do_io - read returned %d : "
				       "errno = %d fd = %d\n", n, 
				       errno, req->fds[bit]);
				return(-1);
			}
			else if(n != len) memset(&buf[n], 0, len - n);
		}
		else {
			n = write(req->fds[bit], buf, len);
			if(n != len){
				printk("do_io - write returned %d : "
				       "errno = %d fd = %d\n", n, 
				       errno, req->fds[bit]);
				return(-1);
			}
		}

		start = end;
	} while(start < nsectors);

	if(req->bitmap_start != -1){
		off = req->bitmap_start * sizeof(req->bitmap[0]);
		off += sizeof(struct cow_header);
		if(lseek64(req->fds[1], off, SEEK_SET) != off){
			printk("do_io - bitmap lseek failed : errno = %d\n",
			       errno);
			return(-1);
		}
		len = (req->bitmap_end - req->bitmap_start) * 
			sizeof(req->bitmap[0]);
		n = write(req->fds[1], &req->bitmap[req->bitmap_start], len);
		if(n != len){
			printk("do_io - bitmap update returned %d : "
			       "errno = %d fd = %d\n", n, errno, req->fds[1]);
			return(-1);
		}
	}
	return(0);
}

int kernel_fds[2] = { -1, -1 };

int io_count = 0;

int io_thread(void *arg)
{
	struct io_thread_req req;
	int n;

	while(1){
		n = read(kernel_fds[0], &req, sizeof(req));
		if(n < 0) printk("io_thread - read returned %d, errno = %d\n",
				 n, errno);
		else if(n < sizeof(req)){
			printk("io_thread - short read : length = %d\n", n);
			continue;
		}
		io_count++;
		do_io(&req);
		if(write(kernel_fds[1], &req, sizeof(req)) != sizeof(req))
			printk("io_thread - write failed, errno = %d\n",
			       errno);
	}
}

int start_io_thread(unsigned long sp, int *fds_out)
{
	int pid;

	if((kernel_fds[0] = get_pty()) < 0) return(-1);
	raw(kernel_fds[0], 1);
	if((fds_out[0] = open(ptsname(kernel_fds[0]), O_RDWR)) < 0){
		printk("Couldn't open tty for slip line\n");
		return(-1);
	}
	fds_out[1] = fds_out[0];
	kernel_fds[1] = kernel_fds[0];
	pid = clone(io_thread, (void *) sp, CLONE_VM | CLONE_FILES | SIGCHLD,
		    NULL);
	if(pid < 0){
		printk("start_io_thread - clone failed : errno = %d\n", errno);
		return(-errno);
	}
	return(pid);
}

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-file-style: "linux"
 * End:
 */
