/* 
 * Copyright (C) 2000 Jeff Dike (jdike@karaya.com)
 * Licensed under the GPL
 */

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <sched.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <asm/ptrace.h>
#include <asm/sigcontext.h>
#ifdef PROFILING
#include <sys/gmon.h>
#endif
#include "user_util.h"
#include "kern_util.h"
#include "user.h"
#include "process.h"
#include "sysdep/ptrace.h"
#include "sysdep/sigcontext.h"

void stop_pid(int pid)
{
	kill(pid, SIGSTOP);
}

void kill_pid(int pid)
{
	kill(pid, SIGKILL);
}

void usr1_pid(int pid)
{
	kill(pid, SIGUSR1);
}

void cont_pid(int pid)
{
	kill(pid, SIGCONT);
}

void set_sigstack(void *sig_stack)
{
	stack_t stack;

	stack.ss_sp = (__ptr_t) sig_stack;
	stack.ss_flags = 0;
	stack.ss_size = 2 * page_size() - sizeof(void *);
	if(sigaltstack(&stack, NULL) != 0)
		panic("sigaltstack failed");
}

void set_handler(int sig, void (*handler)(int), int flags, ...)
{
	struct sigaction action;
	va_list ap;
	int mask;

	va_start(ap, flags);
	action.sa_handler = handler;
	sigemptyset(&action.sa_mask);
	while((mask = va_arg(ap, int)) != -1){
		sigaddset(&action.sa_mask, mask);
	}
	action.sa_flags = flags;
	action.sa_restorer = NULL;
	if(sigaction(sig, &action, NULL) < 0)
		panic("sigaction failed");
}

void init_new_thread(void *sig_stack, void (*usr1_handler)(int))
{
	int flags = 0;

	if(sig_stack != NULL){
		set_sigstack(sig_stack);
		flags = SA_ONSTACK;
	}
	set_handler(SIGSEGV, (__sighandler_t) sig_handler, flags,
		    SIGUSR1, SIGIO, -1);
	set_handler(SIGTRAP, (__sighandler_t) sig_handler, flags, 
		    SIGUSR1, SIGIO, -1);
	set_handler(SIGFPE, (__sighandler_t) sig_handler, flags, 
		    SIGUSR1, SIGIO, -1);
	set_handler(SIGILL, (__sighandler_t) sig_handler, flags, 
		    SIGUSR1, SIGIO, -1);
	set_handler(SIGBUS, (__sighandler_t) sig_handler, flags, 
		    SIGUSR1, SIGIO, -1);
	set_handler(SIGUSR2, process_stack_handler, SA_NODEFER, SIGUSR1, -1);
	change_sig(SIGUSR2, 1);
	if(usr1_handler) set_handler(SIGUSR1, usr1_handler, flags, -1);
	signal(SIGCHLD, SIG_IGN);
	set_timers(1);
	init_irq_signals(sig_stack != NULL);
}

/* This sigstop business works around a bug in gcc's -pg support.  
 * Normally a procedure's mcount call comes after esp has been copied to 
 * ebp and the new frame is constructed.  With procedures with no locals,
 * the mcount comes before, as the first thing that the procedure does.
 * When that procedure is main for a thread, ebp comes in as NULL.  So,
 * when mcount dereferences it, it segfaults.  So, UML works around this
 * by adding a non-optimizable local to the various trampolines, fork_tramp
 * and outer_tramp below, and exec_tramp.
 */

static int sigstop = SIGSTOP;

int fork_tramp(void *sig_stack)
{
	int sig = sigstop;

	block_signals();
	init_new_thread(sig_stack, finish_fork_handler);
	kill(getpid(), sig);
	return(0);
}

struct tramp {
	int (*tramp)(void *);
	unsigned long temp_stack;
	unsigned long stack;
	int flags;
	int pid;
};

/* See above for why sigkill is here */

int sigkill = SIGKILL;

int outer_tramp(void *arg)
{
	struct tramp *t;
	int sig = sigkill;

	t = arg;
	t->pid = clone(t->tramp, (void *) t->temp_stack + page_size()/2,
		       t->flags, (void *) t->stack);
	if(t->pid > 0) wait_for_stop(t->pid, SIGSTOP, PTRACE_CONT);
	kill(getpid(), sig);
	exit(0);
}

int start_fork_tramp(unsigned long sig_stack, unsigned long temp_stack,
		     int clone_vm, int (*tramp)(void *))
{
	struct tramp arg;
	unsigned long sp;
	int new_pid, flags, status, err;

	/* The trampoline will run on the temporary stack */
	sp = stack_sp(temp_stack);

	flags = CLONE_FILES | SIGCHLD;
	if(clone_vm) flags |= CLONE_VM;

	arg.tramp = tramp;
	arg.temp_stack = temp_stack;
	arg.stack = sig_stack;
	arg.flags = flags;

	/* Start the process and wait for it to stop itself */
	new_pid = clone(outer_tramp, (void *) sp, flags, &arg);
	if(new_pid < 0) return(-errno);
	while((err = waitpid(new_pid, &status, 0) < 0) && (errno == EINTR)) ;
	if(err < 0) panic("Waiting for outer trampoline failed - errno = %d", 
			  errno);
	if(!WIFSIGNALED(status) || (WTERMSIG(status) != SIGKILL))
		panic("outer trampoline didn't exit with SIGKILL");

	return(arg.pid);
}

void trace_myself(void)
{
	if(ptrace(PTRACE_TRACEME, 0, 0, 0) < 0)
		panic("ptrace failed in trace_myself");
}

void signal_handler(void *task, unsigned long h, int sig)
{
	void (*handler)(int, struct sigcontext);
	void *regs;
	unsigned long cr2;
	int err;
	UM_ALLOCATE_SC(sc);

	regs = process_state(task, &cr2, &err);
	fill_in_sigcontext(&sc, regs, cr2, err);
	handler = (void (*)(int, struct sigcontext)) h;
	(*handler)(sig, sc);
}

void continue_fork(void *task, int pid, struct sys_pt_regs *regs)
{
	if(ptrace(PTRACE_CONT, pid, 0, SIGUSR1) < 0)
		tracer_panic("Couldn't continue forked process");
	wait_for_stop(pid, SIGSTOP, PTRACE_CONT);
	if(ptrace_setregs(pid, regs) < 0)
		tracer_panic("Couldn't restore registers");
}

int setup_altstack(unsigned long stack, int stack_size)
{
	struct sys_pt_regs *regs;
	unsigned long sp;
	int pid;

	sp = stack + stack_size - sizeof(void *);
	pid = clone(alt_stack_handler, (void *) sp, 
		    CLONE_VM | CLONE_FILES | SIGCHLD, NULL);
	if(pid < 0) return(-errno);
	wait_for_stop(pid, SIGSTOP, PTRACE_CONT);
	regs = altstack_state(NULL, NULL, NULL);
	if(ptrace_getregs(pid, regs) < 0)
		panic("Couldn't get altstack state");
	save_altstack(NULL, UM_SP(regs));
	kill(pid, SIGKILL);
	if(waitpid(pid, NULL, WNOHANG) < 0) 
		printk("setup_altstack : waitpid failed, errno = %d\n",
		       errno);
	return(0);
}

void change_sig(int signal, int on)
{
	sigset_t sigset;

	sigemptyset(&sigset);
	sigaddset(&sigset, signal);
	sigprocmask(on ? SIG_UNBLOCK : SIG_BLOCK, &sigset, NULL);
}

void *last_brk;

void *update_brk(void)
{
	void *current_brk;
	int len, off;

	current_brk = sbrk(0);
	if(current_brk == last_brk) return(current_brk);
	off = last_brk - brk_start;
	if(lseek(brk_fd, off, SEEK_SET) != off){
		panic("update_brk : lseek failed - errno = %d", errno);
	}
	len = current_brk - last_brk;
	if(write(brk_fd, last_brk, len) != len){
		panic("update_brk : write failed - errno = %d", errno);
	}
	last_brk = current_brk;
	return(current_brk);
}

void check_brk(void *process_brk)
{
	if(last_brk == process_brk) return;
	if(mmap(process_brk, last_brk - process_brk,
		PROT_READ | PROT_WRITE, MAP_SHARED | MAP_FIXED,
		brk_fd, process_brk - brk_start) != process_brk){
		panic("check_brk : mmap failed - errno = %d", errno);
	}
}

int get_one_stack(char **stack_out, struct sys_pt_regs *regs_out)
{
	void *stack;
	unsigned long sp;
	int pid, n;

	if((stack = mmap(NULL, page_size(), 
			 PROT_READ | PROT_WRITE | PROT_EXEC,
			 MAP_PRIVATE | MAP_ANONYMOUS, -1, 0)) == MAP_FAILED){
		perror("get_one_stack : couldn't mmap stack");
		exit(1);		
	}
	sp = stack_sp((unsigned long) stack);
	pid = clone(syscall_handler, (void *) sp, SIGCHLD, NULL);
	if(pid < 0){
		perror("get_one_stack : couldn't start thread");
		exit(1);
	}
	wait_for_stop(pid, SIGSTOP, PTRACE_CONT);
	if(ptrace(PTRACE_ATTACH, pid, 0, 0) < 0){
		perror("get_one_stack : couldn't attach to child");
		exit(1);
	}
	if(ptrace_getregs(pid, regs_out) < 0){
		perror("get_one_stack : couldn't get registers");
		exit(1);
	}
	sp = UM_SP(regs_out);
	sp &= ~sizeof(unsigned long);
	while(sp < (unsigned long) stack + page_size()){
		*((unsigned long *) sp) = ptrace(PTRACE_PEEKDATA, pid, 
						  (void *) sp, NULL);
		sp += sizeof(unsigned long);
	}
	ptrace(PTRACE_KILL, pid, 0, 0);
	waitpid(pid, NULL, 0);
	n = page_size() - (UM_SP(regs_out) & ~page_mask());
	*stack_out = stack;
	
	return(n);
}

void attach_process(int pid)
{
	if((ptrace(PTRACE_ATTACH, pid, 0, 0) < 0) ||
	   (ptrace(PTRACE_CONT, pid, 0, 0) < 0))
		tracer_panic("OP_FORK failed to attach pid");
	wait_for_stop(pid, SIGSTOP, PTRACE_CONT);
}

int signal_frame_size;
static void *local_addr;

static void frame_size(int sig)
{
	int n;

	signal_frame_size = (unsigned long) local_addr - (unsigned long) &n;
}

void calc_sigframe_size(void)
{
	int n;

	signal(SIGUSR1, frame_size);
	local_addr = &n;
	usr1_pid(getpid());
}

void tracer_panic(char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	vprintf(format, ap);
	while(1) sleep(10);
}

#ifdef PROFILING
void remap_profiling_buffers(void)
{
	unsigned long low = 1 << 31, high = 0;

	if((unsigned long) _gmonparam.kcount < low) 
		low = (unsigned long) _gmonparam.kcount;
	if((unsigned long) _gmonparam.froms < low) 
		low = (unsigned long) _gmonparam.froms;
	if((unsigned long) _gmonparam.tos < low) 
		low = (unsigned long) _gmonparam.tos;

	if((unsigned long) _gmonparam.kcount + _gmonparam.kcountsize > high)
		high = (unsigned long) _gmonparam.kcount + 
			_gmonparam.kcountsize;
	if((unsigned long) _gmonparam.froms + _gmonparam.fromssize > high)
		high = (unsigned long) _gmonparam.froms + _gmonparam.fromssize;
	if((unsigned long) _gmonparam.tos + _gmonparam.tossize > high)
		high = (unsigned long) _gmonparam.tos + _gmonparam.tossize;

	remap_data((void *) round_down(low), (void *) round_up(high));
}
#endif

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-file-style: "linux"
 * End:
 */
