/* 
 * Copyright (C) 2000 Jeff Dike (jdike@karaya.com)
 * Licensed under the GPL
 */

#include "linux/sched.h"
#include "linux/slab.h"
#include "linux/bootmem.h"
#include "asm/pgtable.h"
#include "asm/pgalloc.h"
#include "asm/a.out.h"
#include "asm/processor.h"
#include "asm/mmu_context.h"
#include "user_util.h"
#include "kern_util.h"
#include "kern.h"

struct mm_struct kernel_maps = EMPTY_MM;

static void fix_range(struct mm_struct *proc_mm, unsigned long start_addr, 
		      unsigned long end_addr, int force)
{
	struct mm_struct *mm;
	pgd_t *npgd;
	pmd_t *npmd;
	pte_t *npte;
	unsigned long addr;
	int r, w, x;

	if((current->thread.extern_pid != -1) && 
	   (current->thread.extern_pid != getpid()))
		panic("fix_range fixing wrong address space");
	mm = proc_mm;
	if(mm == NULL){
		start_addr = start_vm;
		end_addr = end_vm;
	}
	for(addr=start_addr;addr<end_addr;){
                if(force){
			extern unsigned long _stext;
                        struct vm_area_struct *vma;
			if(addr == (unsigned long) ROUND_DOWN(&_stext)){
				addr = (unsigned long) last_brk;
				continue;
			}
                        vma = find_vma(&kernel_maps, addr);
                        if(vma && (vma->vm_start <= addr)){
                                addr = vma->vm_end;
                                continue;
                        }
                }

		if((addr >= start_vm) && (addr < end_vm)) mm = &init_mm;
		else mm = proc_mm;
		npgd = pgd_offset(mm, addr);
		npmd = pmd_offset(npgd, addr);
		if(pmd_present(*npmd)){
			npte = pte_offset(npmd, addr);
			r = pte_read(*npte);
			w = pte_write(*npte);
			x = pte_exec(*npte);
			if(!pte_dirty(*npte)) w = 0;
			if(!pte_young(*npte)){
				r = 0;
				w = 0;
			}
			if(mm == &init_mm){
				unsigned long mask;

				mask = PAGE_MASK | _PAGE_NEWPAGE | 
					_PAGE_NEWPROT;
				if((pte_val(*npte) & ~mask) ==
				   pgprot_val(PAGE_KERNEL)){
					r = 1;
					w = 1;
					x = 1;
				}
				else if((pte_val(*npte) & ~mask) == 
					pgprot_val(PAGE_KERNEL_RO)){
					r = 1;
					w = 0;
					x = 1;
				}
			}
			if(force || !pte_present(*npte) || pte_newpage(*npte)){
				munmap((void *) addr, PAGE_SIZE);
				if(pte_present(*npte))
					map(addr, pte_address(*npte),
					    PAGE_SIZE, r, w, x);
			}
			else if(pte_newprot(*npte))
				protect(addr, PAGE_SIZE, r, w, x);
			if((mm == proc_mm) && pte_present(*npte))
				*npte = pte_mkuptodate(*npte);
			addr += PAGE_SIZE;
		}
		else {
			unsigned long end, len;

			end = (addr + PMD_SIZE) & PMD_MASK;
			len = end - addr;
			if(force || pmd_newpage(*npmd)){
				munmap((void *) addr, len);
				if(mm == proc_mm) 
					pmd_mkuptodate(*npmd);
			}
			addr += len;
		}
	}
}

void init_flush_vm(void)
{
	pgd_t *npgd;
	pmd_t *npmd;
	pte_t *npte;
	unsigned long addr;
	int r, w, x;
  
	for(addr=start_vm;addr<end_vm;){
		npgd = pgd_offset(&init_mm, addr);
		npmd = pmd_offset(npgd, addr);
		if(pmd_present(*npmd)){
			npte = pte_offset(npmd, addr);
			if(pte_present(*npte)){
				unsigned long mask;

				mask = PAGE_MASK | _PAGE_NEWPAGE | 
					_PAGE_NEWPROT;
				r = pte_read(*npte);
				w = pte_write(*npte);
				x = pte_exec(*npte);
				if((pte_val(*npte) & ~mask) == 
				   pgprot_val(PAGE_KERNEL)){
					r = 1;
					w = 1;
					x = 1;
				}
				else if((pte_val(*npte) & ~mask) == 
					pgprot_val(PAGE_KERNEL_RO)){
					r = 1;
					w = 0;
					x = 1;
				}
				map(addr, pte_address(*npte),
				    PAGE_SIZE, r, w, x);
			}
			addr += PAGE_SIZE;
		}
		else addr += PTRS_PER_PMD * PTRS_PER_PTE * PAGE_SIZE;
	}
}

void flush_tlb_range(struct mm_struct *mm, unsigned long start, 
		     unsigned long end)
{
	if(mm == current->mm) fix_range(mm, start, end, 0);
}

void flush_tlb_mm(struct mm_struct *mm)
{
	flush_tlb_range(mm, 0, STACK_TOP);
}

void flush_tlb_kernel_vm(void)
{
	if(current->mm != NULL) fix_range(current->mm, 0, STACK_TOP, 0);
        else fix_range(NULL, start_vm, end_vm, 0);
}

void flush_tlb_page(struct vm_area_struct *vma, unsigned long address)
{
	address &= PAGE_MASK;
	if(vma->vm_mm == current->mm) 
		fix_range(current->mm, address, address + PAGE_SIZE, 0);
}

void flush_tlb_all(void)
{
	flush_tlb_range(current->mm, 0, STACK_TOP);
}

void force_flush_all(void)
{
	fix_range(current->mm, 0, STACK_TOP, 1);
}

static pgprot_t vm_prot(char r, char w, char x, char p)
{
	if((r == '-') && (w == '-') && (x == '-')) return(PAGE_NONE);
	else if(w == '-') return(PAGE_READONLY);
	else if(p == 'p') return(PAGE_COPY);
	else return(PAGE_SHARED);
}

static unsigned short vm_flags(char r, char w, char x, char p)
{
	unsigned short flags;

	flags = 0;
	if(r == 'r') flags |= VM_READ;
	if(w == 'w') flags |= VM_WRITE;
	if(x == 'x') flags |= VM_EXEC;
	if(p == '-') flags |= VM_SHARED;
	return(flags);
}

static struct vm_area_struct *process_vmas;
static int num_process_vmas = 0;

void add_perm_vma(unsigned long start, unsigned long end, char rperm, 
		  char wperm, char xperm, char private)
{
	struct vm_area_struct *vma;

	vma = &process_vmas[num_process_vmas++];
	*vma = ((struct vm_area_struct) {
		&kernel_maps, start, end, NULL, 
		vm_prot(rperm, wperm, xperm, private), 
		vm_flags(rperm, wperm, xperm, private), 0, NULL, NULL, NULL, 
		NULL, NULL, 0, NULL, 0
	});
	insert_vm_struct(&kernel_maps, vma);
}

void add_process_vmas(void)
{
	unsigned long start, end;
	void *maps;
	char rperm, wperm, xperm, private;
	
	if(process_vmas == NULL){
		int count = 0;

		maps = open_maps();
		while(read_map(maps, NULL, NULL, NULL, NULL, NULL, NULL))
			count++;
		close_maps(maps);
		process_vmas = malloc((count + 1) * sizeof(*process_vmas));
		if(process_vmas == NULL)
			panic("Couldn't allocate process_vmas");
	}
	maps = open_maps();
	while(read_map(maps, &start, &end, &rperm, &wperm, &xperm, &private)){
		if(start == 0x40000000) continue;
		add_perm_vma(start, end, rperm, wperm, xperm, private);
	}
	close_maps(maps);	
}

pgd_t *pgd_offset_proc(struct mm_struct *mm, unsigned long address)
{
	return(pgd_offset(mm, address));
}

pmd_t *pmd_offset_proc(pgd_t *pgd, unsigned long address)
{
	return(pmd_offset(pgd, address));
}

pte_t *pte_offset_proc(pmd_t *pmd, unsigned long address)
{
	return(pte_offset(pmd, address));
}

/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-file-style: "linux"
 * End:
 */
