/*
 * PnP bios services
 * 
 * Originally (C) 1998 Christian Schmidt (chr.schmidt@tu-bs.de)
 * Modifications (c) 1998 Tom Lees <tom@lpsg.demon.co.uk>
 * Minor reorganizations by David Hinds <dahinds@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   Reference:
 *   Compaq Computer Corporation, Phoenix Technologies Ltd., Intel 
 *   Corporation. 
 *   Plug and Play BIOS Specification, Version 1.0A, May 5, 1994
 *   Plug and Play BIOS Clarification Paper, October 6, 1994
 *
 */

#include <linux/types.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/linkage.h>
#include <linux/kernel.h>
#include <linux/pnp_bios.h>
#include <asm/page.h>
#include <asm/system.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <asm/desc.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/kmod.h>
#include <linux/completion.h>
#include <linux/spinlock.h>

/* PnP bios signature: "$PnP" */
#define PNP_SIGNATURE   (('$' << 0) + ('P' << 8) + ('n' << 16) + ('P' << 24))

void pnp_proc_init(void);
static void pnpbios_build_devlist(void);
int pnpbios_request_mboard(void);

/*
 * This is the standard structure used to identify the entry point
 * to the Plug and Play bios
 */
#pragma pack(1)
union pnpbios {
	struct {
		u32 signature;    /* "$PnP" */
		u8 version;	  /* in BCD */
		u8 length;	  /* length in bytes, currently 21h */
		u16 control;	  /* system capabilities */
		u8 checksum;	  /* all bytes must add up to 0 */

		u32 eventflag;    /* phys. address of the event flag */
		u16 rmoffset;     /* real mode entry point */
		u16 rmcseg;
		u16 pm16offset;   /* 16 bit protected mode entry */
		u32 pm16cseg;
		u32 deviceID;	  /* EISA encoded system ID or 0 */
		u16 rmdseg;	  /* real mode data segment */
		u32 pm16dseg;	  /* 16 bit pm data segment base */
	} fields;
	char chars[0x21];	  /* To calculate the checksum */
};
#pragma pack()

/*
 * Local Variables
 */
static struct {
	u16	offset;
	u16	segment;
} pnp_bios_callpoint;

static union pnpbios * pnp_bios_inst_struc = NULL;

/* The PnP entries in the GDT */
#define PNP_GDT		0x0060
#define PNP_CS32	(PNP_GDT+0x00)	/* segment for calling fn */
#define PNP_CS16	(PNP_GDT+0x08)	/* code segment for bios */
#define PNP_DS		(PNP_GDT+0x10)	/* data segment for bios */
#define PNP_TS1		(PNP_GDT+0x18)	/* transfer data segment */
#define PNP_TS2		(PNP_GDT+0x20)	/* another data segment */

/* 
 * These are some opcodes for a "static asmlinkage"
 * As this code is *not* executed inside the linux kernel segment, but in a
 * alias at offset 0, we need a far return that can not be compiled by
 * default (please, prove me wrong! this is *really* ugly!) 
 * This is the only way to get the bios to return into the kernel code,
 * because the bios code runs in 16 bit protected mode and therefore can only
 * return to the caller if the call is within the first 64kB, and the linux
 * kernel begins at offset 3GB...
 */
asmlinkage void pnp_bios_callfunc(void);

__asm__(
	".text			\n"
	__ALIGN_STR "\n"
	SYMBOL_NAME_STR(pnp_bios_callfunc) ":\n"
	"	pushl %edx	\n"
	"	pushl %ecx	\n"
	"	pushl %ebx	\n"
	"	pushl %eax	\n"
	"	lcallw " SYMBOL_NAME_STR(pnp_bios_callpoint) "\n"
	"	addl $16, %esp	\n"
	"	lret		\n"
	".previous		\n"
);

#define Q_SET_SEL(selname, address, size) \
set_base (gdt [(selname) >> 3], __va((u32)(address))); \
set_limit (gdt [(selname) >> 3], size)

#define Q2_SET_SEL(selname, address, size) \
set_base (gdt [(selname) >> 3], (u32)(address)); \
set_limit (gdt [(selname) >> 3], size)

/*
 * Callable Functions
 */
#define PNP_GET_NUM_SYS_DEV_NODES       0x00
#define PNP_GET_SYS_DEV_NODE            0x01
#define PNP_SET_SYS_DEV_NODE            0x02
#define PNP_GET_EVENT                   0x03
#define PNP_SEND_MESSAGE                0x04
#define PNP_GET_DOCKING_STATION_INFORMATION 0x05
#define PNP_SET_STATIC_ALLOCED_RES_INFO 0x09
#define PNP_GET_STATIC_ALLOCED_RES_INFO 0x0a
#define PNP_GET_APM_ID_TABLE            0x0b
#define PNP_GET_PNP_ISA_CONFIG_STRUC    0x40
#define PNP_GET_ESCD_INFO               0x41
#define PNP_READ_ESCD                   0x42
#define PNP_WRITE_ESCD                  0x43


/*
 *	At some point we want to use this stack frame pointer to unwind
 *	after PnP BIOS oopses. 
 */
 
u32 pnp_bios_fault_esp;
u32 pnp_bios_fault_eip;
u32 pnp_bios_is_utter_crap = 0;

static spinlock_t pnp_bios_lock;

static inline u16 call_pnp_bios(u16 func, u16 arg1, u16 arg2, u16 arg3,
				u16 arg4, u16 arg5, u16 arg6, u16 arg7)
{
	unsigned long flags;
	u16 status;

	/*
	 *	PnPBIOS is generally not terribly re-entrant.
	 *	Also don't rely on it to save everything correctly
 	 *
 	 *	On some boxes IRQ's during PnP bios calls seem fatal
	 */
	
	if(pnp_bios_is_utter_crap)
		return PNP_FUNCTION_NOT_SUPPORTED;
		
	spin_lock_irqsave(&pnp_bios_lock, flags);
	__cli();
	__asm__ __volatile__(
	        "pushl %%ebp\n\t"
		"pushl %%edi\n\t"
		"pushl %%esi\n\t"
		"pushl %%ds\n\t"
		"pushl %%es\n\t"
		"pushl %%fs\n\t"
		"pushl %%gs\n\t"
		"pushfl\n\t"
		"movl %%esp, pnp_bios_fault_esp\n\t"
		"movl $1f, pnp_bios_fault_eip\n\t"
		"lcall %5,%6\n\t"
		"1:popfl\n\t"
		"popl %%gs\n\t"
		"popl %%fs\n\t"
		"popl %%es\n\t"
		"popl %%ds\n\t"
	        "popl %%esi\n\t"
		"popl %%edi\n\t"
		"popl %%ebp\n\t"
		: "=a" (status)
		: "0" ((func) | (arg1 << 16)),
		  "b" ((arg2) | (arg3 << 16)),
		  "c" ((arg4) | (arg5 << 16)),
		  "d" ((arg6) | (arg7 << 16)),
		  "i" (PNP_CS32),
		  "i" (0)
		: "memory"
	);
	spin_unlock_irqrestore(&pnp_bios_lock, flags);
	
	/* If we got here and this is set the pnp bios faulted on us.. */
	if(pnp_bios_is_utter_crap)
	{
		printk(KERN_ERR "*** Warning: your PnP BIOS caused a fatal error. Attempting to continue\n");
		printk(KERN_ERR "*** You may need to reboot with the \"nobiospnp\" option to operate stably\n");
		printk(KERN_ERR "*** Check with your vendor for an updated BIOS\n");
	}
		
	return status;
}

/*
 * Call pnp bios with function 0x00, "get number of system device nodes"
 */

int pnp_bios_dev_node_info(struct pnp_dev_node_info *data)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, sizeof(struct pnp_dev_node_info));
	status = call_pnp_bios(PNP_GET_NUM_SYS_DEV_NODES, 0, PNP_TS1, 2, PNP_TS1, PNP_DS, 0, 0);
	data->no_nodes &= 0xff;
	return status;
}

/* 
 * Call pnp bios with function 0x01, "get system device node"
 * Input:  *nodenum=desired node, 
 *         static=1: config (dynamic) config, else boot (static) config,
 * Output: *nodenum=next node or 0xff if no more nodes
 */

int pnp_bios_get_dev_node(u8 *nodenum, char config, struct pnp_bios_node *data)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, nodenum, sizeof(char));
	Q2_SET_SEL(PNP_TS2, data, 64 * 1024);
	status = call_pnp_bios(PNP_GET_SYS_DEV_NODE, 0, PNP_TS1, 0, PNP_TS2, config ? 1 : 2, PNP_DS, 0);
	return status;
}

/*
 * Call pnp bios with function 0x02, "set system device node"
 * Input: nodenum=desired node, 
 *        static=1: config (dynamic) config, else boot (static) config,
 */

int pnp_bios_set_dev_node(u8 nodenum, char config, struct pnp_bios_node *data)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, /* *((u16 *) data)*/ 65536);
	status = call_pnp_bios(PNP_SET_SYS_DEV_NODE, nodenum, 0, PNP_TS1, config ? 1 : 2, PNP_DS, 0, 0);
	return status;
}

/*
 * Call pnp bios with function 0x03, "get event"
 */
#if needed

static int pnp_bios_get_event(u16 *event)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, event, sizeof(u16));
	status = call_pnp_bios(PNP_GET_EVENT, 0, PNP_TS1, PNP_DS, 0, 0 ,0 ,0);
	return status;
}
#endif

/* 
 * Call pnp bios with function 0x04, "send message"
 */
#if needed
static int pnp_bios_send_message(u16 message)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	status = call_pnp_bios(PNP_SEND_MESSAGE, message, PNP_DS, 0, 0, 0, 0, 0);
	return status;
}
#endif

#ifdef CONFIG_HOTPLUG
/*
 * Call pnp bios with function 0x05, "get docking station information"
 */

static int pnp_bios_dock_station_info(struct pnp_docking_station_info *data)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, sizeof(struct pnp_docking_station_info));
	status = call_pnp_bios(PNP_GET_DOCKING_STATION_INFORMATION, 0, PNP_TS1, PNP_DS, 0, 0, 0, 0);
	return status;
}
#endif

/*
 * Call pnp bios with function 0x09, "set statically allocated resource
 * information"
 */
#if needed
static int pnp_bios_set_stat_res(char *info)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, info, *((u16 *) info));
	status = call_pnp_bios(PNP_SET_STATIC_ALLOCED_RES_INFO, 0, PNP_TS1, PNP_DS, 0, 0, 0, 0);
	return status;
}
#endif

/*
 * Call pnp bios with function 0x0a, "get statically allocated resource
 * information"
 */
#if needed
static int pnp_bios_get_stat_res(char *info)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, info, 64 * 1024);
	status = call_pnp_bios(PNP_GET_STATIC_ALLOCED_RES_INFO, 0, PNP_TS1, PNP_DS, 0, 0, 0, 0);
	return status;
}
#endif

/*
 * Call pnp bios with function 0x0b, "get APM id table"
 */
#if needed
static int pnp_bios_apm_id_table(char *table, u16 *size)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, table, *size);
	Q2_SET_SEL(PNP_TS2, size, sizeof(u16));
	status = call_pnp_bios(PNP_GET_APM_ID_TABLE, 0, PNP_TS2, 0, PNP_TS1, PNP_DS, 0, 0);
	return status;
}
#endif

/*
 * Call pnp bios with function 0x40, "get isa pnp configuration structure"
 */
#if needed
static int pnp_bios_isapnp_config(struct pnp_isa_config_struc *data)
{
	u16 status;
	if (!pnp_bios_present ())
		return PNP_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, sizeof(struct pnp_isa_config_struc));
	status = call_pnp_bios(PNP_GET_PNP_ISA_CONFIG_STRUC, 0, PNP_TS1, PNP_DS, 0, 0, 0, 0);
	return status;
}
#endif

/*
 * Call pnp bios with function 0x41, "get ESCD info"
 */
#if needed
static int pnp_bios_escd_info(struct escd_info_struc *data)
{
	u16 status;
	if (!pnp_bios_present ())
		return ESCD_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, sizeof(struct escd_info_struc));
	status = call_pnp_bios(PNP_GET_ESCD_INFO, 0, PNP_TS1, 2, PNP_TS1, 4, PNP_TS1, PNP_DS);
	return status;
}
#endif

/*
 * Call pnp bios function 0x42, "read ESCD"
 * nvram_base is determined by calling escd_info
 */
#if needed
static int pnp_bios_read_escd(char *data, u32 nvram_base)
{
	u16 status;
	if (!pnp_bios_present ())
		return ESCD_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, 64 * 1024);
	set_base(gdt[PNP_TS2 >> 3], nvram_base);
	set_limit(gdt[PNP_TS2 >> 3], 64 * 1024);
	status = call_pnp_bios(PNP_READ_ESCD, 0, PNP_TS1, PNP_TS2, PNP_DS, 0, 0, 0);
	return status;
}
#endif

/*
 * Call pnp bios function 0x43, "write ESCD"
 */
#if needed
static int pnp_bios_write_escd(char *data, u32 nvram_base)
{
	u16 status;
	if (!pnp_bios_present ())
		return ESCD_FUNCTION_NOT_SUPPORTED;
	Q2_SET_SEL(PNP_TS1, data, 64 * 1024);
	set_base(gdt[PNP_TS2 >> 3], nvram_base);
	set_limit(gdt[PNP_TS2 >> 3], 64 * 1024);
	status = call_pnp_bios(PNP_WRITE_ESCD, 0, PNP_TS1, PNP_TS2, PNP_DS, 0, 0, 0);
	return status;
}
#endif

int pnp_bios_present(void)
{
	return (pnp_bios_inst_struc != NULL);
}

#ifdef CONFIG_HOTPLUG

/*
 *	Manage PnP docking
 */

static int unloading = 0;
static struct completion unload_sem;

/*
 *	Much of this belongs in a shared routine somewhere
 */
 
static int pnp_dock_event(int dock, struct pnp_docking_station_info *info)
{
	char *argv [3], **envp, *buf, *scratch;
	int i = 0, value;

	if (!hotplug_path [0])
		return -ENOENT;
	if (!current->fs->root) {
		return -EAGAIN;
	}
	if (!(envp = (char **) kmalloc (20 * sizeof (char *), GFP_KERNEL))) {
		return -ENOMEM;
	}
	if (!(buf = kmalloc (256, GFP_KERNEL))) {
		kfree (envp);
		return -ENOMEM;
	}

	/* only one standardized param to hotplug command: type */
	argv [0] = hotplug_path;
	argv [1] = "dock";
	argv [2] = 0;

	/* minimal command environment */
	envp [i++] = "HOME=/";
	envp [i++] = "PATH=/sbin:/bin:/usr/sbin:/usr/bin";

#ifdef	DEBUG
	/* hint that policy agent should enter no-stdout debug mode */
	envp [i++] = "DEBUG=kernel";
#endif
	/* extensible set of named bus-specific parameters,
	 * supporting multiple driver selection algorithms.
	 */
	scratch = buf;

	/* action:  add, remove */
	envp [i++] = scratch;
	scratch += sprintf (scratch, "ACTION=%s", dock?"add":"remove") + 1;

	/* Report the ident for the dock */
	envp [i++] = scratch;
	scratch += sprintf (scratch, "DOCK=%x/%x/%x",
		info->location_id, info->serial, info->capabilities);
	envp[i] = 0;
	
	value = call_usermodehelper (argv [0], argv, envp);
	kfree (buf);
	kfree (envp);
	return 0;
}

/*
 *	Poll the PnP docking at a regular interval
 */
 
static int pnp_dock_thread(void * unused)
{
	static struct pnp_docking_station_info now;
	int docked = -1, d;
	daemonize();
	reparent_to_init();
	strcpy(current->comm, "kpnpbios");
	while(!unloading && !signal_pending(current))
	{
		int err;
		
		/*
		 *	Poll every 2 seconds
		 */
		 
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ*2);
		if(signal_pending(current))
			break;

		err = pnp_bios_dock_station_info(&now);


		switch(err)
		{
			/*
			 *	No dock to manage
			 */
			case PNP_FUNCTION_NOT_SUPPORTED:
				complete_and_exit(&unload_sem, 0);
			case PNP_SYSTEM_NOT_DOCKED:
				d = 0;
				break;
			case PNP_SUCCESS:
				d = 1;
				break;
			default:
				printk(KERN_WARNING "dock: unexpected pnpbios error %d,\n", err);
				continue;
		}
		if(d != docked)
		{
			if(pnp_dock_event(d, &now)==0)
			{
				docked = d;
//				printk(KERN_INFO "Docking station %stached.\n", docked?"at":"de");
			}
		}
	}	
	complete_and_exit(&unload_sem, 0);
}

#endif

/* 
 * Searches the defined area (0xf0000-0xffff0) for a valid PnP BIOS
 * structure and, if found one, sets up the selectors and entry points
 */

static int pnp_bios_disabled;

static int disable_pnp_bios(char *str)
{
	pnp_bios_disabled=1;
	return 0;
}

__setup("nobiospnp", disable_pnp_bios);

void pnp_bios_init(void)
{
	union pnpbios *check;
	u8 sum;
	int i, length;

	spin_lock_init(&pnp_bios_lock);

	if(pnp_bios_disabled)
	{
		printk(KERN_INFO "PNP BIOS services disabled.\n");
		return;
	}
	for (check = (union pnpbios *) __va(0xf0000);
	     check < (union pnpbios *) __va(0xffff0);
	     ((void *) (check)) += 16) {
		if (check->fields.signature != PNP_SIGNATURE)
			continue;
		length = check->fields.length;
		if (!length)
			continue;
		for (sum = 0, i = 0; i < length; i++)
			sum += check->chars[i];
		if (sum)
			continue;
		if (check->fields.version < 0x10) {
			printk(KERN_WARNING "PnP: unsupported version %d.%d",
			       check->fields.version >> 4,
			       check->fields.version & 15);
			continue;
		}
		printk(KERN_INFO "PnP: PNP BIOS installation structure at 0x%p\n",
		       check);
		printk(KERN_INFO "PnP: PNP BIOS version %d.%d, entry at %x:%x, dseg at %x\n",
                       check->fields.version >> 4, check->fields.version & 15,
		       check->fields.pm16cseg, check->fields.pm16offset,
		       check->fields.pm16dseg);
		Q2_SET_SEL(PNP_CS32, &pnp_bios_callfunc, 64 * 1024);
		Q_SET_SEL(PNP_CS16, check->fields.pm16cseg, 64 * 1024);
		Q_SET_SEL(PNP_DS, check->fields.pm16dseg, 64 * 1024);
		pnp_bios_callpoint.offset = check->fields.pm16offset;
		pnp_bios_callpoint.segment = PNP_CS16;
		pnp_bios_inst_struc = check;
		break;
	}
	pnpbios_build_devlist();
	pnpbios_request_mboard();
#ifdef CONFIG_PROC_FS
	pnp_proc_init();
#endif
#ifdef CONFIG_HOTPLUG	
	init_completion(&unload_sem);
	if(kernel_thread(pnp_dock_thread, NULL, CLONE_FS | CLONE_FILES | CLONE_SIGNAL)>0)
		unloading = 0;
#endif		
}

#ifdef MODULE
/* We have to run it early and specifically in non modular.. */
module_init(pnp_bios_init);

#ifdef CONFIG_HOTPLUG
static void pnp_bios_exit(void)
{
	unloading = 1;
	wait_for_completion(&unload_sem);
}

module_exit(pnp_bios_exit);
#endif
#endif

EXPORT_SYMBOL(pnp_bios_get_dev_node);
EXPORT_SYMBOL(pnp_bios_present);
EXPORT_SYMBOL(pnp_bios_dev_node_info);

static void inline pnpbios_add_irqresource(struct pci_dev *dev, int irq)
{
	int i = 0;
	while (dev->irq_resource[i].start && i < DEVICE_COUNT_IRQ) i++;
	if (i < DEVICE_COUNT_IRQ)
		dev->irq_resource[i].start = irq;
}

static void inline pnpbios_add_dmaresource(struct pci_dev *dev, int dma)
{
	int i = 0;
	while (dev->dma_resource[i].start && i < DEVICE_COUNT_DMA) i++;
	if (i < DEVICE_COUNT_DMA)
		dev->dma_resource[i].start = dma;
}

static void __init pnpbios_add_ioresource(struct pci_dev *dev, int io, 
					  int len, int flags)
{
	int i = 0;
	while (dev->resource[i].start && i < DEVICE_COUNT_RESOURCE) i++;
	if (i < DEVICE_COUNT_RESOURCE) {
		dev->resource[i].start = io;
		dev->resource[i].end = io + len;
		dev->resource[i].flags = flags;
	}
}

/* parse PNPBIOS "Allocated Resources Block" and fill IO,IRQ,DMA into pci_dev */
static void __init pnpbios_rawdata_2_pci_dev(struct pnp_bios_node *node, struct pci_dev *pci_dev)
{
	unsigned char *p = node->data, *lastp=NULL;
        int mask,i,io,irq=0,len,dma=-1;

	memset(pci_dev, 0, sizeof(struct pci_dev));
        while ( (char *)p < ((char *)node->data + node->size )) {
        	if(p==lastp) break;

                if( p[0] & 0x80 ) {// large item
			switch (p[0] & 0x7f) {
			case 0x01: // memory
				io = *(short *) &p[4];
				len = *(short *) &p[10];
				pnpbios_add_ioresource(pci_dev, io, len, IORESOURCE_MEM);
				break;
			case 0x02: // device name
				len = *(short *) &p[1];
				memcpy(pci_dev->name, p + 3, len >= 80 ? 79 : len);
				break;
			case 0x05: // 32-bit memory
				io = *(int *) &p[4];
				len = *(int *) &p[16];
				pnpbios_add_ioresource(pci_dev, io, len, IORESOURCE_MEM);
				break;
			case 0x06: // fixed location 32-bit memory
				io = *(int *) &p[4];
				len = *(int *) &p[8];
				pnpbios_add_ioresource(pci_dev, io, len, IORESOURCE_MEM);
				break;
			}
                        lastp = p+3;
                        p = p + p[1] + p[2]*256 + 3;
                        continue;
                }
                if ((p[0]>>3) == 0x0f) // end tag
                        break;
                switch (p[0]>>3) {
                case 0x04: // irq
                        mask= p[1] + p[2]*256;
                        for (i=0;i<16;i++, mask=mask>>1)
                                if(mask &0x01) irq=i;
			pnpbios_add_irqresource(pci_dev, irq);
                        break;
                case 0x05: // dma
                        mask = p[1];
                        for (i=0;i<8;i++, mask = mask>>1)
                                if(mask&0x01) dma=i;
			pnpbios_add_dmaresource(pci_dev, dma);
                        break;
                case 0x08: // io
			io= p[2] + p[3] *256;
			len = p[7];
			pnpbios_add_ioresource(pci_dev, io, len, IORESOURCE_IO);
                        break;
		case 0x09: // fixed location io
			io = p[1] + p[2] * 256;
			len = p[3];
			pnpbios_add_ioresource(pci_dev, io, len, IORESOURCE_IO);
			break;
                }
                lastp=p+1;
                p = p + (p[0] & 0x07) + 1;

        }
}

#define HEX(id,a) hex[((id)>>a) & 15]
#define CHAR(id,a) (0x40 + (((id)>>a) & 31))

static char * __init pnpid32_to_pnpid(u32 id)
{
	const char *hex = "0123456789abcdef";
        static char str[8];
	id = be32_to_cpu(id);
	str[0] = CHAR(id, 26);
	str[1] = CHAR(id, 21);
	str[2] = CHAR(id,16);
	str[3] = HEX(id, 12);
	str[4] = HEX(id, 8);
	str[5] = HEX(id, 4);
	str[6] = HEX(id, 0);
	str[7] = '\0';
	return str;
}                                              

#undef CHAR
#undef HEX  

/*
 *	PnPBIOS public device management layer
 */

static LIST_HEAD(pnpbios_devices);

static int __init pnpbios_insert_device(struct pci_dev *dev)
{
	/* FIXME: Need to check for re-add of existing node */
	list_add_tail(&dev->global_list, &pnpbios_devices);
	return 0;
}

/*
 *	Build the list of pci_dev objects from the PnP table
 */
 
static void __init pnpbios_build_devlist(void)
{
	int i, devs = 0;
	struct pnp_bios_node *node;
        struct pnp_dev_node_info node_info;
	struct pci_dev *dev;
	int num;
	char *pnpid;

	
        if (!pnp_bios_present ())
                return;

        if (pnp_bios_dev_node_info(&node_info) != 0)
                return;

        node = kmalloc(node_info.max_node_size, GFP_KERNEL);
        if (!node)
                return;

	for(i=0;i<0xff;i++) {
		dev =  kmalloc(sizeof (struct pci_dev), GFP_KERNEL);
		if (!dev)
			break;
			
                if (pnp_bios_get_dev_node((u8 *)&num, (char )0 , node))
			continue;

		devs++;
		pnpbios_rawdata_2_pci_dev(node,dev);
		dev->devfn=num;
		pnpid = pnpid32_to_pnpid(node->eisa_id);
		memcpy(dev->name,"PNPBIOS",8);
		memcpy(dev->slot_name,pnpid,8);
		if(pnpbios_insert_device(dev)<0)
			kfree(dev);
	}
	kfree(node);

	if (devs)
		printk(KERN_INFO "PnP: %i device%s detected total\n", devs, devs > 1 ? "s" : "");
	else
		printk(KERN_INFO "PnP: No devices found\n");
}


/*
 *	The public interface to PnP BIOS enumeration
 */
 
struct pci_dev *pnpbios_find_device(char *pnpid, struct pci_dev *prev)
{
	struct pci_dev *dev;
	int num;

	if(prev==NULL)
		num=0; /* Start from beginning */
	else
		num=prev->devfn + 1; /* Encode node number here */
	

	pnpbios_for_each_dev(dev)
	{
		if(dev->devfn >= num)
		{
			if(memcmp(dev->slot_name, pnpid, 7)==0)
				return dev;
		}
	}
	return NULL;
}

EXPORT_SYMBOL(pnpbios_find_device);

/*
 *  Registration of PnPBIOS drivers and handling of hot-pluggable devices.
 */

static LIST_HEAD(pnpbios_drivers);

/**
 * pnpbios_match_device - Tell if a PnPBIOS device structure has a matching PnPBIOS device id structure
 * @ids: array of PnPBIOS device id structures to search in
 * @dev: the PnPBIOS device structure to match against
 * 
 * Used by a driver to check whether a PnPBIOS device present in the
 * system is in its list of supported devices.Returns the matching
 * pnpbios_device_id structure or %NULL if there is no match.
 */

const struct pnpbios_device_id *
pnpbios_match_device(const struct pnpbios_device_id *ids, const struct pci_dev *dev)
{
	while (*ids->id)
	{
		if(memcmp(ids->id, dev->slot_name, 7)==0)
			return ids;
		ids++;
	}
	return NULL;
}

static int
pnpbios_announce_device(struct pnpbios_driver *drv, struct pci_dev *dev)
{
	const struct pnpbios_device_id *id;
	int ret = 0;

	if (drv->id_table) {
		id = pnpbios_match_device(drv->id_table, dev);
		if (!id) {
			ret = 0;
			goto out;
		}
	} else
		id = NULL;

	dev_probe_lock();
	if (drv->probe(dev, id) >= 0) {
		// Hack for 2.4 - in 2.5 this needs to be generic stuff anyway
		dev->driver = (void *)drv;
		ret = 1;
	}
	dev_probe_unlock();
out:
	return ret;
}

EXPORT_SYMBOL(pnpbios_announce_device);

/**
 * pnpbios_register_driver - register a new pci driver
 * @drv: the driver structure to register
 * 
 * Adds the driver structure to the list of registered drivers
 * Returns the number of pci devices which were claimed by the driver
 * during registration.  The driver remains registered even if the
 * return value is zero.
 */
int
pnpbios_register_driver(struct pnpbios_driver *drv)
{
	struct pci_dev *dev;
	int count = 0;

	list_add_tail(&drv->node, &pnpbios_drivers);
	pnpbios_for_each_dev(dev) {
		if (!pnpbios_dev_driver(dev))
			count += pnpbios_announce_device(drv, dev);
	}
	return count;
}

EXPORT_SYMBOL(pnpbios_register_driver);

/**
 * pnpbios_unregister_driver - unregister a pci driver
 * @drv: the driver structure to unregister
 * 
 * Deletes the driver structure from the list of registered PnPBIOS drivers,
 * gives it a chance to clean up by calling its remove() function for
 * each device it was responsible for, and marks those devices as
 * driverless.
 */

void
pnpbios_unregister_driver(struct pnpbios_driver *drv)
{
	struct pci_dev *dev;

	list_del(&drv->node);
	pnpbios_for_each_dev(dev) {
		if (dev->driver == (void *)drv) {
			if (drv->remove)
				drv->remove(dev);
			dev->driver = NULL;
		}
	}
}

EXPORT_SYMBOL(pnpbios_unregister_driver);


