#ifndef __LINUX_UHCI_H
#define __LINUX_UHCI_H

#include <linux/list.h>
#include <linux/usb.h>

#include "uhci-common.h"

struct uhci_frame_list {
	__u32 frame[UHCI_NUMFRAMES];

	void *frame_cpu[UHCI_NUMFRAMES];

	dma_addr_t dma_handle;
};

struct urb_priv;

struct uhci_qh {
	/* Hardware fields */
	__u32 link;			/* Next queue */
	__u32 element;			/* Queue element pointer */

	/* Software fields */
	dma_addr_t dma_handle;

	struct usb_device *dev;
	struct urb_priv *urbp;

	struct list_head list;		/* P: uhci->frame_list_lock */
	struct list_head remove_list;	/* P: uhci->remove_list_lock */
} __attribute__((aligned(16)));



#define uhci_expected_length(info) (((info >> 21) + 1) & TD_CTRL_ACTLEN_MASK) /* 1-based */

/*
 * The documentation says "4 words for hardware, 4 words for software".
 *
 * That's silly, the hardware doesn't care. The hardware only cares that
 * the hardware words are 16-byte aligned, and we can have any amount of
 * sw space after the TD entry as far as I can tell.
 *
 * But let's just go with the documentation, at least for 32-bit machines.
 * On 64-bit machines we probably want to take advantage of the fact that
 * hw doesn't really care about the size of the sw-only area.
 *
 * Alas, not anymore, we have more than 4 words for software, woops.
 * Everything still works tho, surprise! -jerdfelt
 */
struct uhci_td {
	/* Hardware fields */
	__u32 link;
	__u32 status;
	__u32 info;
	__u32 buffer;

	/* Software fields */
	dma_addr_t dma_handle;

	struct usb_device *dev;
	struct urb *urb;

	struct list_head list;		/* P: urb->lock */

	int frame;
	struct list_head fl_list;	/* P: frame_list_lock */
} __attribute__((aligned(16)));

/*
 * There are various standard queues. We set up several different
 * queues for each of the three basic queue types: interrupt,
 * control, and bulk.
 *
 *  - There are various different interrupt latencies: ranging from
 *    every other USB frame (2 ms apart) to every 256 USB frames (ie
 *    256 ms apart). Make your choice according to how obnoxious you
 *    want to be on the wire, vs how critical latency is for you.
 *  - The control list is done every frame.
 *  - There are 4 bulk lists, so that up to four devices can have a
 *    bulk list of their own and when run concurrently all four lists
 *    will be be serviced.
 *
 * This is a bit misleading, there are various interrupt latencies, but they
 * vary a bit, interrupt2 isn't exactly 2ms, it can vary up to 4ms since the
 * other queues can "override" it. interrupt4 can vary up to 8ms, etc. Minor
 * problem
 *
 * In the case of the root hub, these QH's are just head's of qh's. Don't
 * be scared, it kinda makes sense. Look at this wonderful picture care of
 * Linus:
 *
 *  generic-  ->  dev1-  ->  generic-  ->  dev1-  ->  control-  ->  bulk- -> ...
 *   iso-QH      iso-QH       irq-QH      irq-QH        QH           QH
 *      |           |            |           |           |            |
 *     End     dev1-iso-TD1     End     dev1-irq-TD1    ...          ... 
 *                  |
 *             dev1-iso-TD2
 *                  |
 *                ....
 *
 * This may vary a bit (the UHCI docs don't explicitly say you can put iso
 * transfers in QH's and all of their pictures don't have that either) but
 * other than that, that is what we're doing now
 *
 * And now we don't put Iso transfers in QH's, so we don't waste one on it
 * --jerdfelt
 *
 * To keep with Linus' nomenclature, this is called the QH skeleton. These
 * labels (below) are only signficant to the root hub's QH's
 */

#define UHCI_NUM_SKELTD		10
#define skel_int1_td		skeltd[0]
#define skel_int2_td		skeltd[1]
#define skel_int4_td		skeltd[2]
#define skel_int8_td		skeltd[3]
#define skel_int16_td		skeltd[4]
#define skel_int32_td		skeltd[5]
#define skel_int64_td		skeltd[6]
#define skel_int128_td		skeltd[7]
#define skel_int256_td		skeltd[8]
#define skel_term_td		skeltd[9]	/* To work around PIIX UHCI bug */

#define UHCI_NUM_SKELQH		4
#define skel_ls_control_qh	skelqh[0]
#define skel_hs_control_qh	skelqh[1]
#define skel_bulk_qh		skelqh[2]
#define skel_term_qh		skelqh[3]

/*
 * Search tree for determining where <interval> fits in the
 * skelqh[] skeleton.
 *
 * An interrupt request should be placed into the slowest skelqh[]
 * which meets the interval/period/frequency requirement.
 * An interrupt request is allowed to be faster than <interval> but not slower.
 *
 * For a given <interval>, this function returns the appropriate/matching
 * skelqh[] index value.
 *
 * NOTE: For UHCI, we don't really need int256_qh since the maximum interval
 * is 255 ms.  However, we do need an int1_qh since 1 is a valid interval
 * and we should meet that frequency when requested to do so.
 * This will require some change(s) to the UHCI skeleton.
 */
static inline int __interval_to_skel(int interval)
{
	if (interval < 16) {
		if (interval < 4) {
			if (interval < 2)
				return 0;	/* int1 for 0-1 ms */
			return 1;		/* int2 for 2-3 ms */
		}
		if (interval < 8)
			return 2;		/* int4 for 4-7 ms */
		return 3;			/* int8 for 8-15 ms */
	}
	if (interval < 64) {
		if (interval < 32)
			return 4;		/* int16 for 16-31 ms */
		return 5;			/* int32 for 32-63 ms */
	}
	if (interval < 128)
		return 6;			/* int64 for 64-127 ms */
	return 7;				/* int128 for 128-255 ms (Max.) */
}

struct virt_root_hub {
	struct usb_device *dev;
	int devnum;		/* Address of Root Hub endpoint */
	struct urb *urb;
	void *int_addr;
	int send;
	int interval;
	int numports;
	int c_p_r[8];
	struct timer_list rh_int_timer;
};

/*
 * This describes the full uhci information.
 *
 * Note how the "proper" USB information is just
 * a subset of what the full implementation needs.
 */
struct uhci {
	struct pci_dev *dev;

	/* procfs */
	int num;
	struct proc_dir_entry *proc_entry;

	/* Grabbed from PCI */
	int irq;
	unsigned int io_addr;
	unsigned int io_size;

	struct list_head uhci_list;

	struct pci_pool *qh_pool;
	struct pci_pool *td_pool;

	struct usb_bus *bus;

	struct uhci_td *skeltd[UHCI_NUM_SKELTD];	/* Skeleton TD's */
	struct uhci_qh *skelqh[UHCI_NUM_SKELQH];	/* Skeleton QH's */

	spinlock_t frame_list_lock;
	struct uhci_frame_list *fl;		/* Frame list */
	int fsbr;				/* Full speed bandwidth reclamation */
	int is_suspended;

	spinlock_t qh_remove_list_lock;
	struct list_head qh_remove_list;

	spinlock_t urb_remove_list_lock;
	struct list_head urb_remove_list;

	spinlock_t urb_list_lock;
	struct list_head urb_list;

	spinlock_t complete_list_lock;
	struct list_head complete_list;

	struct virt_root_hub rh;	/* private data of the virtual root hub */
};

struct urb_priv {
	struct urb *urb;
	struct usb_device *dev;

	dma_addr_t setup_packet_dma_handle;
	dma_addr_t transfer_buffer_dma_handle;

	struct uhci_qh *qh;		/* QH for this URB */
	struct list_head td_list;

	int fsbr : 1;			/* URB turned on FSBR */
	int fsbr_timeout : 1;		/* URB timed out on FSBR */
	int queued : 1;			/* QH was queued (not linked in) */
	int short_control_packet : 1;	/* If we get a short packet during */
					/*  a control transfer, retrigger */
					/*  the status phase */

	int status;			/* Final status */

	unsigned long inserttime;	/* In jiffies */

	struct list_head queue_list;
	struct list_head complete_list;
};

#endif

