#include <bios/config.h>
#include <bios/bootdev.h>
#include <bios/pci.h>
#include <bios/malloc.h>
#include <bios/time.h>
#include <bios/stdio.h>
#include <bios/stdioint.h>
#include <bios/string.h>
#include <bios/system.h>

typedef enum {
	ram_failure,
	ram_ok
} ram_test_t;

typedef enum {
	boot_failure,
	boot_single,
	boot_multiple
} boot_test_t;

static char extra_args[80];
char *cmdline = "";

extern struct bootdev boot_net;
extern struct bootdev boot_ide;
extern struct bootdev boot_scsi;
extern struct bootdev boot_serial;

static struct bootdev *first;
static struct bootdev *bootdevs[] = {
#ifdef CONFIG_BOOT_IDE
	&boot_ide,
#endif
#ifdef CONFIG_BOOT_SCSI
	&boot_scsi,
#endif
#ifdef CONFIG_BOOT_NET
	&boot_net,
#endif
	&boot_serial,
	NULL
};

extern unsigned long ramtest(unsigned long, unsigned long);

int ram_size;

static ram_test_t ram_test(void)
{
	unsigned int ptr;
	ram_test_t ret = ram_ok;

	printf("       KB SDRAM OK");

	for (ptr = 0; ptr < ram_size; ptr += 64) {
		if (ramtest(ptr, 0x55aacc33) != 0x55aacc33) {
			printf("\nMemory error detected at address 0x%08X\n",
				ptr);
			ret = ram_failure;
			break;
		}
		if ((ptr & 8191) == 0)
			printf("\r%6d", ptr >> 10);
	}

	if (ret == ram_ok)
		printf("\r%6d\n", ram_size >> 10);

	return ret;
}

static void boot_init(void)
{
	struct bootdev **prev;
	int i, found = 0;

	first = NULL;
	prev = &first;

	printf("Initialising devices: ");

	for (i = 0; bootdevs[i]; i++) {
		if (bootdevs[i]->init() == 0) {
			printf(found ? ", %s" : "%s", bootdevs[i]->name);
			*prev = bootdevs[i];
			prev = &bootdevs[i]->next;
			found += 1;
		}
	}

	if (!found)
		printf("no devices found\n");
	else
		printf("\n");
}

static void panic(const char *s)
{
	if (s)
		printf("%s", s);
	printf(" -- System Halted");
	while(1);
}

int img_nr;
static int root_dev;

static void boot(void)
{
	printf("Now booting image...\n");

	sti();

	boot_kernel();
}

static int do_boot(struct bootdev *dev)
{
	int ret;

	ret = dev->start();
	if (ret)
		return ret;

	ret = dev->load();
	dev->stop();

	if (ret == 0)
		boot();

	return ret;
}

static void auto_boot(void)
{
	struct bootdev *dev;

	for (dev = first; dev; dev = dev->next) {
		printf("Trying %s...\n", dev->name);
		do_boot(dev);
	}
}

static void manual_boot(void)
{
	struct bootdev *bd;
	int c;

	bd = first;

	while (1) {
		printf("\014EBSA285 manual boot\n");

		printf("1 - Boot method : %-10s\n", bd->name);
		printf("2 - Image number: %d\n",    img_nr);
		printf("3 - Root device : %04X\n",  root_dev);
		printf("4 - Arguments   : %s\n", extra_args);
		printf("\n");
		printf("b - boot    p - pci config\n");

		printf("\ncfg> ");

		c = getc();

		printf("%c\n", c);

		switch (c) {
		case '1':
			bd = bd->next;
			if (bd == NULL)
				bd = first;
			break;

		case '2':
			img_nr = (img_nr + 1) & 3;

		case '3':
			{
				char str[6];
				printf("\nRoot device: ");
				gets(str, sizeof(str));
				if (str[0])
					root_dev = atol(str);
			}
			break;

		case '4':
			printf("\nExtra arguments: ");
			gets(extra_args, sizeof(extra_args));
			break;

		case 'b':
			do_boot(bd);
			break;

		case 'p':
			pci_print_config();
			getc();
			break;
		}
	}

	boot();
}

/*
 * This is the main entry point for the BIOS.
 */
void start_main(void)
{
	int i;

	debug_init();	/* Initialise serial debug	*/
	malloc_init();	/* Initialise malloc pool	*/
	time_init();	/* Initialise timers		*/
	cli();

	/*
	 * Wait a while until the PCI
	 * devices have properly reset.
	 */
	wait_cs(40);

	pci_init();	/* Initialise PCI sub-system	*/
	vga_init();	/* Initialise VGA adapter	*/

	printf("EBSA285 Linux BIOS v"VERSION"\n\n");

	wait_cs(40);

	isa_init();	/* Initialise ISA		*/

	/* Check integrity of RAM */
	if (ram_test() == ram_failure)
		panic("ram failure");

	printf("\n");

	wait_cs(400);
	boot_init();	/* Initialise & detect devices	*/

	printf("\nHit 's' twice to abort autoboot...");

	i = 0;
	while (1) {
		int c;

		c = getc_timed(400);

		if (c == -1)
			break;

		printf("%c", c);

		if (c == 's')
			i += 1;
		else
			i = 0;

		if (i == 2)
			break;
	}

	if (i == 2)
		manual_boot();
	else {
		printf(" starting autoboot\n");
		auto_boot();
	}

	printf("I'm now waving my legs in the air wondering what to do next!\n");
	while (1);
}
